/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.replication;

import com.google.errorprone.annotations.RestrictedApi;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.client.replication.ReplicationPeerConfigUtil;
import org.apache.hadoop.hbase.exceptions.DeserializationException;
import org.apache.hadoop.hbase.replication.ReplicationException;
import org.apache.hadoop.hbase.replication.ReplicationPeerConfig;
import org.apache.hadoop.hbase.replication.ReplicationPeerStorage;
import org.apache.hadoop.hbase.util.CommonFSUtils;
import org.apache.hadoop.hbase.util.RotateFile;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class FSReplicationPeerStorage
implements ReplicationPeerStorage {
    private static final Logger LOG = LoggerFactory.getLogger(FSReplicationPeerStorage.class);
    public static final String PEERS_DIR = "hbase.replication.peers.directory";
    public static final String PEERS_DIR_DEFAULT = "peers";
    static final String PEER_CONFIG_FILE = "peer_config";
    static final String DISABLED_FILE = "disabled";
    private final FileSystem fs;
    private final Path dir;
    private static final long MAX_FILE_SIZE = 0x1000000L;

    public FSReplicationPeerStorage(FileSystem fs, Configuration conf) throws IOException {
        this.fs = fs;
        this.dir = new Path(CommonFSUtils.getRootDir((Configuration)conf), conf.get(PEERS_DIR, PEERS_DIR_DEFAULT));
    }

    @RestrictedApi(explanation="Should only be called in tests", link="", allowedOnPath=".*/FSReplicationPeerStorage.java|.*/src/test/.*")
    Path getPeerDir(String peerId) {
        return new Path(this.dir, peerId);
    }

    @Override
    public void addPeer(String peerId, ReplicationPeerConfig peerConfig, boolean enabled) throws ReplicationException {
        Path peerDir = this.getPeerDir(peerId);
        try {
            if (this.fs.exists(peerDir) && FSReplicationPeerStorage.read(this.fs, peerDir, PEER_CONFIG_FILE) != null) {
                throw new ReplicationException("Could not add peer with id=" + peerId + ", peerConfig=>" + peerConfig + ", state=" + (enabled ? "ENABLED" : "DISABLED") + ", peer already exists");
            }
            if (!enabled) {
                this.fs.createNewFile(new Path(peerDir, DISABLED_FILE));
            }
            FSReplicationPeerStorage.write(this.fs, peerDir, PEER_CONFIG_FILE, ReplicationPeerConfigUtil.toByteArray((ReplicationPeerConfig)peerConfig));
        }
        catch (IOException e) {
            throw new ReplicationException("Could not add peer with id=" + peerId + ", peerConfig=>" + peerConfig + ", state=" + (enabled ? "ENABLED" : "DISABLED"), (Throwable)e);
        }
    }

    @Override
    public void removePeer(String peerId) throws ReplicationException {
        Path peerDir = this.getPeerDir(peerId);
        try {
            FSReplicationPeerStorage.delete(this.fs, peerDir, PEER_CONFIG_FILE);
            if (!this.fs.delete(peerDir, true)) {
                throw new IOException("Can not delete " + peerDir);
            }
        }
        catch (IOException e) {
            throw new ReplicationException("Could not remove peer with id=" + peerId, (Throwable)e);
        }
    }

    @Override
    public void setPeerState(String peerId, boolean enabled) throws ReplicationException {
        Path disabledFile = new Path(this.getPeerDir(peerId), DISABLED_FILE);
        try {
            if (enabled) {
                if (this.fs.exists(disabledFile) && !this.fs.delete(disabledFile, false)) {
                    throw new IOException("Can not delete " + disabledFile);
                }
            } else if (!this.fs.exists(disabledFile) && !this.fs.createNewFile(disabledFile)) {
                throw new IOException("Can not touch " + disabledFile);
            }
        }
        catch (IOException e) {
            throw new ReplicationException("Unable to change state of the peer with id=" + peerId + " to " + enabled, (Throwable)e);
        }
    }

    @Override
    public void updatePeerConfig(String peerId, ReplicationPeerConfig peerConfig) throws ReplicationException {
        Path peerDir = this.getPeerDir(peerId);
        try {
            FSReplicationPeerStorage.write(this.fs, peerDir, PEER_CONFIG_FILE, ReplicationPeerConfigUtil.toByteArray((ReplicationPeerConfig)peerConfig));
        }
        catch (IOException e) {
            throw new ReplicationException("There was a problem trying to save changes to the replication peer " + peerId, (Throwable)e);
        }
    }

    @Override
    public List<String> listPeerIds() throws ReplicationException {
        try {
            FileStatus[] statuses = this.fs.listStatus(this.dir);
            if (statuses == null || statuses.length == 0) {
                return Collections.emptyList();
            }
            ArrayList<String> peerIds = new ArrayList<String>();
            for (FileStatus status : statuses) {
                String peerId = status.getPath().getName();
                Path peerDir = this.getPeerDir(peerId);
                byte[] peerConfigData = FSReplicationPeerStorage.read(this.fs, peerDir, PEER_CONFIG_FILE);
                if (peerConfigData == null) continue;
                peerIds.add(peerId);
            }
            return Collections.unmodifiableList(peerIds);
        }
        catch (FileNotFoundException e) {
            LOG.debug("Peer directory does not exist yet", (Throwable)e);
            return Collections.emptyList();
        }
        catch (IOException e) {
            throw new ReplicationException("Cannot get the list of peers", (Throwable)e);
        }
    }

    @Override
    public boolean isPeerEnabled(String peerId) throws ReplicationException {
        Path disabledFile = new Path(this.getPeerDir(peerId), DISABLED_FILE);
        try {
            return !this.fs.exists(disabledFile);
        }
        catch (IOException e) {
            throw new ReplicationException("Unable to get status of the peer with id=" + peerId, (Throwable)e);
        }
    }

    @Override
    public ReplicationPeerConfig getPeerConfig(String peerId) throws ReplicationException {
        byte[] data;
        Path peerDir = this.getPeerDir(peerId);
        try {
            data = FSReplicationPeerStorage.read(this.fs, peerDir, PEER_CONFIG_FILE);
        }
        catch (IOException e) {
            throw new ReplicationException("Error getting configuration for peer with id=" + peerId, (Throwable)e);
        }
        if (data == null || data.length == 0) {
            throw new ReplicationException("Replication peer config data shouldn't be empty, peerId=" + peerId);
        }
        try {
            return ReplicationPeerConfigUtil.parsePeerFrom((byte[])data);
        }
        catch (DeserializationException e) {
            throw new ReplicationException("Failed to parse replication peer config for peer with id=" + peerId, (Throwable)e);
        }
    }

    private static byte[] read(FileSystem fs, Path dir, String name) throws IOException {
        RotateFile file = new RotateFile(fs, dir, name, 0x1000000L);
        return file.read();
    }

    private static void write(FileSystem fs, Path dir, String name, byte[] data) throws IOException {
        RotateFile file = new RotateFile(fs, dir, name, 0x1000000L);
        file.read();
        file.write(data);
    }

    private static void delete(FileSystem fs, Path dir, String name) throws IOException {
        RotateFile file = new RotateFile(fs, dir, name, 0x1000000L);
        file.read();
        file.delete();
    }
}

