/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.backup;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Collections2;
import com.google.common.collect.Lists;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.regionserver.HRegion;
import org.apache.hadoop.hbase.regionserver.StoreFile;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.hadoop.hbase.util.FSUtils;
import org.apache.hadoop.hbase.util.HFileArchiveUtil;
import org.apache.hadoop.io.MultipleIOException;

public class HFileArchiver {
    private static final Log LOG = LogFactory.getLog(HFileArchiver.class);
    private static final String SEPARATOR = ".";
    private static final int DEFAULT_RETRIES_NUMBER = 3;

    private HFileArchiver() {
    }

    public static void archiveRegion(Configuration conf, FileSystem fs, HRegionInfo info) throws IOException {
        Path rootDir = FSUtils.getRootDir(conf);
        HFileArchiver.archiveRegion(fs, rootDir, FSUtils.getTableDir(rootDir, info.getTable()), HRegion.getRegionDir(rootDir, info));
    }

    public static boolean archiveRegion(FileSystem fs, Path rootdir, Path tableDir, Path regionDir) throws IOException {
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("ARCHIVING " + regionDir.toString()));
        }
        if (tableDir == null || regionDir == null) {
            LOG.error((Object)("No archive directory could be found because tabledir (" + tableDir + ") or regiondir (" + regionDir + "was null. Deleting files instead."));
            HFileArchiver.deleteRegionWithoutArchiving(fs, regionDir);
            return false;
        }
        Preconditions.checkArgument((boolean)regionDir.toString().startsWith(tableDir.toString()));
        Path regionArchiveDir = HFileArchiveUtil.getRegionArchiveDir(rootdir, FSUtils.getTableName(tableDir), regionDir.getName());
        FileStatusConverter getAsFile = new FileStatusConverter(fs);
        ArrayList<File> toArchive = new ArrayList<File>();
        final FSUtils.DirFilter dirFilter = new FSUtils.DirFilter(fs);
        PathFilter nonHidden = new PathFilter(){

            public boolean accept(Path file) {
                return dirFilter.accept(file) && !file.getName().toString().startsWith(HFileArchiver.SEPARATOR);
            }
        };
        FileStatus[] storeDirs = FSUtils.listStatus(fs, regionDir, nonHidden);
        if (storeDirs == null) {
            LOG.debug((Object)("Region directory (" + regionDir + ") was empty, just deleting and returning!"));
            return HFileArchiver.deleteRegionWithoutArchiving(fs, regionDir);
        }
        toArchive.addAll(Lists.transform(Arrays.asList(storeDirs), (Function)getAsFile));
        LOG.debug((Object)("Archiving " + toArchive));
        boolean success = false;
        try {
            success = HFileArchiver.resolveAndArchive(fs, regionArchiveDir, toArchive);
        }
        catch (IOException e) {
            LOG.error((Object)("Failed to archive " + toArchive), (Throwable)e);
            success = false;
        }
        if (success) {
            return HFileArchiver.deleteRegionWithoutArchiving(fs, regionDir);
        }
        throw new IOException("Received error when attempting to archive files (" + toArchive + "), cannot delete region directory. ");
    }

    public static void archiveFamily(FileSystem fs, Configuration conf, HRegionInfo parent, Path tableDir, byte[] family) throws IOException {
        Path familyDir = new Path(tableDir, new Path(parent.getEncodedName(), Bytes.toString((byte[])family)));
        FileStatus[] storeFiles = FSUtils.listStatus(fs, familyDir);
        if (storeFiles == null) {
            LOG.debug((Object)("No store files to dispose for region=" + parent.getRegionNameAsString() + ", family=" + Bytes.toString((byte[])family)));
            return;
        }
        FileStatusConverter getAsFile = new FileStatusConverter(fs);
        List toArchive = Lists.transform(Arrays.asList(storeFiles), (Function)getAsFile);
        Path storeArchiveDir = HFileArchiveUtil.getStoreArchivePath(conf, parent, tableDir, family);
        if (!HFileArchiver.resolveAndArchive(fs, storeArchiveDir, toArchive)) {
            throw new IOException("Failed to archive/delete all the files for region:" + Bytes.toString((byte[])parent.getRegionName()) + ", family:" + Bytes.toString((byte[])family) + " into " + storeArchiveDir + ". Something is probably awry on the filesystem.");
        }
    }

    public static void archiveStoreFiles(Configuration conf, FileSystem fs, HRegionInfo regionInfo, Path tableDir, byte[] family, Collection<StoreFile> compactedFiles) throws IOException {
        StoreToFile getStorePath;
        Collection storeFiles;
        if (fs == null) {
            LOG.warn((Object)("Passed filesystem is null, so just deleting the files without archiving for region:" + Bytes.toString((byte[])regionInfo.getRegionName()) + ", family:" + Bytes.toString((byte[])family)));
            HFileArchiver.deleteStoreFilesWithoutArchiving(compactedFiles);
            return;
        }
        if (compactedFiles.size() == 0) {
            LOG.debug((Object)"No store files to dispose, done!");
            return;
        }
        if (regionInfo == null || family == null) {
            throw new IOException("Need to have a region and a family to archive from.");
        }
        Path storeArchiveDir = HFileArchiveUtil.getStoreArchivePath(conf, regionInfo, tableDir, family);
        if (!fs.mkdirs(storeArchiveDir)) {
            throw new IOException("Could not make archive directory (" + storeArchiveDir + ") for store:" + Bytes.toString((byte[])family) + ", deleting compacted files instead.");
        }
        if (LOG.isTraceEnabled()) {
            LOG.trace((Object)"Archiving compacted store files.");
        }
        if (!HFileArchiver.resolveAndArchive(fs, storeArchiveDir, storeFiles = Collections2.transform(compactedFiles, (Function)(getStorePath = new StoreToFile(fs))))) {
            throw new IOException("Failed to archive/delete all the files for region:" + Bytes.toString((byte[])regionInfo.getRegionName()) + ", family:" + Bytes.toString((byte[])family) + " into " + storeArchiveDir + ". Something is probably awry on the filesystem.");
        }
    }

    public static void archiveStoreFile(Configuration conf, FileSystem fs, HRegionInfo regionInfo, Path tableDir, byte[] family, Path storeFile) throws IOException {
        Path storeArchiveDir = HFileArchiveUtil.getStoreArchivePath(conf, regionInfo, tableDir, family);
        if (!fs.mkdirs(storeArchiveDir)) {
            throw new IOException("Could not make archive directory (" + storeArchiveDir + ") for store:" + Bytes.toString((byte[])family) + ", deleting compacted files instead.");
        }
        FileablePath file = new FileablePath(fs, storeFile);
        long start = EnvironmentEdgeManager.currentTimeMillis();
        if (!HFileArchiver.resolveAndArchiveFile(storeArchiveDir, file, Long.toString(start))) {
            throw new IOException("Failed to archive/delete the file for region:" + regionInfo.getRegionNameAsString() + ", family:" + Bytes.toString((byte[])family) + " into " + storeArchiveDir + ". Something is probably awry on the filesystem.");
        }
    }

    private static boolean resolveAndArchive(FileSystem fs, Path baseArchiveDir, Collection<File> toArchive) throws IOException {
        long start;
        List<File> failures;
        if (LOG.isTraceEnabled()) {
            LOG.trace((Object)("Starting to archive " + toArchive));
        }
        if ((failures = HFileArchiver.resolveAndArchive(fs, baseArchiveDir, toArchive, start = EnvironmentEdgeManager.currentTimeMillis())).size() > 0) {
            LOG.warn((Object)("Failed to complete archive of: " + failures + ". Those files are still in the original location, and they may slow down reads."));
            return false;
        }
        return true;
    }

    private static List<File> resolveAndArchive(FileSystem fs, Path baseArchiveDir, Collection<File> toArchive, long start) throws IOException {
        if (toArchive.size() == 0) {
            return Collections.emptyList();
        }
        if (LOG.isTraceEnabled()) {
            LOG.trace((Object)("moving files to the archive directory: " + baseArchiveDir));
        }
        if (!fs.exists(baseArchiveDir)) {
            if (!fs.mkdirs(baseArchiveDir)) {
                throw new IOException("Failed to create the archive directory:" + baseArchiveDir + ", quitting archive attempt.");
            }
            if (LOG.isTraceEnabled()) {
                LOG.trace((Object)("Created archive directory:" + baseArchiveDir));
            }
        }
        ArrayList<File> failures = new ArrayList<File>();
        String startTime = Long.toString(start);
        for (File file : toArchive) {
            try {
                if (LOG.isTraceEnabled()) {
                    LOG.trace((Object)("Archiving: " + file));
                }
                if (file.isFile()) {
                    if (HFileArchiver.resolveAndArchiveFile(baseArchiveDir, file, startTime)) continue;
                    LOG.warn((Object)("Couldn't archive " + file + " into backup directory: " + baseArchiveDir));
                    failures.add(file);
                    continue;
                }
                if (LOG.isTraceEnabled()) {
                    LOG.trace((Object)(file + " is a directory, archiving children files"));
                }
                Path parentArchiveDir = new Path(baseArchiveDir, file.getName());
                Collection<File> children = file.getChildren();
                failures.addAll(HFileArchiver.resolveAndArchive(fs, parentArchiveDir, children, start));
            }
            catch (IOException e) {
                LOG.warn((Object)("Failed to archive " + file), (Throwable)e);
                failures.add(file);
            }
        }
        return failures;
    }

    private static boolean resolveAndArchiveFile(Path archiveDir, File currentFile, String archiveStartTime) throws IOException {
        String filename = currentFile.getName();
        Path archiveFile = new Path(archiveDir, filename);
        FileSystem fs = currentFile.getFileSystem();
        if (fs.exists(archiveFile)) {
            Path backedupArchiveFile;
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("File:" + archiveFile + " already exists in archive, moving to " + "timestamped backup and overwriting current."));
            }
            if (!fs.rename(archiveFile, backedupArchiveFile = new Path(archiveDir, filename + SEPARATOR + archiveStartTime))) {
                LOG.error((Object)("Could not rename archive file to backup: " + backedupArchiveFile + ", deleting existing file in favor of newer."));
                if (!fs.delete(archiveFile, false)) {
                    throw new IOException("Couldn't delete existing archive file (" + archiveFile + ") or rename it to the backup file (" + backedupArchiveFile + ") to make room for similarly named file.");
                }
            }
            LOG.debug((Object)("Backed up archive file from " + archiveFile));
        }
        if (LOG.isTraceEnabled()) {
            LOG.trace((Object)("No existing file in archive for: " + archiveFile + ", free to archive original file."));
        }
        boolean success = false;
        for (int i = 0; !success && i < 3; ++i) {
            if (i > 0) {
                try {
                    if (!fs.exists(archiveDir) && fs.mkdirs(archiveDir)) {
                        LOG.debug((Object)("Created archive directory:" + archiveDir));
                    }
                }
                catch (IOException e) {
                    LOG.warn((Object)("Failed to create directory: " + archiveDir), (Throwable)e);
                }
            }
            try {
                success = currentFile.moveAndClose(archiveFile);
                continue;
            }
            catch (IOException e) {
                LOG.warn((Object)("Failed to archive " + currentFile + " on try #" + i), (Throwable)e);
                success = false;
            }
        }
        if (!success) {
            LOG.error((Object)("Failed to archive " + currentFile));
            return false;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Finished archiving from " + currentFile + ", to " + archiveFile));
        }
        return true;
    }

    private static boolean deleteRegionWithoutArchiving(FileSystem fs, Path regionDir) throws IOException {
        if (fs.delete(regionDir, true)) {
            LOG.debug((Object)("Deleted all region files in: " + regionDir));
            return true;
        }
        LOG.debug((Object)("Failed to delete region directory:" + regionDir));
        return false;
    }

    private static void deleteStoreFilesWithoutArchiving(Collection<StoreFile> compactedFiles) throws IOException {
        LOG.debug((Object)"Deleting store files without archiving.");
        ArrayList<IOException> errors = new ArrayList<IOException>(0);
        for (StoreFile hsf : compactedFiles) {
            try {
                hsf.deleteReader();
            }
            catch (IOException e) {
                LOG.error((Object)("Failed to delete store file:" + hsf.getPath()));
                errors.add(e);
            }
        }
        if (errors.size() > 0) {
            throw MultipleIOException.createIOException(errors);
        }
    }

    private static class FileableStoreFile
    extends File {
        StoreFile file;

        public FileableStoreFile(FileSystem fs, StoreFile store) {
            super(fs);
            this.file = store;
        }

        @Override
        public void delete() throws IOException {
            this.file.deleteReader();
        }

        @Override
        public String getName() {
            return this.file.getPath().getName();
        }

        @Override
        public boolean isFile() {
            return true;
        }

        @Override
        public Collection<File> getChildren() throws IOException {
            return Collections.emptyList();
        }

        @Override
        public void close() throws IOException {
            this.file.closeReader(true);
        }

        @Override
        Path getPath() {
            return this.file.getPath();
        }
    }

    private static class FileablePath
    extends File {
        private final Path file;
        private final FileStatusConverter getAsFile;

        public FileablePath(FileSystem fs, Path file) {
            super(fs);
            this.file = file;
            this.getAsFile = new FileStatusConverter(fs);
        }

        @Override
        public void delete() throws IOException {
            if (!this.fs.delete(this.file, true)) {
                throw new IOException("Failed to delete:" + this.file);
            }
        }

        @Override
        public String getName() {
            return this.file.getName();
        }

        @Override
        public Collection<File> getChildren() throws IOException {
            if (this.fs.isFile(this.file)) {
                return Collections.emptyList();
            }
            return Collections2.transform(Arrays.asList(this.fs.listStatus(this.file)), (Function)this.getAsFile);
        }

        @Override
        public boolean isFile() throws IOException {
            return this.fs.isFile(this.file);
        }

        @Override
        public void close() throws IOException {
        }

        @Override
        Path getPath() {
            return this.file;
        }
    }

    private static abstract class File {
        protected final FileSystem fs;

        public File(FileSystem fs) {
            this.fs = fs;
        }

        abstract void delete() throws IOException;

        abstract boolean isFile() throws IOException;

        abstract Collection<File> getChildren() throws IOException;

        abstract void close() throws IOException;

        abstract String getName();

        abstract Path getPath();

        public boolean moveAndClose(Path dest) throws IOException {
            this.close();
            Path p = this.getPath();
            return FSUtils.renameAndSetModifyTime(this.fs, p, dest);
        }

        public FileSystem getFileSystem() {
            return this.fs;
        }

        public String toString() {
            return this.getClass() + ", file:" + this.getPath().toString();
        }
    }

    private static class StoreToFile
    extends FileConverter<StoreFile> {
        public StoreToFile(FileSystem fs) {
            super(fs);
        }

        public File apply(StoreFile input) {
            return new FileableStoreFile(this.fs, input);
        }
    }

    private static class FileStatusConverter
    extends FileConverter<FileStatus> {
        public FileStatusConverter(FileSystem fs) {
            super(fs);
        }

        public File apply(FileStatus input) {
            return new FileablePath(this.fs, input.getPath());
        }
    }

    private static abstract class FileConverter<T>
    implements Function<T, File> {
        protected final FileSystem fs;

        public FileConverter(FileSystem fs) {
            this.fs = fs;
        }
    }
}

