/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.shaded.org.apache.zookeeper.common;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.security.cert.Certificate;
import java.security.cert.CertificateParsingException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.regex.Pattern;
import javax.naming.InvalidNameException;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.ldap.LdapName;
import javax.naming.ldap.Rdn;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLException;
import javax.net.ssl.SSLPeerUnverifiedException;
import javax.net.ssl.SSLSession;
import javax.security.auth.x500.X500Principal;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class ZKHostnameVerifier
implements HostnameVerifier {
    private final Logger log = LoggerFactory.getLogger(ZKHostnameVerifier.class);

    ZKHostnameVerifier() {
    }

    @Override
    public boolean verify(String host, SSLSession session) {
        try {
            Certificate[] certs = session.getPeerCertificates();
            X509Certificate x509 = (X509Certificate)certs[0];
            this.verify(host, x509);
            return true;
        }
        catch (SSLException ex) {
            if (this.log.isDebugEnabled()) {
                this.log.debug(ex.getMessage(), (Throwable)ex);
            }
            return false;
        }
    }

    void verify(String host, X509Certificate cert) throws SSLException {
        HostNameType hostType = ZKHostnameVerifier.determineHostFormat(host);
        List<SubjectName> subjectAlts = ZKHostnameVerifier.getSubjectAltNames(cert);
        if (subjectAlts != null && !subjectAlts.isEmpty()) {
            switch (hostType) {
                case IPv4: {
                    ZKHostnameVerifier.matchIPAddress(host, subjectAlts);
                    break;
                }
                case IPv6: {
                    ZKHostnameVerifier.matchIPv6Address(host, subjectAlts);
                    break;
                }
                default: {
                    ZKHostnameVerifier.matchDNSName(host, subjectAlts);
                    break;
                }
            }
        } else {
            X500Principal subjectPrincipal = cert.getSubjectX500Principal();
            String cn = ZKHostnameVerifier.extractCN(subjectPrincipal.getName("RFC2253"));
            if (cn == null) {
                throw new SSLException("Certificate subject for <" + host + "> doesn't contain a common name and does not have alternative names");
            }
            ZKHostnameVerifier.matchCN(host, cn);
        }
    }

    private static void matchIPAddress(String host, List<SubjectName> subjectAlts) throws SSLException {
        for (int i = 0; i < subjectAlts.size(); ++i) {
            SubjectName subjectAlt = subjectAlts.get(i);
            if (subjectAlt.getType() != 7 || !host.equals(subjectAlt.getValue())) continue;
            return;
        }
        throw new SSLPeerUnverifiedException("Certificate for <" + host + "> doesn't match any of the subject alternative names: " + subjectAlts);
    }

    private static void matchIPv6Address(String host, List<SubjectName> subjectAlts) throws SSLException {
        String normalisedHost = ZKHostnameVerifier.normaliseAddress(host);
        for (int i = 0; i < subjectAlts.size(); ++i) {
            String normalizedSubjectAlt;
            SubjectName subjectAlt = subjectAlts.get(i);
            if (subjectAlt.getType() != 7 || !normalisedHost.equals(normalizedSubjectAlt = ZKHostnameVerifier.normaliseAddress(subjectAlt.getValue()))) continue;
            return;
        }
        throw new SSLPeerUnverifiedException("Certificate for <" + host + "> doesn't match any of the subject alternative names: " + subjectAlts);
    }

    private static void matchDNSName(String host, List<SubjectName> subjectAlts) throws SSLException {
        String normalizedHost = host.toLowerCase(Locale.ROOT);
        for (int i = 0; i < subjectAlts.size(); ++i) {
            String normalizedSubjectAlt;
            SubjectName subjectAlt = subjectAlts.get(i);
            if (subjectAlt.getType() != 2 || !ZKHostnameVerifier.matchIdentityStrict(normalizedHost, normalizedSubjectAlt = subjectAlt.getValue().toLowerCase(Locale.ROOT))) continue;
            return;
        }
        throw new SSLPeerUnverifiedException("Certificate for <" + host + "> doesn't match any of the subject alternative names: " + subjectAlts);
    }

    private static void matchCN(String host, String cn) throws SSLException {
        String normalizedCn;
        String normalizedHost = host.toLowerCase(Locale.ROOT);
        if (!ZKHostnameVerifier.matchIdentityStrict(normalizedHost, normalizedCn = cn.toLowerCase(Locale.ROOT))) {
            throw new SSLPeerUnverifiedException("Certificate for <" + host + "> doesn't match common name of the certificate subject: " + cn);
        }
    }

    private static boolean matchIdentity(String host, String identity, boolean strict) {
        int asteriskIdx = identity.indexOf(42);
        if (asteriskIdx != -1) {
            String remainder;
            String prefix = identity.substring(0, asteriskIdx);
            String suffix = identity.substring(asteriskIdx + 1);
            if (!prefix.isEmpty() && !host.startsWith(prefix)) {
                return false;
            }
            if (!suffix.isEmpty() && !host.endsWith(suffix)) {
                return false;
            }
            return !strict || !(remainder = host.substring(prefix.length(), host.length() - suffix.length())).contains(".");
        }
        return host.equalsIgnoreCase(identity);
    }

    private static boolean matchIdentityStrict(String host, String identity) {
        return ZKHostnameVerifier.matchIdentity(host, identity, true);
    }

    private static String extractCN(String subjectPrincipal) throws SSLException {
        if (subjectPrincipal == null) {
            return null;
        }
        try {
            LdapName subjectDN = new LdapName(subjectPrincipal);
            List<Rdn> rdns = subjectDN.getRdns();
            for (int i = rdns.size() - 1; i >= 0; --i) {
                Rdn rds = rdns.get(i);
                Attributes attributes = rds.toAttributes();
                Attribute cn = attributes.get("cn");
                if (cn == null) continue;
                try {
                    Object value = cn.get();
                    if (value == null) continue;
                    return value.toString();
                }
                catch (NoSuchElementException noSuchElementException) {
                    continue;
                }
                catch (NamingException namingException) {
                    // empty catch block
                }
            }
            return null;
        }
        catch (InvalidNameException e) {
            throw new SSLException(subjectPrincipal + " is not a valid X500 distinguished name");
        }
    }

    private static HostNameType determineHostFormat(String host) {
        if (InetAddressUtils.isIPv4Address(host)) {
            return HostNameType.IPv4;
        }
        String s = host;
        if (s.startsWith("[") && s.endsWith("]")) {
            s = host.substring(1, host.length() - 1);
        }
        if (InetAddressUtils.isIPv6Address(s)) {
            return HostNameType.IPv6;
        }
        return HostNameType.DNS;
    }

    private static List<SubjectName> getSubjectAltNames(X509Certificate cert) {
        try {
            Collection<List<?>> entries = cert.getSubjectAlternativeNames();
            if (entries == null) {
                return Collections.emptyList();
            }
            ArrayList<SubjectName> result = new ArrayList<SubjectName>();
            for (List<?> entry : entries) {
                Integer type = entry.size() >= 2 ? (Integer)entry.get(0) : null;
                if (type == null) continue;
                String s = (String)entry.get(1);
                result.add(new SubjectName(s, type));
            }
            return result;
        }
        catch (CertificateParsingException ignore) {
            return Collections.emptyList();
        }
    }

    private static String normaliseAddress(String hostname) {
        if (hostname == null) {
            return hostname;
        }
        try {
            InetAddress inetAddress = InetAddress.getByName(hostname);
            return inetAddress.getHostAddress();
        }
        catch (UnknownHostException unexpected) {
            return hostname;
        }
    }

    static enum HostNameType {
        IPv4(7),
        IPv6(7),
        DNS(2);

        final int subjectType;

        private HostNameType(int subjectType) {
            this.subjectType = subjectType;
        }
    }

    private static class InetAddressUtils {
        private static final Pattern IPV4_PATTERN = Pattern.compile("^(25[0-5]|2[0-4]\\d|[0-1]?\\d?\\d)(\\.(25[0-5]|2[0-4]\\d|[0-1]?\\d?\\d)){3}$");
        private static final Pattern IPV6_STD_PATTERN = Pattern.compile("^(?:[0-9a-fA-F]{1,4}:){7}[0-9a-fA-F]{1,4}$");
        private static final Pattern IPV6_HEX_COMPRESSED_PATTERN = Pattern.compile("^((?:[0-9A-Fa-f]{1,4}(?::[0-9A-Fa-f]{1,4})*)?)::((?:[0-9A-Fa-f]{1,4}(?::[0-9A-Fa-f]{1,4})*)?)$");

        private InetAddressUtils() {
        }

        static boolean isIPv4Address(String input) {
            return IPV4_PATTERN.matcher(input).matches();
        }

        static boolean isIPv6StdAddress(String input) {
            return IPV6_STD_PATTERN.matcher(input).matches();
        }

        static boolean isIPv6HexCompressedAddress(String input) {
            return IPV6_HEX_COMPRESSED_PATTERN.matcher(input).matches();
        }

        static boolean isIPv6Address(String input) {
            return InetAddressUtils.isIPv6StdAddress(input) || InetAddressUtils.isIPv6HexCompressedAddress(input);
        }
    }

    private static final class SubjectName {
        static final int DNS = 2;
        static final int IP = 7;
        private final String value;
        private final int type;

        static SubjectName IP(String value) {
            return new SubjectName(value, 7);
        }

        static SubjectName DNS(String value) {
            return new SubjectName(value, 2);
        }

        SubjectName(String value, int type) {
            if (type != 2 && type != 7) {
                throw new IllegalArgumentException("Invalid type: " + type);
            }
            this.value = Objects.requireNonNull(value);
            this.type = type;
        }

        public int getType() {
            return this.type;
        }

        public String getValue() {
            return this.value;
        }

        public String toString() {
            return this.value;
        }
    }
}

