/*
 * Decompiled with CFR 0.152.
 */
package com.azure.storage.common.implementation;

import com.azure.core.util.logging.ClientLogger;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.NonReadableChannelException;
import java.nio.channels.NonWritableChannelException;
import java.nio.channels.SeekableByteChannel;
import java.util.Objects;

public final class StorageSeekableByteChannel
implements SeekableByteChannel {
    private static final ClientLogger LOGGER = new ClientLogger(StorageSeekableByteChannel.class);
    private final ReadBehavior readBehavior;
    private final WriteBehavior writeBehavior;
    private boolean isClosed;
    private ByteBuffer buffer;
    private long bufferAbsolutePosition;
    private long absolutePosition;

    public StorageSeekableByteChannel(int chunkSize, ReadBehavior readBehavior, long startingPosition) {
        this(chunkSize, null, Objects.requireNonNull(readBehavior), startingPosition);
        this.buffer.limit(0);
    }

    public StorageSeekableByteChannel(int chunkSize, WriteBehavior writeBehavior, long startingPosition) {
        this(chunkSize, Objects.requireNonNull(writeBehavior), null, startingPosition);
    }

    private StorageSeekableByteChannel(int chunkSize, WriteBehavior writeBehavior, ReadBehavior readBehavior, long startingPosition) {
        if (chunkSize < 1) {
            throw new IllegalArgumentException("'chunkSize' must be a positive number.");
        }
        if (startingPosition < 0L) {
            throw new IllegalArgumentException("'startingPosition' cannot be a negative number.");
        }
        this.readBehavior = readBehavior;
        this.writeBehavior = writeBehavior;
        this.buffer = ByteBuffer.allocate(chunkSize);
        this.absolutePosition = startingPosition;
        this.bufferAbsolutePosition = 0L;
    }

    public ReadBehavior getReadBehavior() {
        return this.readBehavior;
    }

    public WriteBehavior getWriteBehavior() {
        return this.writeBehavior;
    }

    public int getChunkSize() {
        return this.buffer.capacity();
    }

    @Override
    public int read(ByteBuffer dst) throws IOException {
        this.assertOpen();
        this.assertCanRead();
        if (dst.isReadOnly()) {
            throw LOGGER.logExceptionAsError(new IllegalArgumentException("'dst' is read-only and cannot be written to."));
        }
        if (this.buffer.remaining() == 0 && this.refillReadBuffer(this.absolutePosition) == -1) {
            this.absolutePosition = this.readBehavior.getResourceLength();
            return -1;
        }
        if (this.buffer.remaining() == 0) {
            return 0;
        }
        int read = Math.min(this.buffer.remaining(), dst.remaining());
        ByteBuffer temp = this.buffer.duplicate();
        temp.limit(temp.position() + read);
        dst.put(temp);
        this.buffer.position(this.buffer.position() + read);
        this.absolutePosition += (long)read;
        return read;
    }

    private int refillReadBuffer(long newBufferAbsolutePosition) throws IOException {
        this.buffer.clear();
        int read = this.readBehavior.read(this.buffer, newBufferAbsolutePosition);
        this.buffer.rewind();
        this.buffer.limit(Math.max(read, 0));
        this.bufferAbsolutePosition = Math.min(newBufferAbsolutePosition, this.readBehavior.getResourceLength());
        return read;
    }

    @Override
    public int write(ByteBuffer src) throws IOException {
        this.assertOpen();
        this.assertCanWrite();
        int write = Math.min(src.remaining(), this.buffer.remaining());
        if (write > 0) {
            ByteBuffer temp = src.duplicate();
            temp.limit(temp.position() + write);
            this.buffer.put(temp);
            src.position(src.position() + write);
        }
        if (this.buffer.remaining() == 0) {
            try {
                this.flushWriteBuffer();
            }
            catch (RuntimeException e) {
                this.buffer.position(this.buffer.position() - write);
                throw LOGGER.logExceptionAsError(e);
            }
        }
        this.absolutePosition += (long)write;
        return write;
    }

    private void flushWriteBuffer() throws IOException {
        if (this.buffer.position() == 0) {
            return;
        }
        int startingPosition = this.buffer.position();
        this.buffer.limit(this.buffer.position());
        this.buffer.rewind();
        try {
            this.writeBehavior.write(this.buffer, this.bufferAbsolutePosition);
        }
        catch (RuntimeException e) {
            this.buffer.limit(this.buffer.capacity());
            this.buffer.position(startingPosition);
            throw LOGGER.logExceptionAsError(e);
        }
        this.bufferAbsolutePosition += (long)this.buffer.limit();
        this.buffer.clear();
    }

    @Override
    public long position() throws IOException {
        this.assertOpen();
        return this.absolutePosition;
    }

    @Override
    public SeekableByteChannel position(long newPosition) throws IOException {
        this.assertOpen();
        if (this.readBehavior != null) {
            this.readModeSeek(newPosition);
        } else {
            this.writeModeSeek(newPosition);
        }
        return this;
    }

    private void readModeSeek(long newPosition) {
        if (newPosition < this.bufferAbsolutePosition || newPosition > this.bufferAbsolutePosition + (long)this.buffer.limit()) {
            this.buffer.clear();
            this.buffer.limit(0);
        } else {
            this.buffer.position((int)(newPosition - this.bufferAbsolutePosition));
        }
        this.absolutePosition = newPosition;
    }

    private void writeModeSeek(long newPosition) throws IOException {
        this.writeBehavior.assertCanSeek(newPosition);
        this.flushWriteBuffer();
        this.absolutePosition = newPosition;
        this.bufferAbsolutePosition = newPosition;
    }

    @Override
    public long size() throws IOException {
        this.assertOpen();
        if (this.readBehavior != null) {
            return this.readBehavior.getResourceLength();
        }
        return this.absolutePosition;
    }

    @Override
    public SeekableByteChannel truncate(long size) throws IOException {
        this.assertOpen();
        this.writeBehavior.resize(size);
        return this;
    }

    @Override
    public boolean isOpen() {
        return !this.isClosed;
    }

    @Override
    public void close() throws IOException {
        if (this.writeBehavior != null) {
            this.flushWriteBuffer();
            this.writeBehavior.commit(this.absolutePosition);
        }
        this.isClosed = true;
        this.buffer = null;
    }

    private void assertCanRead() {
        if (this.readBehavior == null) {
            throw LOGGER.logExceptionAsError(new NonReadableChannelException());
        }
    }

    private void assertCanWrite() {
        if (this.writeBehavior == null) {
            throw LOGGER.logExceptionAsError(new NonWritableChannelException());
        }
    }

    private void assertOpen() throws ClosedChannelException {
        if (this.isClosed) {
            throw LOGGER.logThrowableAsError(new ClosedChannelException());
        }
    }

    public static interface ReadBehavior {
        public int read(ByteBuffer var1, long var2) throws IOException;

        public long getResourceLength();
    }

    public static interface WriteBehavior {
        public void write(ByteBuffer var1, long var2) throws IOException;

        public void commit(long var1);

        public void assertCanSeek(long var1);

        public void resize(long var1);
    }
}

