/****************************************************************
 * Licensed to the Apache Software Foundation (ASF) under one   *
 * or more contributor license agreements.  See the NOTICE file *
 * distributed with this work for additional information        *
 * regarding copyright ownership.  The ASF licenses this file   *
 * to you under the Apache License, Version 2.0 (the            *
 * "License"); you may not use this file except in compliance   *
 * with the License.  You may obtain a copy of the License at   *
 *                                                              *
 *   http://www.apache.org/licenses/LICENSE-2.0                 *
 *                                                              *
 * Unless required by applicable law or agreed to in writing,   *
 * software distributed under the License is distributed on an  *
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY       *
 * KIND, either express or implied.  See the License for the    *
 * specific language governing permissions and limitations      *
 * under the License.                                           *
 ****************************************************************/

package org.apache.james.imap.encode;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.Locale;

import org.apache.james.imap.api.ImapCommand;
import org.apache.james.imap.api.ImapMessage;
import org.apache.james.imap.api.display.HumanReadableText;
import org.apache.james.imap.api.display.Locales;
import org.apache.james.imap.api.display.Localizer;
import org.apache.james.imap.api.message.response.StatusResponse;
import org.apache.james.imap.api.message.response.StatusResponse.ResponseCode;
import org.apache.james.imap.api.message.response.StatusResponse.Type;
import org.apache.james.imap.api.process.ImapSession;
import org.apache.james.imap.encode.base.AbstractChainedImapEncoder;

public class StatusResponseEncoder extends AbstractChainedImapEncoder {

    private final Localizer localizer;

    public StatusResponseEncoder(ImapEncoder next, final Localizer localizer) {
        super(next);
        this.localizer = localizer;
    }

    protected void doEncode(ImapMessage acceptableMessage, ImapResponseComposer composer, ImapSession session) throws IOException {
        StatusResponse response = (StatusResponse) acceptableMessage;
        final Type serverResponseType = response.getServerResponseType();
        final String type = asString(serverResponseType);
        final ResponseCode responseCode = response.getResponseCode();
        final String code = asString(responseCode);
        final String tag = response.getTag();
        final ImapCommand command = response.getCommand();
        final HumanReadableText textKey = response.getTextKey();
        final String text = asString(textKey, session);
        final Collection<String> parameters;
        final long number;
        final boolean useParens;
        if (responseCode == null) {
            parameters = null;
            number = -1;
            useParens = false;
        } else {
            parameters = responseCode.getParameters();
            number = responseCode.getNumber();
            useParens = responseCode.useParens();
        }
        //composer.statusResponse(tag, command, type, code, parameters, useParens, number, text);
        
        if (tag == null) {
        	composer.untagged();
        } else {
        	composer.tag(tag);
        }
        composer.message(type);
        if (responseCode != null) {
        	composer.openSquareBracket();
        	composer.message(code);
            if (number > -1) {
            	composer.message(number);
            }
            if (parameters != null && !parameters.isEmpty()) {
                if (useParens)
                	composer.openParen();
                for (Iterator<String> it = parameters.iterator(); it.hasNext();) {
                    final String parameter = it.next();
                    composer.message(parameter);
                }
                if (useParens)
                	composer.closeParen();
            }
            composer.closeSquareBracket();
        }
        if (command != null) {
        	composer.commandName(command.getName());
        }
        if (text != null && !"".equals(text)) {
        	composer.message(text);
        }
        composer.end();
    }

    private String asString(HumanReadableText text, ImapSession session) {
        // TODO: calculate locales
        return localizer.localize(text, new Locales(new ArrayList<Locale>(), null));
    }

    private String asString(StatusResponse.ResponseCode code) {
        final String result;
        if (code == null) {
            result = null;
        } else {
            result = code.getCode();
        }
        return result;
    }

    private String asString(StatusResponse.Type type) {
        final String result;
        if (type == null) {
            result = null;
        } else {
            result = type.getCode();
        }
        return result;
    }

    protected boolean isAcceptable(ImapMessage message) {
        return (message instanceof StatusResponse);
    }

}
