/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.network;

import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.SelectionKey;
import java.nio.channels.SocketChannel;
import java.util.Arrays;
import java.util.Map;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLEngine;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.config.types.Password;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.network.CertStores;
import org.apache.kafka.common.network.ChannelBuilder;
import org.apache.kafka.common.network.Mode;
import org.apache.kafka.common.network.NetworkTestUtils;
import org.apache.kafka.common.network.NioEchoServer;
import org.apache.kafka.common.network.Selector;
import org.apache.kafka.common.network.SslChannelBuilder;
import org.apache.kafka.common.network.SslTransportLayer;
import org.apache.kafka.common.protocol.SecurityProtocol;
import org.apache.kafka.common.security.ssl.SslFactory;
import org.apache.kafka.common.utils.MockTime;
import org.apache.kafka.common.utils.Time;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

public class SslTransportLayerTest {
    private static final int BUFFER_SIZE = 4096;
    private NioEchoServer server;
    private Selector selector;
    private ChannelBuilder channelBuilder;
    private CertStores serverCertStores;
    private CertStores clientCertStores;
    private Map<String, Object> sslClientConfigs;
    private Map<String, Object> sslServerConfigs;

    @Before
    public void setup() throws Exception {
        this.serverCertStores = new CertStores(true, "localhost");
        this.clientCertStores = new CertStores(false, "localhost");
        this.sslServerConfigs = this.serverCertStores.getTrustingConfig(this.clientCertStores);
        this.sslClientConfigs = this.clientCertStores.getTrustingConfig(this.serverCertStores);
        this.channelBuilder = new SslChannelBuilder(Mode.CLIENT);
        this.channelBuilder.configure(this.sslClientConfigs);
        this.selector = new Selector(5000L, new Metrics(), (Time)new MockTime(), "MetricGroup", this.channelBuilder);
    }

    @After
    public void teardown() throws Exception {
        if (this.selector != null) {
            this.selector.close();
        }
        if (this.server != null) {
            this.server.close();
        }
    }

    @Test
    public void testValidEndpointIdentification() throws Exception {
        String node = "0";
        this.server = NetworkTestUtils.createEchoServer(SecurityProtocol.SSL, this.sslServerConfigs);
        this.sslClientConfigs.put("ssl.endpoint.identification.algorithm", "HTTPS");
        this.createSelector(this.sslClientConfigs);
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(this.selector, node, 100, 10);
    }

    @Test
    public void testInvalidEndpointIdentification() throws Exception {
        String node = "0";
        this.serverCertStores = new CertStores(true, "notahost");
        this.clientCertStores = new CertStores(false, "localhost");
        this.sslServerConfigs = this.serverCertStores.getTrustingConfig(this.clientCertStores);
        this.sslClientConfigs = this.clientCertStores.getTrustingConfig(this.serverCertStores);
        this.sslClientConfigs.put("ssl.endpoint.identification.algorithm", "HTTPS");
        this.server = NetworkTestUtils.createEchoServer(SecurityProtocol.SSL, this.sslServerConfigs);
        this.createSelector(this.sslClientConfigs);
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.waitForChannelClose(this.selector, node);
    }

    @Test
    public void testEndpointIdentificationDisabled() throws Exception {
        String node = "0";
        String serverHost = InetAddress.getLocalHost().getHostAddress();
        this.server = new NioEchoServer(SecurityProtocol.SSL, this.sslServerConfigs, serverHost);
        this.server.start();
        this.sslClientConfigs.remove("ssl.endpoint.identification.algorithm");
        this.createSelector(this.sslClientConfigs);
        InetSocketAddress addr = new InetSocketAddress(serverHost, this.server.port());
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(this.selector, node, 100, 10);
    }

    @Test
    public void testClientAuthenticationRequiredValidProvided() throws Exception {
        String node = "0";
        this.sslServerConfigs.put("ssl.client.auth", "required");
        this.server = NetworkTestUtils.createEchoServer(SecurityProtocol.SSL, this.sslServerConfigs);
        this.createSelector(this.sslClientConfigs);
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(this.selector, node, 100, 10);
    }

    @Test
    public void testClientAuthenticationRequiredUntrustedProvided() throws Exception {
        String node = "0";
        this.sslServerConfigs = this.serverCertStores.getUntrustingConfig();
        this.sslServerConfigs.put("ssl.client.auth", "required");
        this.server = NetworkTestUtils.createEchoServer(SecurityProtocol.SSL, this.sslServerConfigs);
        this.createSelector(this.sslClientConfigs);
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.waitForChannelClose(this.selector, node);
    }

    @Test
    public void testClientAuthenticationRequiredNotProvided() throws Exception {
        String node = "0";
        this.sslServerConfigs.put("ssl.client.auth", "required");
        this.server = NetworkTestUtils.createEchoServer(SecurityProtocol.SSL, this.sslServerConfigs);
        this.sslClientConfigs.remove("ssl.keystore.location");
        this.sslClientConfigs.remove("ssl.keystore.password");
        this.sslClientConfigs.remove("ssl.key.password");
        this.createSelector(this.sslClientConfigs);
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.waitForChannelClose(this.selector, node);
    }

    @Test
    public void testClientAuthenticationDisabledUntrustedProvided() throws Exception {
        String node = "0";
        this.sslServerConfigs = this.serverCertStores.getUntrustingConfig();
        this.sslServerConfigs.put("ssl.client.auth", "none");
        this.server = NetworkTestUtils.createEchoServer(SecurityProtocol.SSL, this.sslServerConfigs);
        this.createSelector(this.sslClientConfigs);
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(this.selector, node, 100, 10);
    }

    @Test
    public void testClientAuthenticationDisabledNotProvided() throws Exception {
        String node = "0";
        this.sslServerConfigs.put("ssl.client.auth", "none");
        this.server = NetworkTestUtils.createEchoServer(SecurityProtocol.SSL, this.sslServerConfigs);
        this.sslClientConfigs.remove("ssl.keystore.location");
        this.sslClientConfigs.remove("ssl.keystore.password");
        this.sslClientConfigs.remove("ssl.key.password");
        this.createSelector(this.sslClientConfigs);
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(this.selector, node, 100, 10);
    }

    @Test
    public void testClientAuthenticationRequestedValidProvided() throws Exception {
        String node = "0";
        this.sslServerConfigs.put("ssl.client.auth", "requested");
        this.server = NetworkTestUtils.createEchoServer(SecurityProtocol.SSL, this.sslServerConfigs);
        this.createSelector(this.sslClientConfigs);
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(this.selector, node, 100, 10);
    }

    @Test
    public void testClientAuthenticationRequestedNotProvided() throws Exception {
        String node = "0";
        this.sslServerConfigs.put("ssl.client.auth", "requested");
        this.server = NetworkTestUtils.createEchoServer(SecurityProtocol.SSL, this.sslServerConfigs);
        this.sslClientConfigs.remove("ssl.keystore.location");
        this.sslClientConfigs.remove("ssl.keystore.password");
        this.sslClientConfigs.remove("ssl.key.password");
        this.createSelector(this.sslClientConfigs);
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(this.selector, node, 100, 10);
    }

    @Test
    public void testInvalidSecureRandomImplementation() throws Exception {
        SslChannelBuilder channelBuilder = new SslChannelBuilder(Mode.CLIENT);
        try {
            this.sslClientConfigs.put("ssl.secure.random.implementation", "invalid");
            channelBuilder.configure(this.sslClientConfigs);
            Assert.fail((String)"SSL channel configured with invalid SecureRandom implementation");
        }
        catch (KafkaException kafkaException) {
            // empty catch block
        }
    }

    @Test
    public void testInvalidTruststorePassword() throws Exception {
        SslChannelBuilder channelBuilder = new SslChannelBuilder(Mode.CLIENT);
        try {
            this.sslClientConfigs.put("ssl.truststore.password", "invalid");
            channelBuilder.configure(this.sslClientConfigs);
            Assert.fail((String)"SSL channel configured with invalid truststore password");
        }
        catch (KafkaException kafkaException) {
            // empty catch block
        }
    }

    @Test
    public void testInvalidKeystorePassword() throws Exception {
        SslChannelBuilder channelBuilder = new SslChannelBuilder(Mode.CLIENT);
        try {
            this.sslClientConfigs.put("ssl.keystore.password", "invalid");
            channelBuilder.configure(this.sslClientConfigs);
            Assert.fail((String)"SSL channel configured with invalid keystore password");
        }
        catch (KafkaException kafkaException) {
            // empty catch block
        }
    }

    @Test
    public void testInvalidKeyPassword() throws Exception {
        String node = "0";
        this.sslServerConfigs.put("ssl.key.password", new Password("invalid"));
        this.server = NetworkTestUtils.createEchoServer(SecurityProtocol.SSL, this.sslServerConfigs);
        this.createSelector(this.sslClientConfigs);
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.waitForChannelClose(this.selector, node);
    }

    @Test
    public void testUnsupportedTLSVersion() throws Exception {
        String node = "0";
        this.sslServerConfigs.put("ssl.enabled.protocols", Arrays.asList("TLSv1.2"));
        this.server = NetworkTestUtils.createEchoServer(SecurityProtocol.SSL, this.sslServerConfigs);
        this.sslClientConfigs.put("ssl.enabled.protocols", Arrays.asList("TLSv1.1"));
        this.createSelector(this.sslClientConfigs);
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.waitForChannelClose(this.selector, node);
    }

    @Test
    public void testUnsupportedCiphers() throws Exception {
        String node = "0";
        String[] cipherSuites = SSLContext.getDefault().getDefaultSSLParameters().getCipherSuites();
        this.sslServerConfigs.put("ssl.cipher.suites", Arrays.asList(cipherSuites[0]));
        this.server = NetworkTestUtils.createEchoServer(SecurityProtocol.SSL, this.sslServerConfigs);
        this.sslClientConfigs.put("ssl.cipher.suites", Arrays.asList(cipherSuites[1]));
        this.createSelector(this.sslClientConfigs);
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.waitForChannelClose(this.selector, node);
    }

    @Test
    public void testNetReadBufferResize() throws Exception {
        String node = "0";
        this.server = NetworkTestUtils.createEchoServer(SecurityProtocol.SSL, this.sslServerConfigs);
        this.createSelector(this.sslClientConfigs, 10, null, null);
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(this.selector, node, 64000, 10);
    }

    @Test
    public void testNetWriteBufferResize() throws Exception {
        String node = "0";
        this.server = NetworkTestUtils.createEchoServer(SecurityProtocol.SSL, this.sslServerConfigs);
        this.createSelector(this.sslClientConfigs, null, 10, null);
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(this.selector, node, 64000, 10);
    }

    @Test
    public void testApplicationBufferResize() throws Exception {
        String node = "0";
        this.server = NetworkTestUtils.createEchoServer(SecurityProtocol.SSL, this.sslServerConfigs);
        this.createSelector(this.sslClientConfigs, null, null, 10);
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(this.selector, node, 64000, 10);
    }

    private void createSelector(Map<String, Object> sslClientConfigs) {
        this.createSelector(sslClientConfigs, null, null, null);
    }

    private void createSelector(Map<String, Object> sslClientConfigs, final Integer netReadBufSize, final Integer netWriteBufSize, final Integer appBufSize) {
        this.channelBuilder = new SslChannelBuilder(Mode.CLIENT){

            protected SslTransportLayer buildTransportLayer(SslFactory sslFactory, String id, SelectionKey key) throws IOException {
                SocketChannel socketChannel = (SocketChannel)key.channel();
                SSLEngine sslEngine = sslFactory.createSslEngine(socketChannel.socket().getInetAddress().getHostName(), socketChannel.socket().getPort());
                TestSslTransportLayer transportLayer = new TestSslTransportLayer(id, key, sslEngine, netReadBufSize, netWriteBufSize, appBufSize);
                transportLayer.startHandshake();
                return transportLayer;
            }
        };
        this.channelBuilder.configure(sslClientConfigs);
        this.selector = new Selector(5000L, new Metrics(), (Time)new MockTime(), "MetricGroup", this.channelBuilder);
    }

    private static class TestSslTransportLayer
    extends SslTransportLayer {
        private final ResizeableBufferSize netReadBufSize;
        private final ResizeableBufferSize netWriteBufSize;
        private final ResizeableBufferSize appBufSize;

        public TestSslTransportLayer(String channelId, SelectionKey key, SSLEngine sslEngine, Integer netReadBufSize, Integer netWriteBufSize, Integer appBufSize) throws IOException {
            super(channelId, key, sslEngine, false);
            this.netReadBufSize = new ResizeableBufferSize(netReadBufSize);
            this.netWriteBufSize = new ResizeableBufferSize(netWriteBufSize);
            this.appBufSize = new ResizeableBufferSize(appBufSize);
        }

        protected int netReadBufferSize() {
            ByteBuffer netReadBuffer = this.netReadBuffer();
            boolean updateBufSize = netReadBuffer != null && !this.netReadBuffer().hasRemaining();
            return this.netReadBufSize.updateAndGet(super.netReadBufferSize(), updateBufSize);
        }

        protected int netWriteBufferSize() {
            return this.netWriteBufSize.updateAndGet(super.netWriteBufferSize(), true);
        }

        protected int applicationBufferSize() {
            return this.appBufSize.updateAndGet(super.applicationBufferSize(), true);
        }

        private static class ResizeableBufferSize {
            private Integer bufSizeOverride;

            ResizeableBufferSize(Integer bufSizeOverride) {
                this.bufSizeOverride = bufSizeOverride;
            }

            int updateAndGet(int actualSize, boolean update) {
                int size = actualSize;
                if (this.bufSizeOverride != null) {
                    if (update) {
                        this.bufSizeOverride = Math.min(this.bufSizeOverride * 2, size);
                    }
                    size = this.bufSizeOverride;
                }
                return size;
            }
        }
    }
}

