/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.utils;

import java.io.File;
import java.io.IOException;
import java.nio.file.FileSystems;
import java.nio.file.Path;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.time.Duration;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.errors.InvalidConfigurationException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileWatchService {
    private static final Logger log = LoggerFactory.getLogger(FileWatchService.class);
    private static final String POLLING_WATCH_SERVICE_CLASS_NAME = "sun.nio.fs.PollingWatchService";
    private static final String SENSITIVITY_MODIFIER_CLASS_NAME = "com.sun.nio.file.SensitivityWatchEventModifier";
    private static final boolean USES_POLLING_WATCH_SERVICE;
    public static final Duration MIN_WATCH_INTERVAL;
    private static final WatchEvent.Kind<?>[] WATCH_EVENT_KINDS;
    private static WatchEvent.Modifier watchEventModifier;
    private final Map<WatchKey, Watch> watchMap = new HashMap<WatchKey, Watch>();
    private WatchService watchService;
    private ExecutorService executorService;

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public WatchKey add(Listener listener) {
        WatchKey watchKey;
        Path watchDir = this.watchDirectory(listener);
        FileWatchService fileWatchService = this;
        synchronized (fileWatchService) {
            Watch watch = this.watch(watchDir);
            if (watch == null) {
                if (this.watchService == null) {
                    log.info("Starting new file watch service");
                    this.watchService = this.newWatchService();
                    this.executorService = Executors.newSingleThreadExecutor(runnable -> {
                        Thread thread = new Thread(runnable, "file-watch-service");
                        thread.setDaemon(true);
                        return thread;
                    });
                    this.executorService.submit(this::poll);
                }
                watchKey = this.register(watchDir);
                watch = new Watch(watchDir, watchKey);
                this.watchMap.put(watchKey, watch);
            } else {
                watchKey = watch.watchKey;
            }
            watch.add(listener);
        }
        listener.onInit();
        log.info("Added file watch listener for directory {} file {}", (Object)watchDir, (Object)listener.file());
        return watchKey;
    }

    public synchronized void remove(Listener listener) {
        Path watchDir = this.watchDirectory(listener);
        Watch watch = this.watch(watchDir);
        if (watch != null) {
            watch.listeners.remove(listener);
            if (watch.listeners.isEmpty()) {
                this.watchMap.remove(watch.watchKey);
                watch.watchKey.cancel();
                log.debug("Removing file watch for directory {} file {} since no listeners remaining", (Object)watchDir, (Object)listener.file());
            }
            if (this.watchMap.isEmpty()) {
                this.close();
                log.info("Closing file watcher since no watches remaining");
            }
            log.info("Removed file watch listener for directory {} file {}", (Object)watchDir, (Object)listener.file());
        } else {
            log.debug("File watch not found for directory {}", (Object)watchDir);
        }
    }

    public synchronized void close() {
        try {
            if (this.executorService != null) {
                this.executorService.shutdownNow();
                this.executorService = null;
            }
            if (this.watchService != null) {
                this.watchMap.values().forEach(Watch::cancel);
                this.watchService.close();
                this.watchService = null;
            }
        }
        catch (IOException e) {
            log.error("Failed to close file watch service", (Throwable)e);
        }
    }

    protected WatchService newWatchService() {
        try {
            return FileSystems.getDefault().newWatchService();
        }
        catch (IOException e) {
            log.error("Failed to create file watch service", (Throwable)e);
            throw new KafkaException("Failed to create file watch service", e);
        }
    }

    protected WatchKey register(Path watchDir) {
        try {
            log.debug("Register watch for {}", (Object)watchDir);
            if (watchEventModifier == null) {
                return watchDir.register(this.watchService, WATCH_EVENT_KINDS);
            }
            return watchDir.register(this.watchService, WATCH_EVENT_KINDS, watchEventModifier);
        }
        catch (IOException e) {
            log.error("Failed to add watch for " + watchDir, (Throwable)e);
            throw new KafkaException("Failed to add watch for " + watchDir, e);
        }
    }

    private Path watchDirectory(Listener listener) {
        File file = listener.file();
        File dir = file.getParentFile();
        if (dir == null || !dir.isDirectory()) {
            throw new InvalidConfigurationException(String.format("Invalid watch file %s in dir %s", file, dir));
        }
        return dir.getAbsoluteFile().toPath();
    }

    private Watch watch(Path dir) {
        return this.watchMap.values().stream().filter(w -> w.dir.equals(dir)).findFirst().orElse(null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void poll() {
        while (this.watchService != null) {
            try {
                Set<Listener> listenersToUpdate;
                WatchService watchService = this.watchService;
                WatchKey watchKey = watchService.take();
                FileWatchService fileWatchService = this;
                synchronized (fileWatchService) {
                    Watch watch = this.watchMap.get(watchKey);
                    if (watch == null) {
                        log.debug("Ignoring watcher event since watch doesn't exist any more: {}", (Object)watchKey.watchable());
                        listenersToUpdate = Collections.emptySet();
                    } else {
                        listenersToUpdate = this.findListenersToUpdate(watch);
                        watchKey.reset();
                    }
                }
                listenersToUpdate.forEach(Listener::onUpdate);
            }
            catch (InterruptedException e) {
                log.info("Watch service was interrupted, stopping watcher.");
                break;
            }
            catch (Throwable t) {
                log.error("Failed to process events", t);
            }
        }
    }

    private synchronized Set<Listener> findListenersToUpdate(Watch watch) {
        List<WatchEvent<?>> events = watch.watchKey.pollEvents();
        HashSet<Listener> listenersToUpdate = new HashSet<Listener>();
        if (events.stream().anyMatch(e -> e.kind() == StandardWatchEventKinds.OVERFLOW)) {
            log.debug("Overflow event received, notify all listeners of {}", (Object)watch.dir);
            listenersToUpdate.addAll(watch.listeners);
            return listenersToUpdate;
        }
        Iterator<WatchEvent<?>> iterator = events.iterator();
        while (iterator.hasNext()) {
            WatchEvent<?> e2;
            WatchEvent<?> event = e2 = iterator.next();
            log.debug("Processing file event {} for {}, watch dir {}", new Object[]{e2.kind(), e2.context(), watch.dir});
            for (Listener listener : watch.listeners) {
                File listenerFile;
                Path eventFileName = ((Path)event.context()).getFileName();
                if (eventFileName.equals((listenerFile = listener.file()).toPath().getFileName())) {
                    if (event.kind() == StandardWatchEventKinds.ENTRY_DELETE) {
                        log.info("Ignoring file delete event for {}", (Object)listenerFile);
                        continue;
                    }
                    if (event.kind() == StandardWatchEventKinds.ENTRY_CREATE || event.kind() == StandardWatchEventKinds.ENTRY_MODIFY) {
                        log.debug("Notifying listener {} of event {}", (Object)listenerFile, event.kind());
                        listenersToUpdate.add(listener);
                        continue;
                    }
                    log.warn("Ignoring unexpected watcher event {}", event.kind());
                    continue;
                }
                log.debug("Not notifying listener {} of watcher event {}", (Object)listener.file(), event.kind());
            }
        }
        return listenersToUpdate;
    }

    public static void useHighSensitivity() {
        try {
            if (USES_POLLING_WATCH_SERVICE) {
                Class<?> modifierClass = Class.forName(SENSITIVITY_MODIFIER_CLASS_NAME);
                watchEventModifier = (WatchEvent.Modifier)Enum.valueOf(modifierClass, "HIGH");
            }
        }
        catch (Exception e) {
            throw new KafkaException("Could not configure high sensitivity");
        }
    }

    public static void resetSensitivity() {
        watchEventModifier = null;
    }

    static {
        WATCH_EVENT_KINDS = new WatchEvent.Kind[]{StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_MODIFY, StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.OVERFLOW};
        watchEventModifier = null;
        boolean usesPollingService = false;
        try {
            WatchService service = FileSystems.getDefault().newWatchService();
            usesPollingService = service.getClass().getName().equals(POLLING_WATCH_SERVICE_CLASS_NAME);
            service.close();
        }
        catch (Throwable t) {
            log.error("Could not determine watch service type");
        }
        USES_POLLING_WATCH_SERVICE = usesPollingService;
        MIN_WATCH_INTERVAL = usesPollingService ? Duration.ofSeconds(1L) : Duration.ZERO;
    }

    private static class Watch {
        final Path dir;
        final Set<Listener> listeners;
        final WatchKey watchKey;

        public Watch(Path dir, WatchKey watchKey) {
            this.dir = dir;
            this.watchKey = watchKey;
            this.listeners = new HashSet<Listener>();
        }

        public void add(Listener listener) {
            this.listeners.add(listener);
        }

        public void cancel() {
            this.watchKey.cancel();
        }
    }

    public static interface Listener {
        public File file();

        public void onInit();

        public void onUpdate();
    }
}

