/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.network;

import java.io.Closeable;
import java.io.EOFException;
import java.io.IOException;
import java.net.Socket;
import java.nio.ByteBuffer;
import java.nio.channels.CancelledKeyException;
import java.nio.channels.FileChannel;
import java.nio.channels.SelectionKey;
import java.security.Principal;
import java.security.cert.Certificate;
import java.util.Map;
import java.util.Optional;
import javax.net.ssl.SNIHostName;
import javax.net.ssl.SNIServerName;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLEngineResult;
import javax.net.ssl.SSLException;
import javax.net.ssl.SSLHandshakeException;
import javax.net.ssl.SSLKeyException;
import javax.net.ssl.SSLPeerUnverifiedException;
import javax.net.ssl.SSLProtocolException;
import javax.net.ssl.SSLSession;
import org.apache.kafka.common.errors.SslAuthenticationException;
import org.apache.kafka.common.network.AbstractTransportLayer;
import org.apache.kafka.common.network.ChannelMetadataRegistry;
import org.apache.kafka.common.network.CipherInformation;
import org.apache.kafka.common.network.Mode;
import org.apache.kafka.common.network.ProxyProtocolEngine;
import org.apache.kafka.common.network.SslUtil;
import org.apache.kafka.common.security.auth.KafkaPrincipal;
import org.apache.kafka.common.utils.ByteBufferUnmapper;
import org.apache.kafka.common.utils.ByteUtils;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.server.audit.AuditEventStatus;
import org.apache.kafka.server.audit.AuthenticationErrorInfo;
import org.slf4j.Logger;

public class SslTransportLayer
extends AbstractTransportLayer {
    public static final String LKC_PREFIX = "lkc-";
    private static final String TLS13 = "TLSv1.3";
    private final String channelId;
    private final SSLEngine sslEngine;
    private final ChannelMetadataRegistry metadataRegistry;
    private final Closeable closeableSslEngine;
    private final Logger log;
    private final Mode mode;
    private SSLEngineResult.HandshakeStatus handshakeStatus;
    private SSLEngineResult handshakeResult;
    private State state;
    private SslAuthenticationException handshakeException;
    private ByteBuffer netWriteBuffer;
    private ByteBuffer appReadBuffer;
    private ByteBuffer fileChannelBuffer;
    private boolean hasBytesBuffered;
    private SNIHostName sniHostName;
    private boolean shouldParseSNI;
    private boolean shouldParseLkcId;

    public static SslTransportLayer create(String channelId, SelectionKey key, SSLEngine sslEngine, ChannelMetadataRegistry metadataRegistry, Closeable closeableSslEngine, Mode mode, boolean shouldParseSniHostName, boolean shouldParseLkcId, ProxyProtocolEngine proxyProtocolEngine) {
        return SslTransportLayer.create(channelId, key, sslEngine, metadataRegistry, closeableSslEngine, mode, shouldParseSniHostName, shouldParseLkcId, proxyProtocolEngine, false);
    }

    public static SslTransportLayer create(String channelId, SelectionKey key, SSLEngine sslEngine, ChannelMetadataRegistry metadataRegistry, Closeable closeableSslEngine, Mode mode, boolean shouldParseSniHostName, boolean shouldParseLkcId, ProxyProtocolEngine proxyProtocolEngine, boolean proxyModeLocalDefault) {
        return new SslTransportLayer(channelId, key, sslEngine, metadataRegistry, closeableSslEngine, mode, shouldParseSniHostName, shouldParseLkcId, proxyProtocolEngine, proxyModeLocalDefault);
    }

    SslTransportLayer(String channelId, SelectionKey key, SSLEngine sslEngine, ChannelMetadataRegistry metadataRegistry, Mode mode, ProxyProtocolEngine proxyProtocolEngine) {
        this(channelId, key, sslEngine, metadataRegistry, null, mode, false, false, proxyProtocolEngine, false);
    }

    SslTransportLayer(String channelId, SelectionKey key, SSLEngine sslEngine, ChannelMetadataRegistry metadataRegistry, Closeable closeableSslEngine, Mode mode, boolean shouldParseSniHostName, boolean shouldParseLkcId, ProxyProtocolEngine proxyProtocolEngine, boolean proxyModeLocalDefault) {
        super(key, proxyProtocolEngine, proxyModeLocalDefault);
        this.channelId = channelId;
        this.key = key;
        this.sslEngine = sslEngine;
        this.state = State.NOT_INITIALIZED;
        this.metadataRegistry = metadataRegistry;
        this.closeableSslEngine = closeableSslEngine;
        this.mode = mode;
        this.shouldParseSNI = shouldParseSniHostName;
        this.shouldParseLkcId = shouldParseLkcId;
        LogContext logContext = new LogContext(String.format("[SslTransportLayer channelId=%s key=%s] ", channelId, key));
        this.log = logContext.logger(this.getClass());
    }

    protected void startHandshake() throws IOException {
        if (this.state != State.NOT_INITIALIZED) {
            throw new IllegalStateException("startHandshake() can only be called once, state " + (Object)((Object)this.state));
        }
        if (this.netReadBuffer == null) {
            this.netReadBuffer = ByteBuffer.allocate(this.netReadBufferSize());
        }
        this.netReadBuffer.limit(this.netReadBuffer.position());
        if (this.netWriteBuffer == null) {
            this.netWriteBuffer = ByteBuffer.allocate(this.netWriteBufferSize());
            this.netWriteBuffer.limit(0);
        }
        this.appReadBuffer = ByteBuffer.allocate(this.applicationBufferSize());
        this.state = State.HANDSHAKE;
        this.sslEngine.beginHandshake();
        this.handshakeStatus = this.sslEngine.getHandshakeStatus();
    }

    @Override
    public boolean ready() {
        return this.proxyProtocolReady() && (this.state == State.POST_HANDSHAKE || this.state == State.READY);
    }

    @Override
    public void selectionKey(SelectionKey newKey) {
        this.key = newKey;
    }

    @Override
    public boolean isOpen() {
        return this.socketChannel.isOpen();
    }

    @Override
    public boolean isConnected() {
        return this.socketChannel.isConnected();
    }

    @Override
    public void close() throws IOException {
        State prevState = this.state;
        if (this.state == State.CLOSING) {
            return;
        }
        this.state = State.CLOSING;
        this.sslEngine.closeOutbound();
        try {
            if (prevState != State.NOT_INITIALIZED && this.isConnected()) {
                if (!this.flush(this.netWriteBuffer)) {
                    throw new IOException("Remaining data in the network buffer, can't send SSL close message.");
                }
                this.netWriteBuffer.clear();
                SSLEngineResult wrapResult = this.sslEngine.wrap(ByteUtils.EMPTY_BUF, this.netWriteBuffer);
                if (wrapResult.getStatus() != SSLEngineResult.Status.CLOSED) {
                    throw new IOException("Unexpected status returned by SSLEngine.wrap, expected CLOSED, received " + (Object)((Object)wrapResult.getStatus()) + ". Will not send close message to peer.");
                }
                this.netWriteBuffer.flip();
                this.flush(this.netWriteBuffer);
            }
        }
        catch (IOException ie) {
            this.log.debug("Failed to send SSL Close message", (Throwable)ie);
        }
        finally {
            try {
                this.sslEngine.closeInbound();
            }
            catch (SSLException e) {
                this.log.debug("SSLEngine.closeInBound() raised an exception.", (Throwable)e);
            }
            this.socketChannel.socket().close();
            this.socketChannel.close();
            this.netReadBuffer = null;
            this.netWriteBuffer = null;
            this.appReadBuffer = null;
            if (this.fileChannelBuffer != null) {
                ByteBufferUnmapper.unmap("fileChannelBuffer", this.fileChannelBuffer);
                this.fileChannelBuffer = null;
            }
            if (this.closeableSslEngine != null) {
                this.closeableSslEngine.close();
            }
        }
    }

    @Override
    public boolean hasPendingWrites() {
        return this.netWriteBuffer.hasRemaining();
    }

    public Optional<SNIHostName> sniHostName() {
        return Optional.ofNullable(this.sniHostName);
    }

    protected boolean flush(ByteBuffer buf) throws IOException {
        int remaining = buf.remaining();
        if (remaining > 0) {
            boolean completed;
            int written = this.socketChannel.write(buf);
            boolean bl = completed = written >= remaining;
            if (!completed && this.log.isTraceEnabled()) {
                this.log.trace("Flushed {} bytes out of {} to socket channel", (Object)written, (Object)remaining);
            }
            return completed;
        }
        return true;
    }

    @Override
    public void handshake() throws IOException {
        if (this.state == State.NOT_INITIALIZED) {
            try {
                this.startHandshake();
            }
            catch (SSLException e) {
                this.maybeProcessHandshakeFailure(e, false, null);
            }
        }
        if (this.ready()) {
            throw this.renegotiationException();
        }
        if (this.state == State.CLOSING) {
            throw this.closingException();
        }
        int read = 0;
        boolean readable = this.key.isReadable();
        try {
            if (readable) {
                read = this.readFromSocketChannel();
            }
            this.doHandshake();
            if (this.ready()) {
                this.updateBytesBuffered(true);
            }
        }
        catch (SSLException e) {
            this.maybeProcessHandshakeFailure(e, true, null);
        }
        catch (IOException e) {
            this.maybeThrowSslAuthenticationException();
            try {
                do {
                    this.log.trace("Process any available bytes from peer, netReadBuffer {} netWriterBuffer {} handshakeStatus {} readable? {}", new Object[]{this.netReadBuffer, this.netWriteBuffer, this.handshakeStatus, readable});
                    this.handshakeWrapAfterFailure(false);
                    this.handshakeUnwrap(false, true);
                } while (readable && this.readFromSocketChannel() > 0);
            }
            catch (SSLException e1) {
                this.maybeProcessHandshakeFailure(e1, false, e);
            }
            throw e;
        }
        if (read == -1) {
            this.maybeThrowSslAuthenticationException();
            throw new EOFException("EOF during handshake, handshake status is " + (Object)((Object)this.handshakeStatus));
        }
    }

    private void parseSNIHostNameIfPresent() {
        Map<Integer, SNIServerName> sniServerNameMap;
        try {
            sniServerNameMap = SslUtil.parseSniServerName(this.netReadBuffer);
        }
        catch (IOException e) {
            if (this.log.isTraceEnabled()) {
                this.log.trace("Encountered IOException while parsing SNI Host name", (Throwable)e);
            }
            return;
        }
        if (sniServerNameMap.containsKey(0)) {
            SNIServerName sniServerName = sniServerNameMap.get(0);
            if (sniServerName instanceof SNIHostName) {
                this.sniHostName = (SNIHostName)sniServerName;
            } else {
                this.log.warn("Server name: {} not of the type host name", (Object)sniServerName);
            }
        } else {
            this.log.warn("Server name map: {} doesn't contain hostname type", sniServerNameMap);
        }
    }

    private Optional<String> suppliedClusterId() {
        String[] parts;
        String hostName;
        if (this.sniHostName != null && (hostName = this.sniHostName.getAsciiName()).startsWith(LKC_PREFIX) && (parts = hostName.split("-", 3)).length > 2) {
            return Optional.of(parts[0] + "-" + parts[1]);
        }
        return Optional.empty();
    }

    private void doHandshake() throws IOException {
        boolean read = this.key.isReadable();
        boolean write = this.key.isWritable();
        this.handshakeStatus = this.sslEngine.getHandshakeStatus();
        if (!this.flush(this.netWriteBuffer)) {
            this.key.interestOps(this.key.interestOps() | 4);
            return;
        }
        this.maybeThrowSslAuthenticationException();
        switch (this.handshakeStatus) {
            case NEED_TASK: {
                this.log.trace("SSLHandshake NEED_TASK channelId {}, appReadBuffer pos {}, netReadBuffer pos {}, netWriteBuffer pos {}", new Object[]{this.channelId, this.appReadBuffer.position(), this.netReadBuffer.position(), this.netWriteBuffer.position()});
                this.handshakeStatus = this.runDelegatedTasks();
                break;
            }
            case NEED_WRAP: {
                this.log.trace("SSLHandshake NEED_WRAP channelId {}, appReadBuffer pos {}, netReadBuffer pos {}, netWriteBuffer pos {}", new Object[]{this.channelId, this.appReadBuffer.position(), this.netReadBuffer.position(), this.netWriteBuffer.position()});
                this.handshakeResult = this.handshakeWrap(write);
                if (this.handshakeResult.getStatus() == SSLEngineResult.Status.BUFFER_OVERFLOW) {
                    int currentNetWriteBufferSize = this.netWriteBufferSize();
                    this.netWriteBuffer.compact();
                    this.netWriteBuffer = Utils.ensureCapacity(this.netWriteBuffer, currentNetWriteBufferSize);
                    this.netWriteBuffer.flip();
                    if (this.netWriteBuffer.limit() >= currentNetWriteBufferSize) {
                        throw new IllegalStateException("Buffer overflow when available data size (" + this.netWriteBuffer.limit() + ") >= network buffer size (" + currentNetWriteBufferSize + ")");
                    }
                } else {
                    if (this.handshakeResult.getStatus() == SSLEngineResult.Status.BUFFER_UNDERFLOW) {
                        throw new IllegalStateException("Should not have received BUFFER_UNDERFLOW during handshake WRAP.");
                    }
                    if (this.handshakeResult.getStatus() == SSLEngineResult.Status.CLOSED) {
                        throw new EOFException();
                    }
                }
                this.log.trace("SSLHandshake NEED_WRAP channelId {}, handshakeResult {}, appReadBuffer pos {}, netReadBuffer pos {}, netWriteBuffer pos {}", new Object[]{this.channelId, this.handshakeResult, this.appReadBuffer.position(), this.netReadBuffer.position(), this.netWriteBuffer.position()});
                if (this.handshakeStatus != SSLEngineResult.HandshakeStatus.NEED_UNWRAP || !this.flush(this.netWriteBuffer)) {
                    this.key.interestOps(this.key.interestOps() | 4);
                    break;
                }
            }
            case NEED_UNWRAP: {
                this.log.trace("SSLHandshake NEED_UNWRAP channelId {}, appReadBuffer pos {}, netReadBuffer pos {}, netWriteBuffer pos {}", new Object[]{this.channelId, this.appReadBuffer.position(), this.netReadBuffer.position(), this.netWriteBuffer.position()});
                do {
                    this.handshakeResult = this.handshakeUnwrap(read, false);
                    if (this.handshakeResult.getStatus() != SSLEngineResult.Status.BUFFER_OVERFLOW) continue;
                    int currentAppBufferSize = this.applicationBufferSize();
                    this.appReadBuffer = Utils.ensureCapacity(this.appReadBuffer, currentAppBufferSize);
                    if (this.appReadBuffer.position() <= currentAppBufferSize) continue;
                    throw new IllegalStateException("Buffer underflow when available data size (" + this.appReadBuffer.position() + ") > packet buffer size (" + currentAppBufferSize + ")");
                } while (this.handshakeResult.getStatus() == SSLEngineResult.Status.BUFFER_OVERFLOW);
                if (this.handshakeResult.getStatus() == SSLEngineResult.Status.BUFFER_UNDERFLOW) {
                    int currentNetReadBufferSize = this.netReadBufferSize();
                    this.netReadBuffer = Utils.ensureCapacity(this.netReadBuffer, currentNetReadBufferSize);
                    if (this.netReadBuffer.position() >= currentNetReadBufferSize) {
                        throw new IllegalStateException("Buffer underflow when there is available data");
                    }
                } else if (this.handshakeResult.getStatus() == SSLEngineResult.Status.CLOSED) {
                    throw new EOFException("SSL handshake status CLOSED during handshake UNWRAP");
                }
                this.log.trace("SSLHandshake NEED_UNWRAP channelId {}, handshakeResult {}, appReadBuffer pos {}, netReadBuffer pos {}, netWriteBuffer pos {}", new Object[]{this.channelId, this.handshakeResult, this.appReadBuffer.position(), this.netReadBuffer.position(), this.netWriteBuffer.position()});
                if (this.handshakeStatus != SSLEngineResult.HandshakeStatus.FINISHED) {
                    if (this.handshakeStatus == SSLEngineResult.HandshakeStatus.NEED_WRAP) {
                        this.key.interestOps(this.key.interestOps() | 4);
                        break;
                    }
                    if (this.handshakeStatus != SSLEngineResult.HandshakeStatus.NEED_UNWRAP) break;
                    this.key.interestOps(this.key.interestOps() & 0xFFFFFFFB);
                    break;
                }
            }
            case FINISHED: {
                this.handshakeFinished();
                break;
            }
            case NOT_HANDSHAKING: {
                this.handshakeFinished();
                break;
            }
            default: {
                throw new IllegalStateException(String.format("Unexpected status [%s]", new Object[]{this.handshakeStatus}));
            }
        }
    }

    private SSLHandshakeException renegotiationException() {
        return new SSLHandshakeException("Renegotiation is not supported");
    }

    private IllegalStateException closingException() {
        throw new IllegalStateException("Channel is in closing state");
    }

    private SSLEngineResult.HandshakeStatus runDelegatedTasks() {
        Runnable task;
        while ((task = this.delegatedTask()) != null) {
            task.run();
        }
        return this.sslEngine.getHandshakeStatus();
    }

    private void handshakeFinished() throws IOException {
        if (this.handshakeResult.getHandshakeStatus() == SSLEngineResult.HandshakeStatus.FINISHED) {
            if (this.netWriteBuffer.hasRemaining()) {
                this.key.interestOps(this.key.interestOps() | 4);
            } else {
                SSLSession session = this.sslEngine.getSession();
                this.state = session.getProtocol().equals(TLS13) ? State.POST_HANDSHAKE : State.READY;
                this.key.interestOps(this.key.interestOps() & 0xFFFFFFFB);
                this.log.debug("SSL handshake completed successfully with peerHost '{}' peerPort {} peerPrincipal '{}' protocol '{}' cipherSuite '{}'", new Object[]{session.getPeerHost(), session.getPeerPort(), this.peerPrincipal(), session.getProtocol(), session.getCipherSuite()});
                this.metadataRegistry.registerCipherInformation(new CipherInformation(session.getCipherSuite(), session.getProtocol()));
            }
        } else {
            throw new IOException("NOT_HANDSHAKING during handshake");
        }
        this.log.trace("SSLHandshake FINISHED channelId {}, appReadBuffer pos {}, netReadBuffer pos {}, netWriteBuffer pos {} ", new Object[]{this.channelId, this.appReadBuffer.position(), this.netReadBuffer.position(), this.netWriteBuffer.position()});
    }

    private SSLEngineResult handshakeWrap(boolean doWrite) throws IOException {
        SSLEngineResult result;
        this.log.trace("SSLHandshake handshakeWrap {}", (Object)this.channelId);
        if (this.netWriteBuffer.hasRemaining()) {
            throw new IllegalStateException("handshakeWrap called with netWriteBuffer not empty");
        }
        this.netWriteBuffer.clear();
        try {
            result = this.sslEngine.wrap(ByteUtils.EMPTY_BUF, this.netWriteBuffer);
        }
        finally {
            this.netWriteBuffer.flip();
        }
        this.handshakeStatus = result.getHandshakeStatus();
        if (result.getStatus() == SSLEngineResult.Status.OK && result.getHandshakeStatus() == SSLEngineResult.HandshakeStatus.NEED_TASK) {
            this.handshakeStatus = this.runDelegatedTasks();
        }
        if (doWrite) {
            this.flush(this.netWriteBuffer);
        }
        return result;
    }

    SSLEngineResult handshakeUnwrap(boolean doRead, boolean ignoreHandshakeStatus) throws IOException {
        SSLEngineResult result;
        boolean cont;
        this.log.trace("SSLHandshake handshakeUnwrap {}", (Object)this.channelId);
        int read = 0;
        if (doRead) {
            read = this.readFromSocketChannel();
        }
        if (this.shouldParseSNI && this.mode == Mode.SERVER && this.netReadBuffer.position() != 0) {
            int sniPos = this.netReadBuffer.position();
            this.netReadBuffer.position(0);
            this.parseSNIHostNameIfPresent();
            this.netReadBuffer.position(sniPos);
            this.shouldParseSNI = false;
        }
        do {
            int position = this.netReadBuffer.position();
            this.netReadBuffer.flip();
            result = this.sslEngine.unwrap(this.netReadBuffer, this.appReadBuffer);
            this.netReadBuffer.compact();
            this.handshakeStatus = result.getHandshakeStatus();
            if (result.getStatus() == SSLEngineResult.Status.OK && result.getHandshakeStatus() == SSLEngineResult.HandshakeStatus.NEED_TASK) {
                this.handshakeStatus = this.runDelegatedTasks();
            }
            cont = result.getStatus() == SSLEngineResult.Status.OK && this.handshakeStatus == SSLEngineResult.HandshakeStatus.NEED_UNWRAP || ignoreHandshakeStatus && this.netReadBuffer.position() != position;
            this.log.trace("SSLHandshake handshakeUnwrap: handshakeStatus {} status {}", (Object)this.handshakeStatus, (Object)result.getStatus());
        } while (cont);
        if (read == -1) {
            throw new EOFException("EOF during handshake, handshake status is " + (Object)((Object)this.handshakeStatus));
        }
        return result;
    }

    @Override
    public int read(ByteBuffer dst) throws IOException {
        if (this.state == State.CLOSING) {
            return -1;
        }
        if (!this.ready()) {
            return 0;
        }
        int read = 0;
        if (this.appReadBuffer.position() > 0) {
            read = this.readFromAppBuffer(dst);
        }
        boolean readFromNetwork = false;
        boolean isClosed = false;
        while (dst.remaining() > 0) {
            int netread = 0;
            this.netReadBuffer = Utils.ensureCapacity(this.netReadBuffer, this.netReadBufferSize());
            if (this.netReadBuffer.remaining() > 0 && (netread = this.readFromSocketChannel()) > 0) {
                readFromNetwork = true;
            }
            while (this.netReadBuffer.position() > 0) {
                SSLEngineResult unwrapResult;
                this.netReadBuffer.flip();
                try {
                    unwrapResult = this.sslEngine.unwrap(this.netReadBuffer, this.appReadBuffer);
                    if (this.state == State.POST_HANDSHAKE && this.appReadBuffer.position() != 0) {
                        this.state = State.READY;
                    }
                }
                catch (SSLException e) {
                    if (this.state == State.POST_HANDSHAKE) {
                        this.state = State.HANDSHAKE_FAILED;
                        String errorMessage = "Failed to process post-handshake messages, SNI host name: " + (this.sniHostName == null ? "empty" : this.sniHostName.getAsciiName());
                        AuthenticationErrorInfo errorInfo = new AuthenticationErrorInfo(AuditEventStatus.SSL_HANDSHAKE_FAILED, "", "", this.suppliedClusterId().orElse(""));
                        throw new SslAuthenticationException(errorMessage, errorInfo);
                    }
                    throw e;
                }
                this.netReadBuffer.compact();
                if (unwrapResult.getHandshakeStatus() != SSLEngineResult.HandshakeStatus.NOT_HANDSHAKING && unwrapResult.getHandshakeStatus() != SSLEngineResult.HandshakeStatus.FINISHED && unwrapResult.getStatus() == SSLEngineResult.Status.OK && !this.sslEngine.getSession().getProtocol().equals(TLS13)) {
                    this.log.error("Renegotiation requested, but it is not supported, channelId {}, appReadBuffer pos {}, netReadBuffer pos {}, netWriteBuffer pos {} handshakeStatus {}", new Object[]{this.channelId, this.appReadBuffer.position(), this.netReadBuffer.position(), this.netWriteBuffer.position(), unwrapResult.getHandshakeStatus()});
                    throw this.renegotiationException();
                }
                if (unwrapResult.getStatus() == SSLEngineResult.Status.OK) {
                    read += this.readFromAppBuffer(dst);
                    continue;
                }
                if (unwrapResult.getStatus() == SSLEngineResult.Status.BUFFER_OVERFLOW) {
                    int currentApplicationBufferSize = this.applicationBufferSize();
                    this.appReadBuffer = Utils.ensureCapacity(this.appReadBuffer, currentApplicationBufferSize);
                    if (this.appReadBuffer.position() >= currentApplicationBufferSize) {
                        throw new IllegalStateException("Buffer overflow when available data size (" + this.appReadBuffer.position() + ") >= application buffer size (" + currentApplicationBufferSize + ")");
                    }
                    if (!dst.hasRemaining()) break;
                    read += this.readFromAppBuffer(dst);
                    continue;
                }
                if (unwrapResult.getStatus() == SSLEngineResult.Status.BUFFER_UNDERFLOW) {
                    int currentNetReadBufferSize = this.netReadBufferSize();
                    this.netReadBuffer = Utils.ensureCapacity(this.netReadBuffer, currentNetReadBufferSize);
                    if (this.netReadBuffer.position() < currentNetReadBufferSize) break;
                    throw new IllegalStateException("Buffer underflow when available data size (" + this.netReadBuffer.position() + ") > packet buffer size (" + currentNetReadBufferSize + ")");
                }
                if (unwrapResult.getStatus() != SSLEngineResult.Status.CLOSED) continue;
                if (this.appReadBuffer.position() == 0 && read == 0) {
                    throw new EOFException();
                }
                isClosed = true;
                break;
            }
            if (read == 0 && netread < 0) {
                throw new EOFException("EOF during read");
            }
            if (netread > 0 && !isClosed) continue;
            break;
        }
        this.updateBytesBuffered(readFromNetwork || read > 0);
        return read;
    }

    @Override
    public long read(ByteBuffer[] dsts) throws IOException {
        return this.read(dsts, 0, dsts.length);
    }

    @Override
    public long read(ByteBuffer[] dsts, int offset, int length) throws IOException {
        if (offset < 0 || length < 0 || offset > dsts.length - length) {
            throw new IndexOutOfBoundsException();
        }
        int totalRead = 0;
        int i = offset;
        while (i < offset + length) {
            if (dsts[i].hasRemaining()) {
                int read = this.read(dsts[i]);
                if (read <= 0) break;
                totalRead += read;
            }
            if (dsts[i].hasRemaining()) continue;
            ++i;
        }
        return totalRead;
    }

    @Override
    public int write(ByteBuffer src) throws IOException {
        return (int)this.write(new ByteBuffer[]{src});
    }

    @Override
    public long write(ByteBuffer[] srcs, int offset, int length) throws IOException {
        if (offset < 0 || length < 0 || offset > srcs.length - length) {
            throw new IndexOutOfBoundsException();
        }
        if (this.state == State.CLOSING) {
            throw this.closingException();
        }
        if (!this.ready()) {
            return 0L;
        }
        int written = 0;
        int bufferOffset = offset;
        int endOffset = offset + length;
        while (this.flush(this.netWriteBuffer) && bufferOffset < endOffset) {
            this.netWriteBuffer.clear();
            SSLEngineResult wrapResult = this.sslEngine.wrap(srcs, bufferOffset, srcs.length - bufferOffset, this.netWriteBuffer);
            if (!srcs[bufferOffset].hasRemaining()) {
                ++bufferOffset;
            }
            this.netWriteBuffer.flip();
            if (wrapResult.getHandshakeStatus() != SSLEngineResult.HandshakeStatus.NOT_HANDSHAKING && wrapResult.getStatus() == SSLEngineResult.Status.OK && !this.sslEngine.getSession().getProtocol().equals(TLS13)) {
                throw this.renegotiationException();
            }
            if (wrapResult.getStatus() == SSLEngineResult.Status.OK) {
                written += wrapResult.bytesConsumed();
                continue;
            }
            if (wrapResult.getStatus() == SSLEngineResult.Status.BUFFER_OVERFLOW) {
                this.netWriteBuffer = Utils.ensureCapacity(this.netWriteBuffer, this.netWriteBufferSize());
                this.netWriteBuffer.position(this.netWriteBuffer.limit());
                continue;
            }
            if (wrapResult.getStatus() == SSLEngineResult.Status.BUFFER_UNDERFLOW) {
                throw new IllegalStateException("SSL BUFFER_UNDERFLOW during write");
            }
            if (wrapResult.getStatus() != SSLEngineResult.Status.CLOSED) continue;
            throw new EOFException();
        }
        return written;
    }

    @Override
    public long write(ByteBuffer[] srcs) throws IOException {
        return this.write(srcs, 0, srcs.length);
    }

    @Override
    public Principal peerPrincipal() {
        try {
            return this.sslEngine.getSession().getPeerPrincipal();
        }
        catch (SSLPeerUnverifiedException se) {
            this.log.debug("SSL peer is not authenticated, returning ANONYMOUS instead");
            return KafkaPrincipal.ANONYMOUS;
        }
    }

    public SSLSession sslSession() throws IllegalStateException {
        return this.sslEngine.getSession();
    }

    @Override
    public void addInterestOps(int ops) {
        if (!this.key.isValid()) {
            throw new CancelledKeyException();
        }
        if (!this.ready()) {
            throw new IllegalStateException("handshake is not completed");
        }
        this.key.interestOps(this.key.interestOps() | ops);
    }

    @Override
    public void removeInterestOps(int ops) {
        if (!this.key.isValid()) {
            throw new CancelledKeyException();
        }
        if (!this.ready()) {
            throw new IllegalStateException("handshake is not completed");
        }
        this.key.interestOps(this.key.interestOps() & ~ops);
    }

    protected Runnable delegatedTask() {
        return this.sslEngine.getDelegatedTask();
    }

    private int readFromAppBuffer(ByteBuffer dst) {
        this.appReadBuffer.flip();
        int remaining = Math.min(this.appReadBuffer.remaining(), dst.remaining());
        if (remaining > 0) {
            int limit = this.appReadBuffer.limit();
            this.appReadBuffer.limit(this.appReadBuffer.position() + remaining);
            dst.put(this.appReadBuffer);
            this.appReadBuffer.limit(limit);
        }
        this.appReadBuffer.compact();
        return remaining;
    }

    protected int netReadBufferSize() {
        return this.sslEngine.getSession().getPacketBufferSize();
    }

    protected int netWriteBufferSize() {
        return this.sslEngine.getSession().getPacketBufferSize();
    }

    protected int applicationBufferSize() {
        return this.sslEngine.getSession().getApplicationBufferSize();
    }

    protected ByteBuffer netReadBuffer() {
        return this.netReadBuffer;
    }

    protected ByteBuffer appReadBuffer() {
        return this.appReadBuffer;
    }

    private void handshakeFailure(SSLException sslException, boolean flush) {
        this.log.debug("SSL Handshake failed", (Throwable)sslException);
        this.sslEngine.closeOutbound();
        try {
            this.sslEngine.closeInbound();
        }
        catch (SSLException e) {
            this.log.debug("SSLEngine.closeInBound() raised an exception.", (Throwable)e);
        }
        this.state = State.HANDSHAKE_FAILED;
        String errorMessage = "SSL handshake failed";
        if (this.sniHostName != null) {
            errorMessage = errorMessage + ", SNI host name: " + this.sniHostName.getAsciiName();
        }
        AuthenticationErrorInfo errorInfo = new AuthenticationErrorInfo(AuditEventStatus.SSL_HANDSHAKE_FAILED, "", "", this.suppliedClusterId().orElse(""));
        this.handshakeException = new SslAuthenticationException(errorMessage, sslException, errorInfo);
        if (!flush || this.handshakeWrapAfterFailure(flush)) {
            throw this.handshakeException;
        }
        this.log.debug("Delay propagation of handshake exception till {} bytes remaining are flushed", (Object)this.netWriteBuffer.remaining());
    }

    private void maybeProcessHandshakeFailure(SSLException sslException, boolean flush, IOException ioException) throws IOException {
        if (!(sslException instanceof SSLHandshakeException || sslException instanceof SSLProtocolException || sslException instanceof SSLPeerUnverifiedException || sslException instanceof SSLKeyException || sslException.getMessage().contains("Unrecognized SSL message") || sslException.getMessage().contains("Received fatal alert: ") || sslException.getMessage().contains("not an SSL/TLS record"))) {
            if (ioException == null) {
                throw sslException;
            }
            this.log.debug("SSLException while unwrapping data after IOException, original IOException will be propagated", (Throwable)sslException);
            throw ioException;
        }
        this.handshakeFailure(sslException, flush);
    }

    private void maybeThrowSslAuthenticationException() {
        if (this.handshakeException != null) {
            throw this.handshakeException;
        }
    }

    private boolean handshakeWrapAfterFailure(boolean doWrite) {
        try {
            this.log.trace("handshakeWrapAfterFailure status {} doWrite {}", (Object)this.handshakeStatus, (Object)doWrite);
            while (this.handshakeStatus == SSLEngineResult.HandshakeStatus.NEED_WRAP && (!doWrite || this.flush(this.netWriteBuffer))) {
                if (!doWrite) {
                    this.clearWriteBuffer();
                }
                this.handshakeWrap(doWrite);
            }
        }
        catch (Exception e) {
            this.log.debug("Failed to wrap and flush all bytes before closing channel", (Throwable)e);
            this.clearWriteBuffer();
        }
        if (!doWrite) {
            this.clearWriteBuffer();
        }
        return !this.netWriteBuffer.hasRemaining();
    }

    private void clearWriteBuffer() {
        if (this.netWriteBuffer.hasRemaining()) {
            this.log.debug("Discarding write buffer {} since peer has disconnected", (Object)this.netWriteBuffer);
        }
        this.netWriteBuffer.position(0);
        this.netWriteBuffer.limit(0);
    }

    @Override
    public boolean isMute() {
        return this.key.isValid() && (this.key.interestOps() & 1) == 0;
    }

    @Override
    public boolean hasBytesBuffered() {
        return this.hasBytesBuffered;
    }

    private void updateBytesBuffered(boolean madeProgress) {
        this.hasBytesBuffered = madeProgress ? this.netReadBuffer.position() != 0 || this.appReadBuffer != null && this.appReadBuffer.position() != 0 : false;
    }

    @Override
    public long transferFrom(FileChannel fileChannel, long position, long count) throws IOException {
        if (this.state == State.CLOSING) {
            throw this.closingException();
        }
        if (this.state != State.READY) {
            return 0L;
        }
        if (!this.flush(this.netWriteBuffer)) {
            return 0L;
        }
        long channelSize = fileChannel.size();
        if (position > channelSize) {
            return 0L;
        }
        int totalBytesToWrite = (int)Math.min(Math.min(count, channelSize - position), Integer.MAX_VALUE);
        if (this.fileChannelBuffer == null) {
            int transferSize = 32768;
            this.fileChannelBuffer = ByteBuffer.allocateDirect(transferSize);
            this.fileChannelBuffer.position(this.fileChannelBuffer.limit());
        }
        int totalBytesWritten = 0;
        long pos = position;
        try {
            while (totalBytesWritten < totalBytesToWrite) {
                if (!this.fileChannelBuffer.hasRemaining()) {
                    int bytesRead;
                    this.fileChannelBuffer.clear();
                    int bytesRemaining = totalBytesToWrite - totalBytesWritten;
                    if (bytesRemaining < this.fileChannelBuffer.limit()) {
                        this.fileChannelBuffer.limit(bytesRemaining);
                    }
                    if ((bytesRead = fileChannel.read(this.fileChannelBuffer, pos)) <= 0) break;
                    this.fileChannelBuffer.flip();
                }
                int networkBytesWritten = this.write(this.fileChannelBuffer);
                totalBytesWritten += networkBytesWritten;
                if (this.fileChannelBuffer.hasRemaining()) break;
                pos += (long)networkBytesWritten;
            }
            return totalBytesWritten;
        }
        catch (IOException e) {
            if (totalBytesWritten > 0) {
                return totalBytesWritten;
            }
            throw e;
        }
    }

    @Override
    public void handleProxyProtocol() throws IOException {
        if (this.mode == Mode.SERVER) {
            int read;
            if (this.netReadBuffer == null) {
                this.netReadBuffer = ByteBuffer.allocate(this.netReadBufferSize());
            }
            this.updateBytesBuffered((read = this.maybeReadAndProcessProxyHeaders()) > 0);
        } else if (this.mode == Mode.CLIENT && this.proxyProtocolEngine.includeOutgoingProxyHeader()) {
            Socket socket = this.socketChannel.socket();
            byte[] header = this.proxyProtocolEngine.emitHeaders(socket.getInetAddress(), socket.getPort());
            if (this.netWriteBuffer == null) {
                this.netWriteBuffer = ByteBuffer.allocate(this.netWriteBufferSize());
            }
            this.netWriteBuffer.put(header);
            this.netWriteBuffer.flip();
        }
    }

    public void forceClientAuthRequired() {
        this.log.debug("Forcing client authentication as Proxy protocol2 header indicates traffic should be validated using mTLS");
        this.sslEngine.setNeedClientAuth(true);
    }

    public boolean getNeedClientAuth() {
        return this.sslEngine.getNeedClientAuth();
    }

    public boolean getWantClientAuth() {
        return this.sslEngine.getWantClientAuth();
    }

    public boolean hasPeerCertificate() {
        try {
            Certificate[] peerCerts = this.sslEngine.getSession().getPeerCertificates();
            return peerCerts != null && peerCerts.length > 0;
        }
        catch (Exception e) {
            this.log.trace("Encountered exception when getting peer certificates", (Throwable)e);
            return false;
        }
    }

    private static enum State {
        NOT_INITIALIZED,
        HANDSHAKE,
        HANDSHAKE_FAILED,
        POST_HANDSHAKE,
        READY,
        CLOSING;

    }
}

