/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.controller;

import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.TreeSet;
import java.util.function.Function;
import java.util.function.Supplier;
import org.apache.kafka.common.message.BrokerHeartbeatRequestData;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.controller.BrokerControlState;
import org.apache.kafka.controller.BrokerControlStates;
import org.apache.kafka.metadata.placement.UsableBroker;
import org.slf4j.Logger;

public class BrokerHeartbeatManager {
    private final Logger log;
    private final Time time;
    private final long sessionTimeoutNs;
    private final HashMap<Integer, BrokerHeartbeatState> brokers;
    private final BrokerHeartbeatStateList unfenced;
    private final TreeSet<BrokerHeartbeatState> active;

    BrokerHeartbeatManager(LogContext logContext, Time time, long sessionTimeoutNs) {
        this.log = logContext.logger(BrokerHeartbeatManager.class);
        this.time = time;
        this.sessionTimeoutNs = sessionTimeoutNs;
        this.brokers = new HashMap();
        this.unfenced = new BrokerHeartbeatStateList();
        this.active = new TreeSet<BrokerHeartbeatState>(MetadataOffsetComparator.INSTANCE);
    }

    Time time() {
        return this.time;
    }

    BrokerHeartbeatStateList unfenced() {
        return this.unfenced;
    }

    Collection<BrokerHeartbeatState> brokers() {
        return this.brokers.values();
    }

    void fence(int brokerId) {
        BrokerHeartbeatState broker = this.brokers.get(brokerId);
        if (broker != null) {
            this.untrack(broker);
        }
    }

    void remove(int brokerId) {
        BrokerHeartbeatState broker = this.brokers.remove(brokerId);
        if (broker != null) {
            this.untrack(broker);
        }
    }

    private void untrack(BrokerHeartbeatState broker) {
        if (!broker.fenced()) {
            this.unfenced.remove(broker);
            if (!broker.shuttingDown()) {
                this.active.remove(broker);
            }
        }
    }

    boolean hasValidSession(int brokerId) {
        BrokerHeartbeatState broker = this.brokers.get(brokerId);
        if (broker == null) {
            return false;
        }
        return this.hasValidSession(broker);
    }

    private boolean hasValidSession(BrokerHeartbeatState broker) {
        if (broker.fenced()) {
            return false;
        }
        return broker.lastContactNs + this.sessionTimeoutNs >= this.time.nanoseconds();
    }

    void register(int brokerId, boolean fenced) {
        BrokerHeartbeatState broker = this.brokers.get(brokerId);
        if (broker == null) {
            this.touch(brokerId, fenced, -1L);
        } else if (broker.fenced() != fenced) {
            this.touch(brokerId, fenced, broker.metadataOffset);
        }
    }

    void touch(int brokerId, boolean fenced, long metadataOffset) {
        BrokerHeartbeatState broker = this.brokers.get(brokerId);
        if (broker == null) {
            broker = new BrokerHeartbeatState(brokerId);
            this.brokers.put(brokerId, broker);
        } else {
            this.untrack(broker);
        }
        broker.lastContactNs = this.time.nanoseconds();
        broker.metadataOffset = metadataOffset;
        if (fenced) {
            broker.controlledShutDownOffset = -1L;
        } else {
            this.unfenced.add(broker);
            if (!broker.shuttingDown()) {
                this.active.add(broker);
            }
        }
    }

    long lowestActiveOffset() {
        Iterator<BrokerHeartbeatState> iterator = this.active.iterator();
        if (!iterator.hasNext()) {
            return Long.MAX_VALUE;
        }
        BrokerHeartbeatState first = iterator.next();
        return first.metadataOffset;
    }

    void updateControlledShutdownOffset(int brokerId, long controlledShutDownOffset) {
        BrokerHeartbeatState broker = this.brokers.get(brokerId);
        if (broker == null) {
            throw new RuntimeException("Unable to locate broker " + brokerId);
        }
        if (broker.fenced()) {
            throw new RuntimeException("Fenced brokers cannot enter controlled shutdown.");
        }
        this.active.remove(broker);
        broker.controlledShutDownOffset = controlledShutDownOffset;
        this.log.debug("Updated the controlled shutdown offset for broker {} to {}.", (Object)brokerId, (Object)controlledShutDownOffset);
    }

    long nextCheckTimeNs() {
        BrokerHeartbeatState broker = this.unfenced.first();
        if (broker == null) {
            return Long.MAX_VALUE;
        }
        return broker.lastContactNs + this.sessionTimeoutNs;
    }

    Optional<Integer> findOneStaleBroker() {
        BrokerHeartbeatState broker;
        BrokerHeartbeatStateIterator iterator = this.unfenced.iterator();
        if (iterator.hasNext() && !this.hasValidSession(broker = iterator.next())) {
            return Optional.of(broker.id);
        }
        return Optional.empty();
    }

    Iterator<UsableBroker> usableBrokers(Function<Integer, Optional<String>> idToRack) {
        return new UsableBrokerIterator(this.brokers.values().iterator(), idToRack);
    }

    BrokerControlState currentBrokerState(BrokerHeartbeatState broker) {
        if (broker.shuttingDown()) {
            return BrokerControlState.CONTROLLED_SHUTDOWN;
        }
        if (broker.fenced()) {
            return BrokerControlState.FENCED;
        }
        return BrokerControlState.UNFENCED;
    }

    BrokerControlStates calculateNextBrokerState(int brokerId, BrokerHeartbeatRequestData request, long registerBrokerRecordOffset, Supplier<Boolean> hasLeaderships) {
        BrokerHeartbeatState broker = this.brokers.getOrDefault(brokerId, new BrokerHeartbeatState(brokerId));
        BrokerControlState currentState = this.currentBrokerState(broker);
        switch (currentState) {
            case FENCED: {
                if (request.wantShutDown()) {
                    this.log.info("Fenced broker {} has requested and been granted an immediate shutdown.", (Object)brokerId);
                    return new BrokerControlStates(currentState, BrokerControlState.SHUTDOWN_NOW);
                }
                if (!request.wantFence()) {
                    if (request.currentMetadataOffset() >= registerBrokerRecordOffset) {
                        this.log.info("The request from broker {} to unfence has been granted because it has caught up with the offset of it's register broker record {}.", (Object)brokerId, (Object)registerBrokerRecordOffset);
                        return new BrokerControlStates(currentState, BrokerControlState.UNFENCED);
                    }
                    if (this.log.isDebugEnabled()) {
                        this.log.debug("The request from broker {} to unfence cannot yet be granted because it has not caught up with the offset of it's register broker record {}. It is still at offset {}.", new Object[]{brokerId, registerBrokerRecordOffset, request.currentMetadataOffset()});
                    }
                    return new BrokerControlStates(currentState, BrokerControlState.FENCED);
                }
                return new BrokerControlStates(currentState, BrokerControlState.FENCED);
            }
            case UNFENCED: {
                if (request.wantFence()) {
                    if (request.wantShutDown()) {
                        this.log.info("Unfenced broker {} has requested and been granted an immediate shutdown.", (Object)brokerId);
                        return new BrokerControlStates(currentState, BrokerControlState.SHUTDOWN_NOW);
                    }
                    this.log.info("Unfenced broker {} has requested and been granted fencing", (Object)brokerId);
                    return new BrokerControlStates(currentState, BrokerControlState.FENCED);
                }
                if (request.wantShutDown()) {
                    if (hasLeaderships.get().booleanValue()) {
                        this.log.info("Unfenced broker {} has requested and been granted a controlled shutdown.", (Object)brokerId);
                        return new BrokerControlStates(currentState, BrokerControlState.CONTROLLED_SHUTDOWN);
                    }
                    this.log.info("Unfenced broker {} has requested and been granted an immediate shutdown.", (Object)brokerId);
                    return new BrokerControlStates(currentState, BrokerControlState.SHUTDOWN_NOW);
                }
                return new BrokerControlStates(currentState, BrokerControlState.UNFENCED);
            }
            case CONTROLLED_SHUTDOWN: {
                if (hasLeaderships.get().booleanValue()) {
                    this.log.debug("Broker {} is in controlled shutdown state, but can not shut down because more leaders still need to be moved.", (Object)brokerId);
                    return new BrokerControlStates(currentState, BrokerControlState.CONTROLLED_SHUTDOWN);
                }
                long lowestActiveOffset = this.lowestActiveOffset();
                if (broker.controlledShutDownOffset <= lowestActiveOffset) {
                    this.log.info("The request from broker {} to shut down has been granted since the lowest active offset {} is now greater than the broker's controlled shutdown offset {}.", new Object[]{brokerId, lowestActiveOffset, broker.controlledShutDownOffset});
                    return new BrokerControlStates(currentState, BrokerControlState.SHUTDOWN_NOW);
                }
                this.log.debug("The request from broker {} to shut down can not yet be granted because the lowest active offset {} is not greater than the broker's shutdown offset {}.", new Object[]{brokerId, lowestActiveOffset, broker.controlledShutDownOffset});
                return new BrokerControlStates(currentState, BrokerControlState.CONTROLLED_SHUTDOWN);
            }
        }
        return new BrokerControlStates(currentState, BrokerControlState.SHUTDOWN_NOW);
    }

    static class UsableBrokerIterator
    implements Iterator<UsableBroker> {
        private final Iterator<BrokerHeartbeatState> iterator;
        private final Function<Integer, Optional<String>> idToRack;
        private UsableBroker next;

        UsableBrokerIterator(Iterator<BrokerHeartbeatState> iterator, Function<Integer, Optional<String>> idToRack) {
            this.iterator = iterator;
            this.idToRack = idToRack;
            this.next = null;
        }

        @Override
        public boolean hasNext() {
            BrokerHeartbeatState result;
            if (this.next != null) {
                return true;
            }
            do {
                if (this.iterator.hasNext()) continue;
                return false;
            } while ((result = this.iterator.next()).shuttingDown());
            Optional<String> rack = this.idToRack.apply(result.id());
            this.next = new UsableBroker(result.id(), rack, result.fenced());
            return true;
        }

        @Override
        public UsableBroker next() {
            if (!this.hasNext()) {
                throw new NoSuchElementException();
            }
            UsableBroker result = this.next;
            this.next = null;
            return result;
        }
    }

    static class BrokerHeartbeatStateIterator
    implements Iterator<BrokerHeartbeatState> {
        private final BrokerHeartbeatState head;
        private BrokerHeartbeatState cur;

        BrokerHeartbeatStateIterator(BrokerHeartbeatState head) {
            this.head = head;
            this.cur = head;
        }

        @Override
        public boolean hasNext() {
            return this.cur.next != this.head;
        }

        @Override
        public BrokerHeartbeatState next() {
            if (!this.hasNext()) {
                throw new NoSuchElementException();
            }
            BrokerHeartbeatState result = this.cur.next;
            this.cur = this.cur.next;
            return result;
        }
    }

    static class BrokerHeartbeatStateList {
        private final BrokerHeartbeatState head = new BrokerHeartbeatState(-1);

        BrokerHeartbeatStateList() {
            this.head.prev = this.head;
            this.head.next = this.head;
        }

        BrokerHeartbeatState first() {
            BrokerHeartbeatState result = this.head.next;
            return result == this.head ? null : result;
        }

        void add(BrokerHeartbeatState broker) {
            BrokerHeartbeatState cur = this.head.prev;
            while (true) {
                if (cur == this.head || cur.lastContactNs <= broker.lastContactNs) break;
                cur = cur.prev;
            }
            broker.next = cur.next;
            cur.next.prev = broker;
            broker.prev = cur;
            cur.next = broker;
        }

        void remove(BrokerHeartbeatState broker) {
            if (broker.next == null) {
                throw new RuntimeException(broker + " is not in the  list.");
            }
            broker.prev.next = broker.next;
            broker.next.prev = broker.prev;
            broker.prev = null;
            broker.next = null;
        }

        BrokerHeartbeatStateIterator iterator() {
            return new BrokerHeartbeatStateIterator(this.head);
        }
    }

    static class MetadataOffsetComparator
    implements Comparator<BrokerHeartbeatState> {
        static final MetadataOffsetComparator INSTANCE = new MetadataOffsetComparator();

        MetadataOffsetComparator() {
        }

        @Override
        public int compare(BrokerHeartbeatState a, BrokerHeartbeatState b) {
            if (a.metadataOffset < b.metadataOffset) {
                return -1;
            }
            if (a.metadataOffset > b.metadataOffset) {
                return 1;
            }
            if (a.id < b.id) {
                return -1;
            }
            if (a.id > b.id) {
                return 1;
            }
            return 0;
        }
    }

    static class BrokerHeartbeatState {
        private final int id;
        long lastContactNs;
        long metadataOffset;
        private long controlledShutDownOffset;
        private BrokerHeartbeatState prev;
        private BrokerHeartbeatState next;

        BrokerHeartbeatState(int id) {
            this.id = id;
            this.lastContactNs = 0L;
            this.prev = null;
            this.next = null;
            this.metadataOffset = -1L;
            this.controlledShutDownOffset = -1L;
        }

        int id() {
            return this.id;
        }

        boolean fenced() {
            return this.prev == null;
        }

        boolean shuttingDown() {
            return this.controlledShutDownOffset >= 0L;
        }
    }
}

