/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.kafka.streams.processor.internals;

import java.util.List;
import org.apache.kafka.common.TopicPartition;

import java.util.Collection;
import java.util.Map;

/**
 * Performs bulk read operations from a set of partitions. Used to
 * restore  {@link org.apache.kafka.streams.processor.StateStore}s from their
 * changelogs
 */
public interface ChangelogReader {
    /**
     * Register a state store and it's partition for later restoration.
     * @param restorer the state restorer to register
     */
    void register(final StateRestorer restorer);

    /**
     * Restore all registered state stores by reading from their changelogs.
     * @return all topic partitions that have been restored
     */
    Collection<TopicPartition> restore(final RestoringTasks active);

    /**
     * @return the restored offsets for all persistent stores.
     */
    Map<TopicPartition, Long> restoredOffsets();

    /**
     * Removes the passed in partitions from the set of changelogs
     * @param revokedPartitions the set of partitions to remove
     */
    void remove(List<TopicPartition> revokedPartitions);

    /**
     * Clear all partitions
     */
    void clear();

    /**
     * @return whether the changelog reader has just been cleared or is uninitialized.
     */
    boolean isEmpty();
}
