/*
 * Decompiled with CFR 0.152.
 */
package kafka.tier.store;

import com.amazonaws.AmazonClientException;
import com.amazonaws.ClientConfiguration;
import com.amazonaws.auth.AWSCredentialsProvider;
import com.amazonaws.auth.DefaultAWSCredentialsProviderChain;
import com.amazonaws.auth.PropertiesFileCredentialsProvider;
import com.amazonaws.auth.STSAssumeRoleSessionCredentialsProvider;
import com.amazonaws.client.builder.AwsClientBuilder;
import com.amazonaws.regions.Regions;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3ClientBuilder;
import com.amazonaws.services.s3.model.DeleteObjectsRequest;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectInputStream;
import com.amazonaws.services.s3.model.SSEAwsKeyManagementParams;
import com.amazonaws.services.securitytoken.AWSSecurityTokenService;
import com.amazonaws.services.securitytoken.AWSSecurityTokenServiceClient;
import com.amazonaws.services.securitytoken.AWSSecurityTokenServiceClientBuilder;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.security.KeyStore;
import java.util.ArrayList;
import java.util.Map;
import java.util.Optional;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLContext;
import kafka.server.KafkaConfig;
import kafka.tier.exceptions.TierObjectStoreFatalException;
import kafka.tier.exceptions.TierObjectStoreRetriableException;
import kafka.tier.store.AutoAbortingGenericInputStream;
import kafka.tier.store.S3TierObjectStoreConfig;
import kafka.tier.store.TierObjectStore;
import kafka.tier.store.TierObjectStoreResponse;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.ssl.DefaultHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.ssl.SSLContextBuilder;
import org.apache.http.ssl.SSLContexts;
import org.apache.kafka.common.utils.ByteBufferInputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class S3TierObjectStore
implements TierObjectStore {
    private static final Logger log = LoggerFactory.getLogger(S3TierObjectStore.class);
    private final Optional<String> clusterIdOpt;
    private final Optional<Integer> brokerIdOpt;
    private final AmazonS3 client;
    private final String bucket;
    private final String prefix;
    private final String sseAlgorithm;
    private final String sseCustomerEncryptionKey;
    private final int autoAbortThresholdBytes;

    public S3TierObjectStore(S3TierObjectStoreConfig config) {
        this(S3TierObjectStore.client(config), config);
    }

    S3TierObjectStore(AmazonS3 client, S3TierObjectStoreConfig config) {
        this.clusterIdOpt = config.clusterIdOpt;
        this.brokerIdOpt = config.brokerIdOpt;
        this.client = client;
        this.bucket = config.s3Bucket;
        this.prefix = config.s3Prefix;
        this.sseAlgorithm = config.s3SseAlgorithm;
        this.sseCustomerEncryptionKey = config.s3SseCustomerEncryptionKey;
        this.autoAbortThresholdBytes = config.s3AutoAbortThresholdBytes;
        this.expectBucket(this.bucket, config.s3Region);
    }

    @Override
    public TierObjectStore.Backend getBackend() {
        return TierObjectStore.Backend.S3;
    }

    @Override
    public TierObjectStoreResponse getObject(TierObjectStore.ObjectStoreMetadata objectMetadata, TierObjectStore.FileType fileType, Integer byteOffsetStart, Integer byteOffsetEnd) {
        S3Object object;
        String key = this.keyPath(objectMetadata, fileType);
        GetObjectRequest request = new GetObjectRequest(this.bucket, key);
        if (byteOffsetStart != null && byteOffsetEnd != null) {
            request.setRange((long)byteOffsetStart.intValue(), (long)byteOffsetEnd.intValue());
        } else if (byteOffsetStart != null) {
            request.setRange((long)byteOffsetStart.intValue());
        } else if (byteOffsetEnd != null) {
            throw new IllegalStateException("Cannot specify a byteOffsetEnd without specifying a byteOffsetStart");
        }
        log.debug("Fetching object from s3://{}/{}, with range start {}", new Object[]{this.bucket, key, byteOffsetStart});
        try {
            object = this.client.getObject(request);
        }
        catch (AmazonClientException e) {
            throw new TierObjectStoreRetriableException(String.format("Failed to fetch object, metadata: %s type: %s range %s-%s", new Object[]{objectMetadata, fileType, byteOffsetStart, byteOffsetEnd}), e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException(String.format("Unknown exception when fetching object, metadata: %s type: %s range %s-%s", new Object[]{objectMetadata, fileType, byteOffsetStart, byteOffsetEnd}), e);
        }
        S3ObjectInputStream inputStream = object.getObjectContent();
        return new S3TierObjectStoreResponse(inputStream, this.autoAbortThresholdBytes, object.getObjectMetadata().getContentLength());
    }

    @Override
    public TierObjectStore.OpaqueData prepPutSegment() throws TierObjectStoreRetriableException, IOException {
        return TierObjectStore.OpaqueData.ZEROED;
    }

    @Override
    public void putObject(TierObjectStore.ObjectStoreMetadata objectMetadata, File file, TierObjectStore.FileType fileType) {
        Map<String, String> metadata = objectMetadata.objectMetadata(this.clusterIdOpt, this.brokerIdOpt);
        try {
            this.putFile(this.keyPath(objectMetadata, fileType), metadata, file);
        }
        catch (AmazonClientException e) {
            throw new TierObjectStoreRetriableException(String.format("Failed to upload object %s, file %s, type %s", new Object[]{objectMetadata, file, fileType}), e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException(String.format("Failed to upload object %s, file %s, type %s", new Object[]{objectMetadata, file, fileType}), e);
        }
    }

    @Override
    public void putSegment(TierObjectStore.ObjectMetadata objectMetadata, File segmentData, File offsetIndexData, File timestampIndexData, Optional<File> producerStateSnapshotData, Optional<ByteBuffer> transactionIndexData, Optional<ByteBuffer> epochState) {
        Map<String, String> metadata = objectMetadata.objectMetadata(this.clusterIdOpt, this.brokerIdOpt);
        try {
            this.putFile(this.keyPath(objectMetadata, TierObjectStore.FileType.SEGMENT), metadata, segmentData);
            this.putFile(this.keyPath(objectMetadata, TierObjectStore.FileType.OFFSET_INDEX), metadata, offsetIndexData);
            this.putFile(this.keyPath(objectMetadata, TierObjectStore.FileType.TIMESTAMP_INDEX), metadata, timestampIndexData);
            producerStateSnapshotData.ifPresent(file -> this.putFile(this.keyPath(objectMetadata, TierObjectStore.FileType.PRODUCER_STATE), metadata, (File)file));
            transactionIndexData.ifPresent(abortedTxnsBuf -> this.putBuf(this.keyPath(objectMetadata, TierObjectStore.FileType.TRANSACTION_INDEX), metadata, (ByteBuffer)abortedTxnsBuf));
            epochState.ifPresent(buf -> this.putBuf(this.keyPath(objectMetadata, TierObjectStore.FileType.EPOCH_STATE), metadata, (ByteBuffer)buf));
        }
        catch (AmazonClientException e) {
            throw new TierObjectStoreRetriableException("Failed to upload segment: " + objectMetadata, e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException("Unknown exception when uploading segment: " + objectMetadata, e);
        }
    }

    @Override
    public void deleteSegment(TierObjectStore.ObjectMetadata objectMetadata) {
        ArrayList<DeleteObjectsRequest.KeyVersion> keys = new ArrayList<DeleteObjectsRequest.KeyVersion>();
        for (TierObjectStore.FileType type : TierObjectStore.FileType.values()) {
            String keyPath = this.keyPath(objectMetadata, type);
            log.debug("Deleting object s3://{}/{}", (Object)this.bucket, (Object)keyPath);
            keys.add(new DeleteObjectsRequest.KeyVersion(keyPath));
        }
        DeleteObjectsRequest request = new DeleteObjectsRequest(this.bucket).withKeys(keys);
        try {
            this.client.deleteObjects(request);
        }
        catch (AmazonClientException e) {
            throw new TierObjectStoreRetriableException("Failed to delete segment: " + objectMetadata, e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException("Unknown exception when deleting segment: " + objectMetadata, e);
        }
    }

    @Override
    public void close() {
        this.client.shutdown();
    }

    private String keyPath(TierObjectStore.ObjectStoreMetadata objectMetadata, TierObjectStore.FileType fileType) {
        return objectMetadata.toPath(this.prefix, fileType);
    }

    private ObjectMetadata putObjectMetadata(Map<String, String> userMetadata) {
        ObjectMetadata metadata = new ObjectMetadata();
        if (this.sseAlgorithm != null) {
            metadata.setSSEAlgorithm(this.sseAlgorithm);
        }
        if (userMetadata != null) {
            metadata.setUserMetadata(userMetadata);
        }
        return metadata;
    }

    private void putFile(String key, Map<String, String> metadata, File file) {
        PutObjectRequest request = new PutObjectRequest(this.bucket, key, file).withMetadata(this.putObjectMetadata(metadata));
        if (this.sseCustomerEncryptionKey != null && this.sseAlgorithm.equals("aws:kms")) {
            request.setSSEAwsKeyManagementParams(new SSEAwsKeyManagementParams(this.sseCustomerEncryptionKey));
        }
        log.debug("Uploading object to s3://{}/{}", (Object)this.bucket, (Object)key);
        this.client.putObject(request);
    }

    private void putBuf(String key, Map<String, String> metadata, ByteBuffer buf) {
        ObjectMetadata s3metadata = this.putObjectMetadata(metadata);
        s3metadata.setContentLength((long)(buf.limit() - buf.position()));
        PutObjectRequest request = new PutObjectRequest(this.bucket, key, (InputStream)new ByteBufferInputStream(buf.duplicate()), s3metadata);
        if (this.sseCustomerEncryptionKey != null && !this.sseCustomerEncryptionKey.isEmpty() && this.sseAlgorithm.equals("aws:kms")) {
            request.setSSEAwsKeyManagementParams(new SSEAwsKeyManagementParams(this.sseCustomerEncryptionKey));
        }
        log.debug("Uploading object to s3://{}/{}", (Object)this.bucket, (Object)key);
        this.client.putObject(request);
    }

    public static String validateAndGetS3RegionName(String s3Region) {
        try {
            return Regions.fromName((String)s3Region).getName();
        }
        catch (IllegalArgumentException e) {
            throw new IllegalArgumentException("Configured " + KafkaConfig.TierS3RegionProp() + " '" + s3Region + "' is not known");
        }
    }

    public static AmazonS3 client(S3TierObjectStoreConfig config) throws TierObjectStoreFatalException {
        ClientConfiguration clientConfiguration = new ClientConfiguration();
        clientConfiguration.setUserAgentPrefix(config.s3UserAgentPrefix);
        SSLConnectionSocketFactory sslConnectionSocketFactory = S3TierObjectStore.getSSLConnectionSocketFactory(config);
        if (sslConnectionSocketFactory != null) {
            clientConfiguration.getApacheHttpClientConfig().setSslSocketFactory((ConnectionSocketFactory)sslConnectionSocketFactory);
        }
        AmazonS3ClientBuilder builder = AmazonS3ClientBuilder.standard();
        builder.setClientConfiguration(clientConfiguration);
        if (config.s3ForcePathStyleAccess.booleanValue()) {
            builder.setPathStyleAccessEnabled(Boolean.valueOf(true));
        }
        if (config.s3SignerOverride.isPresent() && !config.s3SignerOverride.get().isEmpty()) {
            clientConfiguration.setSignerOverride(config.s3SignerOverride.get());
        }
        if (config.s3EndpointOverride.isPresent() && !config.s3EndpointOverride.get().isEmpty()) {
            String s3RegionName = S3TierObjectStore.validateAndGetS3RegionName(config.s3Region);
            builder.setEndpointConfiguration(new AwsClientBuilder.EndpointConfiguration(config.s3EndpointOverride.get(), s3RegionName));
        } else if (config.s3Region != null && !config.s3Region.isEmpty()) {
            builder.setRegion(config.s3Region);
        }
        AWSCredentialsProvider provider = config.s3CredFilePath.map(PropertiesFileCredentialsProvider::new).orElse((AWSCredentialsProvider)new DefaultAWSCredentialsProviderChain());
        if (config.assumeRoleArn.isPresent()) {
            AWSSecurityTokenService stsClient = (AWSSecurityTokenService)((AWSSecurityTokenServiceClientBuilder)AWSSecurityTokenServiceClient.builder().withCredentials(provider)).build();
            provider = new STSAssumeRoleSessionCredentialsProvider.Builder(config.assumeRoleArn.get(), "tiered-storage").withStsClient(stsClient).build();
        }
        builder.setCredentials(provider);
        return (AmazonS3)builder.build();
    }

    private void expectBucket(String bucket, String expectedRegion) throws TierObjectStoreFatalException {
        try {
            String actualRegion = this.client.getBucketLocation(bucket);
            if (actualRegion.equals("US") && expectedRegion.equals("us-east-1")) {
                return;
            }
            if (!expectedRegion.equals(actualRegion)) {
                log.warn("Bucket region {} does not match expected region {}", (Object)actualRegion, (Object)expectedRegion);
            }
        }
        catch (AmazonClientException ex) {
            throw new TierObjectStoreFatalException("Failed to access bucket " + bucket, ex);
        }
    }

    private static SSLConnectionSocketFactory getSSLConnectionSocketFactory(S3TierObjectStoreConfig config) throws TierObjectStoreFatalException {
        boolean hasCustomKeyStore;
        SSLConnectionSocketFactory sslConnectionSocketFactory = null;
        boolean hasCustomTrustStore = config.s3SslTrustStoreLocation.isPresent() && !config.s3SslTrustStoreLocation.get().isEmpty();
        boolean bl = hasCustomKeyStore = config.s3SslKeyStoreLocation.isPresent() && !config.s3SslKeyStoreLocation.get().isEmpty();
        if (hasCustomTrustStore || hasCustomKeyStore) {
            try {
                SSLContextBuilder sslContextBuilder = SSLContexts.custom();
                if (hasCustomTrustStore) {
                    KeyStore trustStore = KeyStore.getInstance(config.s3SslTrustStoreType.get());
                    FileInputStream trustKeyStoreFile = new FileInputStream(config.s3SslTrustStoreLocation.get());
                    trustStore.load(trustKeyStoreFile, config.s3SslTrustStorePassword.get().value().toCharArray());
                    sslContextBuilder.loadTrustMaterial(trustStore, null);
                }
                if (hasCustomKeyStore) {
                    KeyStore keyStore = KeyStore.getInstance(config.s3SslKeyStoreType.get());
                    FileInputStream identityKeyStoreFile = new FileInputStream(config.s3SslKeyStoreLocation.get());
                    keyStore.load(identityKeyStoreFile, config.s3SslKeyStorePassword.get().value().toCharArray());
                    sslContextBuilder.loadKeyMaterial(keyStore, config.s3SslKeyPassword.get().value().toCharArray(), (aliases, socket) -> "confluent.kafka");
                }
                sslContextBuilder.setProtocol(config.s3SslProtocol);
                SSLContext sslContext = sslContextBuilder.build();
                sslConnectionSocketFactory = new SSLConnectionSocketFactory(sslContext, config.s3SslEnabledProtocols.toArray(new String[0]), null, (HostnameVerifier)new DefaultHostnameVerifier());
            }
            catch (Exception e) {
                throw new TierObjectStoreFatalException("Failed to load keystore or trust store for tiered object store", e);
            }
        }
        return sslConnectionSocketFactory;
    }

    private static class S3TierObjectStoreResponse
    implements TierObjectStoreResponse {
        private final AutoAbortingGenericInputStream inputStream;

        S3TierObjectStoreResponse(S3ObjectInputStream inputStream, long autoAbortSize, long streamSize) {
            this.inputStream = new AutoAbortingGenericInputStream((InputStream)inputStream, autoAbortSize, streamSize);
        }

        @Override
        public void close() {
            this.inputStream.close();
        }

        @Override
        public InputStream getInputStream() {
            return this.inputStream;
        }
    }
}

