/*
 * Decompiled with CFR 0.152.
 */
package kafka.tier.fetcher;

import java.time.Instant;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CancellationException;
import java.util.concurrent.TimeUnit;
import kafka.tier.fetcher.CancellationContext;
import org.apache.kafka.common.MetricName;
import org.apache.kafka.common.metrics.CompoundStat;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.metrics.Sensor;
import org.apache.kafka.common.metrics.stats.Meter;
import org.apache.kafka.common.utils.Time;

public class MemoryTracker
implements AutoCloseable {
    private final Time time;
    private final Metrics metrics;
    private final MetricName leasedMetricName;
    private final MetricName poolSizeMetricName;
    private final MetricName oldestLeaseMetricName;
    private final Map<Long, Instant> creationTimes = new HashMap<Long, Instant>();
    final MetricName memoryTrackerDepletedPercentMetricName;
    final MetricName memoryTrackerDepletedTimeMetricName;
    private final Sensor oomTimeSensor;
    private long oomPeriodStart = 0L;
    private long poolSize;
    private long leased = 0L;
    private long leaseId = Long.MIN_VALUE;
    private boolean closed = false;

    public MemoryTracker(Time time, long poolSize) {
        this(time, null, poolSize);
    }

    public MemoryTracker(Time time, Metrics metrics, long poolSize) {
        if (poolSize < 0L) {
            throw new IllegalArgumentException("MemoryTracker pool size should be >= 0");
        }
        this.time = time;
        this.poolSize = poolSize;
        this.metrics = metrics;
        if (metrics != null) {
            String metricGroupName = "TierFetcherMemoryTracker";
            this.leasedMetricName = metrics.metricName("Leased", metricGroupName, "The amount of memory currently leased in bytes");
            this.poolSizeMetricName = metrics.metricName("PoolSize", metricGroupName, "The size of the memory pool in bytes, 0 if disabled");
            this.oldestLeaseMetricName = metrics.metricName("MaxLeaseLagMs", metricGroupName, "The time difference between the oldest outstanding memory lease and the current time");
            this.memoryTrackerDepletedPercentMetricName = metrics.metricName("MemoryTrackerAvgDepletedPercent", metricGroupName, "The average percentageof time in milliseconds requests were blocked due to memory pressure");
            this.memoryTrackerDepletedTimeMetricName = metrics.metricName("MemoryTrackerDepletedTimeTotal", metricGroupName, "The total amount of time in milliseconds requests were blocked due to memory pressure");
            this.oomTimeSensor = metrics.sensor("MemoryTrackerUtilization");
            this.oomTimeSensor.add((CompoundStat)new Meter(TimeUnit.MILLISECONDS, this.memoryTrackerDepletedPercentMetricName, this.memoryTrackerDepletedTimeMetricName));
            MemoryTracker self = this;
            metrics.addMetric(this.leasedMetricName, (config, now) -> {
                MemoryTracker memoryTracker = self;
                synchronized (memoryTracker) {
                    return this.leased;
                }
            });
            metrics.addMetric(this.poolSizeMetricName, (config, now) -> {
                MemoryTracker memoryTracker = self;
                synchronized (memoryTracker) {
                    return poolSize;
                }
            });
            metrics.addMetric(this.oldestLeaseMetricName, (config, now) -> {
                MemoryTracker memoryTracker = self;
                synchronized (memoryTracker) {
                    Instant smallest = null;
                    for (Instant candidate : this.creationTimes.values()) {
                        if (smallest != null && !candidate.isBefore(smallest)) continue;
                        smallest = candidate;
                    }
                    if (smallest == null) {
                        return 0.0;
                    }
                    return Math.min(0L, now - smallest.toEpochMilli());
                }
            });
        } else {
            this.leasedMetricName = null;
            this.poolSizeMetricName = null;
            this.oldestLeaseMetricName = null;
            this.memoryTrackerDepletedPercentMetricName = null;
            this.memoryTrackerDepletedTimeMetricName = null;
            this.oomTimeSensor = null;
        }
    }

    public synchronized void setPoolSize(long newPoolSize) {
        this.poolSize = newPoolSize;
        this.wakeup();
    }

    public synchronized boolean isDisabled() {
        return this.poolSize == 0L;
    }

    public synchronized MemoryLease newLease(CancellationContext ctx, long amount) {
        while (!ctx.isCancelled()) {
            if (this.closed) {
                throw new IllegalStateException("MemoryTracker closed");
            }
            Optional<MemoryLease> newLease = this.tryLease(amount);
            if (newLease.isPresent()) {
                this.stopOomPeriod();
                return newLease.get();
            }
            try {
                this.startOomPeriod();
                this.wait();
            }
            catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
        }
        throw new CancellationException("Memory lease request cancelled");
    }

    private void startOomPeriod() {
        if (this.oomTimeSensor != null && this.oomPeriodStart == 0L) {
            this.oomPeriodStart = this.time.nanoseconds();
        }
    }

    private void stopOomPeriod() {
        if (this.oomTimeSensor != null) {
            long start = this.oomPeriodStart;
            this.oomPeriodStart = 0L;
            if (start != 0L) {
                this.oomTimeSensor.record((double)(this.time.nanoseconds() - start) / 1000000.0);
            }
        }
    }

    public synchronized Optional<MemoryLease> tryLease(long amount) {
        if (this.closed) {
            throw new IllegalStateException("MemoryTracker closed");
        }
        if (this.poolSize - this.leased > 0L || this.isDisabled()) {
            this.leased += amount;
            ++this.leaseId;
            Instant now = Instant.ofEpochMilli(this.time.hiResClockMs());
            if (!this.isDisabled()) {
                this.creationTimes.put(this.leaseId, now);
            }
            return Optional.of(new MemoryLease(this, this.leaseId, amount));
        }
        return Optional.empty();
    }

    public synchronized long leased() {
        return this.leased;
    }

    public synchronized long poolSize() {
        return this.poolSize;
    }

    public synchronized void wakeup() {
        this.notifyAll();
    }

    private synchronized void release(MemoryLease lease) {
        this.leased -= lease.amount;
        this.creationTimes.remove(lease.leaseID);
        this.notifyAll();
    }

    @Override
    public synchronized void close() {
        if (this.metrics != null) {
            this.metrics.removeMetric(this.leasedMetricName);
            this.metrics.removeMetric(this.poolSizeMetricName);
            this.metrics.removeMetric(this.oldestLeaseMetricName);
            this.metrics.removeMetric(this.memoryTrackerDepletedPercentMetricName);
            this.metrics.removeMetric(this.memoryTrackerDepletedTimeMetricName);
        }
        this.closed = true;
        this.wakeup();
    }

    public static final class MemoryLease {
        private final long leaseID;
        private final MemoryTracker parent;
        private boolean released = false;
        private long amount;

        public MemoryLease(MemoryTracker parent, long leaseId, long amount) {
            this.parent = parent;
            this.leaseID = leaseId;
            this.amount = amount;
        }

        public void release() {
            if (!this.released) {
                this.released = true;
                this.parent.release(this);
            }
        }

        public long leased() {
            if (this.released) {
                throw new IllegalStateException(this + " already reclaimed");
            }
            return this.amount;
        }

        public boolean tryExtendLease(long amount) {
            if (this.released) {
                throw new IllegalStateException("MemoryLease already reclaimed");
            }
            Optional<MemoryLease> newLease = this.parent.tryLease(amount);
            newLease.ifPresent(memoryLease -> this.amount += memoryLease.leased());
            return newLease.isPresent();
        }

        public String toString() {
            return "MemoryLease{leaseID=" + this.leaseID + ", released=" + this.released + ", amount=" + this.amount + '}';
        }
    }
}

