/*
 * Decompiled with CFR 0.152.
 */
package kafka.tier.store;

import com.google.api.gax.paging.Page;
import com.google.auth.Credentials;
import com.google.auth.oauth2.GoogleCredentials;
import com.google.cloud.ReadChannel;
import com.google.cloud.WriteChannel;
import com.google.cloud.storage.Blob;
import com.google.cloud.storage.BlobId;
import com.google.cloud.storage.BlobInfo;
import com.google.cloud.storage.Bucket;
import com.google.cloud.storage.Storage;
import com.google.cloud.storage.StorageException;
import com.google.cloud.storage.StorageOptions;
import com.google.common.collect.Lists;
import com.google.crypto.tink.Aead;
import com.google.crypto.tink.KeyTemplate;
import com.google.crypto.tink.KeysetHandle;
import com.google.crypto.tink.aead.AeadConfig;
import com.google.crypto.tink.aead.KmsAeadKeyManager;
import com.google.crypto.tink.integration.gcpkms.GcpKmsClient;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.nio.file.StandardOpenOption;
import java.security.GeneralSecurityException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalInt;
import kafka.tier.exceptions.TierObjectStoreFatalException;
import kafka.tier.exceptions.TierObjectStoreRetriableException;
import kafka.tier.store.BucketHealthResult;
import kafka.tier.store.GcsTierObjectStoreConfig;
import kafka.tier.store.TierObjectAttribute;
import kafka.tier.store.TierObjectStore;
import kafka.tier.store.TierObjectStoreResponse;
import kafka.tier.store.TierObjectStoreUtils;
import kafka.tier.store.VersionInformation;
import kafka.tier.store.encryption.EncryptionKeyManager;
import kafka.tier.store.encryption.KeyContext;
import kafka.tier.store.encryption.KeySha;
import kafka.utils.CoreUtils;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.utils.Crc32C;
import org.apache.kafka.common.utils.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GcsTierObjectStore
implements TierObjectStore {
    private static final Logger log = LoggerFactory.getLogger(GcsTierObjectStore.class);
    private static final int UNKNOWN_END_RANGE_CHUNK_SIZE = 1000000;
    private final Optional<String> clusterIdOpt;
    private final Optional<Integer> brokerIdOpt;
    private final String bucket;
    private final String prefix;
    private final int writeChunkSize;
    private final Storage storage;
    private final EncryptionKeyManager encryptionKeyManager;
    private static final int DEFAULT_GCS_DELETE_BATCH_SIZE = 500;

    public GcsTierObjectStore(Time time, GcsTierObjectStoreConfig config) {
        this(GcsTierObjectStore.storage(config), GcsTierObjectStore.encryptionKeyManager(config, time, null), config);
    }

    public GcsTierObjectStore(Time time, Metrics metrics, GcsTierObjectStoreConfig config) {
        this(GcsTierObjectStore.storage(config), GcsTierObjectStore.encryptionKeyManager(config, time, metrics), config);
    }

    GcsTierObjectStore(Storage storage, EncryptionKeyManager encryptionKeyManager, GcsTierObjectStoreConfig config) {
        this.clusterIdOpt = config.clusterIdOpt;
        this.brokerIdOpt = config.brokerIdOpt;
        this.storage = storage;
        this.bucket = config.gcsBucket;
        this.prefix = config.gcsPrefix;
        this.writeChunkSize = config.gcsWriteChunkSize;
        this.expectBucket(this.bucket, config.gcsRegion);
        this.encryptionKeyManager = encryptionKeyManager;
        if (this.encryptionKeyManager != null) {
            this.encryptionKeyManager.bindHook(new EncryptionKeyManagerHook());
        }
    }

    @Override
    public TierObjectStore.Backend getBackend() {
        return TierObjectStore.Backend.GCS;
    }

    @Override
    public BucketHealthResult checkBucketHealth() {
        try {
            ByteBuffer payload = TierObjectStoreUtils.timeHealthPayload();
            TierObjectStore.HealthMetadata metadata = new TierObjectStore.HealthMetadata(this.clusterIdOpt, this.brokerIdOpt);
            String key = metadata.toPath(this.prefix, TierObjectStore.FileType.HEALTH_CHECK);
            this.putBuf(key, metadata.objectMetadata(this.clusterIdOpt, this.brokerIdOpt), payload);
            try (InputStream inputStream = this.getObject(metadata, TierObjectStore.FileType.HEALTH_CHECK).getInputStream();){
                int read;
                while ((read = inputStream.read()) > 0) {
                    log.trace("Bucket probe read {} bytes", (Object)read);
                }
            }
            BlobId blobId = BlobId.of((String)this.bucket, (String)key);
            if (!this.storage.delete(blobId)) {
                throw new Exception("Error deleting health key " + key);
            }
            return BucketHealthResult.HEALTHY;
        }
        catch (Exception e) {
            log.error("Bucket health checker returned unclassified error", (Throwable)e);
            return BucketHealthResult.UNCLASSIFIED;
        }
    }

    @Override
    public Map<String, List<VersionInformation>> listObject(String keyPrefix, boolean getVersionInfo) {
        HashMap<String, List<VersionInformation>> results = new HashMap<String, List<VersionInformation>>();
        try {
            Page blobs = this.storage.list(this.bucket, new Storage.BlobListOption[]{Storage.BlobListOption.prefix((String)keyPrefix), Storage.BlobListOption.versions((boolean)getVersionInfo)});
            for (Blob blob : blobs.iterateAll()) {
                results.putIfAbsent(blob.getName(), new ArrayList());
                ((List)results.get(blob.getName())).add(new VersionInformation(blob.getGeneration().toString()));
            }
            if (log.isDebugEnabled()) {
                StringBuilder allBlobs = new StringBuilder();
                results.forEach((key, versions) -> allBlobs.append("[").append((String)key).append("->").append(Arrays.toString(versions.toArray())).append("] "));
                log.debug("TierObjectStore listObjects versions: " + getVersionInfo + " prefix: " + keyPrefix + " " + allBlobs);
            }
        }
        catch (StorageException e) {
            log.info("Google storage client returned exception while listing " + keyPrefix + " " + (Object)((Object)e));
            throw new TierObjectStoreRetriableException(e.getMessage(), e);
        }
        catch (Exception e) {
            log.info("Unknown exception while listing " + keyPrefix + " " + e);
            throw new TierObjectStoreFatalException(e.getMessage(), e);
        }
        return results;
    }

    @Override
    public TierObjectStoreResponse getObject(TierObjectStore.ObjectStoreMetadata objectMetadata, TierObjectStore.FileType fileType, Integer byteOffsetStart, Integer byteOffsetEnd, VersionInformation versionInformation) {
        BlobId blobId;
        String key = this.keyPath(objectMetadata, fileType);
        BlobId blobId2 = blobId = versionInformation != null ? BlobId.of((String)this.bucket, (String)key, (Long)Long.parseLong(versionInformation.getVersionId())) : BlobId.of((String)this.bucket, (String)key);
        if (byteOffsetStart != null && byteOffsetEnd != null && byteOffsetStart > byteOffsetEnd) {
            throw new IllegalStateException("Invalid range of byteOffsetStart and byteOffsetEnd");
        }
        if (byteOffsetStart == null && byteOffsetEnd != null) {
            throw new IllegalStateException("Cannot specify a byteOffsetEnd without specifying a byteOffsetStart");
        }
        try {
            ReadChannel reader = this.getReader(objectMetadata, fileType, blobId);
            long byteOffsetStartLong = byteOffsetStart == null ? 0L : byteOffsetStart.longValue();
            OptionalInt chunkSize = byteOffsetEnd == null ? OptionalInt.empty() : OptionalInt.of(byteOffsetEnd - byteOffsetStart);
            return new GcsTierObjectStoreResponse(reader, byteOffsetStartLong, chunkSize);
        }
        catch (TierObjectStoreRetriableException e) {
            throw e;
        }
        catch (StorageException e) {
            throw new TierObjectStoreRetriableException(String.format("Failed to fetch object, blobId: %s metadata: %s type: %s range %s-%s", new Object[]{blobId, objectMetadata, fileType, byteOffsetStart, byteOffsetEnd}), e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException(String.format("Unknown exception when fetching object, blobId: %s metadata: %s type: %s range %s-%s", new Object[]{blobId, objectMetadata, fileType, byteOffsetStart, byteOffsetEnd}), e);
        }
    }

    private ReadChannel getReader(TierObjectStore.ObjectStoreMetadata objectMetadata, TierObjectStore.FileType fileType, BlobId blobId) {
        if (!objectMetadata.opaqueData().isEmpty() && fileType.equals((Object)TierObjectStore.FileType.SEGMENT)) {
            KeyContext keyContext = this.getKeyContext(objectMetadata.opaqueData(), blobId);
            return this.storage.reader(blobId, new Storage.BlobSourceOption[]{Storage.BlobSourceOption.decryptionKey((String)keyContext.cleartextDataKey.base64Encoded())});
        }
        return this.storage.reader(blobId, new Storage.BlobSourceOption[0]);
    }

    KeyContext getKeyContext(TierObjectStore.OpaqueData opaqueData, BlobId blobId) {
        if (this.encryptionKeyManager == null) {
            throw new TierObjectStoreFatalException("EncryptionKeyManager is not configured");
        }
        KeySha keySha = KeySha.fromRawBytes(opaqueData.intoByteArray());
        KeyContext keyContext = this.encryptionKeyManager.keyContext(keySha);
        if (keyContext == null) {
            log.info("EncryptionKeyManager cache miss while downloading object with KeySha {}, fetching object metadata for cache hydration", (Object)keySha);
            Blob blobMetadata = this.storage.get(blobId, new Storage.BlobGetOption[]{Storage.BlobGetOption.fields((Storage.BlobField[])new Storage.BlobField[]{Storage.BlobField.METADATA})});
            Map extractedMetadata = blobMetadata.getMetadata();
            KeySha restoredKeySha = this.encryptionKeyManager.registerKeyFromObjectMetadata(extractedMetadata);
            if (!restoredKeySha.equals(keySha)) {
                throw new TierObjectStoreFatalException(String.format("KeySha of key material restored from object metadata %s does not match KeySha provided via OpaqueData %s", restoredKeySha, keySha));
            }
            keyContext = this.encryptionKeyManager.keyContext(keySha);
        }
        return keyContext;
    }

    @Override
    public TierObjectStore.OpaqueData prepPutSegment() throws TierObjectStoreRetriableException, IOException {
        if (this.encryptionKeyManager != null) {
            KeySha active = this.encryptionKeyManager.activeKeySha();
            return TierObjectStore.OpaqueData.fromByteArray(active.toRawBytes());
        }
        return TierObjectStore.OpaqueData.ZEROED;
    }

    @Override
    public void putSegment(TierObjectStore.ObjectMetadata objectMetadata, File segmentData, File offsetIndexData, File timestampIndexData, Optional<File> producerStateSnapshotData, Optional<ByteBuffer> transactionIndexData, Optional<ByteBuffer> epochState) {
        Map<String, String> metadata = objectMetadata.objectMetadata(this.clusterIdOpt, this.brokerIdOpt);
        try {
            this.putSegmentFile(metadata, objectMetadata, segmentData);
            this.putFileUnencrypted(this.keyPath(objectMetadata, TierObjectStore.FileType.OFFSET_INDEX), metadata, offsetIndexData);
            this.putFileUnencrypted(this.keyPath(objectMetadata, TierObjectStore.FileType.TIMESTAMP_INDEX), metadata, timestampIndexData);
            if (producerStateSnapshotData.isPresent()) {
                this.putFileUnencrypted(this.keyPath(objectMetadata, TierObjectStore.FileType.PRODUCER_STATE), metadata, producerStateSnapshotData.get());
            }
            if (transactionIndexData.isPresent()) {
                this.putBuf(this.keyPath(objectMetadata, TierObjectStore.FileType.TRANSACTION_INDEX), metadata, transactionIndexData.get());
            }
            if (epochState.isPresent()) {
                this.putBuf(this.keyPath(objectMetadata, TierObjectStore.FileType.EPOCH_STATE), metadata, epochState.get());
            }
        }
        catch (StorageException e) {
            log.warn("Deleting partially uploaded files due to failed to upload segment: " + objectMetadata, (Throwable)e);
            this.deleteObjects(this.objectsForSegment(objectMetadata));
            throw new TierObjectStoreRetriableException("Failed to upload segment: " + objectMetadata, e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException("Unknown exception when uploading segment: " + objectMetadata, e);
        }
    }

    @Override
    public void putInMemorySegment(TierObjectStore.ObjectMetadata objectMetadata, File segmentData, File offsetIndexData, File timestampIndexData, Optional<ByteBuffer> producerStateSnapshotData, Optional<ByteBuffer> transactionIndexData, Optional<ByteBuffer> epochState) {
        Map<String, String> metadata = objectMetadata.objectMetadata(this.clusterIdOpt, this.brokerIdOpt);
        try {
            this.putSegmentFile(metadata, objectMetadata, segmentData);
            this.putFileUnencrypted(this.keyPath(objectMetadata, TierObjectStore.FileType.OFFSET_INDEX), metadata, offsetIndexData);
            this.putFileUnencrypted(this.keyPath(objectMetadata, TierObjectStore.FileType.TIMESTAMP_INDEX), metadata, timestampIndexData);
            if (producerStateSnapshotData.isPresent()) {
                this.putBuf(this.keyPath(objectMetadata, TierObjectStore.FileType.PRODUCER_STATE), metadata, producerStateSnapshotData.get());
            }
            if (transactionIndexData.isPresent()) {
                this.putBuf(this.keyPath(objectMetadata, TierObjectStore.FileType.TRANSACTION_INDEX), metadata, transactionIndexData.get());
            }
            if (epochState.isPresent()) {
                this.putBuf(this.keyPath(objectMetadata, TierObjectStore.FileType.EPOCH_STATE), metadata, epochState.get());
            }
        }
        catch (StorageException e) {
            log.warn("Deleting partially uploaded files due to failed to upload segment: " + objectMetadata, (Throwable)e);
            this.deleteObjects(this.objectsForSegment(objectMetadata));
            throw new TierObjectStoreRetriableException("Failed to upload segment: " + objectMetadata, e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException("Unknown exception when uploading segment: " + objectMetadata, e);
        }
    }

    private void putSegmentFile(Map<String, String> metadata, TierObjectStore.ObjectMetadata objectMetadata, File segmentData) throws IOException {
        if (this.encryptionKeyManager != null) {
            this.putFileEncrypted(this.keyPath(objectMetadata, TierObjectStore.FileType.SEGMENT), metadata, segmentData, objectMetadata.opaqueData());
        } else {
            if (!objectMetadata.opaqueData().isEmpty()) {
                throw new TierObjectStoreFatalException(String.format("Attempted to upload a segment with OpaqueData %s, but encryption is not configured", objectMetadata.opaqueData()));
            }
            this.putFileUnencrypted(this.keyPath(objectMetadata, TierObjectStore.FileType.SEGMENT), metadata, segmentData);
        }
    }

    @Override
    public void putObject(final TierObjectStore.ObjectStoreMetadata objectMetadata, File file, final TierObjectStore.FileType fileType) {
        Map<String, String> metadata = objectMetadata.objectMetadata(this.clusterIdOpt, this.brokerIdOpt);
        try {
            this.putFileUnencrypted(this.keyPath(objectMetadata, fileType), metadata, file);
        }
        catch (StorageException e) {
            log.warn("Deleting partially uploaded files due to failed to upload segment: " + objectMetadata, (Throwable)e);
            this.deleteObjects((List<BlobId>)new ArrayList<BlobId>(){
                {
                    this.add(BlobId.of((String)GcsTierObjectStore.this.bucket, (String)GcsTierObjectStore.this.keyPath(objectMetadata, fileType)));
                }
            });
            throw new TierObjectStoreRetriableException(String.format("Failed to upload object %s, file %s, type %s", new Object[]{objectMetadata, file, fileType}), e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException(String.format("Failed to upload object %s, file %s, type %s", new Object[]{objectMetadata, file, fileType}), e);
        }
    }

    @Override
    public void putBuffer(final TierObjectStore.ObjectStoreMetadata objectMetadata, ByteBuffer buffer, final TierObjectStore.FileType fileType) {
        Map<String, String> metadata = objectMetadata.objectMetadata(this.clusterIdOpt, this.brokerIdOpt);
        try {
            this.putBuf(this.keyPath(objectMetadata, fileType), metadata, buffer);
        }
        catch (StorageException e) {
            log.warn("Deleting partially uploaded files due to failed to upload segment: " + objectMetadata, (Throwable)e);
            this.deleteObjects((List<BlobId>)new ArrayList<BlobId>(){
                {
                    this.add(BlobId.of((String)GcsTierObjectStore.this.bucket, (String)GcsTierObjectStore.this.keyPath(objectMetadata, fileType)));
                }
            });
            throw new TierObjectStoreRetriableException(String.format("Failed to upload object %s, buffer %s, type %s", new Object[]{objectMetadata, buffer, fileType}), e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException(String.format("Failed to upload object %s, buffer %s, type %s", new Object[]{objectMetadata, buffer, fileType}), e);
        }
    }

    @Override
    public void restoreObjectByCopy(TierObjectStore.ObjectMetadata objectMetadata, String key, VersionInformation lastLiveVersion) {
        String lastLiveVersionId = lastLiveVersion.getVersionId();
        try {
            Storage.CopyRequest.Builder copyRequestBuilder = Storage.CopyRequest.newBuilder();
            BlobId source = BlobId.of((String)this.bucket, (String)key, (Long)Long.parseLong(lastLiveVersionId));
            BlobId target = BlobId.of((String)this.bucket, (String)key);
            if (this.encryptionKeyManager != null && key.endsWith(TierObjectStore.FileType.SEGMENT.suffix()) && !objectMetadata.opaqueData().isEmpty()) {
                KeyContext keyContext = this.getKeyContext(objectMetadata.opaqueData(), source);
                if (keyContext == null) {
                    throw new TierObjectStoreFatalException(String.format("No valid KeyContext for copying object '%s'", key));
                }
                String dataKeyBase64 = keyContext.cleartextDataKey.base64Encoded();
                log.debug("Restore encrypted object by copying the last live version {} to gs://{}/{}", new Object[]{lastLiveVersionId, this.bucket, key});
                copyRequestBuilder.setSource(source).setSourceOptions(new Storage.BlobSourceOption[]{Storage.BlobSourceOption.decryptionKey((String)dataKeyBase64)}).setTarget(target, new Storage.BlobTargetOption[]{Storage.BlobTargetOption.encryptionKey((String)dataKeyBase64)});
            } else {
                log.debug("Restore unencrypted object by copying the last live version {} to gs://{}/{}", new Object[]{lastLiveVersionId, this.bucket, key});
                copyRequestBuilder.setSource(source).setTarget(target);
            }
            Storage.CopyRequest copyRequest = copyRequestBuilder.build();
            this.storage.copy(copyRequest);
        }
        catch (StorageException e) {
            throw new TierObjectStoreRetriableException(String.format("Failed to restore object %s (version: %s)", key, lastLiveVersionId), e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException(String.format("Unknown exception when restoring object %s (version: %s)", key, lastLiveVersionId), e);
        }
    }

    private List<BlobId> objectsForSegment(TierObjectStore.ObjectMetadata objectMetadata) {
        ArrayList<BlobId> blobIds = new ArrayList<BlobId>();
        block5: for (TierObjectStore.FileType type : TierObjectStore.FileType.values()) {
            switch (type) {
                case TRANSACTION_INDEX: {
                    if (!objectMetadata.hasAbortedTxns()) continue block5;
                    blobIds.add(BlobId.of((String)this.bucket, (String)this.keyPath(objectMetadata, type)));
                    continue block5;
                }
                case PRODUCER_STATE: {
                    if (!objectMetadata.hasProducerState()) continue block5;
                    blobIds.add(BlobId.of((String)this.bucket, (String)this.keyPath(objectMetadata, type)));
                    continue block5;
                }
                case EPOCH_STATE: {
                    if (!objectMetadata.hasEpochState()) continue block5;
                    blobIds.add(BlobId.of((String)this.bucket, (String)this.keyPath(objectMetadata, type)));
                    continue block5;
                }
                default: {
                    blobIds.add(BlobId.of((String)this.bucket, (String)this.keyPath(objectMetadata, type)));
                }
            }
        }
        return blobIds;
    }

    @Override
    public void deleteSegment(TierObjectStore.ObjectMetadata objectMetadata) {
        List<BlobId> blobIds = this.objectsForSegment(objectMetadata);
        log.debug("Deleting " + blobIds);
        ArrayList<BlobId> foundBlobIds = new ArrayList<BlobId>();
        try {
            List success = this.storage.delete(blobIds);
            log.debug("Deletion result " + success);
            for (int blobIndex = 0; blobIndex < success.size(); ++blobIndex) {
                Blob blob;
                if (((Boolean)success.get(blobIndex)).booleanValue() || (blob = this.storage.get(blobIds.get(blobIndex))) == null) continue;
                log.warn("Found object " + blob.getBlobId() + " that was expected to be deleted of " + objectMetadata);
                foundBlobIds.add(blob.getBlobId());
            }
        }
        catch (StorageException e) {
            throw new TierObjectStoreRetriableException("Failed to delete segment: " + objectMetadata, e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException("Unknown exception when deleting segment: " + objectMetadata, e);
        }
        if (!foundBlobIds.isEmpty()) {
            throw new TierObjectStoreRetriableException("Deletion failed for " + objectMetadata + ". Blobs still exist in object storage with blob ids: " + foundBlobIds);
        }
    }

    @Override
    public void deleteVersions(List<TierObjectStore.KeyAndVersion> keys) {
        ArrayList<BlobId> blobsToDelete = new ArrayList<BlobId>();
        for (TierObjectStore.KeyAndVersion key : keys) {
            BlobId blobId = key.versionId() == null ? BlobId.of((String)this.bucket, (String)key.key()) : BlobId.of((String)this.bucket, (String)key.key(), (Long)Long.parseLong(key.versionId()));
            blobsToDelete.add(blobId);
            log.debug("TierObjectStore sending delete request for " + blobId.getName());
            if (blobsToDelete.size() < 500) continue;
            this.deleteObjects(blobsToDelete);
            blobsToDelete.clear();
        }
        if (!blobsToDelete.isEmpty()) {
            this.deleteObjects(blobsToDelete);
        }
    }

    private void deleteObjects(List<BlobId> blobsToDelete) {
        List results;
        log.info("TierObjectStore sending batch delete request for " + blobsToDelete.size() + " objects");
        try {
            results = this.storage.delete(blobsToDelete);
        }
        catch (StorageException e) {
            log.error("StorageException while deleting versioned objects of size: " + blobsToDelete.size(), (Throwable)e);
            throw new TierObjectStoreRetriableException("StorageException while deleting versioned objects. " + (Object)((Object)e));
        }
        catch (Exception e) {
            log.error("Fatal exception while deleting versioned objects of size: " + blobsToDelete.size(), (Throwable)e);
            throw new TierObjectStoreFatalException("Fatal exception while deleting versioned objects." + e);
        }
        Iterator resultsIter = results.iterator();
        Iterator<BlobId> blobsIter = blobsToDelete.iterator();
        while (resultsIter.hasNext() && blobsIter.hasNext()) {
            BlobId blobId = blobsIter.next();
            Boolean deletionResult = (Boolean)resultsIter.next();
            if (!deletionResult.booleanValue()) {
                log.warn("Unable to delete blob " + blobId.toString() + ". Could be deletion failure or that blob is not found");
                continue;
            }
            log.info("Deleted blob " + blobId.toString());
        }
    }

    @Override
    public TierObjectAttribute objectExists(TierObjectStore.ObjectMetadata objectMetadata, TierObjectStore.FileType fileType) throws TierObjectStoreRetriableException {
        TierObjectAttribute result = new TierObjectAttribute(false);
        try {
            String key = this.keyPath(objectMetadata, fileType);
            Blob blob = this.storage.get(this.bucket, key, new Storage.BlobGetOption[0]);
            if (blob != null) {
                result.exist = true;
                result.size = blob.getSize();
            }
        }
        catch (StorageException e) {
            throw new TierObjectStoreRetriableException("Failed to check object existence: " + objectMetadata + " type: " + (Object)((Object)fileType), e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException("Unknown exception when checking object existence: " + objectMetadata + " type: " + (Object)((Object)fileType), e);
        }
        return result;
    }

    @Override
    public void close() {
        if (this.encryptionKeyManager != null) {
            this.encryptionKeyManager.close();
        }
    }

    private void putFileUnencrypted(String key, Map<String, String> metadata, File file) throws IOException {
        BlobId blobId = BlobId.of((String)this.bucket, (String)key);
        BlobInfo blobInfo = BlobInfo.newBuilder((BlobId)blobId).setMetadata(metadata).build();
        log.debug("Uploading object to gs://{}/{}", (Object)this.bucket, (Object)key);
        this.doFileWrite(file, this.storage.writer(blobInfo, new Storage.BlobWriteOption[]{Storage.BlobWriteOption.doesNotExist()}));
    }

    private void putFileEncrypted(String key, Map<String, String> metadata, File file, TierObjectStore.OpaqueData opaqueData) throws IOException {
        if (opaqueData == null || opaqueData.isEmpty()) {
            throw new TierObjectStoreFatalException("Encryption was enabled but no valid OpaqueData object was provided");
        }
        KeySha keySha = KeySha.fromRawBytes(opaqueData.intoByteArray());
        KeyContext keyContext = this.encryptionKeyManager.keyContext(keySha);
        if (keyContext == null) {
            throw new TierObjectStoreFatalException(String.format("No valid KeyContext for KeySha '%s'", keySha));
        }
        metadata.putAll(keyContext.metadata);
        BlobId blobId = BlobId.of((String)this.bucket, (String)key);
        BlobInfo blobInfo = BlobInfo.newBuilder((BlobId)blobId).setMetadata(metadata).build();
        log.debug("Uploading encrypted object to gs://{}/{} with KeySha {}", new Object[]{this.bucket, key, keyContext.keySha});
        Storage.BlobWriteOption encryption = Storage.BlobWriteOption.encryptionKey((String)keyContext.cleartextDataKey.base64Encoded());
        this.doFileWrite(file, this.storage.writer(blobInfo, new Storage.BlobWriteOption[]{encryption, Storage.BlobWriteOption.doesNotExist()}));
    }

    private void doFileWrite(File file, WriteChannel channel) throws IOException {
        try (WriteChannel writer = channel;
             FileChannel fileChannel = FileChannel.open(file.toPath(), StandardOpenOption.READ);){
            if (this.writeChunkSize > 0) {
                writer.setChunkSize(this.writeChunkSize);
            }
            long fileLength = file.length();
            for (long position = 0L; position < fileLength; position += fileChannel.transferTo(position, fileLength, (WritableByteChannel)writer)) {
            }
        }
    }

    public static String crc32c(ByteBuffer buf) {
        long crc32c = Crc32C.compute((ByteBuffer)buf, (int)0, (int)buf.remaining());
        ByteBuffer buffer = ByteBuffer.allocate(8);
        buffer.putLong(0, crc32c);
        byte[] bs = new byte[4];
        buffer.position(4);
        buffer.get(bs);
        return CoreUtils.toBase64(bs);
    }

    public void putBuf(String key, Map<String, String> metadata, ByteBuffer buf) throws IOException {
        ByteBuffer dupBuf = buf.duplicate();
        BlobId blobId = BlobId.of((String)this.bucket, (String)key);
        BlobInfo blobInfo = BlobInfo.newBuilder((BlobId)blobId).setMetadata(metadata).setCrc32c(GcsTierObjectStore.crc32c(buf)).build();
        log.debug("Uploading object {}", (Object)key);
        try (WriteChannel writer = this.storage.writer(blobInfo, new Storage.BlobWriteOption[]{Storage.BlobWriteOption.crc32cMatch()});){
            if (this.writeChunkSize > 0) {
                writer.setChunkSize(this.writeChunkSize);
            }
            while (dupBuf.hasRemaining()) {
                writer.write(dupBuf);
            }
        }
    }

    private static EncryptionKeyManager encryptionKeyManager(GcsTierObjectStoreConfig config, Time time, Metrics metrics) {
        if (config.gcsSseCustomerEncryptionKey != null && !config.gcsSseCustomerEncryptionKey.isEmpty()) {
            String uri = config.gcsSseCustomerEncryptionKey;
            String prefix = "gcp-kms://";
            if (!uri.startsWith(prefix)) {
                uri = prefix + uri;
            }
            log.info(String.format("Configuring EncryptionKeyManager using KMS key '%s'", uri));
            try {
                GcpKmsClient.register(Optional.of(uri), config.gcsCredFilePath);
                AeadConfig.register();
                KeysetHandle keysetHandle = KeysetHandle.generateNew((KeyTemplate)KmsAeadKeyManager.createKeyTemplate((String)uri));
                Aead masterKey = (Aead)keysetHandle.getPrimitive(Aead.class);
                return new EncryptionKeyManager(time, metrics, masterKey, config.encryptionKeyManagerKeyRotationInterval);
            }
            catch (GeneralSecurityException e) {
                throw new TierObjectStoreFatalException("Could not construct master key AEAD", e);
            }
        }
        return null;
    }

    private static Storage storage(GcsTierObjectStoreConfig config) {
        if (config.gcsCredFilePath.isPresent()) {
            try {
                GoogleCredentials credentials = GoogleCredentials.fromStream((InputStream)new FileInputStream(config.gcsCredFilePath.get())).createScoped((Collection)Lists.newArrayList((Object[])new String[]{"https://www.googleapis.com/auth/cloud-platform"}));
                return (Storage)((StorageOptions.Builder)StorageOptions.newBuilder().setCredentials((Credentials)credentials)).build().getService();
            }
            catch (IOException e) {
                throw new TierObjectStoreFatalException("Error in opening GCS credentials file", e);
            }
        }
        return (Storage)StorageOptions.getDefaultInstance().getService();
    }

    private void expectBucket(String bucket, String expectedRegion) throws TierObjectStoreFatalException {
        Bucket bucketObj;
        try {
            bucketObj = this.storage.get(bucket, new Storage.BucketGetOption[]{Storage.BucketGetOption.fields((Storage.BucketField[])new Storage.BucketField[]{Storage.BucketField.LOCATION})});
        }
        catch (StorageException e) {
            throw new TierObjectStoreFatalException("Unable to access bucket " + bucket, e);
        }
        if (bucketObj == null) {
            throw new TierObjectStoreFatalException("Configured bucket " + bucket + " does not exist or could not be found");
        }
        String actualRegion = bucketObj.getLocation();
        if (!expectedRegion.equalsIgnoreCase(actualRegion)) {
            log.warn("Bucket region {} does not match expected region {}", (Object)actualRegion, (Object)expectedRegion);
        }
    }

    private String lastActiveKeyPath() {
        return this.prefix + TierObjectStore.DataTypePathPrefix.LAST_ACTIVE_ENCRYPTION_KEY.prefix() + "last-active-key";
    }

    private String keyPath(TierObjectStore.ObjectStoreMetadata objectMetadata, TierObjectStore.FileType fileType) {
        return objectMetadata.toPath(this.prefix, fileType);
    }

    private static class GcsTierObjectStoreResponse
    implements TierObjectStoreResponse {
        private final InputStream inputStream;

        GcsTierObjectStoreResponse(ReadChannel channel, long startOffset, OptionalInt chunkSizeOpt) throws IOException {
            int chunkSize = chunkSizeOpt.orElse(1000000);
            channel.seek(startOffset);
            channel.setChunkSize(chunkSize);
            this.inputStream = Channels.newInputStream((ReadableByteChannel)channel);
        }

        @Override
        public void close() throws IOException {
            this.inputStream.close();
        }

        @Override
        public InputStream getInputStream() {
            return this.inputStream;
        }
    }

    private class EncryptionKeyManagerHook
    implements EncryptionKeyManager.WellKnownKeypathHook {
        private EncryptionKeyManagerHook() {
        }

        @Override
        public void writeWellKnownPathMetadata(Map<String, String> metadata) {
            String path = GcsTierObjectStore.this.lastActiveKeyPath();
            log.info("Uploading newly generated key to path {}", (Object)path);
            BlobId blobId = BlobId.of((String)GcsTierObjectStore.this.bucket, (String)path);
            BlobInfo blobInfo = BlobInfo.newBuilder((BlobId)blobId).setMetadata(metadata).build();
            GcsTierObjectStore.this.storage.create(blobInfo, new Storage.BlobTargetOption[0]);
        }

        @Override
        public Map<String, String> fetchWellKnownPathMetadata() {
            String path = GcsTierObjectStore.this.lastActiveKeyPath();
            log.info("Downloading previously generated key from path {}", (Object)path);
            BlobId blobId = BlobId.of((String)GcsTierObjectStore.this.bucket, (String)path);
            Blob blob = GcsTierObjectStore.this.storage.get(blobId, new Storage.BlobGetOption[]{Storage.BlobGetOption.fields((Storage.BlobField[])new Storage.BlobField[]{Storage.BlobField.METADATA})});
            if (blob != null) {
                return blob.getMetadata();
            }
            return new HashMap<String, String>();
        }
    }
}

