/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.processors.hadoop;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.fs.permission.FsAction;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.nifi.annotation.behavior.InputRequirement;
import org.apache.nifi.annotation.behavior.PrimaryNodeOnly;
import org.apache.nifi.annotation.behavior.Stateful;
import org.apache.nifi.annotation.behavior.TriggerSerially;
import org.apache.nifi.annotation.behavior.TriggerWhenEmpty;
import org.apache.nifi.annotation.behavior.WritesAttribute;
import org.apache.nifi.annotation.behavior.WritesAttributes;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.SeeAlso;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.annotation.lifecycle.OnScheduled;
import org.apache.nifi.components.AllowableValue;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.components.ValidationContext;
import org.apache.nifi.components.ValidationResult;
import org.apache.nifi.components.state.Scope;
import org.apache.nifi.components.state.StateMap;
import org.apache.nifi.distributed.cache.client.DistributedMapCacheClient;
import org.apache.nifi.flowfile.FlowFile;
import org.apache.nifi.flowfile.attributes.CoreAttributes;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.ProcessSession;
import org.apache.nifi.processor.ProcessorInitializationContext;
import org.apache.nifi.processor.Relationship;
import org.apache.nifi.processor.exception.ProcessException;
import org.apache.nifi.processor.util.StandardValidators;
import org.apache.nifi.processors.hadoop.AbstractHadoopProcessor;
import org.apache.nifi.processors.hadoop.FetchHDFS;
import org.apache.nifi.processors.hadoop.GetHDFS;
import org.apache.nifi.processors.hadoop.PutHDFS;
import org.apache.nifi.schema.access.SchemaNotFoundException;
import org.apache.nifi.serialization.RecordSetWriter;
import org.apache.nifi.serialization.RecordSetWriterFactory;
import org.apache.nifi.serialization.SimpleRecordSchema;
import org.apache.nifi.serialization.WriteResult;
import org.apache.nifi.serialization.record.MapRecord;
import org.apache.nifi.serialization.record.Record;
import org.apache.nifi.serialization.record.RecordField;
import org.apache.nifi.serialization.record.RecordFieldType;
import org.apache.nifi.serialization.record.RecordSchema;

@PrimaryNodeOnly
@TriggerSerially
@TriggerWhenEmpty
@InputRequirement(value=InputRequirement.Requirement.INPUT_FORBIDDEN)
@Tags(value={"hadoop", "HCFS", "HDFS", "get", "list", "ingest", "source", "filesystem"})
@CapabilityDescription(value="Retrieves a listing of files from HDFS. Each time a listing is performed, the files with the latest timestamp will be excluded and picked up during the next execution of the processor. This is done to ensure that we do not miss any files, or produce duplicates, in the cases where files with the same timestamp are written immediately before and after a single execution of the processor. For each file that is listed in HDFS, this processor creates a FlowFile that represents the HDFS file to be fetched in conjunction with FetchHDFS. This Processor is designed to run on Primary Node only in a cluster. If the primary node changes, the new Primary Node will pick up where the previous node left off without duplicating all of the data. Unlike GetHDFS, this Processor does not delete any data from HDFS.")
@WritesAttributes(value={@WritesAttribute(attribute="filename", description="The name of the file that was read from HDFS."), @WritesAttribute(attribute="path", description="The path is set to the absolute path of the file's directory on HDFS. For example, if the Directory property is set to /tmp, then files picked up from /tmp will have the path attribute set to \"./\". If the Recurse Subdirectories property is set to true and a file is picked up from /tmp/abc/1/2/3, then the path attribute will be set to \"/tmp/abc/1/2/3\"."), @WritesAttribute(attribute="hdfs.owner", description="The user that owns the file in HDFS"), @WritesAttribute(attribute="hdfs.group", description="The group that owns the file in HDFS"), @WritesAttribute(attribute="hdfs.lastModified", description="The timestamp of when the file in HDFS was last modified, as milliseconds since midnight Jan 1, 1970 UTC"), @WritesAttribute(attribute="hdfs.length", description="The number of bytes in the file in HDFS"), @WritesAttribute(attribute="hdfs.replication", description="The number of HDFS replicas for hte file"), @WritesAttribute(attribute="hdfs.permissions", description="The permissions for the file in HDFS. This is formatted as 3 characters for the owner, 3 for the group, and 3 for other users. For example rw-rw-r--")})
@Stateful(scopes={Scope.CLUSTER}, description="After performing a listing of HDFS files, the latest timestamp of all the files listed and the latest timestamp of all the files transferred are both stored. This allows the Processor to list only files that have been added or modified after this date the next time that the Processor is run, without having to store all of the actual filenames/paths which could lead to performance problems. State is stored across the cluster so that this Processor can be run on Primary Node only and if a new Primary Node is selected, the new node can pick up where the previous node left off, without duplicating the data.")
@SeeAlso(value={GetHDFS.class, FetchHDFS.class, PutHDFS.class})
public class ListHDFS
extends AbstractHadoopProcessor {
    private static final RecordSchema RECORD_SCHEMA;
    private static final String FILENAME = "filename";
    private static final String PATH = "path";
    private static final String IS_DIRECTORY = "directory";
    private static final String SIZE = "size";
    private static final String LAST_MODIFIED = "lastModified";
    private static final String PERMISSIONS = "permissions";
    private static final String OWNER = "owner";
    private static final String GROUP = "group";
    private static final String REPLICATION = "replication";
    private static final String IS_SYM_LINK = "symLink";
    private static final String IS_ENCRYPTED = "encrypted";
    private static final String IS_ERASURE_CODED = "erasureCoded";
    @Deprecated
    public static final PropertyDescriptor DISTRIBUTED_CACHE_SERVICE;
    public static final PropertyDescriptor RECURSE_SUBDIRS;
    public static final PropertyDescriptor RECORD_WRITER;
    public static final PropertyDescriptor FILE_FILTER;
    private static final String FILTER_MODE_DIRECTORIES_AND_FILES = "filter-mode-directories-and-files";
    private static final String FILTER_MODE_FILES_ONLY = "filter-mode-files-only";
    private static final String FILTER_MODE_FULL_PATH = "filter-mode-full-path";
    static final AllowableValue FILTER_DIRECTORIES_AND_FILES_VALUE;
    static final AllowableValue FILTER_FILES_ONLY_VALUE;
    static final AllowableValue FILTER_FULL_PATH_VALUE;
    public static final PropertyDescriptor FILE_FILTER_MODE;
    public static final PropertyDescriptor MIN_AGE;
    public static final PropertyDescriptor MAX_AGE;
    public static final Relationship REL_SUCCESS;
    private volatile long latestTimestampListed = -1L;
    private volatile long latestTimestampEmitted = -1L;
    private volatile long lastRunTimestamp = -1L;
    private volatile boolean resetState = false;
    static final String LISTING_TIMESTAMP_KEY = "listing.timestamp";
    static final String EMITTED_TIMESTAMP_KEY = "emitted.timestamp";
    static final long LISTING_LAG_NANOS;
    private Pattern fileFilterRegexPattern;

    protected void init(ProcessorInitializationContext context) {
        super.init(context);
    }

    protected void preProcessConfiguration(Configuration config, ProcessContext context) {
        super.preProcessConfiguration(config, context);
        this.fileFilterRegexPattern = Pattern.compile(context.getProperty(FILE_FILTER).getValue());
    }

    protected File getPersistenceFile() {
        return new File("conf/state/" + this.getIdentifier());
    }

    protected List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        ArrayList<PropertyDescriptor> props = new ArrayList<PropertyDescriptor>(this.properties);
        props.add(DISTRIBUTED_CACHE_SERVICE);
        props.add(DIRECTORY);
        props.add(RECURSE_SUBDIRS);
        props.add(RECORD_WRITER);
        props.add(FILE_FILTER);
        props.add(FILE_FILTER_MODE);
        props.add(MIN_AGE);
        props.add(MAX_AGE);
        return props;
    }

    public Set<Relationship> getRelationships() {
        HashSet<Relationship> relationships = new HashSet<Relationship>();
        relationships.add(REL_SUCCESS);
        return relationships;
    }

    protected Collection<ValidationResult> customValidate(ValidationContext context) {
        long maximumAge;
        ArrayList<ValidationResult> problems = new ArrayList<ValidationResult>(super.customValidate(context));
        Long minAgeProp = context.getProperty(MIN_AGE).asTimePeriod(TimeUnit.MILLISECONDS);
        Long maxAgeProp = context.getProperty(MAX_AGE).asTimePeriod(TimeUnit.MILLISECONDS);
        long minimumAge = minAgeProp == null ? 0L : minAgeProp;
        long l = maximumAge = maxAgeProp == null ? Long.MAX_VALUE : maxAgeProp;
        if (minimumAge > maximumAge) {
            problems.add(new ValidationResult.Builder().valid(false).subject("GetHDFS Configuration").explanation(MIN_AGE.getDisplayName() + " cannot be greater than " + MAX_AGE.getDisplayName()).build());
        }
        return problems;
    }

    protected String getKey(String directory) {
        return this.getIdentifier() + ".lastListingTime." + directory;
    }

    public void onPropertyModified(PropertyDescriptor descriptor, String oldValue, String newValue) {
        super.onPropertyModified(descriptor, oldValue, newValue);
        if (this.isConfigurationRestored() && (descriptor.equals((Object)DIRECTORY) || descriptor.equals((Object)FILE_FILTER))) {
            this.resetState = true;
        }
    }

    Set<FileStatus> determineListable(Set<FileStatus> statuses, ProcessContext context) {
        long minTimestamp = this.latestTimestampListed;
        TreeMap<Long, ArrayList<FileStatus>> orderedEntries = new TreeMap<Long, ArrayList<FileStatus>>();
        Long minAgeProp = context.getProperty(MIN_AGE).asTimePeriod(TimeUnit.MILLISECONDS);
        long minimumAge = minAgeProp == null ? Long.MIN_VALUE : minAgeProp;
        Long maxAgeProp = context.getProperty(MAX_AGE).asTimePeriod(TimeUnit.MILLISECONDS);
        long maximumAge = maxAgeProp == null ? Long.MAX_VALUE : maxAgeProp;
        for (FileStatus status : statuses) {
            boolean newEntry;
            long fileAge;
            if (status.getPath().getName().endsWith("_COPYING_") || minimumAge > (fileAge = System.currentTimeMillis() - status.getModificationTime()) || fileAge > maximumAge) continue;
            long entityTimestamp = status.getModificationTime();
            if (entityTimestamp > this.latestTimestampListed) {
                this.latestTimestampListed = entityTimestamp;
            }
            if (!(newEntry = entityTimestamp >= minTimestamp && entityTimestamp > this.latestTimestampEmitted)) continue;
            ArrayList<FileStatus> entitiesForTimestamp = (ArrayList<FileStatus>)orderedEntries.get(status.getModificationTime());
            if (entitiesForTimestamp == null) {
                entitiesForTimestamp = new ArrayList<FileStatus>();
                orderedEntries.put(status.getModificationTime(), entitiesForTimestamp);
            }
            entitiesForTimestamp.add(status);
        }
        HashSet<FileStatus> toList = new HashSet<FileStatus>();
        if (orderedEntries.size() > 0) {
            long latestListingTimestamp = (Long)orderedEntries.lastKey();
            if (latestListingTimestamp == minTimestamp) {
                if (latestListingTimestamp == this.latestTimestampEmitted) {
                    return Collections.emptySet();
                }
            } else {
                orderedEntries.remove(latestListingTimestamp);
            }
            for (List timestampEntities : orderedEntries.values()) {
                for (FileStatus status : timestampEntities) {
                    toList.add(status);
                }
            }
        }
        return toList;
    }

    @OnScheduled
    public void resetStateIfNecessary(ProcessContext context) throws IOException {
        if (this.resetState) {
            this.getLogger().debug("Property has been modified. Resetting the state values - listing.timestamp and emitted.timestamp to -1L");
            context.getStateManager().clear(Scope.CLUSTER);
            this.resetState = false;
        }
    }

    public void onTrigger(ProcessContext context, ProcessSession session) throws ProcessException {
        Set<FileStatus> statuses;
        long now = System.nanoTime();
        if (now - this.lastRunTimestamp < LISTING_LAG_NANOS) {
            this.lastRunTimestamp = now;
            context.yield();
            return;
        }
        this.lastRunTimestamp = now;
        try {
            StateMap stateMap = session.getState(Scope.CLUSTER);
            if (stateMap.getVersion() == -1L) {
                this.latestTimestampEmitted = -1L;
                this.latestTimestampListed = -1L;
                this.getLogger().debug("Found no state stored");
            } else {
                String emittedString = stateMap.get(EMITTED_TIMESTAMP_KEY);
                if (emittedString == null) {
                    this.latestTimestampEmitted = -1L;
                    this.latestTimestampListed = -1L;
                    this.getLogger().debug("Found no recognized state keys; assuming no relevant state and resetting listing/emitted time to -1");
                } else {
                    this.latestTimestampEmitted = Long.parseLong(emittedString);
                    String listingTimestmapString = stateMap.get(LISTING_TIMESTAMP_KEY);
                    if (listingTimestmapString != null) {
                        this.latestTimestampListed = Long.parseLong(listingTimestmapString);
                    }
                    this.getLogger().debug("Found new-style state stored, latesting timestamp emitted = {}, latest listed = {}", new Object[]{this.latestTimestampEmitted, this.latestTimestampListed});
                }
            }
        }
        catch (IOException ioe) {
            this.getLogger().error("Failed to retrieve timestamp of last listing from the State Manager. Will not perform listing until this is accomplished.");
            context.yield();
            return;
        }
        FileSystem hdfs = this.getFileSystem();
        boolean recursive = context.getProperty(RECURSE_SUBDIRS).asBoolean();
        String fileFilterMode = context.getProperty(FILE_FILTER_MODE).getValue();
        try {
            Path rootPath = this.getNormalizedPath(context, DIRECTORY);
            statuses = this.getStatuses(rootPath, recursive, hdfs, this.createPathFilter(context), fileFilterMode);
            this.getLogger().debug("Found a total of {} files in HDFS", new Object[]{statuses.size()});
        }
        catch (IOException | IllegalArgumentException e) {
            this.getLogger().error("Failed to perform listing of HDFS", (Throwable)e);
            return;
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            this.getLogger().error("Interrupted while performing listing of HDFS", (Throwable)e);
            return;
        }
        Set<FileStatus> listable = this.determineListable(statuses, context);
        this.getLogger().debug("Of the {} files found in HDFS, {} are listable", new Object[]{statuses.size(), listable.size()});
        if (!listable.isEmpty()) {
            if (context.getProperty(RECORD_WRITER).isSet()) {
                try {
                    this.createRecords(listable, context, session);
                }
                catch (IOException | SchemaNotFoundException e) {
                    this.getLogger().error("Failed to write listing of HDFS", e);
                    return;
                }
            } else {
                this.createFlowFiles(listable, session);
            }
        }
        for (FileStatus status : listable) {
            long fileModTime = status.getModificationTime();
            if (fileModTime <= this.latestTimestampEmitted) continue;
            this.latestTimestampEmitted = fileModTime;
        }
        HashMap<String, String> updatedState = new HashMap<String, String>(1);
        updatedState.put(LISTING_TIMESTAMP_KEY, String.valueOf(this.latestTimestampListed));
        updatedState.put(EMITTED_TIMESTAMP_KEY, String.valueOf(this.latestTimestampEmitted));
        this.getLogger().debug("New state map: {}", new Object[]{updatedState});
        try {
            session.setState(updatedState, Scope.CLUSTER);
        }
        catch (IOException ioe) {
            this.getLogger().warn("Failed to save cluster-wide state. If NiFi is restarted, data duplication may occur", (Throwable)ioe);
        }
        int listCount = listable.size();
        if (listCount > 0) {
            this.getLogger().info("Successfully created listing with {} new files from HDFS", new Object[]{listCount});
            session.commitAsync();
        } else {
            this.getLogger().debug("There is no data to list. Yielding.");
            context.yield();
        }
    }

    private void createFlowFiles(Set<FileStatus> fileStatuses, ProcessSession session) {
        for (FileStatus status : fileStatuses) {
            Map<String, String> attributes = this.createAttributes(status);
            FlowFile flowFile = session.create();
            flowFile = session.putAllAttributes(flowFile, attributes);
            session.transfer(flowFile, this.getSuccessRelationship());
        }
    }

    private void createRecords(Set<FileStatus> fileStatuses, ProcessContext context, ProcessSession session) throws IOException, SchemaNotFoundException {
        WriteResult writeResult;
        RecordSetWriterFactory writerFactory = (RecordSetWriterFactory)context.getProperty(RECORD_WRITER).asControllerService(RecordSetWriterFactory.class);
        FlowFile flowFile = session.create();
        try (OutputStream out = session.write(flowFile);
             RecordSetWriter recordSetWriter = writerFactory.createWriter(this.getLogger(), this.getRecordSchema(), out, Collections.emptyMap());){
            recordSetWriter.beginRecordSet();
            for (FileStatus fileStatus : fileStatuses) {
                Record record = this.createRecord(fileStatus);
                recordSetWriter.write(record);
            }
            writeResult = recordSetWriter.finishRecordSet();
        }
        HashMap<String, String> attributes = new HashMap<String, String>(writeResult.getAttributes());
        attributes.put("record.count", String.valueOf(writeResult.getRecordCount()));
        flowFile = session.putAllAttributes(flowFile, attributes);
        session.transfer(flowFile, this.getSuccessRelationship());
    }

    private Record createRecord(FileStatus fileStatus) {
        HashMap<String, Object> values = new HashMap<String, Object>();
        values.put(FILENAME, fileStatus.getPath().getName());
        values.put(PATH, this.getAbsolutePath(fileStatus.getPath().getParent()));
        values.put(OWNER, fileStatus.getOwner());
        values.put(GROUP, fileStatus.getGroup());
        values.put(LAST_MODIFIED, new Timestamp(fileStatus.getModificationTime()));
        values.put(SIZE, fileStatus.getLen());
        values.put(REPLICATION, fileStatus.getReplication());
        FsPermission permission = fileStatus.getPermission();
        String perms = this.getPerms(permission.getUserAction()) + this.getPerms(permission.getGroupAction()) + this.getPerms(permission.getOtherAction());
        values.put(PERMISSIONS, perms);
        values.put(IS_DIRECTORY, fileStatus.isDirectory());
        values.put(IS_SYM_LINK, fileStatus.isSymlink());
        values.put(IS_ENCRYPTED, fileStatus.isEncrypted());
        values.put(IS_ERASURE_CODED, fileStatus.isErasureCoded());
        return new MapRecord(this.getRecordSchema(), values);
    }

    private RecordSchema getRecordSchema() {
        return RECORD_SCHEMA;
    }

    private Set<FileStatus> getStatuses(Path path, boolean recursive, FileSystem hdfs, PathFilter filter, String filterMode) throws IOException, InterruptedException {
        HashSet<FileStatus> statusSet = new HashSet<FileStatus>();
        this.getLogger().debug("Fetching listing for {}", new Object[]{path});
        FileStatus[] statuses = this.isPostListingFilterNeeded(filterMode) ? (FileStatus[])this.getUserGroupInformation().doAs(() -> hdfs.listStatus(path)) : (FileStatus[])this.getUserGroupInformation().doAs(() -> hdfs.listStatus(path, filter));
        for (FileStatus status : statuses) {
            if (status.isDirectory()) {
                if (!recursive) continue;
                try {
                    statusSet.addAll(this.getStatuses(status.getPath(), recursive, hdfs, filter, filterMode));
                }
                catch (IOException ioe) {
                    this.getLogger().error("Failed to retrieve HDFS listing for subdirectory {} due to {}; will continue listing others", new Object[]{status.getPath(), ioe});
                }
                continue;
            }
            if (this.isPostListingFilterNeeded(filterMode)) {
                if (!filter.accept(status.getPath())) continue;
                statusSet.add(status);
                continue;
            }
            statusSet.add(status);
        }
        return statusSet;
    }

    private boolean isPostListingFilterNeeded(String filterMode) {
        return filterMode.equals(FILTER_MODE_FILES_ONLY) || filterMode.equals(FILTER_MODE_FULL_PATH);
    }

    private String getAbsolutePath(Path path) {
        Path parent = path.getParent();
        String prefix = parent == null || parent.getName().equals("") ? "" : this.getAbsolutePath(parent);
        return prefix + "/" + path.getName();
    }

    private Map<String, String> createAttributes(FileStatus status) {
        HashMap<String, String> attributes = new HashMap<String, String>();
        attributes.put(CoreAttributes.FILENAME.key(), status.getPath().getName());
        attributes.put(CoreAttributes.PATH.key(), this.getAbsolutePath(status.getPath().getParent()));
        attributes.put(this.getAttributePrefix() + ".owner", status.getOwner());
        attributes.put(this.getAttributePrefix() + ".group", status.getGroup());
        attributes.put(this.getAttributePrefix() + ".lastModified", String.valueOf(status.getModificationTime()));
        attributes.put(this.getAttributePrefix() + ".length", String.valueOf(status.getLen()));
        attributes.put(this.getAttributePrefix() + ".replication", String.valueOf(status.getReplication()));
        FsPermission permission = status.getPermission();
        String perms = this.getPerms(permission.getUserAction()) + this.getPerms(permission.getGroupAction()) + this.getPerms(permission.getOtherAction());
        attributes.put(this.getAttributePrefix() + ".permissions", perms);
        return attributes;
    }

    private String getPerms(FsAction action) {
        StringBuilder sb = new StringBuilder();
        if (action.implies(FsAction.READ)) {
            sb.append("r");
        } else {
            sb.append("-");
        }
        if (action.implies(FsAction.WRITE)) {
            sb.append("w");
        } else {
            sb.append("-");
        }
        if (action.implies(FsAction.EXECUTE)) {
            sb.append("x");
        } else {
            sb.append("-");
        }
        return sb.toString();
    }

    private PathFilter createPathFilter(ProcessContext context) {
        String filterMode = context.getProperty(FILE_FILTER_MODE).getValue();
        return path -> {
            boolean accepted = FILTER_FULL_PATH_VALUE.getValue().equals(filterMode) ? this.fileFilterRegexPattern.matcher(path.toString()).matches() || this.fileFilterRegexPattern.matcher(Path.getPathWithoutSchemeAndAuthority((Path)path).toString()).matches() : this.fileFilterRegexPattern.matcher(path.getName()).matches();
            return accepted;
        };
    }

    protected Relationship getSuccessRelationship() {
        return REL_SUCCESS;
    }

    protected String getAttributePrefix() {
        return "hdfs";
    }

    static {
        ArrayList<RecordField> recordFields = new ArrayList<RecordField>();
        recordFields.add(new RecordField(FILENAME, RecordFieldType.STRING.getDataType(), false));
        recordFields.add(new RecordField(PATH, RecordFieldType.STRING.getDataType(), false));
        recordFields.add(new RecordField(IS_DIRECTORY, RecordFieldType.BOOLEAN.getDataType(), false));
        recordFields.add(new RecordField(SIZE, RecordFieldType.LONG.getDataType(), false));
        recordFields.add(new RecordField(LAST_MODIFIED, RecordFieldType.TIMESTAMP.getDataType(), false));
        recordFields.add(new RecordField(PERMISSIONS, RecordFieldType.STRING.getDataType()));
        recordFields.add(new RecordField(OWNER, RecordFieldType.STRING.getDataType()));
        recordFields.add(new RecordField(GROUP, RecordFieldType.STRING.getDataType()));
        recordFields.add(new RecordField(REPLICATION, RecordFieldType.INT.getDataType()));
        recordFields.add(new RecordField(IS_SYM_LINK, RecordFieldType.BOOLEAN.getDataType()));
        recordFields.add(new RecordField(IS_ENCRYPTED, RecordFieldType.BOOLEAN.getDataType()));
        recordFields.add(new RecordField(IS_ERASURE_CODED, RecordFieldType.BOOLEAN.getDataType()));
        RECORD_SCHEMA = new SimpleRecordSchema(recordFields);
        DISTRIBUTED_CACHE_SERVICE = new PropertyDescriptor.Builder().name("Distributed Cache Service").description("This property is ignored.  State will be stored in the " + Scope.LOCAL + " or " + Scope.CLUSTER + " scope by the State Manager based on NiFi's configuration.").required(false).identifiesControllerService(DistributedMapCacheClient.class).build();
        RECURSE_SUBDIRS = new PropertyDescriptor.Builder().name("Recurse Subdirectories").description("Indicates whether to list files from subdirectories of the HDFS directory").required(true).allowableValues(new String[]{"true", "false"}).defaultValue("true").build();
        RECORD_WRITER = new PropertyDescriptor.Builder().name("record-writer").displayName("Record Writer").description("Specifies the Record Writer to use for creating the listing. If not specified, one FlowFile will be created for each entity that is listed. If the Record Writer is specified, all entities will be written to a single FlowFile.").required(false).identifiesControllerService(RecordSetWriterFactory.class).build();
        FILE_FILTER = new PropertyDescriptor.Builder().name("File Filter").description("Only files whose names match the given regular expression will be picked up").required(true).defaultValue("[^\\.].*").addValidator(StandardValidators.REGULAR_EXPRESSION_VALIDATOR).build();
        FILTER_DIRECTORIES_AND_FILES_VALUE = new AllowableValue(FILTER_MODE_DIRECTORIES_AND_FILES, "Directories and Files", "Filtering will be applied to the names of directories and files.  If " + RECURSE_SUBDIRS.getDisplayName() + " is set to true, only subdirectories with a matching name will be searched for files that match the regular expression defined in " + FILE_FILTER.getDisplayName() + ".");
        FILTER_FILES_ONLY_VALUE = new AllowableValue(FILTER_MODE_FILES_ONLY, "Files Only", "Filtering will only be applied to the names of files.  If " + RECURSE_SUBDIRS.getDisplayName() + " is set to true, the entire subdirectory tree will be searched for files that match the regular expression defined in " + FILE_FILTER.getDisplayName() + ".");
        FILTER_FULL_PATH_VALUE = new AllowableValue(FILTER_MODE_FULL_PATH, "Full Path", "Filtering will be applied by evaluating the regular expression defined in " + FILE_FILTER.getDisplayName() + " against the full path of files with and without the scheme and authority.  If " + RECURSE_SUBDIRS.getDisplayName() + " is set to true, the entire subdirectory tree will be searched for files in which the full path of the file matches the regular expression defined in " + FILE_FILTER.getDisplayName() + ".  See 'Additional Details' for more information.");
        FILE_FILTER_MODE = new PropertyDescriptor.Builder().name("file-filter-mode").displayName("File Filter Mode").description("Determines how the regular expression in  " + FILE_FILTER.getDisplayName() + " will be used when retrieving listings.").required(true).allowableValues(new AllowableValue[]{FILTER_DIRECTORIES_AND_FILES_VALUE, FILTER_FILES_ONLY_VALUE, FILTER_FULL_PATH_VALUE}).defaultValue(FILTER_DIRECTORIES_AND_FILES_VALUE.getValue()).addValidator(StandardValidators.REGULAR_EXPRESSION_VALIDATOR).build();
        MIN_AGE = new PropertyDescriptor.Builder().name("minimum-file-age").displayName("Minimum File Age").description("The minimum age that a file must be in order to be pulled; any file younger than this amount of time (based on last modification date) will be ignored").required(false).addValidator(StandardValidators.createTimePeriodValidator((long)0L, (TimeUnit)TimeUnit.MILLISECONDS, (long)Long.MAX_VALUE, (TimeUnit)TimeUnit.NANOSECONDS)).build();
        MAX_AGE = new PropertyDescriptor.Builder().name("maximum-file-age").displayName("Maximum File Age").description("The maximum age that a file must be in order to be pulled; any file older than this amount of time (based on last modification date) will be ignored. Minimum value is 100ms.").required(false).addValidator(StandardValidators.createTimePeriodValidator((long)100L, (TimeUnit)TimeUnit.MILLISECONDS, (long)Long.MAX_VALUE, (TimeUnit)TimeUnit.NANOSECONDS)).build();
        REL_SUCCESS = new Relationship.Builder().name("success").description("All FlowFiles are transferred to this relationship").build();
        LISTING_LAG_NANOS = TimeUnit.MILLISECONDS.toNanos(100L);
    }
}

