/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.nar;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileFilter;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.jar.Attributes;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.Manifest;
import org.apache.nifi.bundle.Bundle;
import org.apache.nifi.bundle.BundleCoordinate;
import org.apache.nifi.nar.ExtensionMapping;
import org.apache.nifi.nar.FileDigestUtils;
import org.apache.nifi.nar.NarManifestEntry;
import org.apache.nifi.util.FileUtils;
import org.apache.nifi.util.NiFiProperties;
import org.apache.nifi.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class NarUnpacker {
    public static final String BUNDLED_DEPENDENCIES_DIRECTORY = "NAR-INF/bundled-dependencies";
    private static final Logger logger = LoggerFactory.getLogger(NarUnpacker.class);
    private static final String HASH_FILENAME = "nar-digest";
    private static final FileFilter NAR_FILTER = pathname -> {
        String nameToTest = pathname.getName().toLowerCase();
        return nameToTest.endsWith(".nar") && pathname.isFile();
    };

    public static ExtensionMapping unpackNars(NiFiProperties props, Bundle systemBundle) {
        return NarUnpacker.unpackNars(props, "nifi-framework-nar", systemBundle);
    }

    public static ExtensionMapping unpackNars(NiFiProperties props, String frameworkNarId, Bundle systemBundle) {
        List narLibraryDirs = props.getNarLibraryDirectories();
        File frameworkWorkingDir = props.getFrameworkWorkingDirectory();
        File extensionsWorkingDir = props.getExtensionsWorkingDirectory();
        File docsWorkingDir = props.getComponentDocumentationWorkingDirectory();
        return NarUnpacker.unpackNars(systemBundle, frameworkWorkingDir, frameworkNarId, extensionsWorkingDir, docsWorkingDir, narLibraryDirs);
    }

    public static ExtensionMapping unpackNars(Bundle systemBundle, File frameworkWorkingDir, String frameworkNarId, File extensionsWorkingDir, File docsWorkingDir, List<Path> narLibraryDirs) {
        return NarUnpacker.unpackNars(systemBundle, frameworkWorkingDir, extensionsWorkingDir, docsWorkingDir, narLibraryDirs, true, frameworkNarId, true, true, coordinate -> true);
    }

    public static ExtensionMapping unpackNars(Bundle systemBundle, File frameworkWorkingDir, File extensionsWorkingDir, File docsWorkingDir, List<Path> narLibraryDirs, boolean requireFrameworkNar, String frameworkNarId, boolean requireJettyNar, boolean verifyHash, Predicate<BundleCoordinate> narFilter) {
        HashMap<File, BundleCoordinate> unpackedNars = new HashMap<File, BundleCoordinate>();
        try {
            File unpackedJetty = null;
            File unpackedFramework = null;
            HashSet<File> unpackedExtensions = new HashSet<File>();
            ArrayList<File> narFiles = new ArrayList<File>();
            if (requireFrameworkNar) {
                FileUtils.ensureDirectoryExistAndCanReadAndWrite(frameworkWorkingDir);
            }
            FileUtils.ensureDirectoryExistAndCanReadAndWrite(extensionsWorkingDir);
            if (docsWorkingDir != null) {
                FileUtils.ensureDirectoryExistAndCanReadAndWrite(docsWorkingDir);
            }
            for (Path narLibraryDir : narLibraryDirs) {
                File narDir = narLibraryDir.toFile();
                FileUtils.ensureDirectoryExistAndCanRead(narDir);
                File[] fileArray = narDir.listFiles(NAR_FILTER);
                if (fileArray == null) continue;
                List<File> fileList = Arrays.asList(fileArray);
                narFiles.addAll(fileList);
            }
            if (!narFiles.isEmpty()) {
                File[] extensionsWorkingDirContents;
                File[] frameworkWorkingDirContents;
                long startTime = System.nanoTime();
                logger.info("Expanding " + narFiles.size() + " NAR files with all processors...");
                for (File file : narFiles) {
                    logger.debug("Expanding NAR file: " + file.getAbsolutePath());
                    JarFile nar = new JarFile(file);
                    Throwable throwable = null;
                    try {
                        BundleCoordinate bundleCoordinate = NarUnpacker.createBundleCoordinate(nar.getManifest());
                        if (!narFilter.test(bundleCoordinate)) {
                            logger.debug("Will not expand NAR {} because it does not match the provided filter", (Object)bundleCoordinate);
                            continue;
                        }
                        if (frameworkNarId != null && frameworkNarId.equals(bundleCoordinate.getId())) {
                            if (unpackedFramework != null) {
                                throw new IllegalStateException("Multiple framework NARs discovered. Only one framework is permitted.");
                            }
                            unpackedFramework = NarUnpacker.unpackNar(file, frameworkWorkingDir, verifyHash);
                            continue;
                        }
                        if ("nifi-jetty-bundle".equals(bundleCoordinate.getId())) {
                            if (unpackedJetty != null) {
                                throw new IllegalStateException("Multiple Jetty NARs discovered. Only one Jetty NAR is permitted.");
                            }
                            unpackedJetty = NarUnpacker.unpackNar(file, extensionsWorkingDir, verifyHash);
                            unpackedExtensions.add(unpackedJetty);
                            continue;
                        }
                        File unpackedExtension = NarUnpacker.unpackNar(file, extensionsWorkingDir, verifyHash);
                        unpackedExtensions.add(unpackedExtension);
                    }
                    catch (Throwable bundleCoordinate) {
                        throwable = bundleCoordinate;
                        throw bundleCoordinate;
                    }
                    finally {
                        if (nar == null) continue;
                        if (throwable != null) {
                            try {
                                nar.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                            continue;
                        }
                        nar.close();
                    }
                }
                if (requireFrameworkNar) {
                    if (unpackedFramework == null) {
                        throw new IllegalStateException("No framework NAR found.");
                    }
                    if (!unpackedFramework.canRead()) {
                        throw new IllegalStateException("Framework NAR cannot be read.");
                    }
                }
                if (requireJettyNar) {
                    if (unpackedJetty == null) {
                        throw new IllegalStateException("No Jetty NAR found.");
                    }
                    if (!unpackedJetty.canRead()) {
                        throw new IllegalStateException("Jetty NAR cannot be read.");
                    }
                }
                if (unpackedFramework != null && frameworkWorkingDir != null && (frameworkWorkingDirContents = frameworkWorkingDir.listFiles()) != null) {
                    for (File unpackedNar : frameworkWorkingDirContents) {
                        if (unpackedFramework.equals(unpackedNar)) continue;
                        FileUtils.deleteFile(unpackedNar, true);
                    }
                }
                if ((extensionsWorkingDirContents = extensionsWorkingDir.listFiles()) != null) {
                    for (File unpackedNar : extensionsWorkingDirContents) {
                        if (unpackedExtensions.contains(unpackedNar)) continue;
                        FileUtils.deleteFile(unpackedNar, true);
                    }
                }
                long l = System.nanoTime() - startTime;
                logger.info("NAR loading process took " + l + " nanoseconds (" + (int)TimeUnit.SECONDS.convert(l, TimeUnit.NANOSECONDS) + " seconds).");
            }
            unpackedNars.putAll(NarUnpacker.createUnpackedNarBundleCoordinateMap(extensionsWorkingDir));
            ExtensionMapping extensionMapping = new ExtensionMapping();
            NarUnpacker.mapExtensions(unpackedNars, docsWorkingDir, extensionMapping);
            NarUnpacker.unpackBundleDocs(docsWorkingDir, extensionMapping, systemBundle.getBundleDetails().getCoordinate(), systemBundle.getBundleDetails().getWorkingDirectory());
            return extensionMapping;
        }
        catch (IOException e) {
            logger.warn("Unable to load NAR library bundles due to " + e + " Will proceed without loading any further Nar bundles");
            if (logger.isDebugEnabled()) {
                logger.warn("", (Throwable)e);
            }
            return null;
        }
    }

    private static Map<File, BundleCoordinate> createUnpackedNarBundleCoordinateMap(File extensionsWorkingDir) {
        File[] unpackedDirs;
        HashMap<File, BundleCoordinate> result = new HashMap<File, BundleCoordinate>();
        for (File unpackedDir : unpackedDirs = extensionsWorkingDir.listFiles(file -> file.isDirectory() && file.getName().endsWith("nar-unpacked"))) {
            Path mf = Paths.get(unpackedDir.getAbsolutePath(), "META-INF", "MANIFEST.MF");
            try (InputStream is = Files.newInputStream(mf, new OpenOption[0]);){
                Manifest manifest = new Manifest(is);
                BundleCoordinate bundleCoordinate = NarUnpacker.createBundleCoordinate(manifest);
                result.put(unpackedDir, bundleCoordinate);
            }
            catch (IOException e) {
                logger.error(String.format("Unable to parse NAR information from unpacked nar directory [%s].", unpackedDir.getAbsoluteFile()), (Throwable)e);
            }
        }
        return result;
    }

    private static BundleCoordinate createBundleCoordinate(Manifest manifest) {
        Attributes mainAttributes = manifest.getMainAttributes();
        String groupId = mainAttributes.getValue(NarManifestEntry.NAR_GROUP.getManifestName());
        String narId = mainAttributes.getValue(NarManifestEntry.NAR_ID.getManifestName());
        String version = mainAttributes.getValue(NarManifestEntry.NAR_VERSION.getManifestName());
        BundleCoordinate bundleCoordinate = new BundleCoordinate(groupId, narId, version);
        return bundleCoordinate;
    }

    private static void mapExtensions(Map<File, BundleCoordinate> unpackedNars, File docsDirectory, ExtensionMapping mapping) throws IOException {
        for (Map.Entry<File, BundleCoordinate> entry : unpackedNars.entrySet()) {
            File unpackedNar = entry.getKey();
            BundleCoordinate bundleCoordinate = entry.getValue();
            File bundledDependencies = new File(unpackedNar, BUNDLED_DEPENDENCIES_DIRECTORY);
            if (docsDirectory == null) continue;
            NarUnpacker.unpackBundleDocs(docsDirectory, mapping, bundleCoordinate, bundledDependencies);
        }
    }

    public static void mapExtension(File unpackedNar, BundleCoordinate bundleCoordinate, File docsDirectory, ExtensionMapping mapping) throws IOException {
        File bundledDependencies = new File(unpackedNar, BUNDLED_DEPENDENCIES_DIRECTORY);
        NarUnpacker.unpackBundleDocs(docsDirectory, mapping, bundleCoordinate, bundledDependencies);
    }

    private static void unpackBundleDocs(File docsDirectory, ExtensionMapping mapping, BundleCoordinate bundleCoordinate, File bundledDirectory) throws IOException {
        File[] directoryContents = bundledDirectory.listFiles();
        if (directoryContents != null) {
            for (File file : directoryContents) {
                if (!file.getName().toLowerCase().endsWith(".jar")) continue;
                NarUnpacker.unpackDocumentation(bundleCoordinate, file, docsDirectory, mapping);
            }
        }
    }

    public static File unpackNar(File nar, File baseWorkingDirectory, boolean verifyHash) throws IOException {
        File narWorkingDirectory = new File(baseWorkingDirectory, nar.getName() + "-unpacked");
        if (!narWorkingDirectory.exists()) {
            NarUnpacker.unpack(nar, narWorkingDirectory, FileDigestUtils.getDigest(nar));
        } else if (verifyHash) {
            byte[] narDigest = FileDigestUtils.getDigest(nar);
            File workingHashFile = new File(narWorkingDirectory, HASH_FILENAME);
            if (!workingHashFile.exists()) {
                FileUtils.deleteFile(narWorkingDirectory, true);
                NarUnpacker.unpack(nar, narWorkingDirectory, narDigest);
            } else {
                byte[] hashFileContents = Files.readAllBytes(workingHashFile.toPath());
                if (!Arrays.equals(hashFileContents, narDigest)) {
                    logger.info("Contents of nar {} have changed. Reloading.", new Object[]{nar.getAbsolutePath()});
                    FileUtils.deleteFile(narWorkingDirectory, true);
                    NarUnpacker.unpack(nar, narWorkingDirectory, narDigest);
                }
            }
        } else {
            logger.debug("Directory {} already exists. Will not verify hash. Assuming nothing has changed.", (Object)narWorkingDirectory);
        }
        return narWorkingDirectory;
    }

    private static void unpack(File nar, File workingDirectory, byte[] hash) throws IOException {
        try (JarFile jarFile = new JarFile(nar);){
            Enumeration<JarEntry> jarEntries = jarFile.entries();
            while (jarEntries.hasMoreElements()) {
                JarEntry jarEntry = jarEntries.nextElement();
                String name = jarEntry.getName();
                if (name.contains("META-INF/bundled-dependencies")) {
                    name = name.replace("META-INF/bundled-dependencies", BUNDLED_DEPENDENCIES_DIRECTORY);
                }
                File f = new File(workingDirectory, name);
                if (jarEntry.isDirectory()) {
                    FileUtils.ensureDirectoryExistAndCanReadAndWrite(f);
                    continue;
                }
                NarUnpacker.makeFile(jarFile.getInputStream(jarEntry), f);
            }
        }
        File hashFile = new File(workingDirectory, HASH_FILENAME);
        try (FileOutputStream fos = new FileOutputStream(hashFile);){
            fos.write(hash);
        }
    }

    private static void unpackDocumentation(BundleCoordinate coordinate, File jar, File docsDirectory, ExtensionMapping extensionMapping) throws IOException {
        ExtensionMapping jarExtensionMapping = NarUnpacker.determineDocumentedNiFiComponents(coordinate, jar);
        if (jarExtensionMapping.isEmpty()) {
            return;
        }
        extensionMapping.merge(jarExtensionMapping);
        if (docsDirectory == null) {
            return;
        }
        try (JarFile jarFile = new JarFile(jar);){
            block9: for (String componentName : jarExtensionMapping.getAllExtensionNames().keySet()) {
                String entryName = "docs/" + componentName;
                Enumeration<JarEntry> jarEnumeration = jarFile.entries();
                while (jarEnumeration.hasMoreElements()) {
                    JarEntry jarEntry = jarEnumeration.nextElement();
                    if (!jarEntry.getName().startsWith(entryName)) continue;
                    String name = StringUtils.substringAfter((String)jarEntry.getName(), (String)"docs/");
                    String path = coordinate.getGroup() + "/" + coordinate.getId() + "/" + coordinate.getVersion() + "/" + name;
                    if (jarEntry.isDirectory()) {
                        File componentDocsDirectory = new File(docsDirectory, path);
                        if (componentDocsDirectory.exists() || componentDocsDirectory.mkdirs()) continue;
                        logger.warn("Unable to create docs directory " + componentDocsDirectory.getAbsolutePath());
                        continue block9;
                    }
                    File componentDoc = new File(docsDirectory, path);
                    NarUnpacker.makeFile(jarFile.getInputStream(jarEntry), componentDoc);
                }
            }
        }
    }

    private static ExtensionMapping determineDocumentedNiFiComponents(BundleCoordinate coordinate, File jar) throws IOException {
        ExtensionMapping mapping = new ExtensionMapping();
        try (JarFile jarFile = new JarFile(jar);){
            JarEntry processorEntry = jarFile.getJarEntry("META-INF/services/org.apache.nifi.processor.Processor");
            JarEntry reportingTaskEntry = jarFile.getJarEntry("META-INF/services/org.apache.nifi.reporting.ReportingTask");
            JarEntry controllerServiceEntry = jarFile.getJarEntry("META-INF/services/org.apache.nifi.controller.ControllerService");
            if (processorEntry == null && reportingTaskEntry == null && controllerServiceEntry == null) {
                ExtensionMapping extensionMapping = mapping;
                return extensionMapping;
            }
            mapping.addAllProcessors(coordinate, NarUnpacker.determineDocumentedNiFiComponents(jarFile, processorEntry));
            mapping.addAllReportingTasks(coordinate, NarUnpacker.determineDocumentedNiFiComponents(jarFile, reportingTaskEntry));
            mapping.addAllControllerServices(coordinate, NarUnpacker.determineDocumentedNiFiComponents(jarFile, controllerServiceEntry));
            ExtensionMapping extensionMapping = mapping;
            return extensionMapping;
        }
    }

    private static List<String> determineDocumentedNiFiComponents(JarFile jarFile, JarEntry jarEntry) throws IOException {
        ArrayList<String> componentNames = new ArrayList<String>();
        if (jarEntry == null) {
            return componentNames;
        }
        try (InputStream entryInputStream = jarFile.getInputStream(jarEntry);
             BufferedReader reader = new BufferedReader(new InputStreamReader(entryInputStream));){
            String line;
            while ((line = reader.readLine()) != null) {
                String trimmedLine = line.trim();
                if (trimmedLine.isEmpty() || trimmedLine.startsWith("#")) continue;
                int indexOfPound = trimmedLine.indexOf("#");
                String effectiveLine = indexOfPound > 0 ? trimmedLine.substring(0, indexOfPound) : trimmedLine;
                componentNames.add(effectiveLine);
            }
        }
        return componentNames;
    }

    private static void makeFile(InputStream inputStream, File file) throws IOException {
        try (InputStream in = inputStream;
             FileOutputStream fos = new FileOutputStream(file);){
            int numRead;
            byte[] bytes = new byte[65536];
            while ((numRead = in.read(bytes)) != -1) {
                fos.write(bytes, 0, numRead);
            }
        }
    }

    private NarUnpacker() {
    }
}

