/*
 * Decompiled with CFR 0.152.
 */
package org.apache.parquet.column.values.delta;

import java.io.IOException;
import java.util.Random;
import org.apache.parquet.bytes.BytesInput;
import org.apache.parquet.column.values.ValuesWriter;
import org.apache.parquet.column.values.delta.DeltaBinaryPackingValuesReader;
import org.apache.parquet.column.values.delta.DeltaBinaryPackingValuesWriter;
import org.apache.parquet.io.ParquetDecodingException;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

public class DeltaBinaryPackingValuesWriterTest {
    DeltaBinaryPackingValuesReader reader;
    private int blockSize;
    private int miniBlockNum;
    private ValuesWriter writer;
    private Random random;

    @Before
    public void setUp() {
        this.blockSize = 128;
        this.miniBlockNum = 4;
        this.writer = new DeltaBinaryPackingValuesWriter(this.blockSize, this.miniBlockNum, 100, 200);
        this.random = new Random();
    }

    @Test(expected=IllegalArgumentException.class)
    public void miniBlockSizeShouldBeMultipleOf8() {
        new DeltaBinaryPackingValuesWriter(1281, 4, 100, 100);
    }

    @Test
    public void shouldWriteWhenDataIsAlignedWithBlock() throws IOException {
        int[] data = new int[5 * this.blockSize];
        for (int i = 0; i < this.blockSize * 5; ++i) {
            data[i] = this.random.nextInt();
        }
        this.shouldWriteAndRead(data);
    }

    @Test
    public void shouldWriteAndReadWhenBlockIsNotFullyWritten() throws IOException {
        int[] data = new int[this.blockSize - 3];
        for (int i = 0; i < data.length; ++i) {
            data[i] = this.random.nextInt();
        }
        this.shouldWriteAndRead(data);
    }

    @Test
    public void shouldWriteAndReadWhenAMiniBlockIsNotFullyWritten() throws IOException {
        int miniBlockSize = this.blockSize / this.miniBlockNum;
        int[] data = new int[miniBlockSize - 3];
        for (int i = 0; i < data.length; ++i) {
            data[i] = this.random.nextInt();
        }
        this.shouldWriteAndRead(data);
    }

    @Test
    public void shouldWriteNegativeDeltas() throws IOException {
        int[] data = new int[this.blockSize];
        for (int i = 0; i < data.length; ++i) {
            data[i] = 10 - (i * 32 - this.random.nextInt(6));
        }
        this.shouldWriteAndRead(data);
    }

    @Test
    public void shouldWriteAndReadWhenDeltasAreSame() throws IOException {
        int[] data = new int[2 * this.blockSize];
        for (int i = 0; i < this.blockSize; ++i) {
            data[i] = i * 32;
        }
        this.shouldWriteAndRead(data);
    }

    @Test
    public void shouldWriteAndReadWhenValuesAreSame() throws IOException {
        int[] data = new int[2 * this.blockSize];
        for (int i = 0; i < this.blockSize; ++i) {
            data[i] = 3;
        }
        this.shouldWriteAndRead(data);
    }

    @Test
    public void shouldWriteWhenDeltaIs0ForEachBlock() throws IOException {
        int[] data = new int[5 * this.blockSize + 1];
        for (int i = 0; i < data.length; ++i) {
            data[i] = (i - 1) / this.blockSize;
        }
        this.shouldWriteAndRead(data);
    }

    @Test
    public void shouldReadWriteWhenDataIsNotAlignedWithBlock() throws IOException {
        int[] data = new int[5 * this.blockSize + 3];
        for (int i = 0; i < data.length; ++i) {
            data[i] = this.random.nextInt(20) - 10;
        }
        this.shouldWriteAndRead(data);
    }

    @Test
    public void shouldReadMaxMinValue() throws IOException {
        int[] data = new int[10];
        for (int i = 0; i < data.length; ++i) {
            data[i] = i % 2 == 0 ? Integer.MIN_VALUE : Integer.MAX_VALUE;
        }
        this.shouldWriteAndRead(data);
    }

    @Test
    public void shouldReturnCorrectOffsetAfterInitialization() throws IOException {
        int[] data = new int[2 * this.blockSize + 3];
        for (int i = 0; i < data.length; ++i) {
            data[i] = i * 32;
        }
        this.writeData(data);
        this.reader = new DeltaBinaryPackingValuesReader();
        BytesInput bytes = this.writer.getBytes();
        byte[] valueContent = bytes.toByteArray();
        byte[] pageContent = new byte[valueContent.length * 10];
        int contentOffsetInPage = 33;
        System.arraycopy(valueContent, 0, pageContent, contentOffsetInPage, valueContent.length);
        this.reader.initFromPage(100, pageContent, contentOffsetInPage);
        int offset = this.reader.getNextOffset();
        Assert.assertEquals((long)(valueContent.length + contentOffsetInPage), (long)offset);
        for (int i : data) {
            Assert.assertEquals((long)i, (long)this.reader.readInteger());
        }
    }

    @Test
    public void shouldThrowExceptionWhenReadMoreThanWritten() throws IOException {
        int[] data = new int[5 * this.blockSize + 1];
        for (int i = 0; i < data.length; ++i) {
            data[i] = i * 32;
        }
        this.shouldWriteAndRead(data);
        try {
            this.reader.readInteger();
        }
        catch (ParquetDecodingException e) {
            Assert.assertEquals((Object)("no more value to read, total value count is " + data.length), (Object)e.getMessage());
        }
    }

    @Test
    public void shouldSkip() throws IOException {
        int i;
        int[] data = new int[5 * this.blockSize + 1];
        for (i = 0; i < data.length; ++i) {
            data[i] = i * 32;
        }
        this.writeData(data);
        this.reader = new DeltaBinaryPackingValuesReader();
        this.reader.initFromPage(100, this.writer.getBytes().toByteArray(), 0);
        for (i = 0; i < data.length; ++i) {
            if (i % 3 == 0) {
                this.reader.skip();
                continue;
            }
            Assert.assertEquals((long)(i * 32), (long)this.reader.readInteger());
        }
    }

    @Test
    public void shouldReset() throws IOException {
        this.shouldReadWriteWhenDataIsNotAlignedWithBlock();
        int[] data = new int[5 * this.blockSize];
        for (int i = 0; i < this.blockSize * 5; ++i) {
            data[i] = i * 2;
        }
        this.writer.reset();
        this.shouldWriteAndRead(data);
    }

    @Test
    public void randomDataTest() throws IOException {
        int maxSize = 1000;
        int[] data = new int[maxSize];
        for (int round = 0; round < 100000; ++round) {
            int size = this.random.nextInt(maxSize);
            for (int i = 0; i < size; ++i) {
                data[i] = this.random.nextInt();
            }
            this.shouldReadAndWrite(data, size);
            this.writer.reset();
        }
    }

    private void shouldWriteAndRead(int[] data) throws IOException {
        this.shouldReadAndWrite(data, data.length);
    }

    private void shouldReadAndWrite(int[] data, int length) throws IOException {
        this.writeData(data, length);
        this.reader = new DeltaBinaryPackingValuesReader();
        byte[] page = this.writer.getBytes().toByteArray();
        int miniBlockSize = this.blockSize / this.miniBlockNum;
        double miniBlockFlushed = Math.ceil(((double)length - 1.0) / (double)miniBlockSize);
        double blockFlushed = Math.ceil(((double)length - 1.0) / (double)this.blockSize);
        double estimatedSize = 20.0 + 4.0 * miniBlockFlushed * (double)miniBlockSize + blockFlushed * (double)this.miniBlockNum + 5.0 * blockFlushed;
        Assert.assertTrue((estimatedSize >= (double)page.length ? 1 : 0) != 0);
        this.reader.initFromPage(100, page, 0);
        for (int i = 0; i < length; ++i) {
            Assert.assertEquals((long)data[i], (long)this.reader.readInteger());
        }
    }

    private void writeData(int[] data) {
        this.writeData(data, data.length);
    }

    private void writeData(int[] data, int length) {
        for (int i = 0; i < length; ++i) {
            this.writer.writeInteger(data[i]);
        }
    }
}

