/*
 * Decompiled with CFR 0.152.
 */
package com.gitblit.utils;

import com.gitblit.Constants;
import com.gitblit.models.GitNote;
import com.gitblit.models.PathModel;
import com.gitblit.models.RefModel;
import com.gitblit.models.SubmoduleModel;
import com.gitblit.utils.ArrayUtils;
import com.gitblit.utils.FileUtils;
import com.gitblit.utils.StringUtils;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import org.eclipse.jgit.api.CloneCommand;
import org.eclipse.jgit.api.FetchCommand;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.errors.GitAPIException;
import org.eclipse.jgit.diff.DiffEntry;
import org.eclipse.jgit.diff.DiffFormatter;
import org.eclipse.jgit.diff.RawTextComparator;
import org.eclipse.jgit.errors.ConfigInvalidException;
import org.eclipse.jgit.errors.IncorrectObjectTypeException;
import org.eclipse.jgit.errors.MissingObjectException;
import org.eclipse.jgit.errors.StopWalkException;
import org.eclipse.jgit.lib.AnyObjectId;
import org.eclipse.jgit.lib.BlobBasedConfig;
import org.eclipse.jgit.lib.CommitBuilder;
import org.eclipse.jgit.lib.Config;
import org.eclipse.jgit.lib.FileMode;
import org.eclipse.jgit.lib.ObjectId;
import org.eclipse.jgit.lib.ObjectInserter;
import org.eclipse.jgit.lib.ObjectLoader;
import org.eclipse.jgit.lib.ObjectStream;
import org.eclipse.jgit.lib.PersonIdent;
import org.eclipse.jgit.lib.Ref;
import org.eclipse.jgit.lib.RefUpdate;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.lib.RepositoryCache;
import org.eclipse.jgit.lib.TreeFormatter;
import org.eclipse.jgit.revwalk.RevBlob;
import org.eclipse.jgit.revwalk.RevCommit;
import org.eclipse.jgit.revwalk.RevObject;
import org.eclipse.jgit.revwalk.RevSort;
import org.eclipse.jgit.revwalk.RevTree;
import org.eclipse.jgit.revwalk.RevWalk;
import org.eclipse.jgit.revwalk.filter.CommitTimeRevFilter;
import org.eclipse.jgit.revwalk.filter.RevFilter;
import org.eclipse.jgit.storage.file.FileRepository;
import org.eclipse.jgit.transport.CredentialsProvider;
import org.eclipse.jgit.transport.FetchResult;
import org.eclipse.jgit.transport.RefSpec;
import org.eclipse.jgit.treewalk.TreeWalk;
import org.eclipse.jgit.treewalk.filter.AndTreeFilter;
import org.eclipse.jgit.treewalk.filter.OrTreeFilter;
import org.eclipse.jgit.treewalk.filter.PathFilter;
import org.eclipse.jgit.treewalk.filter.PathFilterGroup;
import org.eclipse.jgit.treewalk.filter.PathSuffixFilter;
import org.eclipse.jgit.treewalk.filter.TreeFilter;
import org.eclipse.jgit.util.FS;
import org.eclipse.jgit.util.io.DisabledOutputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JGitUtils {
    static final Logger LOGGER = LoggerFactory.getLogger(JGitUtils.class);

    private static void error(Throwable t, Repository repository, String pattern, Object ... objects) {
        ArrayList<Object> parameters = new ArrayList<Object>();
        if (objects != null && objects.length > 0) {
            for (Object o : objects) {
                parameters.add(o);
            }
        }
        if (repository != null) {
            parameters.add(0, repository.getDirectory().getAbsolutePath());
        }
        LOGGER.error(MessageFormat.format(pattern, parameters.toArray()), t);
    }

    public static String getDisplayName(PersonIdent person) {
        if (StringUtils.isEmpty(person.getEmailAddress())) {
            return person.getName();
        }
        StringBuilder r = new StringBuilder();
        r.append(person.getName());
        r.append(" <");
        r.append(person.getEmailAddress());
        r.append('>');
        return r.toString().trim();
    }

    public static CloneResult cloneRepository(File repositoriesFolder, String name, String fromUrl) throws Exception {
        return JGitUtils.cloneRepository(repositoriesFolder, name, fromUrl, true, null);
    }

    public static CloneResult cloneRepository(File repositoriesFolder, String name, String fromUrl, boolean bare, CredentialsProvider credentialsProvider) throws Exception {
        CloneResult result = new CloneResult();
        if (bare) {
            if (!name.toLowerCase().endsWith(".git")) {
                name = name + ".git";
            }
        } else if (name.toLowerCase().endsWith(".git")) {
            name = name.substring(0, name.indexOf(".git"));
        }
        result.name = name;
        File folder = new File(repositoriesFolder, name);
        if (folder.exists()) {
            File gitDir = RepositoryCache.FileKey.resolve((File)new File(repositoriesFolder, name), (FS)FS.DETECTED);
            FileRepository repository = new FileRepository(gitDir);
            result.fetchResult = JGitUtils.fetchRepository(credentialsProvider, (Repository)repository, new RefSpec[0]);
            repository.close();
        } else {
            CloneCommand clone = new CloneCommand();
            clone.setBare(bare);
            clone.setCloneAllBranches(true);
            clone.setURI(fromUrl);
            clone.setDirectory(folder);
            if (credentialsProvider != null) {
                clone.setCredentialsProvider(credentialsProvider);
            }
            Repository repository = clone.call().getRepository();
            result.createdRepository = true;
            result.fetchResult = JGitUtils.fetchRepository(credentialsProvider, repository, new RefSpec[0]);
            repository.close();
        }
        return result;
    }

    public static FetchResult fetchRepository(CredentialsProvider credentialsProvider, Repository repository, RefSpec ... refSpecs) throws Exception {
        Git git2 = new Git(repository);
        FetchCommand fetch = git2.fetch();
        ArrayList<RefSpec> specs = new ArrayList<RefSpec>();
        if (refSpecs == null || refSpecs.length == 0) {
            specs.add(new RefSpec("+refs/heads/*:refs/remotes/origin/*"));
            specs.add(new RefSpec("+refs/tags/*:refs/tags/*"));
            specs.add(new RefSpec("+refs/notes/*:refs/notes/*"));
        } else {
            specs.addAll(Arrays.asList(refSpecs));
        }
        if (credentialsProvider != null) {
            fetch.setCredentialsProvider(credentialsProvider);
        }
        fetch.setRefSpecs(specs);
        FetchResult fetchRes = fetch.call();
        return fetchRes;
    }

    public static Repository createRepository(File repositoriesFolder, String name) {
        try {
            Git git2 = Git.init().setDirectory(new File(repositoriesFolder, name)).setBare(true).call();
            return git2.getRepository();
        }
        catch (GitAPIException e) {
            throw new RuntimeException(e);
        }
    }

    public static List<String> getRepositoryList(File repositoriesFolder, boolean onlyBare, boolean searchSubfolders, int depth, List<String> exclusions) {
        ArrayList<String> list = new ArrayList<String>();
        if (repositoriesFolder == null || !repositoriesFolder.exists()) {
            return list;
        }
        ArrayList<Pattern> patterns = new ArrayList<Pattern>();
        if (!ArrayUtils.isEmpty(exclusions)) {
            for (String regex2 : exclusions) {
                patterns.add(Pattern.compile(regex2));
            }
        }
        list.addAll(JGitUtils.getRepositoryList(repositoriesFolder.getAbsolutePath(), repositoriesFolder, onlyBare, searchSubfolders, depth, patterns));
        StringUtils.sortRepositorynames(list);
        return list;
    }

    private static List<String> getRepositoryList(String basePath, File searchFolder, boolean onlyBare, boolean searchSubfolders, int depth, List<Pattern> patterns) {
        File baseFile = new File(basePath);
        ArrayList<String> list = new ArrayList<String>();
        if (depth == 0) {
            return list;
        }
        int nextDepth = depth == -1 ? -1 : depth - 1;
        for (File file : searchFolder.listFiles()) {
            if (!file.isDirectory()) continue;
            boolean exclude = false;
            for (Pattern pattern : patterns) {
                String path;
                if (!pattern.matcher(path = FileUtils.getRelativePath(baseFile, file).replace('\\', '/')).matches()) continue;
                LOGGER.debug(MessageFormat.format("excluding {0} because of rule {1}", path, pattern.pattern()));
                exclude = true;
                break;
            }
            if (exclude) continue;
            File gitDir = RepositoryCache.FileKey.resolve((File)new File(searchFolder, file.getName()), (FS)FS.DETECTED);
            if (gitDir != null) {
                if (onlyBare && gitDir.getName().equals(".git")) continue;
                if (gitDir.equals(file) || gitDir.getParentFile().equals(file)) {
                    String repository = FileUtils.getRelativePath(baseFile, file);
                    list.add(repository);
                    continue;
                }
                if (!searchSubfolders || !file.canRead()) continue;
                list.addAll(JGitUtils.getRepositoryList(basePath, file, onlyBare, searchSubfolders, nextDepth, patterns));
                continue;
            }
            if (!searchSubfolders || !file.canRead()) continue;
            list.addAll(JGitUtils.getRepositoryList(basePath, file, onlyBare, searchSubfolders, nextDepth, patterns));
        }
        return list;
    }

    public static RevCommit getFirstCommit(Repository repository, String branch) {
        if (!JGitUtils.hasCommits(repository)) {
            return null;
        }
        RevCommit commit = null;
        try {
            ObjectId branchObject = StringUtils.isEmpty(branch) ? JGitUtils.getDefaultBranch(repository) : repository.resolve(branch);
            RevWalk walk = new RevWalk(repository);
            walk.sort(RevSort.REVERSE);
            RevCommit head = walk.parseCommit((AnyObjectId)branchObject);
            walk.markStart(head);
            commit = walk.next();
            walk.dispose();
        }
        catch (Throwable t) {
            JGitUtils.error(t, repository, "{0} failed to determine first commit", new Object[0]);
        }
        return commit;
    }

    public static Date getFirstChange(Repository repository, String branch) {
        RevCommit commit = JGitUtils.getFirstCommit(repository, branch);
        if (commit == null) {
            if (repository == null || !repository.getDirectory().exists()) {
                return new Date(0L);
            }
            return new Date(repository.getDirectory().lastModified());
        }
        return JGitUtils.getCommitDate(commit);
    }

    public static boolean hasCommits(Repository repository) {
        if (repository != null && repository.getDirectory().exists()) {
            return new File(repository.getDirectory(), "objects").list().length > 2 || new File(repository.getDirectory(), "objects/pack").list().length > 0;
        }
        return false;
    }

    public static Date getLastChange(Repository repository) {
        if (!JGitUtils.hasCommits(repository)) {
            if (repository == null) {
                return new Date(0L);
            }
            return new Date(repository.getDirectory().lastModified());
        }
        List<RefModel> branchModels = JGitUtils.getLocalBranches(repository, true, -1);
        if (branchModels.size() > 0) {
            Date lastChange = new Date(0L);
            for (RefModel branchModel : branchModels) {
                if (!branchModel.getDate().after(lastChange)) continue;
                lastChange = branchModel.getDate();
            }
            return lastChange;
        }
        return new Date(repository.getDirectory().lastModified());
    }

    public static Date getCommitDate(RevCommit commit) {
        if (commit == null) {
            return new Date(0L);
        }
        return new Date((long)commit.getCommitTime() * 1000L);
    }

    public static Date getAuthorDate(RevCommit commit) {
        if (commit == null) {
            return new Date(0L);
        }
        return commit.getAuthorIdent().getWhen();
    }

    public static RevCommit getCommit(Repository repository, String objectId) {
        if (!JGitUtils.hasCommits(repository)) {
            return null;
        }
        RevCommit commit = null;
        try {
            RevCommit rev;
            ObjectId branchObject = StringUtils.isEmpty(objectId) ? JGitUtils.getDefaultBranch(repository) : repository.resolve(objectId);
            RevWalk walk = new RevWalk(repository);
            commit = rev = walk.parseCommit((AnyObjectId)branchObject);
            walk.dispose();
        }
        catch (Throwable t) {
            JGitUtils.error(t, repository, "{0} failed to get commit {1}", objectId);
        }
        return commit;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static byte[] getByteContent(Repository repository, RevTree tree, String path) {
        RevWalk rw = new RevWalk(repository);
        TreeWalk tw = new TreeWalk(repository);
        tw.setFilter(PathFilterGroup.createFromStrings(Collections.singleton(path)));
        byte[] content = null;
        try {
            if (tree == null) {
                ObjectId object = JGitUtils.getDefaultBranch(repository);
                RevCommit commit = rw.parseCommit((AnyObjectId)object);
                tree = commit.getTree();
            }
            tw.reset((AnyObjectId)tree);
            while (tw.next()) {
                int n;
                if (tw.isSubtree() && !path.equals(tw.getPathString())) {
                    tw.enterSubtree();
                    continue;
                }
                ObjectId entid = tw.getObjectId(0);
                FileMode entmode = tw.getFileMode(0);
                if (entmode == FileMode.GITLINK) continue;
                RevObject ro = rw.lookupAny((AnyObjectId)entid, entmode.getObjectType());
                rw.parseBody(ro);
                ByteArrayOutputStream os = new ByteArrayOutputStream();
                ObjectLoader ldr = repository.open((AnyObjectId)ro.getId(), 3);
                byte[] tmp = new byte[4096];
                ObjectStream in = ldr.openStream();
                while ((n = in.read(tmp)) > 0) {
                    os.write(tmp, 0, n);
                }
                in.close();
                content = os.toByteArray();
            }
        }
        catch (Throwable t) {
            JGitUtils.error(t, repository, "{0} can't find {1} in tree {2}", path, tree.name());
        }
        finally {
            rw.dispose();
            tw.release();
        }
        return content;
    }

    public static String getStringContent(Repository repository, RevTree tree, String blobPath, String ... charsets) {
        byte[] content = JGitUtils.getByteContent(repository, tree, blobPath);
        if (content == null) {
            return null;
        }
        return StringUtils.decodeString(content, charsets);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static byte[] getByteContent(Repository repository, String objectId) {
        RevWalk rw = new RevWalk(repository);
        byte[] content = null;
        try {
            int n;
            RevBlob blob = rw.lookupBlob((AnyObjectId)ObjectId.fromString((String)objectId));
            rw.parseBody((RevObject)blob);
            ByteArrayOutputStream os = new ByteArrayOutputStream();
            ObjectLoader ldr = repository.open((AnyObjectId)blob.getId(), 3);
            byte[] tmp = new byte[4096];
            ObjectStream in = ldr.openStream();
            while ((n = in.read(tmp)) > 0) {
                os.write(tmp, 0, n);
            }
            in.close();
            content = os.toByteArray();
        }
        catch (Throwable t) {
            JGitUtils.error(t, repository, "{0} can't find blob {1}", objectId);
        }
        finally {
            rw.dispose();
        }
        return content;
    }

    public static String getStringContent(Repository repository, String objectId, String ... charsets) {
        byte[] content = JGitUtils.getByteContent(repository, objectId);
        if (content == null) {
            return null;
        }
        return StringUtils.decodeString(content, charsets);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static List<PathModel> getFilesInPath(Repository repository, String path, RevCommit commit) {
        ArrayList<PathModel> list = new ArrayList<PathModel>();
        if (!JGitUtils.hasCommits(repository)) {
            return list;
        }
        if (commit == null) {
            commit = JGitUtils.getCommit(repository, null);
        }
        TreeWalk tw = new TreeWalk(repository);
        try {
            tw.addTree((AnyObjectId)commit.getTree());
            if (!StringUtils.isEmpty(path)) {
                PathFilter f = PathFilter.create((String)path);
                tw.setFilter((TreeFilter)f);
                tw.setRecursive(false);
                boolean foundFolder = false;
                while (tw.next()) {
                    if (!foundFolder && tw.isSubtree()) {
                        tw.enterSubtree();
                    }
                    if (tw.getPathString().equals(path)) {
                        foundFolder = true;
                        continue;
                    }
                    if (!foundFolder) continue;
                    list.add(JGitUtils.getPathModel(tw, path, commit));
                }
            } else {
                tw.setRecursive(false);
                while (tw.next()) {
                    list.add(JGitUtils.getPathModel(tw, null, commit));
                }
            }
        }
        catch (IOException e) {
            JGitUtils.error(e, repository, "{0} failed to get files for commit {1}", commit.getName());
        }
        finally {
            tw.release();
        }
        Collections.sort(list);
        return list;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static List<PathModel.PathChangeModel> getFilesInCommit(Repository repository, RevCommit commit) {
        ArrayList<PathModel.PathChangeModel> list = new ArrayList<PathModel.PathChangeModel>();
        if (!JGitUtils.hasCommits(repository)) {
            return list;
        }
        RevWalk rw = new RevWalk(repository);
        try {
            if (commit == null) {
                ObjectId object = JGitUtils.getDefaultBranch(repository);
                commit = rw.parseCommit((AnyObjectId)object);
            }
            if (commit.getParentCount() == 0) {
                TreeWalk tw = new TreeWalk(repository);
                tw.reset();
                tw.setRecursive(true);
                tw.addTree((AnyObjectId)commit.getTree());
                while (tw.next()) {
                    list.add(new PathModel.PathChangeModel(tw.getPathString(), tw.getPathString(), 0L, tw.getRawMode(0), tw.getObjectId(0).getName(), commit.getId().getName(), DiffEntry.ChangeType.ADD));
                }
                tw.release();
            } else {
                RevCommit parent = rw.parseCommit((AnyObjectId)commit.getParent(0).getId());
                DiffFormatter df = new DiffFormatter((OutputStream)DisabledOutputStream.INSTANCE);
                df.setRepository(repository);
                df.setDiffComparator(RawTextComparator.DEFAULT);
                df.setDetectRenames(true);
                List diffs = df.scan(parent.getTree(), commit.getTree());
                for (DiffEntry diff : diffs) {
                    String objectId = null;
                    if (FileMode.GITLINK.equals(diff.getNewMode())) {
                        objectId = diff.getNewId().name();
                    }
                    if (diff.getChangeType().equals((Object)DiffEntry.ChangeType.DELETE)) {
                        list.add(new PathModel.PathChangeModel(diff.getOldPath(), diff.getOldPath(), 0L, diff.getNewMode().getBits(), objectId, commit.getId().getName(), diff.getChangeType()));
                        continue;
                    }
                    if (diff.getChangeType().equals((Object)DiffEntry.ChangeType.RENAME)) {
                        list.add(new PathModel.PathChangeModel(diff.getOldPath(), diff.getNewPath(), 0L, diff.getNewMode().getBits(), objectId, commit.getId().getName(), diff.getChangeType()));
                        continue;
                    }
                    list.add(new PathModel.PathChangeModel(diff.getNewPath(), diff.getNewPath(), 0L, diff.getNewMode().getBits(), objectId, commit.getId().getName(), diff.getChangeType()));
                }
            }
        }
        catch (Throwable t) {
            JGitUtils.error(t, repository, "{0} failed to determine files in commit!", new Object[0]);
        }
        finally {
            rw.dispose();
        }
        return list;
    }

    public static List<PathModel> getDocuments(Repository repository, List<String> extensions) {
        return JGitUtils.getDocuments(repository, extensions, null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static List<PathModel> getDocuments(Repository repository, List<String> extensions, String objectId) {
        ArrayList<PathModel> list = new ArrayList<PathModel>();
        if (!JGitUtils.hasCommits(repository)) {
            return list;
        }
        RevCommit commit = JGitUtils.getCommit(repository, objectId);
        TreeWalk tw = new TreeWalk(repository);
        try {
            tw.addTree((AnyObjectId)commit.getTree());
            if (extensions != null && extensions.size() > 0) {
                ArrayList<PathSuffixFilter> suffixFilters = new ArrayList<PathSuffixFilter>();
                for (String extension : extensions) {
                    if (extension.charAt(0) == '.') {
                        suffixFilters.add(PathSuffixFilter.create((String)("\\" + extension)));
                        continue;
                    }
                    suffixFilters.add(PathSuffixFilter.create((String)("\\." + extension)));
                }
                TreeFilter filter = suffixFilters.size() == 1 ? (TreeFilter)suffixFilters.get(0) : OrTreeFilter.create(suffixFilters);
                tw.setFilter(filter);
                tw.setRecursive(true);
            }
            while (tw.next()) {
                list.add(JGitUtils.getPathModel(tw, null, commit));
            }
        }
        catch (IOException e) {
            JGitUtils.error(e, repository, "{0} failed to get documents for commit {1}", commit.getName());
        }
        finally {
            tw.release();
        }
        Collections.sort(list);
        return list;
    }

    private static PathModel getPathModel(TreeWalk tw, String basePath, RevCommit commit) {
        long size = 0L;
        String name = StringUtils.isEmpty(basePath) ? tw.getPathString() : tw.getPathString().substring(basePath.length() + 1);
        ObjectId objectId = tw.getObjectId(0);
        try {
            if (!tw.isSubtree() && tw.getFileMode(0) != FileMode.GITLINK) {
                size = tw.getObjectReader().getObjectSize((AnyObjectId)objectId, 3);
            }
        }
        catch (Throwable t) {
            JGitUtils.error(t, null, "failed to retrieve blob size for " + tw.getPathString(), new Object[0]);
        }
        return new PathModel(name, tw.getPathString(), size, tw.getFileMode(0).getBits(), objectId.getName(), commit.getName());
    }

    public static String getPermissionsFromMode(int mode) {
        if (FileMode.TREE.equals(mode)) {
            return "drwxr-xr-x";
        }
        if (FileMode.REGULAR_FILE.equals(mode)) {
            return "-rw-r--r--";
        }
        if (FileMode.EXECUTABLE_FILE.equals(mode)) {
            return "-rwxr-xr-x";
        }
        if (FileMode.SYMLINK.equals(mode)) {
            return "symlink";
        }
        if (FileMode.GITLINK.equals(mode)) {
            return "submodule";
        }
        return "missing";
    }

    public static List<RevCommit> getRevLog(Repository repository, String objectId, Date minimumDate) {
        ArrayList<RevCommit> list = new ArrayList<RevCommit>();
        if (!JGitUtils.hasCommits(repository)) {
            return list;
        }
        try {
            ObjectId branchObject = StringUtils.isEmpty(objectId) ? JGitUtils.getDefaultBranch(repository) : repository.resolve(objectId);
            RevWalk rw = new RevWalk(repository);
            rw.markStart(rw.parseCommit((AnyObjectId)branchObject));
            rw.setRevFilter(CommitTimeRevFilter.after((Date)minimumDate));
            RevWalk revlog = rw;
            for (RevCommit rev : revlog) {
                list.add(rev);
            }
            rw.dispose();
        }
        catch (Throwable t) {
            JGitUtils.error(t, repository, "{0} failed to get {1} revlog for minimum date {2}", objectId, minimumDate);
        }
        return list;
    }

    public static List<RevCommit> getRevLog(Repository repository, int maxCount) {
        return JGitUtils.getRevLog(repository, null, 0, maxCount);
    }

    public static List<RevCommit> getRevLog(Repository repository, String objectId, int offset, int maxCount) {
        return JGitUtils.getRevLog(repository, objectId, null, offset, maxCount);
    }

    public static List<RevCommit> getRevLog(Repository repository, String objectId, String path, int offset, int maxCount) {
        ArrayList<RevCommit> list = new ArrayList<RevCommit>();
        if (maxCount == 0) {
            return list;
        }
        if (!JGitUtils.hasCommits(repository)) {
            return list;
        }
        try {
            ObjectId branchObject = StringUtils.isEmpty(objectId) ? JGitUtils.getDefaultBranch(repository) : repository.resolve(objectId);
            if (branchObject == null) {
                return list;
            }
            RevWalk rw = new RevWalk(repository);
            rw.markStart(rw.parseCommit((AnyObjectId)branchObject));
            if (!StringUtils.isEmpty(path)) {
                TreeFilter filter = AndTreeFilter.create((TreeFilter)PathFilterGroup.createFromStrings(Collections.singleton(path)), (TreeFilter)TreeFilter.ANY_DIFF);
                rw.setTreeFilter(filter);
            }
            RevWalk revlog = rw;
            if (offset > 0) {
                int count = 0;
                for (RevCommit rev : revlog) {
                    if (++count <= offset) continue;
                    list.add(rev);
                    if (maxCount <= 0 || list.size() != maxCount) continue;
                    break;
                }
            } else {
                for (RevCommit rev : revlog) {
                    list.add(rev);
                    if (maxCount <= 0 || list.size() != maxCount) continue;
                    break;
                }
            }
            rw.dispose();
        }
        catch (Throwable t) {
            JGitUtils.error(t, repository, "{0} failed to get {1} revlog for path {2}", objectId, path);
        }
        return list;
    }

    public static List<RevCommit> getRevLog(Repository repository, String startRangeId, String endRangeId) {
        ArrayList<RevCommit> list = new ArrayList<RevCommit>();
        if (!JGitUtils.hasCommits(repository)) {
            return list;
        }
        try {
            ObjectId endRange = repository.resolve(endRangeId);
            ObjectId startRange = repository.resolve(startRangeId);
            RevWalk rw = new RevWalk(repository);
            rw.markStart(rw.parseCommit((AnyObjectId)endRange));
            if (startRange.equals((AnyObjectId)ObjectId.zeroId())) {
                list.add(rw.parseCommit((AnyObjectId)endRange));
                rw.dispose();
                return list;
            }
            rw.markUninteresting(rw.parseCommit((AnyObjectId)startRange));
            RevWalk revlog = rw;
            for (RevCommit rev : revlog) {
                list.add(rev);
            }
            rw.dispose();
        }
        catch (Throwable t) {
            JGitUtils.error(t, repository, "{0} failed to get revlog for {1}..{2}", startRangeId, endRangeId);
        }
        return list;
    }

    public static List<RevCommit> searchRevlogs(Repository repository, String objectId, String value, final Constants.SearchType type, int offset, int maxCount) {
        final String lcValue = value.toLowerCase();
        ArrayList<RevCommit> list = new ArrayList<RevCommit>();
        if (maxCount == 0) {
            return list;
        }
        if (!JGitUtils.hasCommits(repository)) {
            return list;
        }
        try {
            ObjectId branchObject = StringUtils.isEmpty(objectId) ? JGitUtils.getDefaultBranch(repository) : repository.resolve(objectId);
            RevWalk rw = new RevWalk(repository);
            rw.setRevFilter(new RevFilter(){

                public RevFilter clone() {
                    return this;
                }

                public boolean include(RevWalk walker, RevCommit commit) throws StopWalkException, MissingObjectException, IncorrectObjectTypeException, IOException {
                    boolean include = false;
                    switch (type) {
                        case AUTHOR: {
                            include = commit.getAuthorIdent().getName().toLowerCase().indexOf(lcValue) > -1 || commit.getAuthorIdent().getEmailAddress().toLowerCase().indexOf(lcValue) > -1;
                            break;
                        }
                        case COMMITTER: {
                            include = commit.getCommitterIdent().getName().toLowerCase().indexOf(lcValue) > -1 || commit.getCommitterIdent().getEmailAddress().toLowerCase().indexOf(lcValue) > -1;
                            break;
                        }
                        case COMMIT: {
                            include = commit.getFullMessage().toLowerCase().indexOf(lcValue) > -1;
                        }
                    }
                    return include;
                }
            });
            rw.markStart(rw.parseCommit((AnyObjectId)branchObject));
            RevWalk revlog = rw;
            if (offset > 0) {
                int count = 0;
                for (RevCommit rev : revlog) {
                    if (++count <= offset) continue;
                    list.add(rev);
                    if (maxCount <= 0 || list.size() != maxCount) continue;
                    break;
                }
            } else {
                for (RevCommit rev : revlog) {
                    list.add(rev);
                    if (maxCount <= 0 || list.size() != maxCount) continue;
                    break;
                }
            }
            rw.dispose();
        }
        catch (Throwable t) {
            JGitUtils.error(t, repository, "{0} failed to {1} search revlogs for {2}", type.name(), value);
        }
        return list;
    }

    public static ObjectId getDefaultBranch(Repository repository) throws Exception {
        List<RefModel> branchModels;
        ObjectId object = repository.resolve("HEAD");
        if (object == null && (branchModels = JGitUtils.getLocalBranches(repository, true, -1)).size() > 0) {
            RefModel branch = null;
            Date lastDate = new Date(0L);
            for (RefModel branchModel : branchModels) {
                if (!branchModel.getDate().after(lastDate)) continue;
                branch = branchModel;
                lastDate = branch.getDate();
            }
            object = branch.getReferencedObjectId();
        }
        return object;
    }

    public static String getHEADRef(Repository repository) {
        String target = null;
        try {
            List<RefModel> tagModels;
            target = repository.getFullBranch();
            if (!target.startsWith("refs/heads/") && (tagModels = JGitUtils.getTags(repository, true, -1)).size() > 0) {
                RefModel tag = null;
                Date lastDate = new Date(0L);
                for (RefModel tagModel : tagModels) {
                    if (!tagModel.getReferencedObjectId().getName().equals(target) || !tagModel.getDate().after(lastDate)) continue;
                    tag = tagModel;
                    lastDate = tag.getDate();
                }
                target = tag.getName();
            }
        }
        catch (Throwable t) {
            JGitUtils.error(t, repository, "{0} failed to get symbolic HEAD target", new Object[0]);
        }
        return target;
    }

    public static boolean setHEADtoRef(Repository repository, String targetRef) {
        try {
            RefUpdate.Result result;
            boolean detach = !targetRef.startsWith("refs/heads/");
            RefUpdate head = repository.updateRef("HEAD", detach);
            if (detach) {
                RevCommit commit = JGitUtils.getCommit(repository, targetRef);
                head.setNewObjectId((AnyObjectId)commit.getId());
                result = head.forceUpdate();
            } else {
                result = head.link(targetRef);
            }
            switch (result) {
                case NEW: 
                case FORCED: 
                case NO_CHANGE: 
                case FAST_FORWARD: {
                    return true;
                }
            }
            LOGGER.error(MessageFormat.format("{0} HEAD update to {1} returned result {2}", repository.getDirectory().getAbsolutePath(), targetRef, result));
        }
        catch (Throwable t) {
            JGitUtils.error(t, repository, "{0} failed to set HEAD to {1}", targetRef);
        }
        return false;
    }

    public static boolean setBranchRef(Repository repository, String branch, String commitId) {
        String branchName = branch;
        if (!branchName.startsWith("refs/heads/")) {
            branchName = "refs/heads/" + branch;
        }
        try {
            RefUpdate refUpdate = repository.updateRef(branchName, false);
            refUpdate.setNewObjectId((AnyObjectId)ObjectId.fromString((String)commitId));
            RefUpdate.Result result = refUpdate.forceUpdate();
            switch (result) {
                case NEW: 
                case FORCED: 
                case NO_CHANGE: 
                case FAST_FORWARD: {
                    return true;
                }
            }
            LOGGER.error(MessageFormat.format("{0} {1} update to {2} returned result {3}", repository.getDirectory().getAbsolutePath(), branchName, commitId, result));
        }
        catch (Throwable t) {
            JGitUtils.error(t, repository, "{0} failed to set {1} to {2}", branchName, commitId);
        }
        return false;
    }

    public static boolean deleteBranchRef(Repository repository, String branch) {
        String branchName = branch;
        if (!branchName.startsWith("refs/heads/")) {
            branchName = "refs/heads/" + branch;
        }
        try {
            RefUpdate refUpdate = repository.updateRef(branchName, false);
            refUpdate.setForceUpdate(true);
            RefUpdate.Result result = refUpdate.delete();
            switch (result) {
                case NEW: 
                case FORCED: 
                case NO_CHANGE: 
                case FAST_FORWARD: {
                    return true;
                }
            }
            LOGGER.error(MessageFormat.format("{0} failed to delete to {1} returned result {2}", repository.getDirectory().getAbsolutePath(), branchName, result));
        }
        catch (Throwable t) {
            JGitUtils.error(t, repository, "{0} failed to delete {1}", branchName);
        }
        return false;
    }

    public static List<String> getAvailableHeadTargets(Repository repository) {
        ArrayList<String> targets = new ArrayList<String>();
        for (RefModel branchModel : JGitUtils.getLocalBranches(repository, true, -1)) {
            targets.add(branchModel.getName());
        }
        for (RefModel tagModel : JGitUtils.getTags(repository, true, -1)) {
            targets.add(tagModel.getName());
        }
        return targets;
    }

    public static Map<ObjectId, List<RefModel>> getAllRefs(Repository repository) {
        return JGitUtils.getAllRefs(repository, true);
    }

    public static Map<ObjectId, List<RefModel>> getAllRefs(Repository repository, boolean includeRemoteRefs) {
        List<RefModel> list = JGitUtils.getRefs(repository, "", true, -1);
        HashMap<ObjectId, List<RefModel>> refs = new HashMap<ObjectId, List<RefModel>>();
        for (RefModel ref : list) {
            if (!includeRemoteRefs && ref.getName().startsWith("refs/remotes/")) continue;
            ObjectId objectid = ref.getReferencedObjectId();
            if (!refs.containsKey(objectid)) {
                refs.put(objectid, new ArrayList());
            }
            ((List)refs.get(objectid)).add(ref);
        }
        return refs;
    }

    public static List<RefModel> getTags(Repository repository, boolean fullName, int maxCount) {
        return JGitUtils.getRefs(repository, "refs/tags/", fullName, maxCount);
    }

    public static List<RefModel> getLocalBranches(Repository repository, boolean fullName, int maxCount) {
        return JGitUtils.getRefs(repository, "refs/heads/", fullName, maxCount);
    }

    public static List<RefModel> getRemoteBranches(Repository repository, boolean fullName, int maxCount) {
        return JGitUtils.getRefs(repository, "refs/remotes/", fullName, maxCount);
    }

    public static List<RefModel> getNoteBranches(Repository repository, boolean fullName, int maxCount) {
        return JGitUtils.getRefs(repository, "refs/notes/", fullName, maxCount);
    }

    private static List<RefModel> getRefs(Repository repository, String refs, boolean fullName, int maxCount) {
        ArrayList<RefModel> list = new ArrayList<RefModel>();
        if (maxCount == 0) {
            return list;
        }
        if (!JGitUtils.hasCommits(repository)) {
            return list;
        }
        try {
            Map map = repository.getRefDatabase().getRefs(refs);
            RevWalk rw = new RevWalk(repository);
            for (Map.Entry entry : map.entrySet()) {
                Ref ref = (Ref)entry.getValue();
                RevObject object = rw.parseAny((AnyObjectId)ref.getObjectId());
                String name = (String)entry.getKey();
                if (fullName && !StringUtils.isEmpty(refs)) {
                    name = refs + name;
                }
                list.add(new RefModel(name, ref, object));
            }
            rw.dispose();
            Collections.sort(list);
            Collections.reverse(list);
            if (maxCount > 0 && list.size() > maxCount) {
                list = new ArrayList(list.subList(0, maxCount));
            }
        }
        catch (IOException e) {
            JGitUtils.error(e, repository, "{0} failed to retrieve {1}", refs);
        }
        return list;
    }

    public static RefModel getPagesBranch(Repository repository) {
        return JGitUtils.getBranch(repository, "gh-pages");
    }

    public static RefModel getBranch(Repository repository, String name) {
        RefModel branch;
        block4: {
            branch = null;
            try {
                for (RefModel ref : JGitUtils.getLocalBranches(repository, false, -1)) {
                    if (!ref.reference.getName().endsWith(name)) continue;
                    branch = ref;
                    break;
                }
                if (branch != null) break block4;
                for (RefModel ref : JGitUtils.getRemoteBranches(repository, false, -1)) {
                    if (!ref.reference.getName().endsWith(name)) continue;
                    branch = ref;
                    break;
                }
            }
            catch (Throwable t) {
                LOGGER.error(MessageFormat.format("Failed to find {0} branch!", name), t);
            }
        }
        return branch;
    }

    public static List<SubmoduleModel> getSubmodules(Repository repository, String commitId) {
        RevCommit commit = JGitUtils.getCommit(repository, commitId);
        return JGitUtils.getSubmodules(repository, commit.getTree());
    }

    public static List<SubmoduleModel> getSubmodules(Repository repository, RevTree tree) {
        ArrayList<SubmoduleModel> list = new ArrayList<SubmoduleModel>();
        byte[] blob = JGitUtils.getByteContent(repository, tree, ".gitmodules");
        if (blob == null) {
            return list;
        }
        try {
            BlobBasedConfig config = new BlobBasedConfig((Config)repository.getConfig(), blob);
            for (String module : config.getSubsections("submodule")) {
                String path = config.getString("submodule", module, "path");
                String url = config.getString("submodule", module, "url");
                list.add(new SubmoduleModel(module, path, url));
            }
        }
        catch (ConfigInvalidException e) {
            LOGGER.error("Failed to load .gitmodules file for " + repository.getDirectory(), (Throwable)e);
        }
        return list;
    }

    public static SubmoduleModel getSubmoduleModel(Repository repository, String commitId, String path) {
        for (SubmoduleModel model : JGitUtils.getSubmodules(repository, commitId)) {
            if (!model.path.equals(path)) continue;
            return model;
        }
        return null;
    }

    public static List<GitNote> getNotesOnCommit(Repository repository, RevCommit commit) {
        ArrayList<GitNote> list = new ArrayList<GitNote>();
        if (!JGitUtils.hasCommits(repository)) {
            return list;
        }
        List<RefModel> noteBranches = JGitUtils.getNoteBranches(repository, true, -1);
        for (RefModel notesRef : noteBranches) {
            String notePath;
            RevTree notesTree = JGitUtils.getCommit(repository, notesRef.getName()).getTree();
            String text = JGitUtils.getStringContent(repository, notesTree, notePath = commit.getName(), new String[0]);
            if (!StringUtils.isEmpty(text)) {
                List<RevCommit> history = JGitUtils.getRevLog(repository, notesRef.getName(), notePath, 0, -1);
                RefModel noteRef = new RefModel(notesRef.displayName, null, (RevObject)history.get(history.size() - 1));
                GitNote gitNote = new GitNote(noteRef, text);
                list.add(gitNote);
                continue;
            }
            StringBuilder sb = new StringBuilder(commit.getName());
            sb.insert(2, '/');
            notePath = sb.toString();
            text = JGitUtils.getStringContent(repository, notesTree, notePath, new String[0]);
            if (StringUtils.isEmpty(text)) continue;
            List<RevCommit> history = JGitUtils.getRevLog(repository, notesRef.getName(), notePath, 0, -1);
            RefModel noteRef = new RefModel(notesRef.displayName, null, (RevObject)history.get(history.size() - 1));
            GitNote gitNote = new GitNote(noteRef, text);
            list.add(gitNote);
        }
        return list;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public static boolean createOrphanBranch(Repository repository, String branchName, PersonIdent author) {
        boolean success = false;
        String message = "Created branch " + branchName;
        if (author == null) {
            author = new PersonIdent("Gitblit", "gitblit@localhost");
        }
        try {
            ObjectInserter odi = repository.newObjectInserter();
            try {
                ObjectId blobId = odi.insert(3, message.getBytes("UTF-8"));
                TreeFormatter tree = new TreeFormatter();
                tree.append(".branch", FileMode.REGULAR_FILE, (AnyObjectId)blobId);
                ObjectId treeId = odi.insert(tree);
                CommitBuilder commit = new CommitBuilder();
                commit.setAuthor(author);
                commit.setCommitter(author);
                commit.setEncoding("UTF-8");
                commit.setMessage(message);
                commit.setTreeId((AnyObjectId)treeId);
                ObjectId commitId = odi.insert(commit);
                odi.flush();
                RevWalk revWalk = new RevWalk(repository);
                try {
                    RevCommit revCommit = revWalk.parseCommit((AnyObjectId)commitId);
                    if (!branchName.startsWith("refs/")) {
                        branchName = "refs/heads/" + branchName;
                    }
                    RefUpdate ru = repository.updateRef(branchName);
                    ru.setNewObjectId((AnyObjectId)commitId);
                    ru.setRefLogMessage("commit: " + revCommit.getShortMessage(), false);
                    RefUpdate.Result rc = ru.forceUpdate();
                    switch (rc) {
                        case NEW: 
                        case FORCED: 
                        case FAST_FORWARD: {
                            success = true;
                            return success;
                        }
                        default: {
                            success = false;
                            return success;
                        }
                    }
                }
                finally {
                    revWalk.release();
                }
            }
            finally {
                odi.release();
            }
        }
        catch (Throwable t) {
            JGitUtils.error(t, repository, "Failed to create orphan branch {1} in repository {0}", branchName);
        }
        return success;
    }

    public static class CloneResult {
        public String name;
        public FetchResult fetchResult;
        public boolean createdRepository;
    }
}

