package org.apache.tapestry;

import org.mortbay.jetty.Connector;
import org.mortbay.jetty.Server;
import org.mortbay.jetty.nio.BlockingChannelConnector;
import org.mortbay.jetty.webapp.WebAppContext;

import static java.lang.String.format;

/**
 * A utitilty class for running an instance of the <a
 * href="http://jetty.mortbay.com/jetty/index.html">Jetty servlet container</a> from within a unit
 * test. This code is based on the Jetty 6 code base (which is in beta at the time of writing). When
 * combined with <a href="http://jwebunit.sourceforge.net/">jWebUnit</a>, this allows for basic
 * <em>integration</em> testing of an application from within a unit test case.
 *
 * @author Howard M. Lewis Ship
 */
public class JettyRunner
{
    public static final String DEFAULT_CONTEXT_PATH = "/";

    public static final int DEFAULT_PORT = 80;

    public static final String DEFAULT_WAR_PATH = "src/main/webapp";

    private final String _contextPath;

    private final int _port;

    private final String _warPath;

    private final Server _jetty;

    /**
     * Creates and starts a new instance of Jetty, using default configuration values. The
     * contextPath will be /, the port will be 80, the warPath will be src/main/webapp.
     */
    public JettyRunner()
    {
        this(DEFAULT_CONTEXT_PATH, DEFAULT_PORT, DEFAULT_WAR_PATH);
    }

    /**
     * Creates and starts a new instance of Jetty. This should be done from a test case setup
     * method.
     *
     * @param contextPath
     *            the context path for the deployed application
     * @param port
     *            the port number used to access the application
     * @param warPath
     *            the path to the exploded web application (typically, "src/main/webapp")
     */
    public JettyRunner(String contextPath, int port, String warPath)
    {
        _contextPath = contextPath;
        _port = port;
        _warPath = warPath;

        _jetty = new Server();

        startJetty();
    }

    /** Stops the Jetty instance. This should be called from a test case tear down method. */
    public void stop()
    {
        try
        {
            _jetty.stop();
        }
        catch (Exception ex)
        {
            throw new RuntimeException("Error stopping Jetty instance: " + ex.toString(), ex);
        }
    }

    @Override
    public String toString()
    {
        return format("<JettyRunner %s:%d (%s)>", _contextPath, _port, _warPath);
    }

    private void startJetty()
    {
        try
        {
            BlockingChannelConnector connector = new BlockingChannelConnector();
            connector.setPort(_port);

            _jetty.setConnectors(new Connector[]
            { connector });

            WebAppContext context = new WebAppContext();

            context.setContextPath(_contextPath);
            context.setWar(_warPath);

            _jetty.addHandler(context);

            _jetty.start();
        }
        catch (Exception ex)
        {
            throw new RuntimeException("Failure starting Jetty instance: " + ex.toString(), ex);
        }
    }

}
