/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.wicket.request.mapper;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

import org.apache.wicket.request.IRequestHandler;
import org.apache.wicket.request.IRequestMapper;
import org.apache.wicket.request.Request;
import org.apache.wicket.request.Url;


/**
 * Thread safe compound {@link IRequestMapper}. The mappers are searched depending on their
 * compatibility score and the orders they were registered. If two or more {@link IRequestMapper}s
 * have the same compatibility score, the last registered mapper has highest priority.
 * 
 * @author igor.vaynberg
 * @author Matej Knopp
 */
public class CompoundRequestMapper implements ICompoundRequestMapper
{
	/**
	 * 
	 */
	private static class EncoderWithSegmentsCount implements Comparable<EncoderWithSegmentsCount>
	{
		private final IRequestMapper mapper;
		private final int compatibilityScore;

		public EncoderWithSegmentsCount(final IRequestMapper encoder, final int compatibilityScore)
		{
			mapper = encoder;
			this.compatibilityScore = compatibilityScore;
		}

		public int compareTo(final EncoderWithSegmentsCount o)
		{
			return o.compatibilityScore - compatibilityScore;
		}

		public IRequestMapper getMapper()
		{
			return mapper;
		}

		/**
		 * @see java.lang.Object#toString()
		 */
		@Override
		public String toString()
		{
			return "Mapper: " + mapper.getClass().getName() + "; Score: " + compatibilityScore;
		}
	}

	private final List<IRequestMapper> mappers = new CopyOnWriteArrayList<IRequestMapper>();

	/**
	 * Construct.
	 */
	public CompoundRequestMapper()
	{
	}

	/**
	 * @see org.apache.wicket.request.mapper.ICompoundRequestMapper#add(org.apache.wicket.request.IRequestMapper)
	 */
	public CompoundRequestMapper add(final IRequestMapper encoder)
	{
		mappers.add(0, encoder);
		return this;
	}

	/**
	 * @see org.apache.wicket.request.mapper.ICompoundRequestMapper#remove(org.apache.wicket.request.IRequestMapper)
	 */
	public CompoundRequestMapper remove(final IRequestMapper encoder)
	{
		mappers.remove(encoder);
		return this;
	}

	/**
	 * Searches the registered {@link IRequestMapper}s to find one that can decode the
	 * {@link Request}. Each registered {@link IRequestMapper} is asked to provide the matching
	 * segments count. Then the encoders are asked to decode the request in order depending on the
	 * provided segments count.
	 * <p>
	 * The encoder with highest matching segments count that can decode the request is returned.
	 * 
	 * @param request
	 * @return RequestHandler for the request or <code>null</code> if no encoder for the request is
	 *         found.
	 */
	public IRequestHandler mapRequest(final Request request)
	{
		List<EncoderWithSegmentsCount> list = new ArrayList<EncoderWithSegmentsCount>(
			mappers.size());

		for (IRequestMapper encoder : mappers)
		{
			int score = encoder.getCompatibilityScore(request);
			list.add(new EncoderWithSegmentsCount(encoder, score));
		}

		Collections.sort(list);

		for (EncoderWithSegmentsCount encoder : list)
		{
			IRequestHandler handler = encoder.getMapper().mapRequest(request);
			if (handler != null)
			{
				return handler;
			}
		}

		return null;
	}

	/**
	 * Searches the registered {@link IRequestMapper}s to find one that can encode the
	 * {@link IRequestHandler}. Each registered {@link IRequestMapper} is asked to encode the
	 * {@link IRequestHandler} until an encoder that can encode the {@link IRequestHandler} is found
	 * or no more encoders are left.
	 * <p>
	 * The handlers are searched in reverse order as they have been registered. More recently
	 * registered handlers have bigger priority.
	 * 
	 * @param handler
	 * @return Url for the handler or <code>null</code> if no encoder for the handler is found.
	 */
	public Url mapHandler(final IRequestHandler handler)
	{
		for (IRequestMapper encoder : mappers)
		{
			Url url = encoder.mapHandler(handler);
			if (url != null)
			{
				return url;
			}
		}
		return null;
	}

	/**
	 * The scope of the compound mapper is the highest score of the registered mappers.
	 * 
	 * {@inheritDoc}
	 */
	public int getCompatibilityScore(final Request request)
	{
		int score = Integer.MIN_VALUE;
		for (IRequestMapper mapper : mappers)
		{
			score = Math.max(score, mapper.getCompatibilityScore(request));
		}
		return score;
	}

	/** {@inheritDoc} */
	public Iterator<IRequestMapper> iterator()
	{
		return mappers.iterator();
	}
}
