/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.nio;

import java.nio.ByteBuffer;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.IntFunction;
import org.elasticsearch.nio.Page;
import org.elasticsearch.nio.utils.ExceptionsHelper;

public final class InboundChannelBuffer
implements AutoCloseable {
    public static final int PAGE_SIZE = 16384;
    private static final int PAGE_MASK = 16383;
    private static final int PAGE_SHIFT = Integer.numberOfTrailingZeros(16384);
    private static final ByteBuffer[] EMPTY_BYTE_BUFFER_ARRAY = new ByteBuffer[0];
    private static final Page[] EMPTY_BYTE_PAGE_ARRAY = new Page[0];
    private final IntFunction<Page> pageAllocator;
    private final ArrayDeque<Page> pages = new ArrayDeque();
    private final AtomicBoolean isClosed = new AtomicBoolean(false);
    private long capacity = 0L;
    private long internalIndex = 0L;
    private int offset = 0;

    public InboundChannelBuffer(IntFunction<Page> pageAllocator) {
        this.pageAllocator = pageAllocator;
    }

    public static InboundChannelBuffer allocatingInstance() {
        return new InboundChannelBuffer(n -> new Page(ByteBuffer.allocate(n), () -> {}));
    }

    @Override
    public void close() {
        if (this.isClosed.compareAndSet(false, true)) {
            Page page;
            ArrayList<RuntimeException> closingExceptions = new ArrayList<RuntimeException>();
            while ((page = this.pages.pollFirst()) != null) {
                try {
                    page.close();
                }
                catch (RuntimeException e) {
                    closingExceptions.add(e);
                }
            }
            ExceptionsHelper.rethrowAndSuppress(closingExceptions);
        }
    }

    public void ensureCapacity(long requiredCapacity) {
        if (this.isClosed.get()) {
            throw new IllegalStateException("Cannot allocate new pages if the buffer is closed.");
        }
        if (this.capacity < requiredCapacity) {
            int numPages = this.numPages(requiredCapacity + (long)this.offset);
            int pagesToAdd = numPages - this.pages.size();
            for (int i = 0; i < pagesToAdd; ++i) {
                Page page = this.pageAllocator.apply(16384);
                this.pages.addLast(page);
            }
            this.capacity += (long)(pagesToAdd * 16384);
        }
    }

    public void release(long bytesToRelease) {
        if (bytesToRelease > this.capacity) {
            throw new IllegalArgumentException("Releasing more bytes [" + bytesToRelease + "] than buffer capacity [" + this.capacity + "].");
        }
        int pagesToRelease = this.pageIndex((long)this.offset + bytesToRelease);
        for (int i = 0; i < pagesToRelease; ++i) {
            this.pages.removeFirst().close();
        }
        this.capacity -= bytesToRelease;
        this.internalIndex = Math.max(this.internalIndex - bytesToRelease, 0L);
        this.offset = this.indexInPage(bytesToRelease + (long)this.offset);
    }

    public ByteBuffer[] sliceBuffersTo(long to) {
        if (to > this.capacity) {
            throw new IndexOutOfBoundsException("can't slice a channel buffer with capacity [" + this.capacity + "], with slice parameters to [" + to + "]");
        }
        if (to == 0L) {
            return EMPTY_BYTE_BUFFER_ARRAY;
        }
        long indexWithOffset = to + (long)this.offset;
        int pageCount = this.pageIndex(indexWithOffset);
        int finalLimit = this.indexInPage(indexWithOffset);
        if (finalLimit != 0) {
            ++pageCount;
        }
        ByteBuffer[] buffers = new ByteBuffer[pageCount];
        Iterator<Page> pageIterator = this.pages.iterator();
        ByteBuffer firstBuffer = pageIterator.next().byteBuffer().duplicate();
        firstBuffer.position(firstBuffer.position() + this.offset);
        buffers[0] = firstBuffer;
        for (int i = 1; i < buffers.length; ++i) {
            buffers[i] = pageIterator.next().byteBuffer().duplicate();
        }
        if (finalLimit != 0) {
            buffers[buffers.length - 1].limit(finalLimit);
        }
        return buffers;
    }

    public Page[] sliceAndRetainPagesTo(long to) {
        if (to > this.capacity) {
            throw new IndexOutOfBoundsException("can't slice a channel buffer with capacity [" + this.capacity + "], with slice parameters to [" + to + "]");
        }
        if (to == 0L) {
            return EMPTY_BYTE_PAGE_ARRAY;
        }
        long indexWithOffset = to + (long)this.offset;
        int pageCount = this.pageIndex(indexWithOffset);
        int finalLimit = this.indexInPage(indexWithOffset);
        if (finalLimit != 0) {
            ++pageCount;
        }
        Page[] pages = new Page[pageCount];
        Iterator<Page> pageIterator = this.pages.iterator();
        Page firstPage = pageIterator.next().duplicate();
        ByteBuffer firstBuffer = firstPage.byteBuffer();
        firstBuffer.position(firstBuffer.position() + this.offset);
        pages[0] = firstPage;
        for (int i = 1; i < pages.length; ++i) {
            pages[i] = pageIterator.next().duplicate();
        }
        if (finalLimit != 0) {
            pages[pages.length - 1].byteBuffer().limit(finalLimit);
        }
        return pages;
    }

    public ByteBuffer[] sliceBuffersFrom(long from) {
        if (from > this.capacity) {
            throw new IndexOutOfBoundsException("can't slice a channel buffer with capacity [" + this.capacity + "], with slice parameters from [" + from + "]");
        }
        if (from == this.capacity) {
            return EMPTY_BYTE_BUFFER_ARRAY;
        }
        long indexWithOffset = from + (long)this.offset;
        int pageIndex = this.pageIndex(indexWithOffset);
        int indexInPage = this.indexInPage(indexWithOffset);
        ByteBuffer[] buffers = new ByteBuffer[this.pages.size() - pageIndex];
        Iterator<Page> pageIterator = this.pages.descendingIterator();
        for (int i = buffers.length - 1; i > 0; --i) {
            buffers[i] = pageIterator.next().byteBuffer().duplicate();
        }
        ByteBuffer firstPostIndexBuffer = pageIterator.next().byteBuffer().duplicate();
        firstPostIndexBuffer.position(firstPostIndexBuffer.position() + indexInPage);
        buffers[0] = firstPostIndexBuffer;
        return buffers;
    }

    public void incrementIndex(long delta) {
        if (delta < 0L) {
            throw new IllegalArgumentException("Cannot increment an index with a negative delta [" + delta + "]");
        }
        long newIndex = delta + this.internalIndex;
        if (newIndex > this.capacity) {
            throw new IllegalArgumentException("Cannot increment an index [" + this.internalIndex + "] with a delta [" + delta + "] that will result in a new index [" + newIndex + "] that is greater than the capacity [" + this.capacity + "].");
        }
        this.internalIndex = newIndex;
    }

    public long getIndex() {
        return this.internalIndex;
    }

    public long getCapacity() {
        return this.capacity;
    }

    public long getRemaining() {
        long remaining = this.capacity - this.internalIndex;
        assert (remaining >= 0L) : "The remaining [" + remaining + "] number of bytes should not be less than zero.";
        return remaining;
    }

    private int numPages(long capacity) {
        long numPages = capacity + 16383L >>> PAGE_SHIFT;
        if (numPages > Integer.MAX_VALUE) {
            throw new IllegalArgumentException("pageSize=16384 is too small for such as capacity: " + capacity);
        }
        return (int)numPages;
    }

    private int pageIndex(long index) {
        return (int)(index >>> PAGE_SHIFT);
    }

    private int indexInPage(long index) {
        return (int)(index & 0x3FFFL);
    }
}

