/*
 * Copyright (c) 1997, 2022 Oracle and/or its affiliates. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Distribution License v. 1.0, which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

package org.glassfish.jaxb.runtime.v2.runtime.property;


import org.glassfish.jaxb.runtime.api.AccessorException;
import org.glassfish.jaxb.core.v2.model.core.PropertyKind;
import org.glassfish.jaxb.runtime.v2.model.runtime.RuntimeValuePropertyInfo;
import org.glassfish.jaxb.runtime.v2.runtime.JAXBContextImpl;
import org.glassfish.jaxb.runtime.v2.runtime.XMLSerializer;
import org.glassfish.jaxb.runtime.v2.runtime.reflect.Accessor;
import org.glassfish.jaxb.runtime.v2.runtime.reflect.TransducedAccessor;
import org.glassfish.jaxb.runtime.v2.runtime.unmarshaller.ChildLoader;
import org.glassfish.jaxb.runtime.v2.runtime.unmarshaller.ValuePropertyLoader;
import org.glassfish.jaxb.runtime.v2.util.QNameMap;
import jakarta.xml.bind.annotation.XmlValue;
import org.xml.sax.SAXException;

import javax.xml.stream.XMLStreamException;
import java.io.IOException;

/**
 * {@link Property} implementation for {@link XmlValue} properties.
 *
 * <p>
 * This one works for both leaves and nodes, scalars and arrays.
 *
 * @author Bhakti Mehta (bhakti.mehta@sun.com)
 */
public final class ValueProperty<BeanT> extends PropertyImpl<BeanT> {

    /**
     * Heart of the conversion logic.
     */
    private final TransducedAccessor<BeanT> xacc;
    private final Accessor<BeanT,?> acc;


    public ValueProperty(JAXBContextImpl context, RuntimeValuePropertyInfo prop) {
        super(context,prop);
        xacc = TransducedAccessor.get(context,prop);
        acc = prop.getAccessor();   // we only use this for binder, so don't waste memory by optimizing
    }

    @Override
    public void serializeBody(BeanT o, XMLSerializer w, Object outerPeer) throws SAXException, AccessorException, IOException, XMLStreamException {
        if(xacc.hasValue(o))
            xacc.writeText(w,o,fieldName);
    }

    @Override
    public void serializeURIs(BeanT o, XMLSerializer w) throws SAXException, AccessorException {
        xacc.declareNamespace(o,w);
    }

    @Override
    public boolean hasSerializeURIAction() {
        return xacc.useNamespace();
    }

    @Override
    public void buildChildElementUnmarshallers(UnmarshallerChain chainElem, QNameMap<ChildLoader> handlers) {
        handlers.put(TEXT_HANDLER,
                new ChildLoader(new ValuePropertyLoader(xacc),null));
    }

    @Override
    public PropertyKind getKind() {
        return PropertyKind.VALUE;
    }

    @Override
    public void reset(BeanT o) throws AccessorException {
        acc.set(o,null);
    }

    @Override
    public String getIdValue(BeanT bean) throws AccessorException, SAXException {
        return xacc.print(bean).toString();
    }

}
