/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.system.server.profile.basic;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.jboss.deployers.structure.spi.DeploymentContext;
import org.jboss.deployers.structure.spi.main.MainDeployerStructure;
import org.jboss.deployers.vfs.spi.client.VFSDeployment;
import org.jboss.deployers.vfs.spi.structure.VFSDeploymentContext;
import org.jboss.profileservice.spi.ProfileKey;
import org.jboss.virtual.VFSUtils;
import org.jboss.virtual.VirtualFile;

/**
 * Profile monitoring metadata changes.
 *
 * We need to cache files's last modifed timestamps,
 * since file.getChildren re-caches handlers,
 * hence hasBeenModified is useless.
 *
 * @author <a href="mailto:ales.justin@jboss.com">Ales Justin</a>
 */
public class MetaDataAwareProfile extends ProfileImpl
{
   private MainDeployerStructure mainDeployer;
   private Map<String, Long> lastModifiedCache;

   public MetaDataAwareProfile(String profileRoot, ProfileKey key)
   {
      super(profileRoot, key);
      lastModifiedCache = new ConcurrentHashMap<String, Long>();
   }

   /**
    * Set main deployer structure.
    *
    * @param mainDeployer the main deployer structure
    */
   public void setMainDeployer(MainDeployerStructure mainDeployer)
   {
      this.mainDeployer = mainDeployer;
   }

   protected boolean hasBeenModified(VirtualFile root) throws Exception
   {
      // get file:/ schema
      URI uri = VFSUtils.getCompatibleURI(root);
      File file = new File(uri);
      // if root is file check its modification
      if (file.isFile())
         return root.hasBeenModified();

      // else check metadata
      String name = root.toURI().toString();
      VFSDeploymentContext deploymentContext = getDeploymentContext(name);
      if (deploymentContext != null)
         return hasBeenModified(deploymentContext);

      log.debug("Falling back to root name: " + root);
      deploymentContext = getDeploymentContext(root.getName());
      if (deploymentContext != null)
         return hasBeenModified(deploymentContext);

      return false;
   }

   /**
    * Has vfs deployment context been modified.
    *
    * @param deploymentContext the vfs deployment context
    * @return true if modified
    * @throws IOException for any error
    */
   protected boolean hasBeenModified(VFSDeploymentContext deploymentContext) throws IOException
   {
      List<VirtualFile> metadataLocations = deploymentContext.getMetaDataLocations();
      if (metadataLocations != null && metadataLocations.isEmpty() == false)
      {
         for(VirtualFile metadataLocation : metadataLocations)
         {
            List<VirtualFile> children = metadataLocation.getChildren();
            if (children != null && children.isEmpty() == false)
            {
               for(VirtualFile child : children)
               {
                  String pathName = child.getPathName();
                  Long timestamp = lastModifiedCache.get(pathName);
                  long lastModified = child.getLastModified();
                  lastModifiedCache.put(pathName, lastModified);
                  if (timestamp != null && timestamp < lastModified)
                  {
                     if (log.isTraceEnabled())
                        log.trace("Metadata location modified: " + child);
                     return true;
                  }
               }
            }
         }
      }
      List<DeploymentContext> childContexts = deploymentContext.getChildren();
      if (childContexts != null && childContexts.isEmpty() == false)
      {
         for (DeploymentContext childContext : childContexts)
         {
            if (childContext instanceof VFSDeploymentContext)
            {
               if (hasBeenModified((VFSDeploymentContext)childContext))
                  return true;
            }
         }
      }
      return false;
   }

   // expecting all deployments from same context root
   // so path name should group them per deployment unit
   protected void postRemove(VFSDeployment deployment) throws Exception
   {
      VirtualFile root = deployment.getRoot();
      String pathName = root.getPathName();
      if (log.isTraceEnabled())
         log.debug("Removing last modified cache info for: " + pathName);

      Iterator<String> iter = lastModifiedCache.keySet().iterator();
      while (iter.hasNext())
      {
         if (iter.next().startsWith(pathName))
            iter.remove();
      }
   }

   /**
    * Get deployment context.
    *
    * @param name the deployment context name
    * @return vfs deployment context or null if doesn't exist or not vfs based
    */
   @SuppressWarnings("deprecation")
   protected VFSDeploymentContext getDeploymentContext(String name)
   {
      if (mainDeployer == null)
         throw new IllegalArgumentException("Null main deployer");

      DeploymentContext deploymentContext = mainDeployer.getDeploymentContext(name);
      if (deploymentContext == null || deploymentContext instanceof VFSDeploymentContext == false)
         return null;

      return (VFSDeploymentContext)deploymentContext;
   }
}
