/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.system.server.profileservice.repository;

import java.io.File;
import java.io.IOException;
import java.io.NotSerializableException;
import java.util.Map;
import java.util.Map.Entry;

import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.logging.Logger;
import org.jboss.profileservice.aop.TrackingAdvice;
import org.jboss.profileservice.spi.AttachmentsSerializer;

/**
 * A base AttachmentsSerializer that uses a file system based store.
 * 
 * @author Scott.Stark@jboss.org
 * @version $Revision:$
 */
public abstract class AbstractFileAttachmentsSerializer
   implements AttachmentsSerializer
{
   private static final Logger log = Logger.getLogger(AbstractFileAttachmentsSerializer.class);
   /** The deployment pre-processed attachments store dir */
   private File attachmentsStoreDir;

   public File getAttachmentsStoreDir()
   {
      return attachmentsStoreDir;
   }
   public void setAttachmentsStoreDir(File attachmentsStoreDir)
   {
      this.attachmentsStoreDir = attachmentsStoreDir;
   }

   public Map<String, Object> loadAttachments(DeploymentUnit ctx, String deployerID, ClassLoader loader)
      throws Exception
   {
      if( attachmentsStoreDir == null )
         throw new IllegalStateException("attachmentsStoreDir has not been set");

      String vfsPath = ctx.getSimpleName();
      vfsPath += ".attachments";
      File deployerDir = new File(attachmentsStoreDir, deployerID);
      File attachmentsStore = new File(deployerDir, vfsPath);
      if( attachmentsStore.exists() == false )
      {
         return null;
      }

      return loadAttachments(attachmentsStore, loader);
   }

   public void saveAttachments(DeploymentUnit ctx, String deployerID, ClassLoader loader)
      throws Exception
   {
      if( attachmentsStoreDir == null )
         throw new IllegalStateException("attachmentsStoreDir has not been set");

      String vfsPath = ctx.getSimpleName();
      if( vfsPath.length() == 0 )
         vfsPath = ctx.getSimpleName();
      vfsPath += ".attachments";
      File deployerDir = new File(attachmentsStoreDir, deployerID);
      File attachmentsStore = new File(deployerDir, vfsPath);
      File attachmentsParent = attachmentsStore.getParentFile();
      if( attachmentsParent.exists() == false )
      {
         if( attachmentsParent.mkdirs() == false )
            throw new IOException("Failed to create attachmentsParent: "+attachmentsParent.getAbsolutePath());
      }

      Map<String, Object> attachments = TrackingAdvice.clearAttachmentsForTarget(ctx.getTransientManagedObjects());
      if( attachments != null )
      {
         try
         {
            saveAttachments(attachmentsStore, attachments, loader);
         }
         catch(NotSerializableException e)
         {
            // Log what is in the attachments
            StringBuilder tmp = new StringBuilder("Save failed with NSE, attachments contents: ");
            for(Entry<String, Object> entry : attachments.entrySet())
            {
               tmp.append(entry.getKey());
               tmp.append('=');
               tmp.append(entry.getValue());
            }
            log.error(tmp.toString());
            throw e;
         }
      }
   }

   protected abstract Map<String, Object> loadAttachments(File attachmentsStore, ClassLoader loader)
      throws Exception;
   protected abstract void saveAttachments(File attachmentsStore, Map<String, Object> attachments, ClassLoader loader)
      throws Exception;
}
