/*
 * JBoss, Home of Professional Open Source
 * Copyright 2007, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.system.server.profileservice.repository;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import java.util.zip.ZipInputStream;

import org.jboss.deployers.spi.attachments.Attachments;
import org.jboss.deployers.vfs.spi.client.VFSDeployment;
import org.jboss.logging.Logger;
import org.jboss.managed.api.ManagedDeployment.DeploymentPhase;
import org.jboss.profileservice.spi.DeploymentRepository;
import org.jboss.profileservice.spi.ModificationInfo;
import org.jboss.profileservice.spi.NoSuchDeploymentException;
import org.jboss.profileservice.spi.ProfileKey;
import org.jboss.profileservice.spi.ModificationInfo.ModifyStatus;
import org.jboss.profileservice.spi.repository.MutableRepository;
import org.jboss.profileservice.spi.repository.Repository;
import org.jboss.profileservice.spi.repository.RepositoryAdmin;
import org.jboss.profileservice.spi.repository.Resource;
import org.jboss.virtual.VirtualFile;

/**
 * A bridge between the DeploymentRepository and the OBR RepositoryAdmin
 * to validate that the OBR api can be used for the profile service.
 * 
 * TODO: may need separate deployment phase maps for the OBR id to vfspath to
 * track deployments by phase.
 * 
 * @author Scott.Stark@jboss.org
 * @version $Revision: 77864 $
 */
public class RepositoryAdminAdaptor
   implements DeploymentRepository
{
   private static Logger log = Logger.getLogger(RepositoryAdminAdaptor.class);

   /** The OBR repository interface */
   private RepositoryAdmin delegate;
   /** The server root container the deployments */
   private File root;
   /** The bootstrap jboss-service.xml dir */
   private File bootstrapDir;
   /** The server static libraries */
   private File libDir;
   /** The deployers phase deployments dir */
   private File deployersDir;
   /** The application phase deployments dir */
   private File[] applicationDirs;
   /** The deployment post edit */
   private File adminEditsRoot;
   /** The profile key this repository is associated with */
   private ProfileKey key;
   /** The last time the profile was modified */
   private long lastModified;

   public void addDeployment(String vfsPath, VFSDeployment d, DeploymentPhase phase)
      throws Exception
   {
      MutableRepository mrepo = getRepository(d.getRoot().toURI());
      VFSDeploymentResource dres = new VFSDeploymentResource(d, phase, mrepo);
      mrepo.addResource(dres);
   }

   public String addDeploymentContent(String name, InputStream contentIS, DeploymentPhase phase)
      throws IOException
   {
      // TODO
      throw new IOException("Not yet implemented");
   }

   public String[] getRepositoryNames(String[] names, DeploymentPhase phase)
         throws IOException
   {
      // TODO Auto-generated method stub
      return null;
   }

   public VirtualFile getDeploymentContent(String vfsPath, DeploymentPhase phase)
         throws IOException
   {
      // TODO Auto-generated method stub
      return null;
   }

   public int lockDeploymentContent(String vfsPath, DeploymentPhase phase)
   {
      // TODO Auto-generated method stub
      return 0;
   }
   public int unlockDeploymentContent(String vfsPath, DeploymentPhase phase)
   {
      // TODO Auto-generated method stub
      return 0;
   }

   
   public int getDeploymentContentFlags(String vfsPath, DeploymentPhase phase)
   {
      // TODO Auto-generated method stub
      return 0;
   }

   public int clearDeploymentContentFlags(String vfsPath,
         DeploymentPhase phase, int flags)
   {
      // TODO Auto-generated method stub
      return 0;
   }

   public boolean hasDeploymentContentFlags(String vfsPath,
         DeploymentPhase phase, int flag)
   {
      // TODO Auto-generated method stub
      return false;
   }

   public int setDeploymentContentFlags(String vfsPath, DeploymentPhase phase,
         int flags)
   {
      // TODO Auto-generated method stub
      return 0;      
   }

   public void acquireDeploymentContentLock()
   {
      // TODO Auto-generated method stub
   }

   public void releaseDeploymentContentLock()
   {
      // TODO Auto-generated method stub
   }


   public void addManagedObject(String vfsPath, Attachments edits) throws Exception
   {
      MutableRepository repo = this.getRepository(adminEditsRoot.toURI());
      AttachmentsResource attachments = new AttachmentsResource(vfsPath, edits, repo);
      repo.addResource(attachments);
      lastModified = System.currentTimeMillis();
   }
   
   public long getLastModified()
   {
      return this.lastModified;
   }

   public void create() throws Exception
   {
      File profileRoot = new File(root, key.getName());
      if( profileRoot.exists() == true )
         throw new IOException("Profile root already exists: "+profileRoot);
      if( profileRoot.mkdirs() == false )
         throw new IOException("Failed to create profile root: "+profileRoot);
      // server/{name}/bootstrap
      bootstrapDir = new File(profileRoot, "bootstrap");
      if( bootstrapDir.mkdirs() == false )
         throw new IOException("Failed to create profile bootstrap dir: "+bootstrapDir);
      delegate.addRepository(bootstrapDir.toURI());

      // server/{name}/deployers
      deployersDir = new File(profileRoot, "deployers");
      if( deployersDir.mkdirs() == false )
         throw new IOException("Failed to create profile deployers dir: "+deployersDir);
      delegate.addRepository(deployersDir.toURI());

      // server/{name}/deploy dirs
      for (File applicationDir : applicationDirs)
      {
         if( applicationDir.mkdirs() == false )
            throw new IOException("Failed to create profile deploy dir: "+applicationDir);
         delegate.addRepository(applicationDir.toURI());
      }

      // server/{name}/lib
      libDir = new File(profileRoot, "lib");
      if( libDir.mkdirs() == false )
         throw new IOException("Failed to create profile lib dir: "+libDir);
      delegate.addRepository(libDir.toURI());

      adminEditsRoot = new File(profileRoot, "profile/edits");
      if( adminEditsRoot.mkdirs() == false )
         throw new IOException("Failed to create profile adminEdits dir: "+adminEditsRoot);
      delegate.addRepository(adminEditsRoot.toURI());
      lastModified = System.currentTimeMillis();
   }

   public VFSDeployment getDeployment(String name, DeploymentPhase phase)
      throws Exception, NoSuchDeploymentException
   {
      VFSDeployment ctx = null;
      if( phase == null )
      {
         // Try all phases
         try
         {
            ctx = this.getBootstrap(name);
         }
         catch(NoSuchDeploymentException ignore)
         {
         }
         try
         {
            if( ctx == null )
               ctx = this.getDeployer(name);
         }
         catch(NoSuchDeploymentException ignore)
         {
         }
         try
         {
            if( ctx == null )
               ctx = this.getApplication(name);
         }
         catch(NoSuchDeploymentException ignore)
         {
         }
      }
      else
      {
         switch( phase )
         {
            case BOOTSTRAP:
               ctx = this.getBootstrap(name);
               break;
            case DEPLOYER:
               ctx = this.getDeployer(name);
               break;
            case APPLICATION:
               ctx = this.getApplication(name);
               break;
         }
      }
      // Make sure we don't return null
      if( ctx == null )
         throw new NoSuchDeploymentException("name="+name+", phase="+phase);
      return ctx;
   }

   /**
    * Obtain the VFSDeploymentResource SymbolicName from all
    * repositories.
    */
   public Set<String> getDeploymentNames()
   {
      HashSet<String> names = new HashSet<String>();
      Repository[] repositories = delegate.listRepositories();
      for(Repository repo : repositories)
      {
         Resource[] resources = repo.getResources();
         for(Resource res : resources)
         {
            if(res instanceof VFSDeploymentResource)
            {
               VFSDeploymentResource vfsres = (VFSDeploymentResource) res;
               names.add(vfsres.getSymbolicName());
            }
         }
      }
      return names;
   }

   /**
    * Obtain the VFSDeploymentResource SymbolicName from the
    * repository associated with the getDeploymentURI(phase) URI.
    */
   public Set<String> getDeploymentNames(DeploymentPhase phase)
   {
      HashSet<String> names = new HashSet<String>();
      URI uri = getDeploymentURI(phase);
      try
      {
         Repository repo = delegate.getRepository(uri);
         Resource[] resources = repo.getResources();
         for(Resource res : resources)
         {
            if(res instanceof VFSDeploymentResource)
            {
               VFSDeploymentResource vfsres = (VFSDeploymentResource) res;
               names.add(vfsres.getSymbolicName());
            }
         }
      }
      catch(Exception e)
      {
         throw new IllegalStateException(e);
      }
      return names;
   }

   /**
    * TODO: should be a resolver query for matching types via
    * discoverResources(String).
    */
   public Set<String> getDeploymentNamesForType(String type)
   {
      return null;
   }

   /**
    * Obtain the VFSDeploymentResource deployment property from all
    * repositories.
    */
   public Collection<VFSDeployment> getDeployments() throws Exception
   {
      HashSet<VFSDeployment> deployments = new HashSet<VFSDeployment>();
      Repository[] repositories = delegate.listRepositories();
      for(Repository repo : repositories)
      {
         Resource[] resources = repo.getResources();
         for(Resource res : resources)
         {
            if(res instanceof VFSDeploymentResource)
            {
               VFSDeploymentResource vfsres = (VFSDeploymentResource) res;
               VFSDeployment deployment = (VFSDeployment) vfsres.getProperties().get("deployment");
               if(deployment != null)
                  deployments.add(deployment);
            }
         }
      }
      return deployments;
   }

   /**
    * Obtain the VFSDeploymentResource deployment property from the
    * repository
    */
   public Collection<VFSDeployment> getDeployments(DeploymentPhase phase)
      throws Exception
   {
      URI uri = getDeploymentURI(phase);
      Repository repo = delegate.getRepository(uri);
      Resource[] resources = repo.getResources();
      HashSet<VFSDeployment> deployments = new HashSet<VFSDeployment>();
      for(Resource res : resources)
      {
         if(res instanceof VFSDeploymentResource)
         {
            VFSDeploymentResource vfsres = (VFSDeploymentResource) res;
            VFSDeployment deployment = (VFSDeployment) vfsres.getProperties().get("deployment");
            if(deployment != null)
               deployments.add(deployment);
         }
      }
      return deployments;
   }

   public URI getDeploymentURI(DeploymentPhase phase)
   {
      URI uri = null;
      switch( phase )
      {
         case BOOTSTRAP:
            uri = this.getBootstrapURI();
            break;
         case DEPLOYER:
            uri = this.getDeployersURI();
            break;
         case APPLICATION:
            uri = this.getApplicationURI();
            break;
      }
      return uri;
   }

   public Collection<ModificationInfo> getModifiedDeployments() throws Exception
   {
      ArrayList<ModificationInfo> modified = new ArrayList<ModificationInfo>();
      URI appURI = getApplicationURI();
      MutableRepository repo = getRepository(appURI);
      Collection<Resource> resources = repo.getModifiedResources();
      boolean trace = log.isTraceEnabled();
      if( trace )
         log.trace("Checking applications for modifications");
      if( resources != null )
      {
         for(Resource res : resources)
         {
            VFSDeployment ctx = (VFSDeployment) res.getProperties().get("deployment");
            if( ctx == null )
               continue;
            MutableRepository.ModifyStatus status = (MutableRepository.ModifyStatus) res.getProperties().get("modifyStatus");
            Long lastModified = (Long) res.getProperties().get("lastModified");
            VirtualFile root = ctx.getRoot();
            String name = root.getPathName();
            // Check for removal
            if(status == MutableRepository.ModifyStatus.REMOVED)
            {
               ModificationInfo info = new ModificationInfo(ctx, lastModified, ModifyStatus.REMOVED);
               modified.add(info);
               if( trace )
                  log.trace(name+" was removed");
            }
            // Check for modification
            else if(status == MutableRepository.ModifyStatus.MODIFIED)
            {
               if( trace )
                  log.trace(name+" was modified: "+lastModified);
               ModificationInfo info = new ModificationInfo(ctx, lastModified, ModifyStatus.MODIFIED);
               modified.add(info);
            }
            // Check for additions
            else if(status == MutableRepository.ModifyStatus.ADDED)
            {
               if( trace )
                  log.trace(name+" was added: "+lastModified);
               ModificationInfo info = new ModificationInfo(ctx, lastModified, ModifyStatus.ADDED);
               modified.add(info);               
            }
         }
      }
      if(modified.size() > 0)
         lastModified = System.currentTimeMillis();
      return modified;
   }

   public void load() throws Exception
   {
   }

   public void remove() throws Exception
   {
      Repository[] repositories = delegate.listRepositories();
      for(Repository repo : repositories)
      {
         delegate.removeRepository(repo.getURI());
      }
   }

   public VFSDeployment removeDeployment(String name, DeploymentPhase phase)
      throws Exception
   {
      URI uri = null;
      switch( phase )
      {
         case BOOTSTRAP:
            uri = getBootstrapURI();
            break;
         case DEPLOYER:
            uri = getDeployersURI();
            break;
         case APPLICATION:
            uri = getApplicationURI();
            break;
      }
      VFSDeployment ctx = removeDeployment(name, uri);
      return ctx;
   }

   public void setDeploymentURI(URI uri, DeploymentPhase phase)
   {
      switch( phase )
      {
         case BOOTSTRAP:
            this.setBootstrapURI(uri);
            break;
         case DEPLOYER:
            this.setDeployersURI(uri);
            break;
         case APPLICATION:
            this.setApplicationURIs(new URI[]{uri});
            break;
      }      
   }

   public void updateDeployment(String vfsPath, VFSDeployment d, DeploymentPhase phase) throws Exception
   {      
   }

   /** This requires that the RepositoryAdmin be able to walk up the
      deployment URI to a matching repository URI
      @param uri a URI pointing to a resource under a repository
   */
   protected MutableRepository getRepository(URI uri)
      throws Exception
   {
      Repository repo = delegate.getRepository(uri);
      if ( (repo instanceof MutableRepository) == false )
         throw new IllegalStateException("Repository does not support MutableRepository");
      MutableRepository mrepo = (MutableRepository) repo;
      return mrepo;
   }

   protected VFSDeployment getBootstrap(String vfsPath)
      throws Exception
   {
      Resource res = delegate.getResource(vfsPath);
      if (res == null || (res instanceof VFSDeploymentResource) == false)
         throw new NoSuchDeploymentException(vfsPath);
      VFSDeploymentResource vfsres = (VFSDeploymentResource) res;
      VFSDeployment deployment = (VFSDeployment) vfsres.getProperties().get("deployment");
      return deployment;
   }
   
   protected VFSDeployment getDeployer(String vfsPath)
      throws Exception
   {
      Resource res = delegate.getResource(vfsPath);
      if (res == null || (res instanceof VFSDeploymentResource) == false)
         throw new NoSuchDeploymentException(vfsPath);
      VFSDeploymentResource vfsres = (VFSDeploymentResource) res;
      VFSDeployment deployment = (VFSDeployment) vfsres.getProperties().get("deployment");
      return deployment;
   }

   protected VFSDeployment getApplication(String vfsPath)
      throws Exception
   {
      Resource res = delegate.getResource(vfsPath);
      if (res == null || (res instanceof VFSDeploymentResource) == false)
         throw new NoSuchDeploymentException(vfsPath);
      VFSDeploymentResource vfsres = (VFSDeploymentResource) res;
      VFSDeployment deployment = (VFSDeployment) vfsres.getProperties().get("deployment");
      return deployment;
   }

   protected URI getBootstrapURI()
   {
      return bootstrapDir.toURI();
   }
   protected URI getDeployersURI()
   {
      return deployersDir.toURI();
   }
   protected URI getApplicationURI()
   {
      File applicationDir = applicationDirs[0];
      return applicationDir.toURI();
   }

   protected void setBootstrapURI(URI uri)
   {
      bootstrapDir = new File(uri);
   }
   protected void setDeployersURI(URI uri)
   {
      deployersDir = new File(uri);
   }
   public void setApplicationURIs(URI[] uris)
   {
      applicationDirs = new File[uris.length];
      for (int n = 0; n < uris.length; n ++)
      {
         URI uri = uris[n];
         applicationDirs[n] = new File(uri);
      }
   }

   protected VFSDeployment removeDeployment(String name, URI uri) throws Exception
   {
      MutableRepository mrepo = getRepository(uri);
      Resource res = mrepo.removeResource(name);
      VFSDeploymentResource vfsres = (VFSDeploymentResource) res;
      VFSDeployment deployment = (VFSDeployment) vfsres.getProperties().get("deployment");
      return deployment;
   }
}
