/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.system.server.profileservice;

import java.lang.annotation.Annotation;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;

import org.jboss.beans.metadata.spi.BeanMetaData;
import org.jboss.bootstrap.spi.Bootstrap;
import org.jboss.bootstrap.spi.Server;
import org.jboss.bootstrap.spi.ServerConfig;
import org.jboss.bootstrap.spi.microcontainer.MCServer;
import org.jboss.dependency.spi.ControllerContext;
import org.jboss.dependency.spi.ControllerState;
import org.jboss.deployers.client.spi.IncompleteDeploymentException;
import org.jboss.deployers.client.spi.main.MainDeployer;
import org.jboss.deployers.plugins.main.MainDeployerImpl;
import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.spi.deployer.managed.ManagedDeploymentCreator;
import org.jboss.deployers.spi.management.KnownComponentTypes;
import org.jboss.deployers.structure.spi.DeploymentContext;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.deployers.vfs.deployer.kernel.KernelDeploymentDeployer.KernelDeploymentVisitor;
import org.jboss.deployers.vfs.spi.client.VFSDeployment;
import org.jboss.kernel.Kernel;
import org.jboss.kernel.spi.dependency.KernelController;
import org.jboss.kernel.spi.deployment.KernelDeployment;
import org.jboss.logging.Logger;
import org.jboss.managed.api.ComponentType;
import org.jboss.managed.api.Fields;
import org.jboss.managed.api.ManagedDeployment;
import org.jboss.managed.api.ManagedObject;
import org.jboss.managed.api.ManagedProperty;
import org.jboss.managed.api.MutableManagedObject;
import org.jboss.managed.api.ManagedDeployment.DeploymentPhase;
import org.jboss.managed.api.annotation.ManagementComponent;
import org.jboss.managed.api.annotation.ViewUse;
import org.jboss.managed.api.factory.ManagedObjectFactory;
import org.jboss.managed.plugins.DefaultFieldsImpl;
import org.jboss.managed.plugins.ManagedComponentImpl;
import org.jboss.managed.plugins.ManagedPropertyImpl;
import org.jboss.metatype.api.types.ArrayMetaType;
import org.jboss.metatype.api.types.MetaType;
import org.jboss.metatype.api.types.SimpleMetaType;
import org.jboss.metatype.api.values.ArrayValueSupport;
import org.jboss.metatype.api.values.EnumValue;
import org.jboss.metatype.api.values.EnumValueSupport;
import org.jboss.profileservice.spi.Profile;
import org.jboss.profileservice.spi.ProfileKey;
import org.jboss.profileservice.spi.ProfileService;
import org.jboss.profileservice.spi.types.ControllerStateMetaType;

/**
 * Bootstraps the profile service
 * 
 * @author Scott.Stark@jboss.org
 * @author adrian@jboss.org
 * @version $Revision: 81835 $
 */
public class ProfileServiceBootstrap implements Bootstrap
{
   /** The log */
   private static final Logger log = Logger.getLogger(ProfileServiceBootstrap.class);
   
   /** The name of the profile that is being booted */
   protected String profileName = "default";

   /** The server MainDeployer */
   protected MainDeployer mainDeployer;

   /** The server ProfileService */
   protected ProfileService profileService;

   /** The ManagedDeploymentCreator plugin */
   private ManagedDeploymentCreator mgtDeploymentCreator = null;
   /** The ManagedObjectFactory for building the bootstrap deployment ManagedObjects */
   private ManagedObjectFactory mof;
   /** The ManagedDeployment map for the MCServer KernelDeployments */
   private Map<String, ManagedDeployment> bootstrapMDs = new HashMap<String, ManagedDeployment>();

   /** Whether we are shutdown */
   private AtomicBoolean shutdown = new AtomicBoolean(false);
   
   /**
    * Create a new ProfileServiceBootstrap.
    */
   public ProfileServiceBootstrap()
   {
   }

   /**
    * Return the MainDeployer bean.
    * 
    * @return the MainDeployer bean if bootstrap succeeded, null otherwise.
    */
   public MainDeployer getMainDeployer()
   {
      return mainDeployer;
   }
   public void setMainDeployer(MainDeployer mainDeployer)
   {
      this.mainDeployer = mainDeployer;
   }

   /**
    * Return the ProfileService bean.
    * 
    * @return the ProfileService bean if bootstrap succeeded, null otherwise
    */
   public ProfileService getProfileService()
   {
      return profileService;
   }
   public void setProfileService(ProfileService profileService)
   {
      this.profileService = profileService;
   }

   public String getProfileName()
   {
      return profileName;
   }
   public void setProfileName(String profileName)
   {
      this.profileName = profileName;
   }

   public ManagedObjectFactory getMof()
   {
      return mof;
   }
   public void setMof(ManagedObjectFactory mof)
   {
      this.mof = mof;
   }

   public ManagedDeploymentCreator getMgtDeploymentCreator()
   {
      return mgtDeploymentCreator;
   }
   public void setMgtDeploymentCreator(ManagedDeploymentCreator mgtDeploymentCreator)
   {
      this.mgtDeploymentCreator = mgtDeploymentCreator;
   }

   public Map<String, ManagedDeployment> getBootstrapMDs()
   {
      return bootstrapMDs;
   }
   public void setBootstrapMDs(Map<String, ManagedDeployment> bootstrapMDs)
   {
      this.bootstrapMDs = bootstrapMDs;
   }

   /**
    * 
    */
   public void start(Server server)
      throws Exception
   {
      shutdown.set(false);

      if(profileService == null)
         throw new IllegalStateException("The ProfileService has not been injected"); 
      log.debug("Using ProfileService: " + profileService);
      if(mainDeployer == null)
         throw new IllegalStateException("The MainDeployer has not been injected"); 
      log.debug("Using MainDeployer: " + mainDeployer);

      // Validate that everything is ok
      mainDeployer.checkComplete();

      // Expose the bootstrap ManagedDeployments
      initBootstrapMDs(server);

      // Load the profile beans
      try
      {
         profileName = server.getConfig().getServerName();
         loadProfile(profileName);
      }
      catch (IncompleteDeploymentException e)
      {
         log.error("Failed to load profile: " + e.getMessage());
      }
      catch (Exception e)
      {
         log.error("Failed to load profile: ", e);
      }
      // Mark the profile as ready for hotdeployment if supported
      Profile profile = profileService.getActiveProfile();
      if( profile != null )
         profile.enableModifiedDeploymentChecks(true);
   }

   public void prepareShutdown(Server server)
   {
      shutdown.set(true);
      if (mainDeployer != null)
         mainDeployer.prepareShutdown();
   }

   public void shutdown(Server server)
   {
      unloadProfile(profileName);
      try
      {
         mainDeployer.shutdown();
      }
      catch (Throwable t)
      {
         log.warn("Error shutting down the main deployer", t);
      }
   }

   /**
    * Load the deployments associated with the named profile and deploy them
    * using the MainDeployer.
    * 
    * @param name
    * @throws Exception for any error
    * @throws NullPointerException if either the MainDeployer or ProfileService
    * have not been injected.
    */
   protected void loadProfile(String name) throws Exception
   {
      MainDeployer deployer = getMainDeployer();
      if (deployer == null)
         throw new NullPointerException("MainDeployer has not been set");
      ProfileService ps = getProfileService();
      if (ps == null)
         throw new NullPointerException("ProfileService has not been set");

      // Load the named profile
      ProfileKey key = new ProfileKey(name);
      Profile profile = ps.getProfile(key);

      // HACK
      VFSDeployment first = null;
      
      // Deploy the profile bootstrap deployments
      Collection<VFSDeployment> boostraps = profile.getDeployments(DeploymentPhase.BOOTSTRAP);
      for (VFSDeployment d : boostraps)
      {
         deployer.addDeployment(d);
         if (first == null)
            first = d;
      }
      deployer.process();
      deployer.checkComplete();

      Thread thread = Thread.currentThread();
      ClassLoader old = thread.getContextClassLoader();
      // FIXME remove this hack
      MainDeployerImpl hack = (MainDeployerImpl) deployer;
      ClassLoader cl = null;
      if (first != null)
      {
         DeploymentContext ctx = hack.getDeploymentContext(first.getName());
         if (ctx != null)
            cl = ctx.getClassLoader();
      }
      //if (cl != null)
      //   thread.setContextClassLoader(cl);
      try
      {
         
         // Deploy the profile deployers
         Collection<VFSDeployment> profileDeployers = profile.getDeployments(DeploymentPhase.DEPLOYER);
         for (VFSDeployment d : profileDeployers)
            deployer.addDeployment(d);
         deployer.process();
         deployer.checkComplete();

         // Deploy the profile applications
         Collection<VFSDeployment> profileDeployments = profile.getDeployments(DeploymentPhase.APPLICATION);
         for (VFSDeployment d : profileDeployments)
            deployer.addDeployment(d);
         deployer.process();
         deployer.checkComplete();
      }
      finally
      {
         thread.setContextClassLoader(old);
      }
   }

   /**
    * Unload the deployments associated with the named profile and undeploy them
    * using the MainDeployer in reverse phase order.
    * 
    * @param name the profile name
    * @throws NullPointerException if either the MainDeployer or ProfileService
    * have not been injected.
    */
   protected void unloadProfile(String name)
   {
      MainDeployer deployer = getMainDeployer();
      if (deployer == null)
      {
         log.warn("MainDeployer has not been set");
         return;
      }
      ProfileService ps = getProfileService();
      if (ps == null)
      {
         log.warn("ProfileService has not been set");
         return;
      }

      try
      {
         // Load the named profile
         ProfileKey key = new ProfileKey(name);
         Profile profile = ps.getProfile(key);

         // HACK
         VFSDeployment first = null;
         
         // Deploy the bootstrap
         Collection<VFSDeployment> boostraps = profile.getDeployments(DeploymentPhase.BOOTSTRAP);
         for (VFSDeployment d : boostraps)
         {
            if (first == null)
            {
               first = d;
               break;
            }
         }

         Thread thread = Thread.currentThread();
         ClassLoader old = thread.getContextClassLoader();
         // FIXME remove this hack
         MainDeployerImpl hack = (MainDeployerImpl) deployer;
         ClassLoader cl = null;
         if (first != null)
         {
            try
            {
               DeploymentContext ctx = hack.getDeploymentContext(first.getName());
               if (ctx != null)
                  cl = ctx.getClassLoader();
            }
            catch (Exception e)
            {
               log.debug("Unable to get first deployment", e);
            }
         }
         //if (cl != null)
         //   thread.setContextClassLoader(cl);
         try
         {
            // Undeploy the applications
            unload(deployer, profile.getDeployments(DeploymentPhase.APPLICATION));
            // Undeploy the deployers
            unload(deployer, profile.getDeployments(DeploymentPhase.DEPLOYER));
            // Undeploy the bootstrap
            unload(deployer, profile.getDeployments(DeploymentPhase.BOOTSTRAP));
         }
         finally
         {
            thread.setContextClassLoader(old);
         }
      }
      catch (Throwable t)
      {
         log.warn("Error unloading profile", t);
      }
   }

   /**
    * Unload a set of deployments
    * 
    * @param deployer the main deployer
    * @param deployments the deployments
    */
   protected void unload(MainDeployer deployer, Collection<VFSDeployment> deployments)
   {
      if (deployments == null || deployments.isEmpty())
         return;
      
      for (VFSDeployment d : deployments)
      {
         try
         {
            deployer.removeDeployment(d);
         }
         catch (Exception e)
         {
            log.warn("Unable to remove deployment: " + d);
         }
      }
      deployer.process();
   }

   /**
    * Create ManagedDeployments for the MCServer KernelDeployments. This allows
    * the bootstrap deployments outside of the profile service to be visible in
    * the ManagementView
    * @see {@link ManagementView}
    * 
    * @param server - the Bootstrap.start Server instance. This must be an
    * MCServer in order for there to be KernelDeployments available.
    */
   protected void initBootstrapMDs(Server server)
   {
      if(mof == null || mgtDeploymentCreator == null)
      {
         log.warn("Skipping ManagedDeployment creation due to missing mof, mgtDeploymentCreator");
         return;
      }

      Map<String, KernelDeployment> serverDeployments = null;
      if(server instanceof MCServer)
      {
         MCServer mcserver = MCServer.class.cast(server);
         Kernel kernel = mcserver.getKernel();
         serverDeployments = mcserver.getDeployments();
         ManagedDeployment firstDeployment = null;
         for(KernelDeployment kd : serverDeployments.values())
         {
            ManagedObject kdMO = mof.initManagedObject(kd, null);
            Map<String, ManagedObject> kdMOs = Collections.singletonMap(kd.getName(), kdMO);
            BootstrapDeployment deploymentUnit = new BootstrapDeployment(kd);
            KernelDeploymentVisitor visitor = new KernelDeploymentVisitor();
            try
            {
               visitor.deploy(deploymentUnit, kd);
            }
            catch(DeploymentException e)
            {
               log.debug("Failed to build ManagedDeployment for: "+kd, e);
               continue;
            }
            ManagedDeployment md = mgtDeploymentCreator.build(deploymentUnit, kdMOs, null);
            if(firstDeployment == null)
               firstDeployment = md;
            for(DeploymentUnit compUnit : deploymentUnit.getComponents())
            {
               BeanMetaData bmd = compUnit.getAttachment(BeanMetaData.class);
               ManagedObject bmdMO = mof.initManagedObject(bmd, compUnit.getMetaData());
               if(bmdMO == null)
                  continue;
               // Reset the name to the bean name
               if(bmdMO instanceof MutableManagedObject)
               {
                  MutableManagedObject mmo = (MutableManagedObject) bmdMO;
                  mmo.setName(bmd.getName());
                  mmo.setParent(kdMO);
                  // Add an alias property
                  Set<Object> bmdAliases = bmd.getAliases();
                  if(bmdAliases != null && bmdAliases.size() > 0)
                  {
                     Map<String, ManagedProperty> oldProps = mmo.getProperties();
                     Map<String, ManagedProperty> newProps = new HashMap<String, ManagedProperty>(oldProps);
                     ArrayMetaType aliasType = new ArrayMetaType(SimpleMetaType.STRING, false);
                     DefaultFieldsImpl fields = getFields("alias", aliasType);
                     fields.setDescription("Aliases of the bean");
                     String[] aliases = new String[bmdAliases.size()];
                     Iterator<?> i = bmdAliases.iterator();
                     for(int n = 0; i.hasNext(); n++)
                     {
                        aliases[n] = i.next().toString();
                     }
                     ArrayValueSupport value = new ArrayValueSupport(aliasType, aliases);
                     fields.setValue(value);
                     ManagedPropertyImpl aliasesMP = new ManagedPropertyImpl(bmdMO, fields);
                     newProps.put("alias", aliasesMP);
                     // Add a state property
                     DefaultFieldsImpl stateFields = getFields("state", ControllerStateMetaType.TYPE);
                     stateFields.setViewUse(new ViewUse[]{ViewUse.STATISTIC});
                     EnumValue stateValue = getState(bmd.getName(), kernel);
                     stateFields.setValue(stateValue);
                     stateFields.setDescription("The bean controller state");
                     ManagedPropertyImpl stateMP = new ManagedPropertyImpl(mmo, stateFields);
                     newProps.put("state", stateMP);
                     // Update the properties
                     mmo.setProperties(newProps);
                  }
               }
               log.debug("Created ManagedObject: "+bmdMO+" for bean: "+bmd.getName());

               ComponentType type = KnownComponentTypes.MCBean.Any.getType();
               Map<String, Annotation> moAnns = bmdMO.getAnnotations();
               ManagementComponent mc = (ManagementComponent) moAnns.get(ManagementComponent.class.getName());
               if(mc != null)
               {
                  type = new ComponentType(mc.type(), mc.subtype());
               }
               ManagedComponentImpl comp = new ManagedComponentImpl(type, md, bmdMO);
               md.addComponent(bmdMO.getName(), comp);
               log.debug("Created ManagedComponent of type: "+type+" for bean: "+bmd.getName());
            }
            if(md != null)
               bootstrapMDs.put(kd.getName(), md);
         }

         // Add other Server managed objects
         if(firstDeployment != null)
         {
            ComponentType type = KnownComponentTypes.MCBean.Any.getType();
            // MCServer
            ManagedObject serverMO = mof.initManagedObject(mcserver, null);
            ManagedComponentImpl serverComp = new ManagedComponentImpl(type, firstDeployment, serverMO);
            firstDeployment.addComponent("MCServer", serverComp);
            // ServerConfig
            ServerConfig config = mcserver.getConfig();
            ManagedObject mo = mof.initManagedObject(config, null);
            ManagedComponentImpl configComp = new ManagedComponentImpl(type, firstDeployment, mo);
            firstDeployment.addComponent("ServerConfig", configComp);
            log.debug("Created ManagedComponent of type: "+type+" for ServerConfig");
         }
      }
   }

   /**
    * Create a DefaultFieldsImpl for the given property name and type
    * @param name - the property name
    * @param type - the property type
    * @return return the fields implementation
    */
   protected DefaultFieldsImpl getFields(String name, MetaType type)
   {
      DefaultFieldsImpl fields = new DefaultFieldsImpl();
      fields.setMetaType(type);
      fields.setName(name);
      fields.setMandatory(false);

      return fields;
   }
   /**
    * Get the state of a bean
    * 
    * @param name the bean name
    * @return state enum value
    */
   protected EnumValue getState(Object name, Kernel kernel)
   {
      KernelController controller = kernel.getController();
      ControllerContext context = controller.getContext(name, null);
      if (context == null)
         throw new IllegalStateException("Context not installed: " + name);

      ControllerState state = context.getState();
      return new EnumValueSupport(ControllerStateMetaType.TYPE, state.getStateString());
   }
}
