/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.system.server.profileservice.hotdeploy;

import java.util.Collection;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

import org.jboss.deployers.client.spi.main.MainDeployer;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.deployers.structure.spi.main.MainDeployerStructure;
import org.jboss.deployers.vfs.spi.client.VFSDeployment;
import org.jboss.kernel.spi.dependency.KernelController;
import org.jboss.logging.Logger;
import org.jboss.managed.api.ManagedDeployment.DeploymentPhase;
import org.jboss.profileservice.spi.ModificationInfo;
import org.jboss.profileservice.spi.Profile;
import org.jboss.profileservice.spi.ProfileService;

/**
 * A DeploymentScanner built on the ProfileService and MainDeployer. This
 * is really just a simple ExecutorService Runnable that knows nothing
 * about how to detect changed deployers. The ProfileService determines
 * this.
 * 
 * @see MainDeployer
 * @see ProfileService
 * 
 * @author <a href="mailto:dimitris@jboss.org">Dimitris Andreadis</a>
 * @author Scott.Stark@jboss.org
 * @author adrian@jboss.org
 * @version $Revision: 80843 $
 */
public class HDScanner
   implements Runnable
{
   private static final Logger log = Logger.getLogger(HDScanner.class);
   // Private Data --------------------------------------------------
   /** The MainDeployer used to deploy modifications */
   private MainDeployer mainDeployer;
   /** The controller */
   private KernelController controller;

   /** The ProfileService used to determine modified deployments */
   private ProfileService profileService;

   /** The ExecutorService/ThreadPool for performing scans */
   private ScheduledExecutorService scanExecutor;
   private ScheduledFuture activeScan;
   /** Thread name used when the ScheduledExecutorService is created internally */
   private String scanThreadName = "HDScanner";

   /** Period in ms between deployment scans */
   private long scanPeriod = 5000;
   /** The number of scans that have been done */
   private int scanCount;

   // Constructor ---------------------------------------------------
   
   public HDScanner()
   {
      // empty
   }
   
   // Attributes ----------------------------------------------------

   public void setMainDeployer(MainDeployer deployer)
   {
      this.mainDeployer = deployer;
   }

   public KernelController getController()
   {
      return controller;
   }

   public void setController(KernelController controller)
   {
      this.controller = controller;
   }

   public ProfileService getProfileService()
   {
      return profileService;
   }
   public void setProfileService(ProfileService profileService)
   {
      this.profileService = profileService;
   }

   /**
    * @return Returns the scanExecutor.
    */
   public ScheduledExecutorService getScanExecutor()
   {
      return this.scanExecutor;
   }

   /**
    * @param scanExecutor The scanExecutor to set.
    */
   public void setScanExecutor(ScheduledExecutorService scanExecutor)
   {
      this.scanExecutor = scanExecutor;
   }

   public String getScanThreadName()
   {
      return scanThreadName;
   }

   public void setScanThreadName(String scanThreadName)
   {
      this.scanThreadName = scanThreadName;
   }

   /* (non-Javadoc)
    * @see org.jboss.deployment.scanner.VFSDeploymentScanner#getScanPeriod()
    */
   public long getScanPeriod()
   {
      return scanPeriod;
   }
   /* (non-Javadoc)
    * @see org.jboss.deployment.scanner.VFSDeploymentScanner#setScanPeriod(long)
    */
   public void setScanPeriod(long period)
   {
      this.scanPeriod = period;
   }

   /** 
    * Are deployment scans enabled.
    * 
    * @return whether scan is enabled
    */
   public boolean isScanEnabled()
   {
      return activeScan != null;
   }

   public synchronized int getScanCount()
   {
      return scanCount;
   }
   public synchronized void resetScanCount()
   {
      this.scanCount = 0;
   }

   /**
    * Enable/disable deployment scans.
    * @param scanEnabled true to enable scans, false to disable.
    */
   public synchronized void setScanEnabled(boolean scanEnabled)
   {
      if( scanEnabled == true && activeScan == null )
      {
         activeScan = this.scanExecutor.scheduleWithFixedDelay(this, 0,
               scanPeriod, TimeUnit.MILLISECONDS);
      }
      else if( scanEnabled == false && activeScan != null )
      {
         activeScan.cancel(true);
         activeScan = null;
      }
   }


   // Operations ----------------------------------------------------
   
   public void start() throws Exception
   {
      // Default to a single thread executor
      if( scanExecutor == null )
      {
         scanExecutor = Executors.newSingleThreadScheduledExecutor(
            new ThreadFactory()
            {
               public Thread newThread(Runnable r)
               {
                  return new Thread(r, HDScanner.this.getScanThreadName());
               }
            }
        );
      }
      activeScan = scanExecutor.scheduleWithFixedDelay(this, 0,
            scanPeriod, TimeUnit.MILLISECONDS);
   }

   /**
    * Executes scan 
    *
    */
   public void run()
   {
      try
      {
         scan();
      }
      catch(Throwable e)
      {
         log.warn("Scan failed", e);
      }
      finally
      {
         incScanCount();
      }
   }

   public void stop()
   {
      if( activeScan != null )
      {
         activeScan.cancel(true);
         activeScan = null;
      }
   }

   public synchronized void scan() throws Exception
   {
      boolean trace = log.isTraceEnabled();

      // Query the ProfileService for deployments
      if( trace )
         log.trace("Begin deployment scan");

      
      // Get the modified deployments
      Profile activeProfile = profileService.getActiveProfile();
      if( activeProfile == null )
      {
         if( trace )
            log.trace("End deployment scan, no activeProfile");
         return;
      }

      Collection<ModificationInfo> modified = activeProfile.getModifiedDeployments();
      for(ModificationInfo info : modified)
      {
         VFSDeployment ctx = info.getDeployment();
         switch( info.getStatus() )
         {
            case ADDED:
               mainDeployer.addDeployment(ctx);
               break;
            case MODIFIED:
               mainDeployer.addDeployment(ctx);
               break;
            case REMOVED:
               mainDeployer.removeDeployment(ctx.getName());
               break;
         }
      }

      // Process the changes
      try
      {
         if( modified.size() > 0 )
         {
            // Current workaround for JBAS-4206
            ClassLoader oldTCL = Thread.currentThread().getContextClassLoader();
            ClassLoader tcl = getTCL(activeProfile);
            try
            {
               if (tcl != null)
                  Thread.currentThread().setContextClassLoader(tcl);
               mainDeployer.process();
               // Can be nulled by a shutdown
               if(mainDeployer != null)
                  mainDeployer.checkComplete();
            }
            finally
            {
               Thread.currentThread().setContextClassLoader(oldTCL);
            }
         }
      }
      catch (Exception e)
      {
         log.warn("Failed to process changes", e);
         // TODO: somehow need to ignore bad deployments to avoid repeated errors
         return;
      }

      if( trace )
         log.trace("End deployment scan");
   }

   /**
    * Inc the scanCount and to a notifyAll.
    *
    */
   protected synchronized void incScanCount()
   {
      scanCount ++;
      notifyAll();
   }

   // Private -------------------------------------------------------

   /**
    * Current workaround for JBAS-4206
    */
   private ClassLoader getTCL(Profile activeProfile)
      throws Exception
   {
      MainDeployerStructure structure = (MainDeployerStructure) mainDeployer;
      Collection<VFSDeployment> ctxs = activeProfile.getDeployments(DeploymentPhase.BOOTSTRAP);
      if (ctxs != null && ctxs.isEmpty() == false)
      {
         for (VFSDeployment deployment : ctxs)
         {
            DeploymentUnit unit = structure.getDeploymentUnit(deployment.getName());
            if (unit != null)
            {
               try
               {
                  ClassLoader cl = unit.getClassLoader();
                  if (cl != null)
                     return cl;
               }
               catch (Exception ignored)
               {
               }
            }
            
         }
      }
      log.warn("No bootstrap deployments? profile=" + activeProfile);
      return null;
   }
}
