/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.system.server.profileservice.persistence;

import java.util.Set;

import org.jboss.beans.info.spi.PropertyInfo;
import org.jboss.logging.Logger;
import org.jboss.managed.api.Fields;
import org.jboss.managed.api.ManagedObject;
import org.jboss.managed.api.ManagedProperty;
import org.jboss.managed.api.annotation.ViewUse;
import org.jboss.metatype.api.types.ArrayMetaType;
import org.jboss.metatype.api.types.CompositeMetaType;
import org.jboss.metatype.api.types.MetaType;
import org.jboss.metatype.api.values.ArrayValue;
import org.jboss.metatype.api.values.CollectionValue;
import org.jboss.metatype.api.values.CompositeValue;
import org.jboss.metatype.api.values.EnumValue;
import org.jboss.metatype.api.values.GenericValue;
import org.jboss.metatype.api.values.MetaValue;
import org.jboss.metatype.api.values.MetaValueFactory;
import org.jboss.metatype.api.values.SimpleValue;
import org.jboss.system.server.profileservice.persistence.xml.PersistedArrayValue;
import org.jboss.system.server.profileservice.persistence.xml.PersistedCollectionValue;
import org.jboss.system.server.profileservice.persistence.xml.PersistedCompositeValue;
import org.jboss.system.server.profileservice.persistence.xml.PersistedEnumValue;
import org.jboss.system.server.profileservice.persistence.xml.PersistedGenericValue;
import org.jboss.system.server.profileservice.persistence.xml.PersistedManagedObject;
import org.jboss.system.server.profileservice.persistence.xml.PersistedProperty;
import org.jboss.system.server.profileservice.persistence.xml.PersistedSimpleValue;
import org.jboss.system.server.profileservice.persistence.xml.PersistedTableValue;
import org.jboss.system.server.profileservice.persistence.xml.PersistedValue;

/**
 * Create a xml representation of a Managed Object.
 * 
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision$
 */
public class ManagedObjectPeristenceHandler
{
   /** The meta value factory. */
   MetaValueFactory metaValueFactory = MetaValueFactory.getInstance();
   
   /** The logger. */
   private static final Logger log = Logger.getLogger(ManagedObjectPeristenceHandler.class);
   
   /**
    * Process a managed object.
    * 
    * @param moElement the xml meta data.
    * @param mo the managed object
    * @return isModified
    */
   public boolean processManagedObject(PersistedManagedObject moElement, ManagedObject mo)
   {
      if(moElement == null)
         throw new IllegalArgumentException("Null rootElement.");
      if(mo == null)
         throw new IllegalArgumentException("null managedObject");
      
      moElement.setName(mo.getName());
      moElement.setClassName(mo.getAttachmentName());
      
      // Store the original name, if it does not exist.
      if(moElement.getOriginalName() == null)
         moElement.setOriginalName(mo.getName());
      
      boolean changed = false;
      Set<String> propertyNames = mo.getPropertyNames();
      for(String propertyName : propertyNames)
      {
         ManagedProperty property = mo.getProperty(propertyName);
         if(property == null)
            throw new IllegalStateException("unable to find propery: "+ property);
         
         if(! property.hasViewUse(ViewUse.CONFIGURATION))
            continue;
         
         // getProperty
         PersistedProperty propertyElement = moElement.get(propertyName);
         if(propertyElement == null)
            propertyElement = new PersistedProperty(property.getName());
         boolean propertyChanged = processManagedProperty(propertyElement, propertyName, property);
         
         if(propertyChanged)
         {
            changed = true;
            moElement.put(propertyName, propertyElement);
         }
      }
      return changed;
   }
   
   /**
    * Process a ManagedProperty.
    * 
    * @param persistedProperty the xml property meta data.
    * @param name the property name.
    * @param property the managed property.
    * @return isModified.
    */
   protected boolean processManagedProperty(PersistedProperty persistedProperty, String name, ManagedProperty property)
   {
      boolean trace = log.isTraceEnabled();
      PropertyInfo propertyInfo = property.getField(Fields.PROPERTY_INFO, PropertyInfo.class);
      // Skip not readable properties
      if(propertyInfo != null && propertyInfo.isReadable() == false)
      {
         if(trace)
            log.trace("property "+ name + "is not readable");
         return false;
      }
      
      // TODO Check if property was modified
      // MetaType metaType = property.getMetaType();
      // if( ! metaType.isCollection() && ! metaType.isArray() )
      // {
      //    if(property.isModified() == false)
      //       return false;
      // }
      
      // Get MetaValue
      MetaValue value = property.getValue();
      // Check if there is a previous persisted property value
      PersistedValue persistedValue = persistedProperty.getValue();
      // Process
      boolean changed = false;
      if(value != null)
      {
         // Create a new value, if needed
         if(persistedValue == null)
            persistedValue = createPersistedValue(value.getMetaType());
         // Process
         changed = processMetaValue(persistedValue, value);
      }
      
      persistedProperty.setValue(persistedValue);
      
      return changed;
   }
   
   /**
    * Process a MetaValue.
    * 
    * @param valueElement the xml metadata.
    * @param value the meta value.
    * @return isModified.
    */
   protected boolean processMetaValue(PersistedValue valueElement, MetaValue value)
   {
      // FIXME a MetaValue should never be null ?
      if(value == null)
         return false;
      
      // TODO we need to know if the MetaValue was modified too.
      boolean changed = false;
      MetaType metaType = value.getMetaType();
      
      if(metaType.isSimple())
      {
         changed = processSimpleValue((PersistedSimpleValue) valueElement, (SimpleValue) value);
      }
      else if(metaType.isEnum())
      {
         changed = processEnumValue((PersistedEnumValue) valueElement, (EnumValue) value);
      }
      else if(metaType.isCollection())
      {
         changed = processCollectionValue((PersistedCollectionValue) valueElement, (CollectionValue) value);
      }
      else if(metaType.isGeneric())
      {
         changed = processGenericValue((PersistedGenericValue) valueElement, (GenericValue) value);
      }
      else if(metaType.isComposite())
      {
         changed = processCompositeValue((PersistedCompositeValue) valueElement, (CompositeValue) value);
      }
      else if(metaType.isArray())
      {
         changed = processArrayValue((PersistedArrayValue) valueElement, (ArrayValue) value);
      }
      else if(metaType.isTable())
      {
         throw new UnsupportedOperationException("Persisting a table value.");
      }
      else
      {
         throw new IllegalStateException("unknown metaType");
      }
      return changed;
   }
   
   /**
    * Process a simple value.
    * 
    * @param valueElement the xml meta data.
    * @param value the simple value.
    * @return isModified.
    */
   protected boolean processSimpleValue(PersistedSimpleValue valueElement, SimpleValue value)
   {
      valueElement.setValue(convertSimple2String(value));
      return true;
   }
   
   /**
    * Process a enumValue.
    * 
    * @param enumElement the xml meta data.
    * @param value the enum value.
    */
   protected boolean processEnumValue(PersistedEnumValue enumElement, EnumValue value)
   {
      enumElement.setValue(value.getValue());
      return true;
   }
   
   /**
    * Process a collection value.
    * 
    * @param valueElement the xml meta data.
    * @param value the collection value.
    * @return isModified.
    */
   protected boolean processCollectionValue(PersistedCollectionValue valueElement, CollectionValue value)
   {
      boolean changed = false;
      for(MetaValue child : value.getElements())
      {
         PersistedValue persistedValue = createPersistedValue(child.getMetaType());
         
         if( processMetaValue(persistedValue, child) == true)
            changed = true;
         
         valueElement.addValue(persistedValue);
      }
      return changed;
   }
   
   /**
    * Process a generic value.
    * 
    * @param genericElement the xml metadata.
    * @param value the generic value.
    * @return isModified.
    */
   protected boolean processGenericValue(PersistedGenericValue genericElement, GenericValue value)
   {
      Object o = value.getValue();
      if(o == null)
         return false;
      
      boolean changed = false;
      if(o instanceof ManagedObject)
      {
         PersistedManagedObject mo = new PersistedManagedObject();
         if(processManagedObject(mo, (ManagedObject) o) == true)
            changed = true;
         genericElement.setManagedObject(mo);
         return changed;
      }
      else
      {
         throw new IllegalStateException("The value of GenericValue must be a ManagedObject: " + value);
      }
   }
   
   /**
    * Process the array value.
    * 
    * @param array the xml meta data.
    * @param value the array value.
    * @return isModified.
    */
   protected boolean processArrayValue(PersistedArrayValue array, ArrayValue value)
   {
      ArrayMetaType metaType = value.getMetaType();
      
      if(! metaType.getElementType().isSimple())
         throw new UnsupportedOperationException("Persisting a non primitive array.");
         
      boolean changed = false;
      for(int i = 0; i < value.getLength(); i++)
      {
         MetaValue metaValue = (MetaValue) value.getValue(i);
         PersistedValue persistedValue = createPersistedValue(metaType.getElementType());
         
         if(processMetaValue(persistedValue, metaValue) == true)
            changed = true;
         
         array.addValue(persistedValue);
      }
      return changed;
   }
   
   /**
    * Process a composite value.
    * 
    * @param composite the xml meta data.
    * @param value the composite value.
    * @return isModified.
    */
   protected boolean processCompositeValue(PersistedCompositeValue composite, CompositeValue value)
   {
      CompositeMetaType metaType = value.getMetaType();
      
      boolean changed = false;
      for(String item : metaType.itemSet())
      {
         MetaValue itemValue = value.get(item);
         
         // FIXME a value should never be null ?
         MetaType itemType = null;
         if(itemValue != null)
         {
            itemType = itemValue.getMetaType();
         }
         else
         {
            itemType = metaType.getType(item);
         }
         
         PersistedValue persistedValue = createPersistedValue(itemType);
         persistedValue.setName(item);
         
         if(processMetaValue(persistedValue, itemValue) == true)
            changed = true;
         
         composite.put(item, persistedValue);
      }
      return changed;
   }
  
   /**
    * Create an empty xml value representation. 
    * 
    * @param metaType the meta type.
    * @return a empty xml meta data, based on the meta type.
    */
   protected static PersistedValue createPersistedValue(MetaType metaType)
   {
      if(metaType.isSimple())
      {
         return new PersistedSimpleValue(); 
      }
      else if(metaType.isEnum())
      {
         return new PersistedEnumValue();
      }
      else if(metaType.isCollection())
      {
         return new PersistedCollectionValue();
      }
      else if(metaType.isGeneric())
      {
         return new PersistedGenericValue();
      }
      else if(metaType.isComposite())
      {
         return new PersistedCompositeValue();
      }
      else if(metaType.isTable())
      {
         return new PersistedTableValue();
      }
      else if(metaType.isArray())
      {
         return new PersistedArrayValue();
      }
      else
      {
         throw new IllegalStateException("unknown metaType");
      }
   }
   
   /**
    * Convert a simple meta value to a String.
    * 
    * @param value the simple meta value.
    * @return the string.
    */
   protected String convertSimple2String(SimpleValue value)
   {       
      // TODO a metaValue should never be null?
      if(value == null)
         return null;
      
      Object unwrappedValue = metaValueFactory.unwrap(value);
      
      if(unwrappedValue == null)
         return null; 
      
      return ("" + unwrappedValue);
   }
   
}