/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.system.server.profileservice.persistence;

import java.io.Serializable;
import java.lang.reflect.Proxy;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.jboss.beans.info.spi.BeanInfo;
import org.jboss.beans.info.spi.PropertyInfo;
import org.jboss.logging.Logger;
import org.jboss.managed.api.Fields;
import org.jboss.managed.api.ManagedObject;
import org.jboss.managed.api.ManagedProperty;
import org.jboss.managed.api.annotation.ViewUse;
import org.jboss.managed.api.factory.ManagedObjectFactory;
import org.jboss.managed.spi.factory.InstanceClassFactory;
import org.jboss.metatype.api.types.ArrayMetaType;
import org.jboss.metatype.api.types.CollectionMetaType;
import org.jboss.metatype.api.types.CompositeMetaType;
import org.jboss.metatype.api.types.EnumMetaType;
import org.jboss.metatype.api.types.MetaType;
import org.jboss.metatype.api.types.Name;
import org.jboss.metatype.api.types.SimpleMetaType;
import org.jboss.metatype.api.types.TableMetaType;
import org.jboss.metatype.api.values.ArrayValue;
import org.jboss.metatype.api.values.ArrayValueSupport;
import org.jboss.metatype.api.values.CollectionValue;
import org.jboss.metatype.api.values.CollectionValueSupport;
import org.jboss.metatype.api.values.CompositeValue;
import org.jboss.metatype.api.values.CompositeValueSupport;
import org.jboss.metatype.api.values.EnumValue;
import org.jboss.metatype.api.values.EnumValueSupport;
import org.jboss.metatype.api.values.GenericValue;
import org.jboss.metatype.api.values.MetaValue;
import org.jboss.metatype.api.values.MetaValueFactory;
import org.jboss.metatype.api.values.SimpleValue;
import org.jboss.metatype.api.values.SimpleValueSupport;
import org.jboss.metatype.api.values.TableValueSupport;
import org.jboss.metatype.plugins.types.StringName;
import org.jboss.reflect.plugins.ValueConvertor;
import org.jboss.system.server.profileservice.persistence.xml.PersistedArrayValue;
import org.jboss.system.server.profileservice.persistence.xml.PersistedCollectionValue;
import org.jboss.system.server.profileservice.persistence.xml.PersistedCompositeValue;
import org.jboss.system.server.profileservice.persistence.xml.PersistedEnumValue;
import org.jboss.system.server.profileservice.persistence.xml.PersistedGenericValue;
import org.jboss.system.server.profileservice.persistence.xml.PersistedManagedObject;
import org.jboss.system.server.profileservice.persistence.xml.PersistedProperty;
import org.jboss.system.server.profileservice.persistence.xml.PersistedSimpleValue;
import org.jboss.system.server.profileservice.persistence.xml.PersistedValue;


/**
 * Handler for merging the persisted information with a ManagedObject.
 * 
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision$
 */
public class ManagedObjectUpdateHandler
{
   
   /** The managed object factory. */
   ManagedObjectFactory managedObjectFactory = ManagedObjectFactory.getInstance();
   
   /** The meta value factory. */
   MetaValueFactory metaValueFactory = MetaValueFactory.getInstance();
   
   /** The simple types. */
   private static final Map<String, Class<? extends Serializable>> simpleTypes = new HashMap<String, Class<? extends Serializable>>();
   
   /** The logger. */
   private final static Logger log = Logger.getLogger(ManagedObjectUpdateHandler.class);
   
   static
   {
      // Fill simple types map.
      simpleTypes.put(BigDecimal.class.getName(), BigDecimal.class);
      simpleTypes.put(BigInteger.class.getName(), BigInteger.class);
      simpleTypes.put(Boolean.class.getName(), Boolean.class);
      simpleTypes.put(Byte.class.getName(), Byte.class);
      simpleTypes.put(Character.class.getName(), Character.class);
      simpleTypes.put(Date.class.getName(), Date.class);
      simpleTypes.put(Double.class.getName(), Double.class);
      simpleTypes.put(Float.class.getName(), Float.class);
      simpleTypes.put(Integer.class.getName(), Integer.class);
      simpleTypes.put(Long.class.getName(), Long.class);
      simpleTypes.put(Short.class.getName(), Short.class);
      simpleTypes.put(String.class.getName(), String.class);
      simpleTypes.put(Name.class.getName(), Name.class);
   }
   
   /**
    * Process a ManagedObject.
    * 
    * @param moElement the persisted xml meta data.
    * @param mo the managed object
    * @return the attachment meta data.
    */
   public Object processManagedObject(PersistedManagedObject moElement, ManagedObject mo)
   {
      if(moElement == null)
         throw new IllegalArgumentException("Null rootElement.");
      if(mo == null)
         throw new IllegalArgumentException("null managedObject");

      Object attachment = mo.getAttachment();

      // Get the xml to see what we need to merge
      Set<String> propertyNames = moElement.keySet();
      for(String propertyName : propertyNames)
      {
         ManagedProperty property = mo.getProperty(propertyName);
         if(property == null)
            throw new IllegalStateException("unable to find propery: "+ property);
         
         // Skip statistic - although they should not be persisted anyway
         if(property.hasViewUse(ViewUse.STATISTIC))
            continue;
     
         // getProperty
         PersistedProperty propertyElement = moElement.get(propertyName);
         processManagedProperty(propertyElement, propertyName, property, attachment);
      }
      return attachment;
   }
   
   /**
    * Process a ManagedProperty.
    * 
    * @param propertyElement the persisted xml meta data.
    * @param name the property name.
    * @param property the managed property.
    * @param attachment the managed object attachment.
    */
   protected void processManagedProperty(PersistedProperty propertyElement, String name, ManagedProperty property, Object attachment)
   {
      boolean trace = log.isTraceEnabled();
      PropertyInfo propertyInfo = property.getField(Fields.PROPERTY_INFO, PropertyInfo.class);
      
      // Skip not readable properties
      if(propertyInfo != null && propertyInfo.isReadable() == false)
      {
         if(trace)
            log.trace("property "+ name + "is not readable");
         return;
      }
     
      // Get MetaValue
      MetaValue value = property.getValue();
      if(value == null)
      {
         value = createEmptyMetaValue(property.getMetaType());
      }
      
      // Get the xml value
      PersistedValue persistedValue = propertyElement.getValue();
      // Skip null values
      if(value != null && persistedValue != null)
      {
         // Process the meta value.
         MetaValue merged = processMetaValue(persistedValue, value);
         // If merged != null
         if(merged != null)
         {
            // TODO Can this be null anyway ?
            if(propertyInfo == null)
               return;
            // throw new IllegalStateException("null propertyInfo"); 
            
            // Skip not writable properties
            if(propertyInfo.isWritable() == false)
            {
               if(trace)
                  log.trace("property "+ name + "is not writable");
               return;
            }
            
            // FIXME Ignore some metaTypes for now
            MetaType metaType = merged.getMetaType();
            if( ! metaType.isCollection() 
                  && ! metaType.isArray() 
                  && ! metaType.isTable() )
            {
               
               try
               {
                  // set the value Field
                  property.setField(Fields.VALUE, merged);
                  
                  // FIXME skip CompositeValueInvocationHandler
                  if( metaType.isComposite() )
                  {
                     // unwrap
                     Object unwrapped = metaValueFactory.unwrap(merged, propertyInfo.getType());
                     if(Proxy.isProxyClass(unwrapped.getClass()))
                        return;                     
                  }
                  
                  // Set value
                  InstanceClassFactory icf = managedObjectFactory.getInstanceClassFactory(attachment.getClass(), null);
                  BeanInfo beanInfo = propertyInfo.getBeanInfo();
                  icf.setValue(beanInfo, property, attachment, merged);

               }
               catch(Throwable t)
               {
                  log.debug("failed to set value to property: "+ propertyInfo, t);
               }
            }
         }
      }
      else 
      {
         return;
      }
   }
   
   /**
    * Process a MetaValue.
    * 
    * @param valueElement the persisted xml meta data.
    * @param value the meta value.
    * @return a meta value.
    */
   protected MetaValue processMetaValue(PersistedValue valueElement, MetaValue value)
   {
      MetaType metaType = value.getMetaType();
      
      MetaValue metaValue = null;
      if(metaType.isSimple())
      {
         metaValue = processSimpleValue(
               (PersistedSimpleValue) valueElement,
               (SimpleValue) value);
      }
      else if(metaType.isEnum())
      {
         metaValue = processEnumValue(
               (PersistedEnumValue) valueElement,
               (EnumValue) value);
      }
      else if(metaType.isCollection())
      {
         metaValue = processCollectionValue(
               (PersistedCollectionValue) valueElement,
               (CollectionValue) value);
      }
      else if(metaType.isGeneric())
      {
         metaValue = processGenericValue(
               (PersistedGenericValue) valueElement,
               (GenericValue) value);
      }
      else if(metaType.isComposite())
      {
         metaValue = processCompositeValue(
               (PersistedCompositeValue) valueElement,
               (CompositeValue) value);
      }
      else if(metaType.isTable())
      {
         // FIXME process Table 
      }
      else if(metaType.isArray())
      {
         metaValue = processArrayValue(
               (PersistedArrayValue) valueElement,
               (ArrayValue) value);
      }
      else
      {
         throw new IllegalStateException("unknown metaType");
      }
      return metaValue;
   }
   
   /**
    * Process an Enum value.
    * 
    * @param enumElement the persisted xml meta data.
    * @param value the enum value.
    * @return a enum value.
    */
   protected EnumValue processEnumValue(PersistedEnumValue enumElement, EnumValue value)
   {
      return new EnumValueSupport(value.getMetaType(), enumElement.getValue());
   }
   
   /**
    * Process a collection.
    * TODO - support merging of collections.
    * 
    * @param collection the persisted xml meta data.
    * @param value the collection value.
    * @return a collection value.
    */
   protected CollectionValue processCollectionValue(PersistedCollectionValue collection, CollectionValue value)
   {
      if(collection.size() == 0)
         return value;

      // FIXME merge collections
      if(collection.size() != value.getSize())
      {
         log.warn("unable to merge collection: " + value);
         return value;
      }
      
      ArrayList<MetaValue> elementList = new ArrayList<MetaValue>();
      Iterator<PersistedValue> i = collection.getValues().iterator();
      for(MetaValue item : value)
      {
         MetaValue newValue = processMetaValue(i.next(), item);
         if(newValue != null)
         {
            elementList.add(newValue);
         }
      }
      return new CollectionValueSupport(value.getMetaType(), elementList.toArray(new MetaValue[elementList.size()]));
   }
   
   /**
    * Process a GenericValue.
    * 
    * @param genericElement the persisted xml meta data.
    * @param value the generic value.
    * @return a generic value.
    */
   protected MetaValue processGenericValue(PersistedGenericValue genericElement, GenericValue value)
   {
      PersistedManagedObject po = genericElement.getManagedObject();
      if(po == null || value.getValue() == null)
         return value;

      if(value.getValue() instanceof ManagedObject)
      {
         ManagedObject mo = (ManagedObject) value.getValue();
         processManagedObject(po, mo);
         
         return value;
      }
      else
      {
         throw new IllegalStateException("The value of GenericValue must be a ManagedObject: " + value);
      }
   }
   
   /**
    * Process composite value.
    * 
    * @param composite the persisted xml meta data.
    * @param value the composite value.
    * @return a composite value.
    */
   protected CompositeValue processCompositeValue(PersistedCompositeValue composite, CompositeValue value)
   {
      CompositeMetaType metaType = value.getMetaType();
      Map<String, MetaValue> values = new HashMap<String, MetaValue>();
      
      for(String key : composite.keySet())
      {
         MetaValue metaValue = value.get(key);
         if(metaValue == null)
         {
            metaValue = createEmptyMetaValue(metaType.getType(key));
         }
         PersistedValue persistedValue = composite.get(key);

         metaValue = processMetaValue(persistedValue, metaValue);
         
         values.put(key, metaValue);
      }
      return new CompositeValueSupport(metaType, values);
   }
   
   /**
    * process an arrayValue.
    * FIXME - support merging of arrays.
    * 
    * @param array the persisted xml meta data.
    * @param value the array value.
    * @return a array value.
    */
   protected ArrayValue processArrayValue(PersistedArrayValue array, ArrayValue value)
   {
      // FIXME 
      if(array.size() != value.getLength())
      {
         log.debug("cannot merge array: " + value);
         return null; 
      }
      
      ArrayList<MetaValue> arrayList = new ArrayList<MetaValue>();
      for(int i = 0; i < value.getLength(); i++)
      {
         PersistedValue persisted = array.getValue(i);
         MetaValue restored = processMetaValue(persisted, (MetaValue) value.getValue(i));
         if(restored != null)
            arrayList.add(restored);
      }

      return new ArrayValueSupport(value.getMetaType(), arrayList.toArray(new MetaValue[arrayList.size()]));
   }
   
   /**
    * Process simple value.
    * 
    * @param valueElement the persisted xml meta data.
    * @param value the simple value.
    * @return a simple value.
    */
   protected SimpleValue processSimpleValue(PersistedSimpleValue valueElement, SimpleValue value)
   {
      SimpleMetaType metaType = value.getMetaType();
      String elementValue = valueElement.getValue();
      
      Serializable converted = null;
      if(elementValue != null)
      {
         if(metaType.equals(SimpleMetaType.STRING))
         {
            converted = (String) elementValue;
         }
         else if (metaType.equals(SimpleMetaType.NAMEDOBJECT))
         {
            converted = new StringName(elementValue);
         }
         else if (metaType.equals(SimpleMetaType.VOID))
         {
            // 
         }
         else
         {
            converted = convert2Type(metaType.getTypeName(), elementValue);
         }
      }
      return SimpleValueSupport.wrap(converted);
   }
 
   /**
    * Create a empty meta value, based on a given MetaType.
    * 
    * @param metaType the meta type.
    * @return a meta value.
    */
   protected MetaValue createEmptyMetaValue(MetaType metaType)
   {
      MetaValue metaValue = null;
      if(metaType.isSimple())
      {
         metaValue = new SimpleValueSupport((SimpleMetaType) metaType, null);
      }
      else if(metaType.isEnum())
      {
         metaValue = new EnumValueSupport((EnumMetaType) metaType, (String) null);
      }
      else if(metaType.isCollection())
      {
         metaValue = new CollectionValueSupport((CollectionMetaType) metaType);
      }
      else if(metaType.isGeneric())
      {
         // TODO
      }
      else if(metaType.isComposite())
      {
         metaValue = new CompositeValueSupport((CompositeMetaType) metaType);
      }
      else if(metaType.isTable())
      {
         metaValue = new TableValueSupport((TableMetaType) metaType);
      }
      else if(metaType.isArray())
      {
         metaValue = new ArrayValueSupport((ArrayMetaType) metaType);
      }      
      return metaValue;
   }
   
   /**
    * Convert simple types.
    * 
    * @param clazz a primitive serializable class.
    * @param value the String
    * @return the converted object, null in case of any failure.
    */
   public Serializable convert2Type(String className, String value)
   {
      if(value == null)
         return null;
      
      Class<?> clazz = simpleTypes.get(className);
      if(clazz == null)
         throw new IllegalStateException("Cannot find simple type entry for "+ value + " and class "+ className);
      
      try
      {
         return (Serializable) ValueConvertor.convertValue(clazz, value);
      }
      catch(Throwable t)
      {
         log.debug("could convert "+ value +" to " + clazz.getName());
         return null;
      }
   }
   
}
