/*
 * JBoss, Home of Professional Open Source.
 *
 * Copyright 2018 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.commons.logging.impl;

import java.io.InputStream;
import java.io.Serializable;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Properties;

import org.apache.commons.logging.Log;

/**
 * Simple implementation of Log that sends all enabled log messages,
 * for all defined loggers, to System.err.  The following system properties
 * are supported to configure the behavior of this logger:
 * <ul>
 * <li><code>org.apache.commons.logging.simplelog.defaultlog</code> -
 * Default org.apache.commons.logging detail level for all instances of SimpleLog.
 * Must be one of ("trace", "debug", "info", "warn", "error", or "fatal").
 * If not specified, defaults to "info". </li>
 * <li><code>org.apache.commons.logging.simplelog.log.xxxxx</code> -
 * Logging detail level for a SimpleLog instance named "xxxxx".
 * Must be one of ("trace", "debug", "info", "warn", "error", or "fatal").
 * If not specified, the default org.apache.commons.logging detail level is used.</li>
 * <li><code>org.apache.commons.logging.simplelog.showlogname</code> -
 * Set to <code>true</code> if you want the Log instance name to be
 * included in output messages. Defaults to <code>false</code>.</li>
 * <li><code>org.apache.commons.logging.simplelog.showShortLogname</code> -
 * Set to <code>true</code> if you want the last component of the name to be
 * included in output messages. Defaults to <code>true</code>.</li>
 * <li><code>org.apache.commons.logging.simplelog.showdatetime</code> -
 * Set to <code>true</code> if you want the current date and time
 * to be included in output messages. Default is <code>false</code>.</li>
 * <li><code>org.apache.commons.logging.simplelog.dateTimeFormat</code> -
 * The date and time format to be used in the output messages.
 * The pattern describing the date and time format is the same that is
 * used in <code>java.text.SimpleDateFormat</code>. If the format is not
 * specified or is invalid, the default format is used.
 * The default format is <code>yyyy/MM/dd HH:mm:ss:SSS zzz</code>.</li>
 * </ul>
 * <p>
 * In addition to looking for system properties with the names specified
 * above, this implementation also checks for a class loader resource named
 * <code>"simplelog.properties"</code>, and includes any matching definitions
 * from this resource (if it exists).
 *
 * @version $Id: SimpleLog.java 1435115 2013-01-18 12:40:19Z tn $
 */
public class SimpleLog implements Log, Serializable {

    /**
     * Serializable version identifier.
     */
    private static final long serialVersionUID = 136942970684951178L;

    // ------------------------------------------------------- Class Attributes

    /**
     * All system properties used by <code>SimpleLog</code> start with this
     */
    static protected final String systemPrefix = "org.apache.commons.logging.simplelog.";

    /**
     * Properties loaded from simplelog.properties
     */
    static protected final Properties simpleLogProps = new Properties();

    /**
     * The default format to use when formating dates
     */
    static protected final String DEFAULT_DATE_TIME_FORMAT = "yyyy/MM/dd HH:mm:ss:SSS zzz";

    /**
     * Include the instance name in the log message?
     */
    static volatile protected boolean showLogName = false;

    /**
     * Include the short name ( last component ) of the logger in the log
     * message. Defaults to true - otherwise we'll be lost in a flood of
     * messages without knowing who sends them.
     */
    static volatile protected boolean showShortName = true;

    /**
     * Include the current time in the log message
     */
    static volatile protected boolean showDateTime = false;

    /**
     * The date and time format to use in the log message
     */
    static volatile protected String dateTimeFormat = DEFAULT_DATE_TIME_FORMAT;

    /**
     * Used to format times.
     * <p>
     * Any code that accesses this object should first obtain a lock on it,
     * ie use synchronized(dateFormatter); this requirement was introduced
     * in 1.1.1 to fix an existing thread safety bug (SimpleDateFormat.format
     * is not thread-safe).
     */
    static protected DateFormat dateFormatter = null;

    // ---------------------------------------------------- Log Level Constants

    /**
     * "Trace" level org.apache.commons.logging.
     */
    public static final int LOG_LEVEL_TRACE = 1;
    /**
     * "Debug" level org.apache.commons.logging.
     */
    public static final int LOG_LEVEL_DEBUG = 2;
    /**
     * "Info" level org.apache.commons.logging.
     */
    public static final int LOG_LEVEL_INFO = 3;
    /**
     * "Warn" level org.apache.commons.logging.
     */
    public static final int LOG_LEVEL_WARN = 4;
    /**
     * "Error" level org.apache.commons.logging.
     */
    public static final int LOG_LEVEL_ERROR = 5;
    /**
     * "Fatal" level org.apache.commons.logging.
     */
    public static final int LOG_LEVEL_FATAL = 6;

    /**
     * Enable all org.apache.commons.logging levels
     */
    public static final int LOG_LEVEL_ALL = LOG_LEVEL_TRACE - 1;

    /**
     * Enable no org.apache.commons.logging levels
     */
    public static final int LOG_LEVEL_OFF = LOG_LEVEL_FATAL + 1;

    // ------------------------------------------------------------ Initializer

    private static String getStringProperty(String name) {
        String prop = null;
        try {
            prop = System.getProperty(name);
        } catch (SecurityException e) {
            // Ignore
        }
        return prop == null ? simpleLogProps.getProperty(name) : prop;
    }

    private static String getStringProperty(String name, String dephault) {
        String prop = getStringProperty(name);
        return prop == null ? dephault : prop;
    }

    private static boolean getBooleanProperty(String name, boolean dephault) {
        String prop = getStringProperty(name);
        return prop == null ? dephault : "true".equalsIgnoreCase(prop);
    }

    // Initialize class attributes.
    // Load properties file, if found.
    // Override with system properties.
    static {
        // Add props from the resource simplelog.properties
        InputStream in = getResourceAsStream("simplelog.properties");
        if (null != in) {
            try {
                simpleLogProps.load(in);
                in.close();
            } catch (java.io.IOException e) {
                // ignored
            }
        }

        showLogName = getBooleanProperty(systemPrefix + "showlogname", showLogName);
        showShortName = getBooleanProperty(systemPrefix + "showShortLogname", showShortName);
        showDateTime = getBooleanProperty(systemPrefix + "showdatetime", showDateTime);

        if (showDateTime) {
            dateTimeFormat = getStringProperty(systemPrefix + "dateTimeFormat",
                    dateTimeFormat);
            try {
                dateFormatter = new SimpleDateFormat(dateTimeFormat);
            } catch (IllegalArgumentException e) {
                // If the format pattern is invalid - use the default format
                dateTimeFormat = DEFAULT_DATE_TIME_FORMAT;
                dateFormatter = new SimpleDateFormat(dateTimeFormat);
            }
        }
    }

    // ------------------------------------------------------------- Attributes

    /**
     * The name of this simple log instance
     */
    protected volatile String logName = null;
    /**
     * The current log level
     */
    protected volatile int currentLogLevel;
    /**
     * The short name of this simple log instance
     */
    private volatile String shortLogName = null;

    // ------------------------------------------------------------ Constructor

    /**
     * Construct a simple log with given name.
     *
     * @param name log name
     */
    public SimpleLog(String name) {
        logName = name;

        // Set initial log level
        // Used to be: set default log level to ERROR
        // IMHO it should be lower, but at least info ( costin ).
        setLevel(LOG_LEVEL_INFO);

        // Set log level from properties
        String lvl = getStringProperty(systemPrefix + "log." + logName);
        int i = String.valueOf(name).lastIndexOf(".");
        while (null == lvl && i > -1) {
            name = name.substring(0, i);
            lvl = getStringProperty(systemPrefix + "log." + name);
            i = String.valueOf(name).lastIndexOf(".");
        }

        if (null == lvl) {
            lvl = getStringProperty(systemPrefix + "defaultlog");
        }

        if ("all".equalsIgnoreCase(lvl)) {
            setLevel(LOG_LEVEL_ALL);
        } else if ("trace".equalsIgnoreCase(lvl)) {
            setLevel(LOG_LEVEL_TRACE);
        } else if ("debug".equalsIgnoreCase(lvl)) {
            setLevel(LOG_LEVEL_DEBUG);
        } else if ("info".equalsIgnoreCase(lvl)) {
            setLevel(LOG_LEVEL_INFO);
        } else if ("warn".equalsIgnoreCase(lvl)) {
            setLevel(LOG_LEVEL_WARN);
        } else if ("error".equalsIgnoreCase(lvl)) {
            setLevel(LOG_LEVEL_ERROR);
        } else if ("fatal".equalsIgnoreCase(lvl)) {
            setLevel(LOG_LEVEL_FATAL);
        } else if ("off".equalsIgnoreCase(lvl)) {
            setLevel(LOG_LEVEL_OFF);
        }
    }

    // -------------------------------------------------------- Properties

    /**
     * Set org.apache.commons.logging level.
     *
     * @param currentLogLevel new org.apache.commons.logging level
     */
    public void setLevel(int currentLogLevel) {
        this.currentLogLevel = currentLogLevel;
    }

    /**
     * Get org.apache.commons.logging level.
     */
    public int getLevel() {
        return currentLogLevel;
    }

    // -------------------------------------------------------- Logging Methods

    /**
     * Do the actual org.apache.commons.logging.
     * <p>
     * This method assembles the message and then calls <code>write()</code>
     * to cause it to be written.
     *
     * @param type    One of the LOG_LEVEL_XXX constants defining the log level
     * @param message The message itself (typically a String)
     * @param t       The exception whose stack trace should be logged
     */
    protected void log(int type, Object message, Throwable t) {
        // Use a string buffer for better performance
        final StringBuffer buf = new StringBuffer();

        // Append date-time if so configured
        if (showDateTime) {
            final Date now = new Date();
            String dateText;
            synchronized (dateFormatter) {
                dateText = dateFormatter.format(now);
            }
            buf.append(dateText);
            buf.append(" ");
        }

        // Append a readable representation of the log level
        switch (type) {
            case LOG_LEVEL_TRACE:
                buf.append("[TRACE] ");
                break;
            case LOG_LEVEL_DEBUG:
                buf.append("[DEBUG] ");
                break;
            case LOG_LEVEL_INFO:
                buf.append("[INFO] ");
                break;
            case LOG_LEVEL_WARN:
                buf.append("[WARN] ");
                break;
            case LOG_LEVEL_ERROR:
                buf.append("[ERROR] ");
                break;
            case LOG_LEVEL_FATAL:
                buf.append("[FATAL] ");
                break;
        }

        // Append the name of the log instance if so configured
        if (showShortName) {
            if (shortLogName == null) {
                // Cut all but the last component of the name for both styles
                final String slName = logName.substring(logName.lastIndexOf(".") + 1);
                shortLogName = slName.substring(slName.lastIndexOf("/") + 1);
            }
            buf.append(String.valueOf(shortLogName)).append(" - ");
        } else if (showLogName) {
            buf.append(String.valueOf(logName)).append(" - ");
        }

        // Append the message
        buf.append(String.valueOf(message));

        // Append stack trace if not null
        if (t != null) {
            buf.append(" <");
            buf.append(t.toString());
            buf.append(">");

            final java.io.StringWriter sw = new java.io.StringWriter(1024);
            final java.io.PrintWriter pw = new java.io.PrintWriter(sw);
            t.printStackTrace(pw);
            pw.close();
            buf.append(sw.toString());
        }

        // Print to the appropriate destination
        write(buf);
    }

    /**
     * Write the content of the message accumulated in the specified
     * <code>StringBuffer</code> to the appropriate output destination.  The
     * default implementation writes to <code>System.err</code>.
     *
     * @param buffer A <code>StringBuffer</code> containing the accumulated
     *               text to be logged
     */
    protected void write(StringBuffer buffer) {
        System.err.println(buffer.toString());
    }

    /**
     * Is the given log level currently enabled?
     *
     * @param logLevel is this level enabled?
     */
    protected boolean isLevelEnabled(int logLevel) {
        // log level are numerically ordered so can use simple numeric
        // comparison
        return logLevel >= currentLogLevel;
    }

    // -------------------------------------------------------- Log Implementation

    /**
     * Logs a message with
     * <code>org.apache.commons.logging.impl.SimpleLog.LOG_LEVEL_DEBUG</code>.
     *
     * @param message to log
     *
     * @see Log#debug(Object)
     */
    public final void debug(Object message) {
        if (isLevelEnabled(LOG_LEVEL_DEBUG)) {
            log(LOG_LEVEL_DEBUG, message, null);
        }
    }

    /**
     * Logs a message with
     * <code>org.apache.commons.logging.impl.SimpleLog.LOG_LEVEL_DEBUG</code>.
     *
     * @param message to log
     * @param t       log this cause
     *
     * @see Log#debug(Object, Throwable)
     */
    public final void debug(Object message, Throwable t) {
        if (isLevelEnabled(LOG_LEVEL_DEBUG)) {
            log(LOG_LEVEL_DEBUG, message, t);
        }
    }

    /**
     * Logs a message with <code>org.apache.commons.logging.impl.SimpleLog.LOG_LEVEL_TRACE</code>.
     *
     * @param message to log
     *
     * @see Log#trace(Object)
     */
    public final void trace(Object message) {
        if (isLevelEnabled(LOG_LEVEL_TRACE)) {
            log(LOG_LEVEL_TRACE, message, null);
        }
    }

    /**
     * Logs a message with <code>org.apache.commons.logging.impl.SimpleLog.LOG_LEVEL_TRACE</code>.
     *
     * @param message to log
     * @param t       log this cause
     *
     * @see Log#trace(Object, Throwable)
     */
    public final void trace(Object message, Throwable t) {
        if (isLevelEnabled(LOG_LEVEL_TRACE)) {
            log(LOG_LEVEL_TRACE, message, t);
        }
    }

    /**
     * Logs a message with <code>org.apache.commons.logging.impl.SimpleLog.LOG_LEVEL_INFO</code>.
     *
     * @param message to log
     *
     * @see Log#info(Object)
     */
    public final void info(Object message) {
        if (isLevelEnabled(LOG_LEVEL_INFO)) {
            log(LOG_LEVEL_INFO, message, null);
        }
    }

    /**
     * Logs a message with <code>org.apache.commons.logging.impl.SimpleLog.LOG_LEVEL_INFO</code>.
     *
     * @param message to log
     * @param t       log this cause
     *
     * @see Log#info(Object, Throwable)
     */
    public final void info(Object message, Throwable t) {
        if (isLevelEnabled(LOG_LEVEL_INFO)) {
            log(LOG_LEVEL_INFO, message, t);
        }
    }

    /**
     * Logs a message with <code>org.apache.commons.logging.impl.SimpleLog.LOG_LEVEL_WARN</code>.
     *
     * @param message to log
     *
     * @see Log#warn(Object)
     */
    public final void warn(Object message) {
        if (isLevelEnabled(LOG_LEVEL_WARN)) {
            log(LOG_LEVEL_WARN, message, null);
        }
    }

    /**
     * Logs a message with <code>org.apache.commons.logging.impl.SimpleLog.LOG_LEVEL_WARN</code>.
     *
     * @param message to log
     * @param t       log this cause
     *
     * @see Log#warn(Object, Throwable)
     */
    public final void warn(Object message, Throwable t) {
        if (isLevelEnabled(LOG_LEVEL_WARN)) {
            log(LOG_LEVEL_WARN, message, t);
        }
    }

    /**
     * Logs a message with <code>org.apache.commons.logging.impl.SimpleLog.LOG_LEVEL_ERROR</code>.
     *
     * @param message to log
     *
     * @see Log#error(Object)
     */
    public final void error(Object message) {
        if (isLevelEnabled(LOG_LEVEL_ERROR)) {
            log(LOG_LEVEL_ERROR, message, null);
        }
    }

    /**
     * Logs a message with <code>org.apache.commons.logging.impl.SimpleLog.LOG_LEVEL_ERROR</code>.
     *
     * @param message to log
     * @param t       log this cause
     *
     * @see Log#error(Object, Throwable)
     */
    public final void error(Object message, Throwable t) {
        if (isLevelEnabled(LOG_LEVEL_ERROR)) {
            log(LOG_LEVEL_ERROR, message, t);
        }
    }

    /**
     * Log a message with <code>org.apache.commons.logging.impl.SimpleLog.LOG_LEVEL_FATAL</code>.
     *
     * @param message to log
     *
     * @see Log#fatal(Object)
     */
    public final void fatal(Object message) {
        if (isLevelEnabled(LOG_LEVEL_FATAL)) {
            log(LOG_LEVEL_FATAL, message, null);
        }
    }

    /**
     * Logs a message with <code>org.apache.commons.logging.impl.SimpleLog.LOG_LEVEL_FATAL</code>.
     *
     * @param message to log
     * @param t       log this cause
     *
     * @see Log#fatal(Object, Throwable)
     */
    public final void fatal(Object message, Throwable t) {
        if (isLevelEnabled(LOG_LEVEL_FATAL)) {
            log(LOG_LEVEL_FATAL, message, t);
        }
    }

    /**
     * Are debug messages currently enabled?
     * <p>
     * This allows expensive operations such as <code>String</code>
     * concatenation to be avoided when the message will be ignored by the
     * logger.
     */
    public final boolean isDebugEnabled() {
        return isLevelEnabled(LOG_LEVEL_DEBUG);
    }

    /**
     * Are error messages currently enabled?
     * <p>
     * This allows expensive operations such as <code>String</code>
     * concatenation to be avoided when the message will be ignored by the
     * logger.
     */
    public final boolean isErrorEnabled() {
        return isLevelEnabled(LOG_LEVEL_ERROR);
    }

    /**
     * Are fatal messages currently enabled?
     * <p>
     * This allows expensive operations such as <code>String</code>
     * concatenation to be avoided when the message will be ignored by the
     * logger.
     */
    public final boolean isFatalEnabled() {
        return isLevelEnabled(LOG_LEVEL_FATAL);
    }

    /**
     * Are info messages currently enabled?
     * <p>
     * This allows expensive operations such as <code>String</code>
     * concatenation to be avoided when the message will be ignored by the
     * logger.
     */
    public final boolean isInfoEnabled() {
        return isLevelEnabled(LOG_LEVEL_INFO);
    }

    /**
     * Are trace messages currently enabled?
     * <p>
     * This allows expensive operations such as <code>String</code>
     * concatenation to be avoided when the message will be ignored by the
     * logger.
     */
    public final boolean isTraceEnabled() {
        return isLevelEnabled(LOG_LEVEL_TRACE);
    }

    /**
     * Are warn messages currently enabled?
     * <p>
     * This allows expensive operations such as <code>String</code>
     * concatenation to be avoided when the message will be ignored by the
     * logger.
     */
    public final boolean isWarnEnabled() {
        return isLevelEnabled(LOG_LEVEL_WARN);
    }

    private static InputStream getResourceAsStream(final String name) {
        return (InputStream) AccessController.doPrivileged(
                new PrivilegedAction() {
                    public Object run() {
                        ClassLoader threadCL = Thread.currentThread().getContextClassLoader();

                        if (threadCL != null) {
                            return threadCL.getResourceAsStream(name);
                        } else {
                            return ClassLoader.getSystemResourceAsStream(name);
                        }
                    }
                });
    }
}

