package org.kiwiproject.consul;

import okhttp3.MediaType;
import okhttp3.RequestBody;
import okhttp3.ResponseBody;
import org.kiwiproject.consul.async.Callback;
import org.kiwiproject.consul.async.ConsulResponseCallback;
import org.kiwiproject.consul.config.ClientConfig;
import org.kiwiproject.consul.model.ConsulResponse;
import org.kiwiproject.consul.monitoring.ClientEventCallback;
import org.kiwiproject.consul.option.QueryOptions;
import retrofit2.Call;
import retrofit2.Retrofit;
import retrofit2.http.Body;
import retrofit2.http.GET;
import retrofit2.http.Headers;
import retrofit2.http.PUT;
import retrofit2.http.QueryMap;
import retrofit2.http.Streaming;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.Map;

/**
 * HTTP Client for /v1/snapshot/ endpoints.
 *
 * @see <a href="https://developer.hashicorp.com/consul/api-docs/snapshot">The Consul API Docs</a> for Snapshots
 */
public class SnapshotClient extends BaseClient {

    private static final String CLIENT_NAME = "snapshot";

    private final Api api;

    /**
     * Constructs an instance of this class.
     * @param retrofit The {@link Retrofit} to build a client from.
     */
    SnapshotClient(Retrofit retrofit, ClientConfig config, ClientEventCallback eventCallback) {
        super(CLIENT_NAME, config, eventCallback);
        this.api = retrofit.create(Api.class);
    }

    /**
     * Requests a new snapshot and save it in a file.
     * Only a subset of the QueryOptions is supported: datacenter, consistency mode, and token.
     *
     * @param destinationFile file in which the snapshot is to be saved.
     * @param queryOptions    query options. Only a subset of the QueryOptions is supported: datacenter, consistency mode, and token.
     * @param callback        callback called once the operation is over. It the save operation is successful, the X-Consul-Index is sent.
     */
    public void save(File destinationFile, QueryOptions queryOptions, Callback<BigInteger> callback) {
        http.extractConsulResponse(api.generateSnapshot(queryOptions.toQuery()), new ConsulResponseCallback<>() {
            @Override
            public void onComplete(ConsulResponse<ResponseBody> consulResponse) {
                // Note that response.body() and response.body().byteStream() should be closed.
                // see: https://square.github.io/okhttp/4.x/okhttp/okhttp3/-response-body/
                try (ResponseBody responseBody = consulResponse.getResponse()) {
                    try (InputStream inputStream = responseBody.byteStream()) {
                        Files.copy(inputStream, destinationFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
                        callback.onResponse(consulResponse.getIndex());
                    }
                } catch (IOException e) {
                    callback.onFailure(e);
                }
            }

            @Override
            public void onFailure(Throwable t) {
                callback.onFailure(t);
            }
        });
    }

    /**
     * Restores a snapshot stored in a file.
     * @param sourceFile source file where the snapshot is stored.
     * @param queryOptions query options. Only a subset of the QueryOptions is supported: datacenter, token.
     * @param callback callback called once the operation is over.
     */
    public void restore(File sourceFile, QueryOptions queryOptions, Callback<Void> callback) {
        RequestBody requestBody = RequestBody.create(sourceFile, MediaType.parse("application/binary"));
        http.extractBasicResponse(api.restoreSnapshot(queryOptions.toQuery(), requestBody), callback);
    }

    /**
     * Retrofit API interface.
     */
    interface Api {

        @Streaming
        @GET("snapshot")
        Call<ResponseBody> generateSnapshot(@QueryMap Map<String, Object> query);

        @PUT("snapshot")
        @Headers("Content-Type: application/binary")
        Call<Void> restoreSnapshot(@QueryMap Map<String, Object> query,
                                   @Body RequestBody requestBody);
    }
}
