/*
 * Decompiled with CFR 0.152.
 */
package org.knowm.xchange.binance;

import java.math.BigDecimal;
import java.math.MathContext;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.knowm.xchange.binance.dto.account.AssetDetail;
import org.knowm.xchange.binance.dto.marketdata.BinanceKline;
import org.knowm.xchange.binance.dto.marketdata.BinancePriceQuantity;
import org.knowm.xchange.binance.dto.trade.BinanceOrder;
import org.knowm.xchange.binance.dto.trade.OrderSide;
import org.knowm.xchange.binance.dto.trade.OrderStatus;
import org.knowm.xchange.binance.dto.trade.OrderType;
import org.knowm.xchange.binance.service.BinanceTradeService;
import org.knowm.xchange.currency.Currency;
import org.knowm.xchange.currency.CurrencyPair;
import org.knowm.xchange.dto.Order;
import org.knowm.xchange.dto.marketdata.CandleStick;
import org.knowm.xchange.dto.marketdata.CandleStickData;
import org.knowm.xchange.dto.marketdata.Ticker;
import org.knowm.xchange.dto.meta.CurrencyMetaData;
import org.knowm.xchange.dto.meta.WalletHealth;
import org.knowm.xchange.dto.trade.LimitOrder;
import org.knowm.xchange.dto.trade.MarketOrder;
import org.knowm.xchange.dto.trade.StopOrder;
import org.knowm.xchange.instrument.Instrument;

public class BinanceAdapters {
    private static final DateTimeFormatter DATE_TIME_FMT = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

    private BinanceAdapters() {
    }

    public static Date toDate(String dateTime) {
        return Date.from(Instant.from(BinanceAdapters.toLocalDateTime(dateTime).atZone(ZoneId.of("UTC"))));
    }

    public static LocalDateTime toLocalDateTime(String dateTime) {
        return LocalDateTime.parse(dateTime, DATE_TIME_FMT);
    }

    public static String toSymbol(CurrencyPair pair) {
        if (pair.equals((Object)CurrencyPair.IOTA_BTC)) {
            return "IOTABTC";
        }
        return pair.base.getCurrencyCode() + pair.counter.getCurrencyCode();
    }

    public static String toSymbol(Currency currency) {
        if (Currency.IOT.equals((Object)currency)) {
            return "IOTA";
        }
        return currency.getSymbol();
    }

    public static Order.OrderType convert(OrderSide side) {
        switch (side) {
            case BUY: {
                return Order.OrderType.BID;
            }
            case SELL: {
                return Order.OrderType.ASK;
            }
        }
        throw new RuntimeException("Not supported order side: " + (Object)((Object)side));
    }

    public static OrderSide convert(Order.OrderType type) {
        switch (type) {
            case ASK: {
                return OrderSide.SELL;
            }
            case BID: {
                return OrderSide.BUY;
            }
        }
        throw new RuntimeException("Not supported order type: " + type);
    }

    public static CurrencyPair convert(String symbol) {
        for (Currency base : Arrays.asList(Currency.BTC, Currency.ETH, Currency.BNB, Currency.USDT)) {
            if (!symbol.contains(base.toString())) continue;
            String counter = symbol.replace(base.toString(), "");
            return new CurrencyPair(base, new Currency(counter));
        }
        throw new IllegalArgumentException("Could not parse currency pair from '" + symbol + "'");
    }

    public static long id(String id) {
        try {
            return Long.valueOf(id);
        }
        catch (Throwable e) {
            throw new IllegalArgumentException("Binance id must be a valid long number.", e);
        }
    }

    public static Order.OrderStatus adaptOrderStatus(OrderStatus orderStatus) {
        switch (orderStatus) {
            case NEW: {
                return Order.OrderStatus.NEW;
            }
            case FILLED: {
                return Order.OrderStatus.FILLED;
            }
            case EXPIRED: {
                return Order.OrderStatus.EXPIRED;
            }
            case CANCELED: {
                return Order.OrderStatus.CANCELED;
            }
            case REJECTED: {
                return Order.OrderStatus.REJECTED;
            }
            case PENDING_CANCEL: {
                return Order.OrderStatus.PENDING_CANCEL;
            }
            case PARTIALLY_FILLED: {
                return Order.OrderStatus.PARTIALLY_FILLED;
            }
        }
        return Order.OrderStatus.UNKNOWN;
    }

    public static Order.OrderType convertType(boolean isBuyer) {
        return isBuyer ? Order.OrderType.BID : Order.OrderType.ASK;
    }

    public static CurrencyPair adaptSymbol(String symbol) {
        int pairLength = symbol.length();
        if (symbol.endsWith("USDT")) {
            return new CurrencyPair(symbol.substring(0, pairLength - 4), "USDT");
        }
        if (symbol.endsWith("USDC")) {
            return new CurrencyPair(symbol.substring(0, pairLength - 4), "USDC");
        }
        if (symbol.endsWith("TUSD")) {
            return new CurrencyPair(symbol.substring(0, pairLength - 4), "TUSD");
        }
        if (symbol.endsWith("USDS")) {
            return new CurrencyPair(symbol.substring(0, pairLength - 4), "USDS");
        }
        if (symbol.endsWith("BUSD")) {
            return new CurrencyPair(symbol.substring(0, pairLength - 4), "BUSD");
        }
        return new CurrencyPair(symbol.substring(0, pairLength - 3), symbol.substring(pairLength - 3));
    }

    public static Order adaptOrder(BinanceOrder order) {
        Order.OrderType type = BinanceAdapters.convert(order.side);
        CurrencyPair currencyPair = BinanceAdapters.adaptSymbol(order.symbol);
        Object builder = order.type.equals((Object)OrderType.MARKET) ? new MarketOrder.Builder(type, (Instrument)currencyPair) : (order.type.equals((Object)OrderType.LIMIT) || order.type.equals((Object)OrderType.LIMIT_MAKER) ? new LimitOrder.Builder(type, (Instrument)currencyPair).limitPrice(order.price) : new StopOrder.Builder(type, (Instrument)currencyPair).stopPrice(order.stopPrice));
        builder.orderStatus(BinanceAdapters.adaptOrderStatus(order.status)).originalAmount(order.origQty).id(Long.toString(order.orderId)).timestamp(order.getTime()).cumulativeAmount(order.executedQty);
        if (order.executedQty.signum() != 0 && order.cummulativeQuoteQty.signum() != 0) {
            builder.averagePrice(order.cummulativeQuoteQty.divide(order.executedQty, MathContext.DECIMAL32));
        }
        if (order.clientOrderId != null) {
            builder.flag((Order.IOrderFlags)BinanceTradeService.BinanceOrderFlags.withClientId(order.clientOrderId));
        }
        return builder.build();
    }

    private static Ticker adaptPriceQuantity(BinancePriceQuantity priceQuantity) {
        return new Ticker.Builder().currencyPair(BinanceAdapters.adaptSymbol(priceQuantity.symbol)).ask(priceQuantity.askPrice).askSize(priceQuantity.askQty).bid(priceQuantity.bidPrice).bidSize(priceQuantity.bidQty).build();
    }

    public static List<Ticker> adaptPriceQuantities(List<BinancePriceQuantity> priceQuantities) {
        return priceQuantities.stream().map(BinanceAdapters::adaptPriceQuantity).collect(Collectors.toList());
    }

    static CurrencyMetaData adaptCurrencyMetaData(Map<Currency, CurrencyMetaData> currencies, Currency currency, Map<String, AssetDetail> assetDetailMap, int precision) {
        AssetDetail asset;
        if (assetDetailMap != null && (asset = assetDetailMap.get(currency.getCurrencyCode())) != null) {
            BigDecimal withdrawalFee = asset.getWithdrawFee().stripTrailingZeros();
            BigDecimal minWithdrawalAmount = new BigDecimal(asset.getMinWithdrawAmount()).stripTrailingZeros();
            WalletHealth walletHealth = BinanceAdapters.getWalletHealth(asset.isDepositStatus(), asset.isWithdrawStatus());
            return new CurrencyMetaData(Integer.valueOf(precision), withdrawalFee, minWithdrawalAmount, walletHealth);
        }
        BigDecimal withdrawalFee = null;
        BigDecimal minWithdrawalAmount = null;
        if (currencies.containsKey(currency)) {
            CurrencyMetaData currencyMetaData = currencies.get(currency);
            withdrawalFee = currencyMetaData.getWithdrawalFee();
            minWithdrawalAmount = currencyMetaData.getMinWithdrawalAmount();
        }
        return new CurrencyMetaData(Integer.valueOf(precision), withdrawalFee, minWithdrawalAmount);
    }

    private static WalletHealth getWalletHealth(boolean depositEnabled, boolean withdrawEnabled) {
        if (depositEnabled && withdrawEnabled) {
            return WalletHealth.ONLINE;
        }
        if (!depositEnabled && withdrawEnabled) {
            return WalletHealth.DEPOSITS_DISABLED;
        }
        if (depositEnabled) {
            return WalletHealth.WITHDRAWALS_DISABLED;
        }
        return WalletHealth.OFFLINE;
    }

    public static OrderType adaptOrderType(StopOrder order) {
        if (order.getIntention() == null) {
            throw new IllegalArgumentException("Missing intention");
        }
        switch (order.getIntention()) {
            case STOP_LOSS: {
                return order.getLimitPrice() == null ? OrderType.STOP_LOSS : OrderType.STOP_LOSS_LIMIT;
            }
            case TAKE_PROFIT: {
                return order.getLimitPrice() == null ? OrderType.TAKE_PROFIT : OrderType.TAKE_PROFIT_LIMIT;
            }
        }
        throw new IllegalStateException("Unexpected value: " + order.getIntention());
    }

    public static CandleStickData adaptBinanceCandleStickData(List<BinanceKline> klines, CurrencyPair currencyPair) {
        CandleStickData candleStickData = null;
        if (klines.size() != 0) {
            ArrayList<CandleStick> candleSticks = new ArrayList<CandleStick>();
            for (BinanceKline chartData : klines) {
                candleSticks.add(new CandleStick.Builder().timestamp(new Date(chartData.getCloseTime())).open(chartData.getOpenPrice()).high(chartData.getHighPrice()).low(chartData.getLowPrice()).close(chartData.getClosePrice()).volume(chartData.getVolume()).quotaVolume(chartData.getQuoteAssetVolume()).build());
            }
            candleStickData = new CandleStickData((Instrument)currencyPair, candleSticks);
        }
        return candleStickData;
    }
}

