/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.index.internal.gbptree;

import java.io.EOFException;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.CopyOption;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Comparator;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Function;
import java.util.function.LongPredicate;
import org.neo4j.index.internal.gbptree.StructureWriteLog;
import org.neo4j.internal.helpers.Args;
import org.neo4j.internal.helpers.Format;
import org.neo4j.io.ByteUnit;
import org.neo4j.io.IOUtils;
import org.neo4j.io.fs.DefaultFileSystemAbstraction;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.fs.FlushableChannel;
import org.neo4j.io.fs.InputStreamReadableChannel;
import org.neo4j.io.fs.OutputStreamWritableChannel;
import org.neo4j.time.Clocks;
import org.neo4j.time.SystemNanoClock;

class LoggingStructureWriteLog
implements StructureWriteLog {
    private static final int ENTRY_HEADER_SIZE = 25;
    private static final Function<Path, Path> PATH_FUNCTION = gbptreePath -> gbptreePath.resolveSibling(String.valueOf(gbptreePath.getFileName()) + ".slog");
    private final FileSystemAbstraction fs;
    private final Path path;
    private final SystemNanoClock clock;
    private FlushableChannel channel;
    private final AtomicLong position = new AtomicLong();
    private final long rotationThreshold;
    private final long pruneThreshold = TimeUnit.DAYS.toMillis(1L);
    private final AtomicLong nextSessionId = new AtomicLong();
    static final Type[] TYPES = Type.values();

    LoggingStructureWriteLog(FileSystemAbstraction fs, Path path, long rotationThreshold) {
        this.fs = fs;
        this.path = path;
        this.clock = Clocks.nanoClock();
        this.rotationThreshold = rotationThreshold;
        try {
            if (fs.fileExists(path)) {
                this.moveAwayFile();
            }
            this.channel = this.instantiateChannel();
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public static LoggingStructureWriteLog forGBPTree(FileSystemAbstraction fs, Path gbptreeFile) {
        return new LoggingStructureWriteLog(fs, PATH_FUNCTION.apply(gbptreeFile), ByteUnit.mebiBytes((long)50L));
    }

    @Override
    public StructureWriteLog.Session newSession() {
        return new SessionImpl(this.nextSessionId.getAndIncrement());
    }

    @Override
    public void createRoot(long generation, long id) {
        this.writeEntry(Type.CREATE_ROOT, -1L, generation, id);
    }

    @Override
    public void deleteRoot(long generation, long id) {
        this.writeEntry(Type.DELETE_ROOT, -1L, generation, id);
    }

    @Override
    public synchronized void checkpoint(long previousStableGeneration, long newStableGeneration, long newUnstableGeneration) {
        this.writeEntry(Type.CHECKPOINT, -1L, newStableGeneration, previousStableGeneration, newUnstableGeneration);
        this.checkRotation();
    }

    private void checkRotation() {
        if (this.position.longValue() >= this.rotationThreshold) {
            try {
                this.channel.prepareForFlush().flush();
                this.channel.close();
                this.moveAwayFile();
                this.position.set(0L);
                this.channel = this.instantiateChannel();
                long time = this.clock.millis();
                long threshold = time - this.pruneThreshold;
                for (Path file2 : this.fs.listFiles(this.path.getParent(), file -> file.getFileName().toString().startsWith(String.valueOf(this.path.getFileName()) + "-"))) {
                    if (LoggingStructureWriteLog.millisOf(file2) >= threshold) continue;
                    this.fs.deleteFile(file2);
                }
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }
    }

    @Override
    public synchronized void close() {
        IOUtils.closeAllUnchecked((AutoCloseable[])new FlushableChannel[]{this.channel});
    }

    private void moveAwayFile() throws IOException {
        Path to;
        while (this.fs.fileExists(to = this.timestampedFile())) {
        }
        this.fs.renameFile(this.path, to, new CopyOption[0]);
    }

    private Path timestampedFile() {
        return this.path.resolveSibling(String.valueOf(this.path.getFileName()) + "-" + this.clock.millis());
    }

    static long millisOf(Path file) {
        String name = file.getFileName().toString();
        int dashIndex = name.lastIndexOf(45);
        if (dashIndex == -1) {
            return Long.MAX_VALUE;
        }
        return Long.parseLong(name.substring(dashIndex + 1));
    }

    private FlushableChannel instantiateChannel() throws IOException {
        return new OutputStreamWritableChannel(this.fs.openAsOutputStream(this.path, false));
    }

    private void writeHeader(Type type, long sessionId, long generation) throws IOException {
        this.channel.put(type.type);
        this.channel.putLong(sessionId);
        this.channel.putLong(this.clock.millis());
        this.channel.putLong(generation);
    }

    private synchronized void writeEntry(Type type, long sessionId, long generation, long ... ids) {
        try {
            this.writeHeader(type, sessionId, generation);
            for (long id : ids) {
                this.channel.putLong(id);
            }
            this.position.addAndGet(25 + 8 * ids.length);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public static void read(FileSystemAbstraction fs, Path baseFile, Events events) throws IOException {
        Path[] files = fs.listFiles(baseFile.getParent(), file -> file.getFileName().toString().startsWith(baseFile.getFileName().toString()) && !file.getFileName().toString().endsWith(".txt"));
        Arrays.sort(files, Comparator.comparing(LoggingStructureWriteLog::millisOf));
        for (Path file2 : files) {
            LoggingStructureWriteLog.readFile(fs, file2, events);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static void readFile(FileSystemAbstraction fs, Path path, Events events) throws IOException {
        try {
            InputStreamReadableChannel channel = new InputStreamReadableChannel(fs.openAsInputStream(path));
            try {
                Type type;
                block17: while (true) {
                    byte typeByte;
                    if ((typeByte = channel.get()) < 0 || typeByte >= TYPES.length) {
                        System.out.println("Unknown type " + typeByte);
                        continue;
                    }
                    type = TYPES[typeByte];
                    long sessionId = channel.getLong();
                    long timeMillis = channel.getLong();
                    long generation = channel.getLong();
                    switch (type) {
                        case SPLIT: {
                            events.split(timeMillis, sessionId, generation, channel.getLong(), channel.getLong(), channel.getLong());
                            continue block17;
                        }
                        case MERGE: {
                            events.merge(timeMillis, sessionId, generation, channel.getLong(), channel.getLong(), channel.getLong());
                            continue block17;
                        }
                        case SUCCESSOR: {
                            events.createSuccessor(timeMillis, sessionId, generation, channel.getLong(), channel.getLong(), channel.getLong());
                            continue block17;
                        }
                        case FREELIST: {
                            events.addToFreeList(timeMillis, sessionId, generation, channel.getLong());
                            continue block17;
                        }
                        case TREE_GROW: {
                            events.growTree(timeMillis, sessionId, generation, channel.getLong());
                            continue block17;
                        }
                        case TREE_SHRINK: {
                            events.shrinkTree(timeMillis, sessionId, generation, channel.getLong());
                            continue block17;
                        }
                        case CHECKPOINT: {
                            events.checkpoint(timeMillis, channel.getLong(), generation, channel.getLong());
                            continue block17;
                        }
                        case CREATE_ROOT: {
                            events.createRoot(timeMillis, generation, channel.getLong());
                            continue block17;
                        }
                        case DELETE_ROOT: {
                            events.deleteRoot(timeMillis, generation, channel.getLong());
                            continue block17;
                        }
                    }
                    break;
                }
                throw new UnsupportedOperationException(type.toString());
            }
            catch (Throwable throwable) {
                try {
                    channel.close();
                    throw throwable;
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
        }
        catch (EOFException eOFException) {
            return;
        }
    }

    public static void main(String[] args) throws IOException {
        Args arguments = Args.parse((String[])args);
        Path basePath = Path.of((String)arguments.orphans().get(0), new String[0]);
        String rawFilter = arguments.get("filter", null);
        LongPredicate idFilter = id -> true;
        if (rawFilter != null) {
            long idToFilterOn = Long.parseLong(rawFilter);
            idFilter = id -> id == idToFilterOn;
        }
        try (DefaultFileSystemAbstraction fs = new DefaultFileSystemAbstraction();){
            LoggingStructureWriteLog.read((FileSystemAbstraction)fs, basePath, new Dumper(idFilter));
        }
    }

    private class SessionImpl
    implements StructureWriteLog.Session {
        private final long sessionId;

        SessionImpl(long sessionId) {
            this.sessionId = sessionId;
        }

        @Override
        public void split(long generation, long parentId, long childId, long createdChildId) {
            LoggingStructureWriteLog.this.writeEntry(Type.SPLIT, this.sessionId, generation, parentId, childId, createdChildId);
        }

        @Override
        public void merge(long generation, long parentId, long childId, long deletedChildId) {
            LoggingStructureWriteLog.this.writeEntry(Type.MERGE, this.sessionId, generation, parentId, childId, deletedChildId);
        }

        @Override
        public void createSuccessor(long generation, long parentId, long oldId, long newId) {
            LoggingStructureWriteLog.this.writeEntry(Type.SUCCESSOR, this.sessionId, generation, parentId, oldId, newId);
        }

        @Override
        public void addToFreelist(long generation, long id) {
            LoggingStructureWriteLog.this.writeEntry(Type.FREELIST, this.sessionId, generation, id);
        }

        @Override
        public void growTree(long generation, long createdRootId) {
            LoggingStructureWriteLog.this.writeEntry(Type.TREE_GROW, this.sessionId, generation, createdRootId);
        }

        @Override
        public void shrinkTree(long generation, long deletedRootId) {
            LoggingStructureWriteLog.this.writeEntry(Type.TREE_SHRINK, this.sessionId, generation, deletedRootId);
        }
    }

    static enum Type {
        SPLIT(0),
        MERGE(1),
        SUCCESSOR(2),
        FREELIST(3),
        TREE_GROW(4),
        TREE_SHRINK(5),
        CHECKPOINT(6),
        CREATE_ROOT(7),
        DELETE_ROOT(8);

        private final byte type;

        private Type(byte value) {
            this.type = value;
        }
    }

    static interface Events {
        public void split(long var1, long var3, long var5, long var7, long var9, long var11);

        public void merge(long var1, long var3, long var5, long var7, long var9, long var11);

        public void createSuccessor(long var1, long var3, long var5, long var7, long var9, long var11);

        public void addToFreeList(long var1, long var3, long var5, long var7);

        public void checkpoint(long var1, long var3, long var5, long var7);

        public void growTree(long var1, long var3, long var5, long var7);

        public void shrinkTree(long var1, long var3, long var5, long var7);

        public void createRoot(long var1, long var3, long var5);

        public void deleteRoot(long var1, long var3, long var5);
    }

    private static class Dumper
    implements Events {
        private final LongPredicate idFilter;

        Dumper(LongPredicate idFilter) {
            this.idFilter = idFilter;
        }

        @Override
        public void split(long timeMillis, long sessionId, long generation, long parentId, long childId, long createdChildId) {
            if (this.idFilter.test(parentId) || this.idFilter.test(createdChildId)) {
                System.out.printf("%s %d %d SP %d -> %d -> %d%n", Format.date((long)timeMillis), sessionId, generation, parentId, childId, createdChildId);
            }
        }

        @Override
        public void merge(long timeMillis, long sessionId, long generation, long parentId, long childId, long deletedChildId) {
            if (this.idFilter.test(parentId) || this.idFilter.test(deletedChildId)) {
                System.out.printf("%s %d %d ME %d -> %d -X-> %d%n", Format.date((long)timeMillis), sessionId, generation, parentId, childId, deletedChildId);
            }
        }

        @Override
        public void createSuccessor(long timeMillis, long sessionId, long generation, long parentId, long oldId, long newId) {
            if (this.idFilter.test(oldId) || this.idFilter.test(newId)) {
                System.out.printf("%s %d %d SU %d -> %d -> %d%n", Format.date((long)timeMillis), sessionId, generation, parentId, oldId, newId);
            }
        }

        @Override
        public void addToFreeList(long timeMillis, long sessionId, long generation, long id) {
            if (this.idFilter.test(id)) {
                System.out.printf("%s %d %d FR %d%n", Format.date((long)timeMillis), sessionId, generation, id);
            }
        }

        @Override
        public void growTree(long timeMillis, long sessionId, long generation, long createdRootId) {
            if (this.idFilter.test(createdRootId)) {
                System.out.printf("%s %d %d GT %d%n", Format.date((long)timeMillis), sessionId, generation, createdRootId);
            }
        }

        @Override
        public void shrinkTree(long timeMillis, long sessionId, long generation, long deletedRootId) {
            if (this.idFilter.test(deletedRootId)) {
                System.out.printf("%s %d %d ST %d%n", Format.date((long)timeMillis), sessionId, generation, deletedRootId);
            }
        }

        @Override
        public void checkpoint(long timeMillis, long previousStableGeneration, long newStableGeneration, long newUnstableGeneration) {
            System.out.printf("%s CP %d %d %d%n", Format.date((long)timeMillis), previousStableGeneration, newStableGeneration, newUnstableGeneration);
        }

        @Override
        public void createRoot(long timeMillis, long generation, long id) {
            if (this.idFilter.test(id)) {
                System.out.printf("%s CR %d %d%n", Format.date((long)timeMillis), generation, id);
            }
        }

        @Override
        public void deleteRoot(long timeMillis, long generation, long id) {
            if (this.idFilter.test(id)) {
                System.out.printf("%s DR %d %d%n", Format.date((long)timeMillis), generation, id);
            }
        }
    }
}

