/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 */

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/*
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

package org.opensearch.client.opensearch.cluster.allocation_explain;

import jakarta.json.stream.JsonGenerator;
import java.util.Map;
import java.util.function.Function;
import org.opensearch.client.json.JsonpDeserializable;
import org.opensearch.client.json.JsonpDeserializer;
import org.opensearch.client.json.JsonpMapper;
import org.opensearch.client.json.ObjectBuilderDeserializer;
import org.opensearch.client.json.ObjectDeserializer;
import org.opensearch.client.json.PlainJsonSerializable;
import org.opensearch.client.util.ApiTypeHelper;
import org.opensearch.client.util.ObjectBuilder;
import org.opensearch.client.util.ObjectBuilderBase;

// typedef: cluster.allocation_explain.CurrentNode

@JsonpDeserializable
public class CurrentNode implements PlainJsonSerializable {
    private final String id;

    private final String name;

    private final Map<String, String> attributes;

    private final String transportAddress;

    private final int weightRanking;

    // ---------------------------------------------------------------------------------------------

    private CurrentNode(Builder builder) {

        this.id = ApiTypeHelper.requireNonNull(builder.id, this, "id");
        this.name = ApiTypeHelper.requireNonNull(builder.name, this, "name");
        this.attributes = ApiTypeHelper.unmodifiableRequired(builder.attributes, this, "attributes");
        this.transportAddress = ApiTypeHelper.requireNonNull(builder.transportAddress, this, "transportAddress");
        this.weightRanking = ApiTypeHelper.requireNonNull(builder.weightRanking, this, "weightRanking");

    }

    public static CurrentNode of(Function<Builder, ObjectBuilder<CurrentNode>> fn) {
        return fn.apply(new Builder()).build();
    }

    /**
     * Required - API name: {@code id}
     */
    public final String id() {
        return this.id;
    }

    /**
     * Required - API name: {@code name}
     */
    public final String name() {
        return this.name;
    }

    /**
     * Required - API name: {@code attributes}
     */
    public final Map<String, String> attributes() {
        return this.attributes;
    }

    /**
     * Required - API name: {@code transport_address}
     */
    public final String transportAddress() {
        return this.transportAddress;
    }

    /**
     * Required - API name: {@code weight_ranking}
     */
    public final int weightRanking() {
        return this.weightRanking;
    }

    /**
     * Serialize this object to JSON.
     */
    public void serialize(JsonGenerator generator, JsonpMapper mapper) {
        generator.writeStartObject();
        serializeInternal(generator, mapper);
        generator.writeEnd();
    }

    protected void serializeInternal(JsonGenerator generator, JsonpMapper mapper) {

        generator.writeKey("id");
        generator.write(this.id);

        generator.writeKey("name");
        generator.write(this.name);

        if (ApiTypeHelper.isDefined(this.attributes)) {
            generator.writeKey("attributes");
            generator.writeStartObject();
            for (Map.Entry<String, String> item0 : this.attributes.entrySet()) {
                generator.writeKey(item0.getKey());
                generator.write(item0.getValue());

            }
            generator.writeEnd();

        }
        generator.writeKey("transport_address");
        generator.write(this.transportAddress);

        generator.writeKey("weight_ranking");
        generator.write(this.weightRanking);

    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Builder for {@link CurrentNode}.
     */

    public static class Builder extends ObjectBuilderBase implements ObjectBuilder<CurrentNode> {
        private String id;

        private String name;

        private Map<String, String> attributes;

        private String transportAddress;

        private Integer weightRanking;

        /**
         * Required - API name: {@code id}
         */
        public final Builder id(String value) {
            this.id = value;
            return this;
        }

        /**
         * Required - API name: {@code name}
         */
        public final Builder name(String value) {
            this.name = value;
            return this;
        }

        /**
         * Required - API name: {@code attributes}
         * <p>
         * Adds all entries of <code>map</code> to <code>attributes</code>.
         */
        public final Builder attributes(Map<String, String> map) {
            this.attributes = _mapPutAll(this.attributes, map);
            return this;
        }

        /**
         * Required - API name: {@code attributes}
         * <p>
         * Adds an entry to <code>attributes</code>.
         */
        public final Builder attributes(String key, String value) {
            this.attributes = _mapPut(this.attributes, key, value);
            return this;
        }

        /**
         * Required - API name: {@code transport_address}
         */
        public final Builder transportAddress(String value) {
            this.transportAddress = value;
            return this;
        }

        /**
         * Required - API name: {@code weight_ranking}
         */
        public final Builder weightRanking(int value) {
            this.weightRanking = value;
            return this;
        }

        /**
         * Builds a {@link CurrentNode}.
         *
         * @throws NullPointerException
         *             if some of the required fields are null.
         */
        public CurrentNode build() {
            _checkSingleUse();

            return new CurrentNode(this);
        }
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Json deserializer for {@link CurrentNode}
     */
    public static final JsonpDeserializer<CurrentNode> _DESERIALIZER = ObjectBuilderDeserializer.lazy(
        Builder::new,
        CurrentNode::setupCurrentNodeDeserializer
    );

    protected static void setupCurrentNodeDeserializer(ObjectDeserializer<CurrentNode.Builder> op) {

        op.add(Builder::id, JsonpDeserializer.stringDeserializer(), "id");
        op.add(Builder::name, JsonpDeserializer.stringDeserializer(), "name");
        op.add(Builder::attributes, JsonpDeserializer.stringMapDeserializer(JsonpDeserializer.stringDeserializer()), "attributes");
        op.add(Builder::transportAddress, JsonpDeserializer.stringDeserializer(), "transport_address");
        op.add(Builder::weightRanking, JsonpDeserializer.integerDeserializer(), "weight_ranking");

    }

}
