/*
 * Decompiled with CFR 0.152.
 */
package android.database.sqlite;

import android.database.sqlite.SQLiteCompatibilityWalFlags;
import android.database.sqlite.SQLiteConnection;
import android.database.sqlite.SQLiteDatabaseConfiguration;
import android.database.sqlite.SQLiteDebug;
import android.database.sqlite.SQLiteGlobal;
import android.os.CancellationSignal;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.os.OperationCanceledException;
import android.os.SystemClock;
import android.text.TextUtils;
import android.util.ArraySet;
import android.util.Log;
import android.util.PrefixPrinter;
import android.util.Printer;
import com.android.internal.annotations.GuardedBy;
import com.android.internal.annotations.VisibleForTesting;
import dalvik.system.CloseGuard;
import java.io.Closeable;
import java.io.File;
import java.util.ArrayList;
import java.util.Map;
import java.util.WeakHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.LockSupport;

public final class SQLiteConnectionPool
implements Closeable {
    private static final String TAG = "SQLiteConnectionPool";
    private static final long CONNECTION_POOL_BUSY_MILLIS = 30000L;
    private final CloseGuard mCloseGuard = CloseGuard.get();
    private final Object mLock = new Object();
    private final AtomicBoolean mConnectionLeaked = new AtomicBoolean();
    private final SQLiteDatabaseConfiguration mConfiguration;
    private int mMaxConnectionPoolSize;
    private boolean mIsOpen;
    private int mNextConnectionId;
    private ConnectionWaiter mConnectionWaiterPool;
    private ConnectionWaiter mConnectionWaiterQueue;
    private final ArrayList<SQLiteConnection> mAvailableNonPrimaryConnections = new ArrayList();
    private SQLiteConnection mAvailablePrimaryConnection;
    @GuardedBy(value={"mLock"})
    private IdleConnectionHandler mIdleConnectionHandler;
    private final AtomicLong mTotalExecutionTimeCounter = new AtomicLong(0L);
    private final WeakHashMap<SQLiteConnection, AcquiredConnectionStatus> mAcquiredConnections = new WeakHashMap();
    public static final int CONNECTION_FLAG_READ_ONLY = 1;
    public static final int CONNECTION_FLAG_PRIMARY_CONNECTION_AFFINITY = 2;
    public static final int CONNECTION_FLAG_INTERACTIVE = 4;

    private SQLiteConnectionPool(SQLiteDatabaseConfiguration configuration) {
        this.mConfiguration = new SQLiteDatabaseConfiguration(configuration);
        this.setMaxConnectionPoolSizeLocked();
        if (this.mConfiguration.idleConnectionTimeoutMs != Long.MAX_VALUE) {
            this.setupIdleConnectionHandler(Looper.getMainLooper(), this.mConfiguration.idleConnectionTimeoutMs);
        }
    }

    protected void finalize() throws Throwable {
        try {
            this.dispose(true);
        }
        finally {
            super.finalize();
        }
    }

    public static SQLiteConnectionPool open(SQLiteDatabaseConfiguration configuration) {
        if (configuration == null) {
            throw new IllegalArgumentException("configuration must not be null.");
        }
        SQLiteConnectionPool pool = new SQLiteConnectionPool(configuration);
        pool.open();
        return pool;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void open() {
        this.mAvailablePrimaryConnection = this.openConnectionLocked(this.mConfiguration, true);
        Object object = this.mLock;
        synchronized (object) {
            if (this.mIdleConnectionHandler != null) {
                this.mIdleConnectionHandler.connectionReleased(this.mAvailablePrimaryConnection);
            }
        }
        this.mIsOpen = true;
        this.mCloseGuard.open("close");
    }

    @Override
    public void close() {
        this.dispose(false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void dispose(boolean finalized) {
        if (this.mCloseGuard != null) {
            if (finalized) {
                this.mCloseGuard.warnIfOpen();
            }
            this.mCloseGuard.close();
        }
        if (!finalized) {
            Object object = this.mLock;
            synchronized (object) {
                this.throwIfClosedLocked();
                this.mIsOpen = false;
                this.closeAvailableConnectionsAndLogExceptionsLocked();
                int pendingCount = this.mAcquiredConnections.size();
                if (pendingCount != 0) {
                    Log.i(TAG, "The connection pool for " + this.mConfiguration.label + " has been closed but there are still " + pendingCount + " connections in use.  They will be closed as they are released back to the pool.");
                }
                this.wakeConnectionWaitersLocked();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void reconfigure(SQLiteDatabaseConfiguration configuration) {
        if (configuration == null) {
            throw new IllegalArgumentException("configuration must not be null.");
        }
        Object object = this.mLock;
        synchronized (object) {
            boolean onlyCompatWalChanged;
            boolean foreignKeyModeChanged;
            boolean walModeChanged;
            this.throwIfClosedLocked();
            boolean bl = walModeChanged = ((configuration.openFlags ^ this.mConfiguration.openFlags) & 0x20000000) != 0;
            if (walModeChanged) {
                if (!this.mAcquiredConnections.isEmpty()) {
                    throw new IllegalStateException("Write Ahead Logging (WAL) mode cannot be enabled or disabled while there are transactions in progress.  Finish all transactions and release all active database connections first.");
                }
                this.closeAvailableNonPrimaryConnectionsAndLogExceptionsLocked();
                assert (this.mAvailableNonPrimaryConnections.isEmpty());
            }
            boolean bl2 = foreignKeyModeChanged = configuration.foreignKeyConstraintsEnabled != this.mConfiguration.foreignKeyConstraintsEnabled;
            if (foreignKeyModeChanged && !this.mAcquiredConnections.isEmpty()) {
                throw new IllegalStateException("Foreign Key Constraints cannot be enabled or disabled while there are transactions in progress.  Finish all transactions and release all active database connections first.");
            }
            boolean bl3 = onlyCompatWalChanged = (this.mConfiguration.openFlags ^ configuration.openFlags) == Integer.MIN_VALUE;
            if (!onlyCompatWalChanged && this.mConfiguration.openFlags != configuration.openFlags) {
                if (walModeChanged) {
                    this.closeAvailableConnectionsAndLogExceptionsLocked();
                }
                SQLiteConnection newPrimaryConnection = this.openConnectionLocked(configuration, true);
                this.closeAvailableConnectionsAndLogExceptionsLocked();
                this.discardAcquiredConnectionsLocked();
                this.mAvailablePrimaryConnection = newPrimaryConnection;
                this.mConfiguration.updateParametersFrom(configuration);
                this.setMaxConnectionPoolSizeLocked();
            } else {
                this.mConfiguration.updateParametersFrom(configuration);
                this.setMaxConnectionPoolSizeLocked();
                this.closeExcessConnectionsAndLogExceptionsLocked();
                this.reconfigureAllConnectionsLocked();
            }
            this.wakeConnectionWaitersLocked();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public SQLiteConnection acquireConnection(String sql, int connectionFlags, CancellationSignal cancellationSignal) {
        SQLiteConnection con = this.waitForConnection(sql, connectionFlags, cancellationSignal);
        Object object = this.mLock;
        synchronized (object) {
            if (this.mIdleConnectionHandler != null) {
                this.mIdleConnectionHandler.connectionAcquired(con);
            }
        }
        return con;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void releaseConnection(SQLiteConnection connection) {
        Object object = this.mLock;
        synchronized (object) {
            AcquiredConnectionStatus status;
            if (this.mIdleConnectionHandler != null) {
                this.mIdleConnectionHandler.connectionReleased(connection);
            }
            if ((status = this.mAcquiredConnections.remove(connection)) == null) {
                throw new IllegalStateException("Cannot perform this operation because the specified connection was not acquired from this pool or has already been released.");
            }
            if (!this.mIsOpen) {
                this.closeConnectionAndLogExceptionsLocked(connection);
            } else if (connection.isPrimaryConnection()) {
                if (this.recycleConnectionLocked(connection, status)) {
                    assert (this.mAvailablePrimaryConnection == null);
                    this.mAvailablePrimaryConnection = connection;
                }
                this.wakeConnectionWaitersLocked();
            } else if (this.mAvailableNonPrimaryConnections.size() >= this.mMaxConnectionPoolSize - 1) {
                this.closeConnectionAndLogExceptionsLocked(connection);
            } else {
                if (this.recycleConnectionLocked(connection, status)) {
                    this.mAvailableNonPrimaryConnections.add(connection);
                }
                this.wakeConnectionWaitersLocked();
            }
        }
    }

    @GuardedBy(value={"mLock"})
    private boolean recycleConnectionLocked(SQLiteConnection connection, AcquiredConnectionStatus status) {
        if (status == AcquiredConnectionStatus.RECONFIGURE) {
            try {
                connection.reconfigure(this.mConfiguration);
            }
            catch (RuntimeException ex) {
                Log.e(TAG, "Failed to reconfigure released connection, closing it: " + connection, ex);
                status = AcquiredConnectionStatus.DISCARD;
            }
        }
        if (status == AcquiredConnectionStatus.DISCARD) {
            this.closeConnectionAndLogExceptionsLocked(connection);
            return false;
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean shouldYieldConnection(SQLiteConnection connection, int connectionFlags) {
        Object object = this.mLock;
        synchronized (object) {
            if (!this.mAcquiredConnections.containsKey(connection)) {
                throw new IllegalStateException("Cannot perform this operation because the specified connection was not acquired from this pool or has already been released.");
            }
            if (!this.mIsOpen) {
                return false;
            }
            return this.isSessionBlockingImportantConnectionWaitersLocked(connection.isPrimaryConnection(), connectionFlags);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void collectDbStats(ArrayList<SQLiteDebug.DbStats> dbStatsList) {
        Object object = this.mLock;
        synchronized (object) {
            if (this.mAvailablePrimaryConnection != null) {
                this.mAvailablePrimaryConnection.collectDbStats(dbStatsList);
            }
            for (SQLiteConnection connection : this.mAvailableNonPrimaryConnections) {
                connection.collectDbStats(dbStatsList);
            }
            for (SQLiteConnection connection : this.mAcquiredConnections.keySet()) {
                connection.collectDbStatsUnsafe(dbStatsList);
            }
        }
    }

    private SQLiteConnection openConnectionLocked(SQLiteDatabaseConfiguration configuration, boolean primaryConnection) {
        int connectionId = this.mNextConnectionId++;
        return SQLiteConnection.open(this, configuration, connectionId, primaryConnection);
    }

    void onConnectionLeaked() {
        Log.w(TAG, "A SQLiteConnection object for database '" + this.mConfiguration.label + "' was leaked!  Please fix your application to end transactions in progress properly and to close the database when it is no longer needed.");
        this.mConnectionLeaked.set(true);
    }

    void onStatementExecuted(long executionTimeMs) {
        this.mTotalExecutionTimeCounter.addAndGet(executionTimeMs);
    }

    @GuardedBy(value={"mLock"})
    private void closeAvailableConnectionsAndLogExceptionsLocked() {
        this.closeAvailableNonPrimaryConnectionsAndLogExceptionsLocked();
        if (this.mAvailablePrimaryConnection != null) {
            this.closeConnectionAndLogExceptionsLocked(this.mAvailablePrimaryConnection);
            this.mAvailablePrimaryConnection = null;
        }
    }

    @GuardedBy(value={"mLock"})
    private boolean closeAvailableConnectionLocked(int connectionId) {
        int count = this.mAvailableNonPrimaryConnections.size();
        for (int i = count - 1; i >= 0; --i) {
            SQLiteConnection c = this.mAvailableNonPrimaryConnections.get(i);
            if (c.getConnectionId() != connectionId) continue;
            this.closeConnectionAndLogExceptionsLocked(c);
            this.mAvailableNonPrimaryConnections.remove(i);
            return true;
        }
        if (this.mAvailablePrimaryConnection != null && this.mAvailablePrimaryConnection.getConnectionId() == connectionId) {
            this.closeConnectionAndLogExceptionsLocked(this.mAvailablePrimaryConnection);
            this.mAvailablePrimaryConnection = null;
            return true;
        }
        return false;
    }

    @GuardedBy(value={"mLock"})
    private void closeAvailableNonPrimaryConnectionsAndLogExceptionsLocked() {
        int count = this.mAvailableNonPrimaryConnections.size();
        for (int i = 0; i < count; ++i) {
            this.closeConnectionAndLogExceptionsLocked(this.mAvailableNonPrimaryConnections.get(i));
        }
        this.mAvailableNonPrimaryConnections.clear();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void closeAvailableNonPrimaryConnectionsAndLogExceptions() {
        Object object = this.mLock;
        synchronized (object) {
            this.closeAvailableNonPrimaryConnectionsAndLogExceptionsLocked();
        }
    }

    @GuardedBy(value={"mLock"})
    private void closeExcessConnectionsAndLogExceptionsLocked() {
        int availableCount = this.mAvailableNonPrimaryConnections.size();
        while (availableCount-- > this.mMaxConnectionPoolSize - 1) {
            SQLiteConnection connection = this.mAvailableNonPrimaryConnections.remove(availableCount);
            this.closeConnectionAndLogExceptionsLocked(connection);
        }
    }

    @GuardedBy(value={"mLock"})
    private void closeConnectionAndLogExceptionsLocked(SQLiteConnection connection) {
        try {
            connection.close();
            if (this.mIdleConnectionHandler != null) {
                this.mIdleConnectionHandler.connectionClosed(connection);
            }
        }
        catch (RuntimeException ex) {
            Log.e(TAG, "Failed to close connection, its fate is now in the hands of the merciful GC: " + connection, ex);
        }
    }

    private void discardAcquiredConnectionsLocked() {
        this.markAcquiredConnectionsLocked(AcquiredConnectionStatus.DISCARD);
    }

    @GuardedBy(value={"mLock"})
    private void reconfigureAllConnectionsLocked() {
        if (this.mAvailablePrimaryConnection != null) {
            try {
                this.mAvailablePrimaryConnection.reconfigure(this.mConfiguration);
            }
            catch (RuntimeException ex) {
                Log.e(TAG, "Failed to reconfigure available primary connection, closing it: " + this.mAvailablePrimaryConnection, ex);
                this.closeConnectionAndLogExceptionsLocked(this.mAvailablePrimaryConnection);
                this.mAvailablePrimaryConnection = null;
            }
        }
        int count = this.mAvailableNonPrimaryConnections.size();
        for (int i = 0; i < count; ++i) {
            SQLiteConnection connection = this.mAvailableNonPrimaryConnections.get(i);
            try {
                connection.reconfigure(this.mConfiguration);
                continue;
            }
            catch (RuntimeException ex) {
                Log.e(TAG, "Failed to reconfigure available non-primary connection, closing it: " + connection, ex);
                this.closeConnectionAndLogExceptionsLocked(connection);
                this.mAvailableNonPrimaryConnections.remove(i--);
                --count;
            }
        }
        this.markAcquiredConnectionsLocked(AcquiredConnectionStatus.RECONFIGURE);
    }

    private void markAcquiredConnectionsLocked(AcquiredConnectionStatus status) {
        if (!this.mAcquiredConnections.isEmpty()) {
            ArrayList<SQLiteConnection> keysToUpdate = new ArrayList<SQLiteConnection>(this.mAcquiredConnections.size());
            for (Map.Entry<SQLiteConnection, AcquiredConnectionStatus> entry : this.mAcquiredConnections.entrySet()) {
                AcquiredConnectionStatus oldStatus = entry.getValue();
                if (status == oldStatus || oldStatus == AcquiredConnectionStatus.DISCARD) continue;
                keysToUpdate.add(entry.getKey());
            }
            int updateCount = keysToUpdate.size();
            for (int i = 0; i < updateCount; ++i) {
                this.mAcquiredConnections.put((SQLiteConnection)keysToUpdate.get(i), status);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     * Converted monitor instructions to comments
     * Lifted jumps to return sites
     */
    private SQLiteConnection waitForConnection(String sql, int connectionFlags, CancellationSignal cancellationSignal) {
        boolean wantPrimaryConnection = (connectionFlags & 2) != 0;
        Object object = this.mLock;
        // MONITORENTER : object
        this.throwIfClosedLocked();
        if (cancellationSignal != null) {
            cancellationSignal.throwIfCanceled();
        }
        SQLiteConnection connection = null;
        if (!wantPrimaryConnection) {
            connection = this.tryAcquireNonPrimaryConnectionLocked(sql, connectionFlags);
        }
        if (connection == null) {
            connection = this.tryAcquirePrimaryConnectionLocked(connectionFlags);
        }
        if (connection != null) {
            // MONITOREXIT : object
            return connection;
        }
        int priority = SQLiteConnectionPool.getPriority(connectionFlags);
        long startTime = SystemClock.uptimeMillis();
        final ConnectionWaiter waiter = this.obtainConnectionWaiterLocked(Thread.currentThread(), startTime, priority, wantPrimaryConnection, sql, connectionFlags);
        ConnectionWaiter predecessor = null;
        ConnectionWaiter successor = this.mConnectionWaiterQueue;
        while (successor != null) {
            if (priority > successor.mPriority) {
                waiter.mNext = successor;
                break;
            }
            predecessor = successor;
            successor = successor.mNext;
        }
        if (predecessor != null) {
            predecessor.mNext = waiter;
        } else {
            this.mConnectionWaiterQueue = waiter;
        }
        final int nonce = waiter.mNonce;
        // MONITOREXIT : object
        if (cancellationSignal != null) {
            cancellationSignal.setOnCancelListener(new CancellationSignal.OnCancelListener(){

                /*
                 * WARNING - Removed try catching itself - possible behaviour change.
                 */
                @Override
                public void onCancel() {
                    Object object = SQLiteConnectionPool.this.mLock;
                    synchronized (object) {
                        if (waiter.mNonce == nonce) {
                            SQLiteConnectionPool.this.cancelConnectionWaiterLocked(waiter);
                        }
                    }
                }
            });
        }
        try {
            long busyTimeoutMillis = 30000L;
            long nextBusyTimeoutTime = waiter.mStartTime + busyTimeoutMillis;
            while (true) {
                Object object2;
                if (this.mConnectionLeaked.compareAndSet(true, false)) {
                    object2 = this.mLock;
                    // MONITORENTER : object2
                    this.wakeConnectionWaitersLocked();
                    // MONITOREXIT : object2
                }
                LockSupport.parkNanos(this, busyTimeoutMillis * 1000000L);
                Thread.interrupted();
                object2 = this.mLock;
                // MONITORENTER : object2
                this.throwIfClosedLocked();
                SQLiteConnection connection2 = waiter.mAssignedConnection;
                RuntimeException ex = waiter.mException;
                if (connection2 != null || ex != null) {
                    this.recycleConnectionWaiterLocked(waiter);
                    if (connection2 == null) throw ex;
                    SQLiteConnection sQLiteConnection = connection2;
                    // MONITOREXIT : object2
                    return sQLiteConnection;
                }
                long now = SystemClock.uptimeMillis();
                if (now < nextBusyTimeoutTime) {
                    busyTimeoutMillis = now - nextBusyTimeoutTime;
                } else {
                    this.logConnectionPoolBusyLocked(now - waiter.mStartTime, connectionFlags);
                    busyTimeoutMillis = 30000L;
                    nextBusyTimeoutTime = now + busyTimeoutMillis;
                }
                // MONITOREXIT : object2
            }
        }
        finally {
            if (cancellationSignal != null) {
                cancellationSignal.setOnCancelListener(null);
            }
        }
    }

    @GuardedBy(value={"mLock"})
    private void cancelConnectionWaiterLocked(ConnectionWaiter waiter) {
        if (waiter.mAssignedConnection != null || waiter.mException != null) {
            return;
        }
        ConnectionWaiter predecessor = null;
        ConnectionWaiter current = this.mConnectionWaiterQueue;
        while (current != waiter) {
            assert (current != null);
            predecessor = current;
            current = current.mNext;
        }
        if (predecessor != null) {
            predecessor.mNext = waiter.mNext;
        } else {
            this.mConnectionWaiterQueue = waiter.mNext;
        }
        waiter.mException = new OperationCanceledException();
        LockSupport.unpark(waiter.mThread);
        this.wakeConnectionWaitersLocked();
    }

    private void logConnectionPoolBusyLocked(long waitMillis, int connectionFlags) {
        Thread thread = Thread.currentThread();
        StringBuilder msg = new StringBuilder();
        msg.append("The connection pool for database '").append(this.mConfiguration.label);
        msg.append("' has been unable to grant a connection to thread ");
        msg.append(thread.getId()).append(" (").append(thread.getName()).append(") ");
        msg.append("with flags 0x").append(Integer.toHexString(connectionFlags));
        msg.append(" for ").append((float)waitMillis * 0.001f).append(" seconds.\n");
        ArrayList<String> requests = new ArrayList<String>();
        int activeConnections = 0;
        int idleConnections = 0;
        if (!this.mAcquiredConnections.isEmpty()) {
            for (SQLiteConnection connection : this.mAcquiredConnections.keySet()) {
                String description = connection.describeCurrentOperationUnsafe();
                if (description != null) {
                    requests.add(description);
                    ++activeConnections;
                    continue;
                }
                ++idleConnections;
            }
        }
        int availableConnections = this.mAvailableNonPrimaryConnections.size();
        if (this.mAvailablePrimaryConnection != null) {
            ++availableConnections;
        }
        msg.append("Connections: ").append(activeConnections).append(" active, ");
        msg.append(idleConnections).append(" idle, ");
        msg.append(availableConnections).append(" available.\n");
        if (!requests.isEmpty()) {
            msg.append("\nRequests in progress:\n");
            for (String request : requests) {
                msg.append("  ").append(request).append("\n");
            }
        }
        Log.w(TAG, msg.toString());
    }

    @GuardedBy(value={"mLock"})
    private void wakeConnectionWaitersLocked() {
        ConnectionWaiter predecessor = null;
        ConnectionWaiter waiter = this.mConnectionWaiterQueue;
        boolean primaryConnectionNotAvailable = false;
        boolean nonPrimaryConnectionNotAvailable = false;
        while (waiter != null) {
            boolean unpark = false;
            if (!this.mIsOpen) {
                unpark = true;
            } else {
                try {
                    SQLiteConnection connection = null;
                    if (!waiter.mWantPrimaryConnection && !nonPrimaryConnectionNotAvailable && (connection = this.tryAcquireNonPrimaryConnectionLocked(waiter.mSql, waiter.mConnectionFlags)) == null) {
                        nonPrimaryConnectionNotAvailable = true;
                    }
                    if (connection == null && !primaryConnectionNotAvailable && (connection = this.tryAcquirePrimaryConnectionLocked(waiter.mConnectionFlags)) == null) {
                        primaryConnectionNotAvailable = true;
                    }
                    if (connection != null) {
                        waiter.mAssignedConnection = connection;
                        unpark = true;
                    } else if (nonPrimaryConnectionNotAvailable && primaryConnectionNotAvailable) {
                        break;
                    }
                }
                catch (RuntimeException ex) {
                    waiter.mException = ex;
                    unpark = true;
                }
            }
            ConnectionWaiter successor = waiter.mNext;
            if (unpark) {
                if (predecessor != null) {
                    predecessor.mNext = successor;
                } else {
                    this.mConnectionWaiterQueue = successor;
                }
                waiter.mNext = null;
                LockSupport.unpark(waiter.mThread);
            } else {
                predecessor = waiter;
            }
            waiter = successor;
        }
    }

    @GuardedBy(value={"mLock"})
    private SQLiteConnection tryAcquirePrimaryConnectionLocked(int connectionFlags) {
        SQLiteConnection connection = this.mAvailablePrimaryConnection;
        if (connection != null) {
            this.mAvailablePrimaryConnection = null;
            this.finishAcquireConnectionLocked(connection, connectionFlags);
            return connection;
        }
        for (SQLiteConnection acquiredConnection : this.mAcquiredConnections.keySet()) {
            if (!acquiredConnection.isPrimaryConnection()) continue;
            return null;
        }
        connection = this.openConnectionLocked(this.mConfiguration, true);
        this.finishAcquireConnectionLocked(connection, connectionFlags);
        return connection;
    }

    @GuardedBy(value={"mLock"})
    private SQLiteConnection tryAcquireNonPrimaryConnectionLocked(String sql, int connectionFlags) {
        SQLiteConnection connection;
        int availableCount = this.mAvailableNonPrimaryConnections.size();
        if (availableCount > 1 && sql != null) {
            for (int i = 0; i < availableCount; ++i) {
                connection = this.mAvailableNonPrimaryConnections.get(i);
                if (!connection.isPreparedStatementInCache(sql)) continue;
                this.mAvailableNonPrimaryConnections.remove(i);
                this.finishAcquireConnectionLocked(connection, connectionFlags);
                return connection;
            }
        }
        if (availableCount > 0) {
            connection = this.mAvailableNonPrimaryConnections.remove(availableCount - 1);
            this.finishAcquireConnectionLocked(connection, connectionFlags);
            return connection;
        }
        int openConnections = this.mAcquiredConnections.size();
        if (this.mAvailablePrimaryConnection != null) {
            ++openConnections;
        }
        if (openConnections >= this.mMaxConnectionPoolSize) {
            return null;
        }
        connection = this.openConnectionLocked(this.mConfiguration, false);
        this.finishAcquireConnectionLocked(connection, connectionFlags);
        return connection;
    }

    @GuardedBy(value={"mLock"})
    private void finishAcquireConnectionLocked(SQLiteConnection connection, int connectionFlags) {
        try {
            boolean readOnly = (connectionFlags & 1) != 0;
            connection.setOnlyAllowReadOnlyOperations(readOnly);
            this.mAcquiredConnections.put(connection, AcquiredConnectionStatus.NORMAL);
        }
        catch (RuntimeException ex) {
            Log.e(TAG, "Failed to prepare acquired connection for session, closing it: " + connection + ", connectionFlags=" + connectionFlags);
            this.closeConnectionAndLogExceptionsLocked(connection);
            throw ex;
        }
    }

    private boolean isSessionBlockingImportantConnectionWaitersLocked(boolean holdingPrimaryConnection, int connectionFlags) {
        ConnectionWaiter waiter = this.mConnectionWaiterQueue;
        if (waiter != null) {
            int priority = SQLiteConnectionPool.getPriority(connectionFlags);
            while (priority <= waiter.mPriority) {
                if (holdingPrimaryConnection || !waiter.mWantPrimaryConnection) {
                    return true;
                }
                waiter = waiter.mNext;
                if (waiter != null) continue;
            }
        }
        return false;
    }

    private static int getPriority(int connectionFlags) {
        return (connectionFlags & 4) != 0 ? 1 : 0;
    }

    private void setMaxConnectionPoolSizeLocked() {
        this.mMaxConnectionPoolSize = !this.mConfiguration.isInMemoryDb() && (this.mConfiguration.openFlags & 0x20000000) != 0 ? SQLiteGlobal.getWALConnectionPoolSize() : 1;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    public void setupIdleConnectionHandler(Looper looper, long timeoutMs) {
        Object object = this.mLock;
        synchronized (object) {
            this.mIdleConnectionHandler = new IdleConnectionHandler(looper, timeoutMs);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void disableIdleConnectionHandler() {
        Object object = this.mLock;
        synchronized (object) {
            this.mIdleConnectionHandler = null;
        }
    }

    private void throwIfClosedLocked() {
        if (!this.mIsOpen) {
            throw new IllegalStateException("Cannot perform this operation because the connection pool has been closed.");
        }
    }

    private ConnectionWaiter obtainConnectionWaiterLocked(Thread thread, long startTime, int priority, boolean wantPrimaryConnection, String sql, int connectionFlags) {
        ConnectionWaiter waiter = this.mConnectionWaiterPool;
        if (waiter != null) {
            this.mConnectionWaiterPool = waiter.mNext;
            waiter.mNext = null;
        } else {
            waiter = new ConnectionWaiter();
        }
        waiter.mThread = thread;
        waiter.mStartTime = startTime;
        waiter.mPriority = priority;
        waiter.mWantPrimaryConnection = wantPrimaryConnection;
        waiter.mSql = sql;
        waiter.mConnectionFlags = connectionFlags;
        return waiter;
    }

    private void recycleConnectionWaiterLocked(ConnectionWaiter waiter) {
        waiter.mNext = this.mConnectionWaiterPool;
        waiter.mThread = null;
        waiter.mSql = null;
        waiter.mAssignedConnection = null;
        waiter.mException = null;
        ++waiter.mNonce;
        this.mConnectionWaiterPool = waiter;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void dump(Printer printer, boolean verbose, ArraySet<String> directories) {
        Printer indentedPrinter = PrefixPrinter.create(printer, "    ");
        Object object = this.mLock;
        synchronized (object) {
            if (directories != null) {
                directories.add(new File(this.mConfiguration.path).getParent());
            }
            boolean isCompatibilityWalEnabled = this.mConfiguration.isLegacyCompatibilityWalEnabled();
            printer.println("Connection pool for " + this.mConfiguration.path + ":");
            printer.println("  Open: " + this.mIsOpen);
            printer.println("  Max connections: " + this.mMaxConnectionPoolSize);
            printer.println("  Total execution time: " + this.mTotalExecutionTimeCounter);
            printer.println("  Configuration: openFlags=" + this.mConfiguration.openFlags + ", isLegacyCompatibilityWalEnabled=" + isCompatibilityWalEnabled + ", journalMode=" + TextUtils.emptyIfNull(this.mConfiguration.journalMode) + ", syncMode=" + TextUtils.emptyIfNull(this.mConfiguration.syncMode));
            if (isCompatibilityWalEnabled) {
                printer.println("  Compatibility WAL enabled: wal_syncmode=" + SQLiteCompatibilityWalFlags.getWALSyncMode());
            }
            if (this.mConfiguration.isLookasideConfigSet()) {
                printer.println("  Lookaside config: sz=" + this.mConfiguration.lookasideSlotSize + " cnt=" + this.mConfiguration.lookasideSlotCount);
            }
            if (this.mConfiguration.idleConnectionTimeoutMs != Long.MAX_VALUE) {
                printer.println("  Idle connection timeout: " + this.mConfiguration.idleConnectionTimeoutMs);
            }
            printer.println("  Available primary connection:");
            if (this.mAvailablePrimaryConnection != null) {
                this.mAvailablePrimaryConnection.dump(indentedPrinter, verbose);
            } else {
                indentedPrinter.println("<none>");
            }
            printer.println("  Available non-primary connections:");
            if (!this.mAvailableNonPrimaryConnections.isEmpty()) {
                int count = this.mAvailableNonPrimaryConnections.size();
                for (int i = 0; i < count; ++i) {
                    this.mAvailableNonPrimaryConnections.get(i).dump(indentedPrinter, verbose);
                }
            } else {
                indentedPrinter.println("<none>");
            }
            printer.println("  Acquired connections:");
            if (!this.mAcquiredConnections.isEmpty()) {
                for (Map.Entry<SQLiteConnection, AcquiredConnectionStatus> entry : this.mAcquiredConnections.entrySet()) {
                    SQLiteConnection connection = entry.getKey();
                    connection.dumpUnsafe(indentedPrinter, verbose);
                    indentedPrinter.println("  Status: " + (Object)((Object)entry.getValue()));
                }
            } else {
                indentedPrinter.println("<none>");
            }
            printer.println("  Connection waiters:");
            if (this.mConnectionWaiterQueue != null) {
                int i = 0;
                long now = SystemClock.uptimeMillis();
                ConnectionWaiter waiter = this.mConnectionWaiterQueue;
                while (waiter != null) {
                    indentedPrinter.println(i + ": waited for " + (float)(now - waiter.mStartTime) * 0.001f + " ms - thread=" + waiter.mThread + ", priority=" + waiter.mPriority + ", sql='" + waiter.mSql + "'");
                    waiter = waiter.mNext;
                    ++i;
                }
            } else {
                indentedPrinter.println("<none>");
            }
        }
    }

    public String toString() {
        return "SQLiteConnectionPool: " + this.mConfiguration.path;
    }

    public String getPath() {
        return this.mConfiguration.path;
    }

    private class IdleConnectionHandler
    extends Handler {
        private final long mTimeout;

        IdleConnectionHandler(Looper looper, long timeout) {
            super(looper);
            this.mTimeout = timeout;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void handleMessage(Message msg) {
            Object object = SQLiteConnectionPool.this.mLock;
            synchronized (object) {
                if (this != SQLiteConnectionPool.this.mIdleConnectionHandler) {
                    return;
                }
                if (SQLiteConnectionPool.this.closeAvailableConnectionLocked(msg.what) && Log.isLoggable(SQLiteConnectionPool.TAG, 3)) {
                    Log.d(SQLiteConnectionPool.TAG, "Closed idle connection " + ((SQLiteConnectionPool)SQLiteConnectionPool.this).mConfiguration.label + " " + msg.what + " after " + this.mTimeout);
                }
            }
        }

        void connectionReleased(SQLiteConnection con) {
            this.sendEmptyMessageDelayed(con.getConnectionId(), this.mTimeout);
        }

        void connectionAcquired(SQLiteConnection con) {
            this.removeMessages(con.getConnectionId());
        }

        void connectionClosed(SQLiteConnection con) {
            this.removeMessages(con.getConnectionId());
        }
    }

    private static final class ConnectionWaiter {
        public ConnectionWaiter mNext;
        public Thread mThread;
        public long mStartTime;
        public int mPriority;
        public boolean mWantPrimaryConnection;
        public String mSql;
        public int mConnectionFlags;
        public SQLiteConnection mAssignedConnection;
        public RuntimeException mException;
        public int mNonce;

        private ConnectionWaiter() {
        }
    }

    static enum AcquiredConnectionStatus {
        NORMAL,
        RECONFIGURE,
        DISCARD;

    }
}

