/*
 * Decompiled with CFR 0.152.
 */
package com.android.internal.os;

import android.os.Process;
import android.util.Slog;
import com.android.internal.annotations.VisibleForTesting;
import com.android.internal.os.ProcStatsUtil;
import com.android.internal.os.ProcTimeInStateReader;
import com.android.internal.util.ArrayUtils;
import com.android.internal.util.Preconditions;
import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.function.Predicate;

public class KernelCpuThreadReader {
    private static final String TAG = "KernelCpuThreadReader";
    private static final boolean DEBUG = false;
    private static final String CPU_STATISTICS_FILENAME = "time_in_state";
    private static final String PROCESS_NAME_FILENAME = "cmdline";
    private static final String THREAD_NAME_FILENAME = "comm";
    private static final String PROCESS_DIRECTORY_FILTER = "[0-9]*";
    private static final String DEFAULT_PROCESS_NAME = "unknown_process";
    private static final String DEFAULT_THREAD_NAME = "unknown_thread";
    private static final Path DEFAULT_PROC_PATH = Paths.get("/proc", new String[0]);
    private static final Path DEFAULT_INITIAL_TIME_IN_STATE_PATH = DEFAULT_PROC_PATH.resolve("self/time_in_state");
    private static final int ID_ERROR = -1;
    private Predicate<Integer> mUidPredicate;
    private final Path mProcPath;
    private int[] mFrequenciesKhz;
    private final ProcTimeInStateReader mProcTimeInStateReader;
    private FrequencyBucketCreator mFrequencyBucketCreator;
    private final Injector mInjector;

    @VisibleForTesting
    public KernelCpuThreadReader(int numBuckets, Predicate<Integer> uidPredicate, Path procPath, Path initialTimeInStatePath, Injector injector) throws IOException {
        this.mUidPredicate = uidPredicate;
        this.mProcPath = procPath;
        this.mProcTimeInStateReader = new ProcTimeInStateReader(initialTimeInStatePath);
        this.mInjector = injector;
        this.setNumBuckets(numBuckets);
    }

    public static KernelCpuThreadReader create(int numBuckets, Predicate<Integer> uidPredicate) {
        try {
            return new KernelCpuThreadReader(numBuckets, uidPredicate, DEFAULT_PROC_PATH, DEFAULT_INITIAL_TIME_IN_STATE_PATH, new Injector());
        }
        catch (IOException e) {
            Slog.e(TAG, "Failed to initialize KernelCpuThreadReader", e);
            return null;
        }
    }

    public ArrayList<ProcessCpuUsage> getProcessCpuUsage() {
        ArrayList<ProcessCpuUsage> processCpuUsages = new ArrayList<ProcessCpuUsage>();
        try (DirectoryStream<Path> processPaths = Files.newDirectoryStream(this.mProcPath, PROCESS_DIRECTORY_FILTER);){
            for (Path processPath : processPaths) {
                ProcessCpuUsage processCpuUsage;
                int processId = this.getProcessId(processPath);
                int uid = this.mInjector.getUidForPid(processId);
                if (uid == -1 || processId == -1 || !this.mUidPredicate.test(uid) || (processCpuUsage = this.getProcessCpuUsage(processPath, processId, uid)) == null) continue;
                processCpuUsages.add(processCpuUsage);
            }
        }
        catch (IOException e) {
            Slog.w(TAG, "Failed to iterate over process paths", e);
            return null;
        }
        if (processCpuUsages.isEmpty()) {
            Slog.w(TAG, "Didn't successfully get any process CPU information for UIDs specified");
            return null;
        }
        return processCpuUsages;
    }

    public int[] getCpuFrequenciesKhz() {
        return this.mFrequenciesKhz;
    }

    void setNumBuckets(int numBuckets) {
        if (numBuckets < 1) {
            Slog.w(TAG, "Number of buckets must be at least 1, but was " + numBuckets);
            return;
        }
        if (this.mFrequenciesKhz != null && this.mFrequenciesKhz.length == numBuckets) {
            return;
        }
        this.mFrequencyBucketCreator = new FrequencyBucketCreator(this.mProcTimeInStateReader.getFrequenciesKhz(), numBuckets);
        this.mFrequenciesKhz = this.mFrequencyBucketCreator.bucketFrequencies(this.mProcTimeInStateReader.getFrequenciesKhz());
    }

    void setUidPredicate(Predicate<Integer> uidPredicate) {
        this.mUidPredicate = uidPredicate;
    }

    private ProcessCpuUsage getProcessCpuUsage(Path processPath, int processId, int uid) {
        Path allThreadsPath = processPath.resolve("task");
        ArrayList<ThreadCpuUsage> threadCpuUsages = new ArrayList<ThreadCpuUsage>();
        try (DirectoryStream<Path> threadPaths = Files.newDirectoryStream(allThreadsPath);){
            for (Path threadDirectory : threadPaths) {
                ThreadCpuUsage threadCpuUsage = this.getThreadCpuUsage(threadDirectory);
                if (threadCpuUsage == null) continue;
                threadCpuUsages.add(threadCpuUsage);
            }
        }
        catch (IOException e) {
            return null;
        }
        if (threadCpuUsages.isEmpty()) {
            return null;
        }
        return new ProcessCpuUsage(processId, this.getProcessName(processPath), uid, threadCpuUsages);
    }

    private ThreadCpuUsage getThreadCpuUsage(Path threadDirectory) {
        int threadId;
        try {
            String directoryName = threadDirectory.getFileName().toString();
            threadId = Integer.parseInt(directoryName);
        }
        catch (NumberFormatException e) {
            Slog.w(TAG, "Failed to parse thread ID when iterating over /proc/*/task", e);
            return null;
        }
        String threadName = this.getThreadName(threadDirectory);
        Path threadCpuStatPath = threadDirectory.resolve(CPU_STATISTICS_FILENAME);
        long[] cpuUsagesLong = this.mProcTimeInStateReader.getUsageTimesMillis(threadCpuStatPath);
        if (cpuUsagesLong == null) {
            return null;
        }
        int[] cpuUsages = this.mFrequencyBucketCreator.bucketValues(cpuUsagesLong);
        return new ThreadCpuUsage(threadId, threadName, cpuUsages);
    }

    private String getProcessName(Path processPath) {
        Path processNamePath = processPath.resolve(PROCESS_NAME_FILENAME);
        String processName = ProcStatsUtil.readSingleLineProcFile(processNamePath.toString());
        if (processName != null) {
            return processName;
        }
        return DEFAULT_PROCESS_NAME;
    }

    private String getThreadName(Path threadPath) {
        Path threadNamePath = threadPath.resolve(THREAD_NAME_FILENAME);
        String threadName = ProcStatsUtil.readNullSeparatedFile(threadNamePath.toString());
        if (threadName == null) {
            return DEFAULT_THREAD_NAME;
        }
        return threadName;
    }

    private int getProcessId(Path processPath) {
        String fileName = processPath.getFileName().toString();
        try {
            return Integer.parseInt(fileName);
        }
        catch (NumberFormatException e) {
            Slog.w(TAG, "Failed to parse " + fileName + " as process ID", e);
            return -1;
        }
    }

    @VisibleForTesting
    public static class Injector {
        public int getUidForPid(int pid) {
            return Process.getUidForPid(pid);
        }
    }

    public static class ThreadCpuUsage {
        public final int threadId;
        public final String threadName;
        public int[] usageTimesMillis;

        @VisibleForTesting
        public ThreadCpuUsage(int threadId, String threadName, int[] usageTimesMillis) {
            this.threadId = threadId;
            this.threadName = threadName;
            this.usageTimesMillis = usageTimesMillis;
        }
    }

    public static class ProcessCpuUsage {
        public final int processId;
        public final String processName;
        public final int uid;
        public ArrayList<ThreadCpuUsage> threadCpuUsages;

        @VisibleForTesting
        public ProcessCpuUsage(int processId, String processName, int uid, ArrayList<ThreadCpuUsage> threadCpuUsages) {
            this.processId = processId;
            this.processName = processName;
            this.uid = uid;
            this.threadCpuUsages = threadCpuUsages;
        }
    }

    @VisibleForTesting
    public static class FrequencyBucketCreator {
        private final int mNumFrequencies;
        private final int mNumBuckets;
        private final int[] mBucketStartIndices;

        @VisibleForTesting
        public FrequencyBucketCreator(long[] frequencies, int targetNumBuckets) {
            this.mNumFrequencies = frequencies.length;
            int[] clusterStartIndices = FrequencyBucketCreator.getClusterStartIndices(frequencies);
            this.mBucketStartIndices = FrequencyBucketCreator.getBucketStartIndices(clusterStartIndices, targetNumBuckets, this.mNumFrequencies);
            this.mNumBuckets = this.mBucketStartIndices.length;
        }

        @VisibleForTesting
        public int[] bucketValues(long[] values) {
            Preconditions.checkArgument(values.length == this.mNumFrequencies);
            int[] buckets = new int[this.mNumBuckets];
            for (int bucketIdx = 0; bucketIdx < this.mNumBuckets; ++bucketIdx) {
                int bucketStartIdx = FrequencyBucketCreator.getLowerBound(bucketIdx, this.mBucketStartIndices);
                int bucketEndIdx = FrequencyBucketCreator.getUpperBound(bucketIdx, this.mBucketStartIndices, values.length);
                for (int valuesIdx = bucketStartIdx; valuesIdx < bucketEndIdx; ++valuesIdx) {
                    int n = bucketIdx;
                    buckets[n] = (int)((long)buckets[n] + values[valuesIdx]);
                }
            }
            return buckets;
        }

        @VisibleForTesting
        public int[] bucketFrequencies(long[] frequencies) {
            Preconditions.checkArgument(frequencies.length == this.mNumFrequencies);
            int[] buckets = new int[this.mNumBuckets];
            for (int i = 0; i < buckets.length; ++i) {
                buckets[i] = (int)frequencies[this.mBucketStartIndices[i]];
            }
            return buckets;
        }

        private static int[] getClusterStartIndices(long[] frequencies) {
            ArrayList<Integer> indices = new ArrayList<Integer>();
            indices.add(0);
            for (int i = 0; i < frequencies.length - 1; ++i) {
                if (frequencies[i] < frequencies[i + 1]) continue;
                indices.add(i + 1);
            }
            return ArrayUtils.convertToIntArray(indices);
        }

        private static int[] getBucketStartIndices(int[] clusterStartIndices, int targetNumBuckets, int numFrequencies) {
            int numClusters = clusterStartIndices.length;
            if (numClusters > targetNumBuckets) {
                return Arrays.copyOfRange(clusterStartIndices, 0, targetNumBuckets);
            }
            ArrayList<Integer> bucketStartIndices = new ArrayList<Integer>();
            for (int clusterIdx = 0; clusterIdx < numClusters; ++clusterIdx) {
                int bucketStartIdx;
                int numBucketsInCluster;
                int clusterStartIdx = FrequencyBucketCreator.getLowerBound(clusterIdx, clusterStartIndices);
                int clusterEndIdx = FrequencyBucketCreator.getUpperBound(clusterIdx, clusterStartIndices, numFrequencies);
                if (clusterIdx != numClusters - 1) {
                    numBucketsInCluster = targetNumBuckets / numClusters;
                } else {
                    int previousBucketsInCluster = targetNumBuckets / numClusters;
                    numBucketsInCluster = targetNumBuckets - previousBucketsInCluster * (numClusters - 1);
                }
                int numFrequenciesInCluster = clusterEndIdx - clusterStartIdx;
                int numFrequenciesInBucket = Math.max(1, numFrequenciesInCluster / numBucketsInCluster);
                for (int bucketIdx = 0; bucketIdx < numBucketsInCluster && (bucketStartIdx = clusterStartIdx + bucketIdx * numFrequenciesInBucket) < clusterEndIdx; ++bucketIdx) {
                    bucketStartIndices.add(bucketStartIdx);
                }
            }
            return ArrayUtils.convertToIntArray(bucketStartIndices);
        }

        private static int getLowerBound(int index, int[] startIndices) {
            return startIndices[index];
        }

        private static int getUpperBound(int index, int[] startIndices, int max) {
            if (index != startIndices.length - 1) {
                return startIndices[index + 1];
            }
            return max;
        }
    }
}

