/*
 * Decompiled with CFR 0.152.
 */
package com.android.server;

import android.content.ContentResolver;
import android.content.Context;
import android.os.Build;
import android.os.Environment;
import android.os.FileUtils;
import android.os.RecoverySystem;
import android.os.SystemClock;
import android.os.SystemProperties;
import android.provider.Settings;
import android.util.ExceptionUtils;
import android.util.MathUtils;
import android.util.Slog;
import android.util.SparseArray;
import android.util.StatsLog;
import com.android.internal.annotations.VisibleForTesting;
import com.android.internal.util.ArrayUtils;
import com.android.server.EventLogTags;
import com.android.server.am.SettingsToPropertiesMapper;
import com.android.server.pm.PackageManagerServiceUtils;
import com.android.server.utils.FlagNamespaceUtils;
import java.io.File;
import java.util.Arrays;

public class RescueParty {
    @VisibleForTesting
    static final String PROP_ENABLE_RESCUE = "persist.sys.enable_rescue";
    @VisibleForTesting
    static final int TRIGGER_COUNT = 5;
    @VisibleForTesting
    static final String PROP_RESCUE_LEVEL = "sys.rescue_level";
    @VisibleForTesting
    static final int LEVEL_NONE = 0;
    @VisibleForTesting
    static final int LEVEL_RESET_SETTINGS_UNTRUSTED_DEFAULTS = 1;
    @VisibleForTesting
    static final int LEVEL_RESET_SETTINGS_UNTRUSTED_CHANGES = 2;
    @VisibleForTesting
    static final int LEVEL_RESET_SETTINGS_TRUSTED_DEFAULTS = 3;
    @VisibleForTesting
    static final int LEVEL_FACTORY_RESET = 4;
    @VisibleForTesting
    static final String PROP_RESCUE_BOOT_COUNT = "sys.rescue_boot_count";
    @VisibleForTesting
    static final long BOOT_TRIGGER_WINDOW_MILLIS = 600000L;
    @VisibleForTesting
    static final long PERSISTENT_APP_CRASH_TRIGGER_WINDOW_MILLIS = 30000L;
    @VisibleForTesting
    static final String TAG = "RescueParty";
    private static final String PROP_DISABLE_RESCUE = "persist.sys.disable_rescue";
    private static final String PROP_RESCUE_BOOT_START = "sys.rescue_boot_start";
    private static final String PROP_VIRTUAL_DEVICE = "ro.hardware.virtual_device";
    private static final Threshold sBoot = new BootThreshold();
    private static SparseArray<Threshold> sApps = new SparseArray();

    private static boolean isDisabled() {
        if (SystemProperties.getBoolean(PROP_ENABLE_RESCUE, false)) {
            return false;
        }
        if (Build.IS_ENG) {
            Slog.v(TAG, "Disabled because of eng build");
            return true;
        }
        if (Build.IS_USERDEBUG && RescueParty.isUsbActive()) {
            Slog.v(TAG, "Disabled because of active USB connection");
            return true;
        }
        if (SystemProperties.getBoolean(PROP_DISABLE_RESCUE, false)) {
            Slog.v(TAG, "Disabled because of manual property");
            return true;
        }
        return false;
    }

    public static void noteBoot(Context context) {
        if (RescueParty.isDisabled()) {
            return;
        }
        if (sBoot.incrementAndTest()) {
            sBoot.reset();
            RescueParty.incrementRescueLevel(sBoot.uid);
            RescueParty.executeRescueLevel(context);
        }
    }

    public static void noteAppCrash(Context context, int uid) {
        if (RescueParty.isDisabled()) {
            return;
        }
        Threshold t = sApps.get(uid);
        if (t == null) {
            t = new AppThreshold(uid);
            sApps.put(uid, t);
        }
        if (t.incrementAndTest()) {
            t.reset();
            RescueParty.incrementRescueLevel(t.uid);
            RescueParty.executeRescueLevel(context);
        }
    }

    public static boolean isAttemptingFactoryReset() {
        return SystemProperties.getInt(PROP_RESCUE_LEVEL, 0) == 4;
    }

    public static void onSettingsProviderPublished(Context context) {
        RescueParty.handleNativeRescuePartyResets();
        RescueParty.executeRescueLevel(context);
    }

    @VisibleForTesting
    static void resetAllThresholds() {
        sBoot.reset();
        for (int i = 0; i < sApps.size(); ++i) {
            Threshold appThreshold = sApps.get(sApps.keyAt(i));
            appThreshold.reset();
        }
    }

    @VisibleForTesting
    static long getElapsedRealtime() {
        return SystemClock.elapsedRealtime();
    }

    private static void handleNativeRescuePartyResets() {
        if (SettingsToPropertiesMapper.isNativeFlagsResetPerformed()) {
            FlagNamespaceUtils.resetDeviceConfig(4, Arrays.asList(SettingsToPropertiesMapper.getResetNativeCategories()));
        }
    }

    private static void incrementRescueLevel(int triggerUid) {
        int level = MathUtils.constrain(SystemProperties.getInt(PROP_RESCUE_LEVEL, 0) + 1, 0, 4);
        SystemProperties.set(PROP_RESCUE_LEVEL, Integer.toString(level));
        EventLogTags.writeRescueLevel(level, triggerUid);
        PackageManagerServiceUtils.logCriticalInfo(5, "Incremented rescue level to " + RescueParty.levelToString(level) + " triggered by UID " + triggerUid);
    }

    private static void executeRescueLevel(Context context) {
        int level = SystemProperties.getInt(PROP_RESCUE_LEVEL, 0);
        if (level == 0) {
            return;
        }
        Slog.w(TAG, "Attempting rescue level " + RescueParty.levelToString(level));
        try {
            RescueParty.executeRescueLevelInternal(context, level);
            EventLogTags.writeRescueSuccess(level);
            PackageManagerServiceUtils.logCriticalInfo(3, "Finished rescue level " + RescueParty.levelToString(level));
        }
        catch (Throwable t) {
            String msg = ExceptionUtils.getCompleteMessage(t);
            EventLogTags.writeRescueFailure(level, msg);
            PackageManagerServiceUtils.logCriticalInfo(6, "Failed rescue level " + RescueParty.levelToString(level) + ": " + msg);
        }
    }

    private static void executeRescueLevelInternal(Context context, int level) throws Exception {
        StatsLog.write(122, level);
        switch (level) {
            case 1: {
                RescueParty.resetAllSettings(context, 2);
                break;
            }
            case 2: {
                RescueParty.resetAllSettings(context, 3);
                break;
            }
            case 3: {
                RescueParty.resetAllSettings(context, 4);
                break;
            }
            case 4: {
                RecoverySystem.rebootPromptAndWipeUserData(context, TAG);
            }
        }
        FlagNamespaceUtils.addToKnownResetNamespaces("no_package");
    }

    private static void resetAllSettings(Context context, int mode) throws Exception {
        RuntimeException res = null;
        ContentResolver resolver = context.getContentResolver();
        try {
            FlagNamespaceUtils.resetDeviceConfig(mode);
        }
        catch (Exception e) {
            res = new RuntimeException("Failed to reset config settings", e);
        }
        try {
            Settings.Global.resetToDefaultsAsUser(resolver, null, mode, 0);
        }
        catch (Exception e) {
            res = new RuntimeException("Failed to reset global settings", e);
        }
        for (int userId : RescueParty.getAllUserIds()) {
            try {
                Settings.Secure.resetToDefaultsAsUser(resolver, null, mode, userId);
            }
            catch (Exception e) {
                res = new RuntimeException("Failed to reset secure settings for " + userId, e);
            }
        }
        if (res != null) {
            throw res;
        }
    }

    private static int[] getAllUserIds() {
        int[] userIds = new int[]{0};
        try {
            for (File file : FileUtils.listFilesOrEmpty(Environment.getDataSystemDeDirectory())) {
                try {
                    int userId = Integer.parseInt(file.getName());
                    if (userId == 0) continue;
                    userIds = ArrayUtils.appendInt(userIds, userId);
                }
                catch (NumberFormatException numberFormatException) {
                    // empty catch block
                }
            }
        }
        catch (Throwable t) {
            Slog.w(TAG, "Trouble discovering users", t);
        }
        return userIds;
    }

    private static boolean isUsbActive() {
        if (SystemProperties.getBoolean(PROP_VIRTUAL_DEVICE, false)) {
            Slog.v(TAG, "Assuming virtual device is connected over USB");
            return true;
        }
        try {
            String state = FileUtils.readTextFile(new File("/sys/class/android_usb/android0/state"), 128, "");
            return "CONFIGURED".equals(state.trim());
        }
        catch (Throwable t) {
            Slog.w(TAG, "Failed to determine if device was on USB", t);
            return false;
        }
    }

    private static String levelToString(int level) {
        switch (level) {
            case 0: {
                return "NONE";
            }
            case 1: {
                return "RESET_SETTINGS_UNTRUSTED_DEFAULTS";
            }
            case 2: {
                return "RESET_SETTINGS_UNTRUSTED_CHANGES";
            }
            case 3: {
                return "RESET_SETTINGS_TRUSTED_DEFAULTS";
            }
            case 4: {
                return "FACTORY_RESET";
            }
        }
        return Integer.toString(level);
    }

    private static class AppThreshold
    extends Threshold {
        private int count;
        private long start;

        public AppThreshold(int uid) {
            super(uid, 5, 30000L);
        }

        @Override
        public int getCount() {
            return this.count;
        }

        @Override
        public void setCount(int count) {
            this.count = count;
        }

        @Override
        public long getStart() {
            return this.start;
        }

        @Override
        public void setStart(long start) {
            this.start = start;
        }
    }

    private static class BootThreshold
    extends Threshold {
        public BootThreshold() {
            super(0, 5, 600000L);
        }

        @Override
        public int getCount() {
            return SystemProperties.getInt(RescueParty.PROP_RESCUE_BOOT_COUNT, 0);
        }

        @Override
        public void setCount(int count) {
            SystemProperties.set(RescueParty.PROP_RESCUE_BOOT_COUNT, Integer.toString(count));
        }

        @Override
        public long getStart() {
            return SystemProperties.getLong(RescueParty.PROP_RESCUE_BOOT_START, 0L);
        }

        @Override
        public void setStart(long start) {
            SystemProperties.set(RescueParty.PROP_RESCUE_BOOT_START, Long.toString(start));
        }
    }

    private static abstract class Threshold {
        private final int uid;
        private final int triggerCount;
        private final long triggerWindow;

        public abstract int getCount();

        public abstract void setCount(int var1);

        public abstract long getStart();

        public abstract void setStart(long var1);

        public Threshold(int uid, int triggerCount, long triggerWindow) {
            this.uid = uid;
            this.triggerCount = triggerCount;
            this.triggerWindow = triggerWindow;
        }

        public void reset() {
            this.setCount(0);
            this.setStart(0L);
        }

        public boolean incrementAndTest() {
            long now = RescueParty.getElapsedRealtime();
            long window = now - this.getStart();
            if (window > this.triggerWindow) {
                this.setCount(1);
                this.setStart(now);
                return false;
            }
            int count = this.getCount() + 1;
            this.setCount(count);
            EventLogTags.writeRescueNote(this.uid, count, window);
            Slog.w(RescueParty.TAG, "Noticed " + count + " events for UID " + this.uid + " in last " + window / 1000L + " sec");
            return count >= this.triggerCount;
        }
    }
}

