/*
 * Decompiled with CFR 0.152.
 */
package com.android.server.connectivity;

import android.app.usage.NetworkStatsManager;
import android.content.BroadcastReceiver;
import android.content.ContentResolver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.database.ContentObserver;
import android.net.ConnectivityManager;
import android.net.Network;
import android.net.NetworkCapabilities;
import android.net.NetworkIdentity;
import android.net.NetworkPolicy;
import android.net.NetworkPolicyManager;
import android.net.NetworkRequest;
import android.net.NetworkTemplate;
import android.net.StringNetworkSpecifier;
import android.net.Uri;
import android.os.BestClock;
import android.os.Handler;
import android.os.SystemClock;
import android.os.UserHandle;
import android.provider.Settings;
import android.telephony.TelephonyManager;
import android.util.DebugUtils;
import android.util.Range;
import android.util.Slog;
import com.android.internal.annotations.VisibleForTesting;
import com.android.internal.util.IndentingPrintWriter;
import com.android.server.LocalServices;
import com.android.server.net.NetworkPolicyManagerInternal;
import com.android.server.net.NetworkStatsManagerInternal;
import java.time.Clock;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

public class MultipathPolicyTracker {
    private static String TAG = MultipathPolicyTracker.class.getSimpleName();
    private static final boolean DBG = false;
    private final Context mContext;
    private final Handler mHandler;
    private final Clock mClock;
    private final Dependencies mDeps;
    private final ContentResolver mResolver;
    private final ConfigChangeReceiver mConfigChangeReceiver;
    @VisibleForTesting
    final ContentObserver mSettingsObserver;
    private ConnectivityManager mCM;
    private NetworkPolicyManager mNPM;
    private NetworkStatsManager mStatsManager;
    private ConnectivityManager.NetworkCallback mMobileNetworkCallback;
    private NetworkPolicyManager.Listener mPolicyListener;
    private static final int OPQUOTA_USER_SETTING_DIVIDER = 20;
    private final ConcurrentHashMap<Network, MultipathTracker> mMultipathTrackers = new ConcurrentHashMap();

    public MultipathPolicyTracker(Context ctx, Handler handler) {
        this(ctx, handler, new Dependencies());
    }

    public MultipathPolicyTracker(Context ctx, Handler handler, Dependencies deps) {
        this.mContext = ctx;
        this.mHandler = handler;
        this.mClock = deps.getClock();
        this.mDeps = deps;
        this.mResolver = this.mContext.getContentResolver();
        this.mSettingsObserver = new SettingsObserver(this.mHandler);
        this.mConfigChangeReceiver = new ConfigChangeReceiver();
    }

    public void start() {
        this.mCM = this.mContext.getSystemService(ConnectivityManager.class);
        this.mNPM = this.mContext.getSystemService(NetworkPolicyManager.class);
        this.mStatsManager = this.mContext.getSystemService(NetworkStatsManager.class);
        this.registerTrackMobileCallback();
        this.registerNetworkPolicyListener();
        Uri defaultSettingUri = Settings.Global.getUriFor("network_default_daily_multipath_quota_bytes");
        this.mResolver.registerContentObserver(defaultSettingUri, false, this.mSettingsObserver);
        IntentFilter intentFilter = new IntentFilter();
        intentFilter.addAction("android.intent.action.CONFIGURATION_CHANGED");
        this.mContext.registerReceiverAsUser(this.mConfigChangeReceiver, UserHandle.ALL, intentFilter, null, this.mHandler);
    }

    public void shutdown() {
        this.maybeUnregisterTrackMobileCallback();
        this.unregisterNetworkPolicyListener();
        for (MultipathTracker t : this.mMultipathTrackers.values()) {
            t.shutdown();
        }
        this.mMultipathTrackers.clear();
        this.mResolver.unregisterContentObserver(this.mSettingsObserver);
        this.mContext.unregisterReceiver(this.mConfigChangeReceiver);
    }

    public Integer getMultipathPreference(Network network) {
        if (network == null) {
            return null;
        }
        MultipathTracker t = this.mMultipathTrackers.get(network);
        if (t != null) {
            return t.getMultipathPreference();
        }
        return null;
    }

    private static long getActiveWarning(NetworkPolicy policy, long cycleStart) {
        return policy.lastWarningSnooze < cycleStart ? policy.warningBytes : -1L;
    }

    private static long getActiveLimit(NetworkPolicy policy, long cycleStart) {
        return policy.lastLimitSnooze < cycleStart ? policy.limitBytes : -1L;
    }

    private long getDefaultDailyMultipathQuotaBytes() {
        String setting = Settings.Global.getString(this.mContext.getContentResolver(), "network_default_daily_multipath_quota_bytes");
        if (setting != null) {
            try {
                return Long.parseLong(setting);
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
        }
        return this.mContext.getResources().getInteger(17694850);
    }

    private void registerTrackMobileCallback() {
        NetworkRequest request = new NetworkRequest.Builder().addCapability(12).addTransportType(0).build();
        this.mMobileNetworkCallback = new ConnectivityManager.NetworkCallback(){

            @Override
            public void onCapabilitiesChanged(Network network, NetworkCapabilities nc) {
                MultipathTracker existing = (MultipathTracker)MultipathPolicyTracker.this.mMultipathTrackers.get(network);
                if (existing != null) {
                    existing.setNetworkCapabilities(nc);
                    existing.updateMultipathBudget();
                    return;
                }
                try {
                    MultipathPolicyTracker.this.mMultipathTrackers.put(network, new MultipathTracker(network, nc));
                }
                catch (IllegalStateException e) {
                    Slog.e(TAG, "Can't track mobile network " + network + ": " + e.getMessage());
                }
            }

            @Override
            public void onLost(Network network) {
                MultipathTracker existing = (MultipathTracker)MultipathPolicyTracker.this.mMultipathTrackers.get(network);
                if (existing != null) {
                    existing.shutdown();
                    MultipathPolicyTracker.this.mMultipathTrackers.remove(network);
                }
            }
        };
        this.mCM.registerNetworkCallback(request, this.mMobileNetworkCallback, this.mHandler);
    }

    private void updateAllMultipathBudgets() {
        for (MultipathTracker t : this.mMultipathTrackers.values()) {
            t.updateMultipathBudget();
        }
    }

    private void maybeUnregisterTrackMobileCallback() {
        if (this.mMobileNetworkCallback != null) {
            this.mCM.unregisterNetworkCallback(this.mMobileNetworkCallback);
        }
        this.mMobileNetworkCallback = null;
    }

    private void registerNetworkPolicyListener() {
        this.mPolicyListener = new NetworkPolicyManager.Listener(){

            @Override
            public void onMeteredIfacesChanged(String[] meteredIfaces) {
                MultipathPolicyTracker.this.mHandler.post(() -> MultipathPolicyTracker.this.updateAllMultipathBudgets());
            }
        };
        this.mNPM.registerListener(this.mPolicyListener);
    }

    private void unregisterNetworkPolicyListener() {
        this.mNPM.unregisterListener(this.mPolicyListener);
    }

    public void dump(IndentingPrintWriter pw) {
        pw.println("MultipathPolicyTracker:");
        pw.increaseIndent();
        for (MultipathTracker t : this.mMultipathTrackers.values()) {
            pw.println(String.format("Network %s: quota %d, budget %d. Preference: %s", t.network, t.getQuota(), t.getMultipathBudget(), DebugUtils.flagsToString(ConnectivityManager.class, "MULTIPATH_PREFERENCE_", t.getMultipathPreference())));
        }
        pw.decreaseIndent();
    }

    private final class ConfigChangeReceiver
    extends BroadcastReceiver {
        private ConfigChangeReceiver() {
        }

        @Override
        public void onReceive(Context context, Intent intent) {
            MultipathPolicyTracker.this.updateAllMultipathBudgets();
        }
    }

    private final class SettingsObserver
    extends ContentObserver {
        public SettingsObserver(Handler handler) {
            super(handler);
        }

        @Override
        public void onChange(boolean selfChange) {
            Slog.wtf(TAG, "Should never be reached.");
        }

        @Override
        public void onChange(boolean selfChange, Uri uri) {
            if (!Settings.Global.getUriFor("network_default_daily_multipath_quota_bytes").equals(uri)) {
                Slog.wtf(TAG, "Unexpected settings observation: " + uri);
            }
            MultipathPolicyTracker.this.updateAllMultipathBudgets();
        }
    }

    class MultipathTracker {
        final Network network;
        final int subId;
        final String subscriberId;
        private long mQuota;
        private long mMultipathBudget;
        private final NetworkTemplate mNetworkTemplate;
        private final NetworkStatsManager.UsageCallback mUsageCallback;
        private NetworkCapabilities mNetworkCapabilities;

        public MultipathTracker(final Network network, NetworkCapabilities nc) {
            this.network = network;
            this.mNetworkCapabilities = new NetworkCapabilities(nc);
            try {
                this.subId = Integer.parseInt(((StringNetworkSpecifier)nc.getNetworkSpecifier()).toString());
            }
            catch (ClassCastException | NullPointerException | NumberFormatException e) {
                throw new IllegalStateException(String.format("Can't get subId from mobile network %s (%s): %s", network, nc, e.getMessage()));
            }
            TelephonyManager tele = MultipathPolicyTracker.this.mContext.getSystemService(TelephonyManager.class);
            if (tele == null) {
                throw new IllegalStateException(String.format("Missing TelephonyManager", new Object[0]));
            }
            if ((tele = tele.createForSubscriptionId(this.subId)) == null) {
                throw new IllegalStateException(String.format("Can't get TelephonyManager for subId %d", this.subId));
            }
            this.subscriberId = tele.getSubscriberId();
            this.mNetworkTemplate = new NetworkTemplate(1, this.subscriberId, new String[]{this.subscriberId}, null, -1, -1, 0);
            this.mUsageCallback = new NetworkStatsManager.UsageCallback(){

                @Override
                public void onThresholdReached(int networkType, String subscriberId) {
                    MultipathTracker.this.mMultipathBudget = 0L;
                    MultipathTracker.this.updateMultipathBudget();
                }
            };
            this.updateMultipathBudget();
        }

        public void setNetworkCapabilities(NetworkCapabilities nc) {
            this.mNetworkCapabilities = new NetworkCapabilities(nc);
        }

        private long getDailyNonDefaultDataUsage() {
            ZonedDateTime end = ZonedDateTime.ofInstant(MultipathPolicyTracker.this.mClock.instant(), ZoneId.systemDefault());
            ZonedDateTime start = end.truncatedTo(ChronoUnit.DAYS);
            long bytes = this.getNetworkTotalBytes(start.toInstant().toEpochMilli(), end.toInstant().toEpochMilli());
            return bytes;
        }

        private long getNetworkTotalBytes(long start, long end) {
            try {
                return LocalServices.getService(NetworkStatsManagerInternal.class).getNetworkTotalBytes(this.mNetworkTemplate, start, end);
            }
            catch (RuntimeException e) {
                Slog.w(TAG, "Failed to get data usage: " + e);
                return -1L;
            }
        }

        private NetworkIdentity getTemplateMatchingNetworkIdentity(NetworkCapabilities nc) {
            return new NetworkIdentity(0, 0, this.subscriberId, null, !nc.hasCapability(18), !nc.hasCapability(11), false);
        }

        private long getRemainingDailyBudget(long limitBytes, Range<ZonedDateTime> cycle) {
            long end;
            long start = cycle.getLower().toInstant().toEpochMilli();
            long totalBytes = this.getNetworkTotalBytes(start, end = cycle.getUpper().toInstant().toEpochMilli());
            long remainingBytes = totalBytes == -1L ? 0L : Math.max(0L, limitBytes - totalBytes);
            long remainingDays = 1L + (end - MultipathPolicyTracker.this.mClock.millis() - 1L) / TimeUnit.DAYS.toMillis(1L);
            return remainingBytes / Math.max(1L, remainingDays);
        }

        private long getUserPolicyOpportunisticQuotaBytes() {
            NetworkPolicy[] policies;
            long minQuota = Long.MAX_VALUE;
            NetworkIdentity identity = this.getTemplateMatchingNetworkIdentity(this.mNetworkCapabilities);
            for (NetworkPolicy policy : policies = MultipathPolicyTracker.this.mNPM.getNetworkPolicies()) {
                long policyBytes;
                if (!policy.hasCycle() || !policy.template.matches(identity)) continue;
                long cycleStart = policy.cycleIterator().next().getLower().toInstant().toEpochMilli();
                long activeWarning = MultipathPolicyTracker.getActiveWarning(policy, cycleStart);
                long l = policyBytes = activeWarning == -1L ? MultipathPolicyTracker.getActiveLimit(policy, cycleStart) : activeWarning;
                if (policyBytes == -1L || policyBytes == -1L) continue;
                long policyBudget = this.getRemainingDailyBudget(policyBytes, policy.cycleIterator().next());
                minQuota = Math.min(minQuota, policyBudget);
            }
            if (minQuota == Long.MAX_VALUE) {
                return -1L;
            }
            return minQuota / 20L;
        }

        void updateMultipathBudget() {
            long budget;
            long quota = LocalServices.getService(NetworkPolicyManagerInternal.class).getSubscriptionOpportunisticQuota(this.network, 2);
            if (quota == -1L) {
                quota = this.getUserPolicyOpportunisticQuotaBytes();
            }
            if (quota == -1L) {
                quota = MultipathPolicyTracker.this.getDefaultDailyMultipathQuotaBytes();
            }
            if (this.haveMultipathBudget() && quota == this.mQuota) {
                return;
            }
            this.mQuota = quota;
            long usage = this.getDailyNonDefaultDataUsage();
            long l = budget = usage == -1L ? 0L : Math.max(0L, quota - usage);
            if (budget > NetworkStatsManager.MIN_THRESHOLD_BYTES) {
                this.registerUsageCallback(budget);
            } else {
                this.maybeUnregisterUsageCallback();
            }
        }

        public int getMultipathPreference() {
            if (this.haveMultipathBudget()) {
                return 3;
            }
            return 0;
        }

        public long getQuota() {
            return this.mQuota;
        }

        public long getMultipathBudget() {
            return this.mMultipathBudget;
        }

        private boolean haveMultipathBudget() {
            return this.mMultipathBudget > 0L;
        }

        private void registerUsageCallback(long budget) {
            this.maybeUnregisterUsageCallback();
            MultipathPolicyTracker.this.mStatsManager.registerUsageCallback(this.mNetworkTemplate, 0, budget, this.mUsageCallback, MultipathPolicyTracker.this.mHandler);
            this.mMultipathBudget = budget;
        }

        private void maybeUnregisterUsageCallback() {
            if (this.haveMultipathBudget()) {
                MultipathPolicyTracker.this.mStatsManager.unregisterUsageCallback(this.mUsageCallback);
                this.mMultipathBudget = 0L;
            }
        }

        void shutdown() {
            this.maybeUnregisterUsageCallback();
        }
    }

    public static class Dependencies {
        public Clock getClock() {
            return new BestClock(ZoneOffset.UTC, SystemClock.currentNetworkTimeClock(), Clock.systemUTC());
        }
    }
}

