/*
 * Decompiled with CFR 0.152.
 */
package com.android.server.connectivity;

import android.net.IDnsResolver;
import android.net.INetd;
import android.net.InetAddresses;
import android.net.InterfaceConfiguration;
import android.net.IpPrefix;
import android.net.LinkAddress;
import android.net.LinkProperties;
import android.net.NetworkInfo;
import android.net.RouteInfo;
import android.os.INetworkManagementService;
import android.os.RemoteException;
import android.os.ServiceSpecificException;
import android.util.Slog;
import com.android.internal.annotations.VisibleForTesting;
import com.android.internal.util.ArrayUtils;
import com.android.server.connectivity.NetworkAgentInfo;
import com.android.server.net.BaseNetworkObserver;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.util.Objects;

public class Nat464Xlat
extends BaseNetworkObserver {
    private static final String TAG = Nat464Xlat.class.getSimpleName();
    private static final String CLAT_PREFIX = "v4-";
    private static final int[] NETWORK_TYPES = new int[]{0, 1, 9};
    private static final NetworkInfo.State[] NETWORK_STATES = new NetworkInfo.State[]{NetworkInfo.State.CONNECTED, NetworkInfo.State.SUSPENDED};
    private final IDnsResolver mDnsResolver;
    private final INetd mNetd;
    private final INetworkManagementService mNMService;
    private final NetworkAgentInfo mNetwork;
    private IpPrefix mNat64Prefix;
    private String mBaseIface;
    private String mIface;
    private Inet6Address mIPv6Address;
    private State mState = State.IDLE;

    public Nat464Xlat(NetworkAgentInfo nai, INetd netd, IDnsResolver dnsResolver, INetworkManagementService nmService) {
        this.mDnsResolver = dnsResolver;
        this.mNetd = netd;
        this.mNMService = nmService;
        this.mNetwork = nai;
    }

    @VisibleForTesting
    protected static boolean requiresClat(NetworkAgentInfo nai) {
        boolean supported = ArrayUtils.contains(NETWORK_TYPES, nai.networkInfo.getType());
        boolean connected = ArrayUtils.contains(NETWORK_STATES, nai.networkInfo.getState());
        LinkProperties lp = nai.linkProperties;
        boolean isIpv6OnlyNetwork = lp != null && lp.hasGlobalIpv6Address() && !lp.hasIpv4Address();
        boolean skip464xlat = nai.netMisc() != null && nai.netMisc().skip464xlat;
        return supported && connected && isIpv6OnlyNetwork && !skip464xlat;
    }

    @VisibleForTesting
    protected static boolean shouldStartClat(NetworkAgentInfo nai) {
        LinkProperties lp = nai.linkProperties;
        return Nat464Xlat.requiresClat(nai) && lp != null && lp.getNat64Prefix() != null;
    }

    public boolean isPrefixDiscoveryStarted() {
        return this.mState == State.DISCOVERING || this.isStarted();
    }

    public boolean isStarted() {
        return this.mState == State.STARTING || this.mState == State.RUNNING;
    }

    public boolean isStarting() {
        return this.mState == State.STARTING;
    }

    public boolean isRunning() {
        return this.mState == State.RUNNING;
    }

    private void enterStartingState(String baseIface) {
        try {
            this.mNMService.registerObserver(this);
        }
        catch (RemoteException e) {
            Slog.e(TAG, "Can't register interface observer for clat on " + this.mNetwork.name());
            return;
        }
        String addrStr = null;
        try {
            addrStr = this.mNetd.clatdStart(baseIface, this.mNat64Prefix.toString());
        }
        catch (RemoteException | ServiceSpecificException e) {
            Slog.e(TAG, "Error starting clatd on " + baseIface + ": " + e);
        }
        this.mIface = CLAT_PREFIX + baseIface;
        this.mBaseIface = baseIface;
        this.mState = State.STARTING;
        try {
            this.mIPv6Address = (Inet6Address)InetAddresses.parseNumericAddress(addrStr);
        }
        catch (ClassCastException | IllegalArgumentException | NullPointerException e) {
            Slog.e(TAG, "Invalid IPv6 address " + addrStr);
        }
    }

    private void enterRunningState() {
        this.mState = State.RUNNING;
    }

    private void leaveStartedState() {
        try {
            this.mNMService.unregisterObserver(this);
        }
        catch (RemoteException | IllegalStateException e) {
            Slog.e(TAG, "Error unregistering clatd observer on " + this.mBaseIface + ": " + e);
        }
        this.mIface = null;
        this.mBaseIface = null;
        this.mState = State.IDLE;
        if (Nat464Xlat.requiresClat(this.mNetwork)) {
            this.mState = State.DISCOVERING;
        } else {
            this.stopPrefixDiscovery();
            this.mState = State.IDLE;
        }
    }

    @VisibleForTesting
    protected void start() {
        if (this.isStarted()) {
            Slog.e(TAG, "startClat: already started");
            return;
        }
        if (this.mNetwork.linkProperties == null) {
            Slog.e(TAG, "startClat: Can't start clat with null LinkProperties");
            return;
        }
        String baseIface = this.mNetwork.linkProperties.getInterfaceName();
        if (baseIface == null) {
            Slog.e(TAG, "startClat: Can't start clat on null interface");
            return;
        }
        Slog.i(TAG, "Starting clatd on " + baseIface);
        this.enterStartingState(baseIface);
    }

    @VisibleForTesting
    protected void stop() {
        if (!this.isStarted()) {
            Slog.e(TAG, "stopClat: already stopped");
            return;
        }
        Slog.i(TAG, "Stopping clatd on " + this.mBaseIface);
        try {
            this.mNetd.clatdStop(this.mBaseIface);
        }
        catch (RemoteException | ServiceSpecificException e) {
            Slog.e(TAG, "Error stopping clatd on " + this.mBaseIface + ": " + e);
        }
        String iface = this.mIface;
        boolean wasRunning = this.isRunning();
        this.leaveStartedState();
        if (wasRunning) {
            LinkProperties lp = new LinkProperties(this.mNetwork.linkProperties);
            lp.removeStackedLink(iface);
            this.mNetwork.connService().handleUpdateLinkProperties(this.mNetwork, lp);
        }
    }

    private void startPrefixDiscovery() {
        try {
            this.mDnsResolver.startPrefix64Discovery(this.getNetId());
            this.mState = State.DISCOVERING;
        }
        catch (RemoteException | ServiceSpecificException e) {
            Slog.e(TAG, "Error starting prefix discovery on netId " + this.getNetId() + ": " + e);
        }
    }

    private void stopPrefixDiscovery() {
        try {
            this.mDnsResolver.stopPrefix64Discovery(this.getNetId());
        }
        catch (RemoteException | ServiceSpecificException e) {
            Slog.e(TAG, "Error stopping prefix discovery on netId " + this.getNetId() + ": " + e);
        }
    }

    public void update() {
        if (Nat464Xlat.requiresClat(this.mNetwork)) {
            if (!this.isPrefixDiscoveryStarted()) {
                this.startPrefixDiscovery();
            } else if (Nat464Xlat.shouldStartClat(this.mNetwork)) {
                this.start();
            } else {
                this.stop();
            }
        } else if (this.isStarted()) {
            this.stop();
        } else if (this.isPrefixDiscoveryStarted()) {
            this.leaveStartedState();
        }
    }

    public void setNat64Prefix(IpPrefix nat64Prefix) {
        this.mNat64Prefix = nat64Prefix;
    }

    public void fixupLinkProperties(LinkProperties oldLp, LinkProperties lp) {
        lp.setNat64Prefix(this.mNat64Prefix);
        if (!this.isRunning()) {
            return;
        }
        if (lp == null || lp.getAllInterfaceNames().contains(this.mIface)) {
            return;
        }
        Slog.d(TAG, "clatd running, updating NAI for " + this.mIface);
        for (LinkProperties stacked : oldLp.getStackedLinks()) {
            if (!Objects.equals(this.mIface, stacked.getInterfaceName())) continue;
            lp.addStackedLink(stacked);
            return;
        }
    }

    private LinkProperties makeLinkProperties(LinkAddress clatAddress) {
        LinkProperties stacked = new LinkProperties();
        stacked.setInterfaceName(this.mIface);
        RouteInfo ipv4Default = new RouteInfo(new LinkAddress(Inet4Address.ANY, 0), clatAddress.getAddress(), this.mIface);
        stacked.addRoute(ipv4Default);
        stacked.addLinkAddress(clatAddress);
        return stacked;
    }

    private LinkAddress getLinkAddress(String iface) {
        try {
            InterfaceConfiguration config = this.mNMService.getInterfaceConfig(iface);
            return config.getLinkAddress();
        }
        catch (RemoteException | IllegalStateException e) {
            Slog.e(TAG, "Error getting link properties: " + e);
            return null;
        }
    }

    private void handleInterfaceLinkStateChanged(String iface, boolean up) {
        if (!(this.isStarting() && up && Objects.equals(this.mIface, iface))) {
            return;
        }
        LinkAddress clatAddress = this.getLinkAddress(iface);
        if (clatAddress == null) {
            Slog.e(TAG, "clatAddress was null for stacked iface " + iface);
            return;
        }
        Slog.i(TAG, String.format("interface %s is up, adding stacked link %s on top of %s", this.mIface, this.mIface, this.mBaseIface));
        this.enterRunningState();
        LinkProperties lp = new LinkProperties(this.mNetwork.linkProperties);
        lp.addStackedLink(this.makeLinkProperties(clatAddress));
        this.mNetwork.connService().handleUpdateLinkProperties(this.mNetwork, lp);
    }

    private void handleInterfaceRemoved(String iface) {
        if (!Objects.equals(this.mIface, iface)) {
            return;
        }
        if (!this.isRunning()) {
            return;
        }
        Slog.i(TAG, "interface " + iface + " removed");
        this.stop();
    }

    @Override
    public void interfaceLinkStateChanged(String iface, boolean up) {
        this.mNetwork.handler().post(() -> this.handleInterfaceLinkStateChanged(iface, up));
    }

    @Override
    public void interfaceRemoved(String iface) {
        this.mNetwork.handler().post(() -> this.handleInterfaceRemoved(iface));
    }

    public String toString() {
        return "mBaseIface: " + this.mBaseIface + ", mIface: " + this.mIface + ", mState: " + (Object)((Object)this.mState);
    }

    @VisibleForTesting
    protected int getNetId() {
        return this.mNetwork.network.netId;
    }

    private static enum State {
        IDLE,
        DISCOVERING,
        STARTING,
        RUNNING;

    }
}

