/*
 * Decompiled with CFR 0.152.
 */
package com.android.server.display;

import android.content.res.Resources;
import android.content.res.TypedArray;
import android.hardware.display.BrightnessConfiguration;
import android.hardware.display.BrightnessCorrection;
import android.util.MathUtils;
import android.util.Pair;
import android.util.Slog;
import android.util.Spline;
import com.android.internal.annotations.VisibleForTesting;
import com.android.internal.util.Preconditions;
import com.android.server.display.utils.Plog;
import java.io.PrintWriter;
import java.util.Arrays;

public abstract class BrightnessMappingStrategy {
    private static final String TAG = "BrightnessMappingStrategy";
    private static final float LUX_GRAD_SMOOTHING = 0.25f;
    private static final float MAX_GRAD = 1.0f;
    protected boolean mLoggingEnabled;
    private static final Plog PLOG = Plog.createSystemPlog("BrightnessMappingStrategy");

    public static BrightnessMappingStrategy create(Resources resources) {
        int[] backlightRange;
        float[] luxLevels = BrightnessMappingStrategy.getLuxLevels(resources.getIntArray(17235989));
        int[] brightnessLevelsBacklight = resources.getIntArray(17235988);
        float[] brightnessLevelsNits = BrightnessMappingStrategy.getFloatArray(resources.obtainTypedArray(17235986));
        float autoBrightnessAdjustmentMaxGamma = resources.getFraction(0x1130000, 1, 1);
        float[] nitsRange = BrightnessMappingStrategy.getFloatArray(resources.obtainTypedArray(17236055));
        if (BrightnessMappingStrategy.isValidMapping(nitsRange, backlightRange = resources.getIntArray(17236054)) && BrightnessMappingStrategy.isValidMapping(luxLevels, brightnessLevelsNits)) {
            int minimumBacklight = resources.getInteger(17694887);
            int maximumBacklight = resources.getInteger(17694886);
            if (backlightRange[0] > minimumBacklight || backlightRange[backlightRange.length - 1] < maximumBacklight) {
                Slog.w(TAG, "Screen brightness mapping does not cover whole range of available backlight values, autobrightness functionality may be impaired.");
            }
            BrightnessConfiguration.Builder builder = new BrightnessConfiguration.Builder(luxLevels, brightnessLevelsNits);
            return new PhysicalMappingStrategy(builder.build(), nitsRange, backlightRange, autoBrightnessAdjustmentMaxGamma);
        }
        if (BrightnessMappingStrategy.isValidMapping(luxLevels, brightnessLevelsBacklight)) {
            return new SimpleMappingStrategy(luxLevels, brightnessLevelsBacklight, autoBrightnessAdjustmentMaxGamma);
        }
        return null;
    }

    private static float[] getLuxLevels(int[] lux) {
        float[] levels = new float[lux.length + 1];
        for (int i = 0; i < lux.length; ++i) {
            levels[i + 1] = lux[i];
        }
        return levels;
    }

    private static float[] getFloatArray(TypedArray array2) {
        int N = array2.length();
        float[] vals = new float[N];
        for (int i = 0; i < N; ++i) {
            vals[i] = array2.getFloat(i, -1.0f);
        }
        array2.recycle();
        return vals;
    }

    private static boolean isValidMapping(float[] x, float[] y) {
        if (x == null || y == null || x.length == 0 || y.length == 0) {
            return false;
        }
        if (x.length != y.length) {
            return false;
        }
        int N = x.length;
        float prevX = x[0];
        float prevY = y[0];
        if (prevX < 0.0f || prevY < 0.0f || Float.isNaN(prevX) || Float.isNaN(prevY)) {
            return false;
        }
        for (int i = 1; i < N; ++i) {
            if (prevX >= x[i] || prevY > y[i]) {
                return false;
            }
            if (Float.isNaN(x[i]) || Float.isNaN(y[i])) {
                return false;
            }
            prevX = x[i];
            prevY = y[i];
        }
        return true;
    }

    private static boolean isValidMapping(float[] x, int[] y) {
        if (x == null || y == null || x.length == 0 || y.length == 0) {
            return false;
        }
        if (x.length != y.length) {
            return false;
        }
        int N = x.length;
        float prevX = x[0];
        int prevY = y[0];
        if (prevX < 0.0f || prevY < 0 || Float.isNaN(prevX)) {
            return false;
        }
        for (int i = 1; i < N; ++i) {
            if (prevX >= x[i] || prevY > y[i]) {
                return false;
            }
            if (Float.isNaN(x[i])) {
                return false;
            }
            prevX = x[i];
            prevY = y[i];
        }
        return true;
    }

    public boolean setLoggingEnabled(boolean loggingEnabled) {
        if (this.mLoggingEnabled == loggingEnabled) {
            return false;
        }
        this.mLoggingEnabled = loggingEnabled;
        return true;
    }

    public abstract boolean setBrightnessConfiguration(BrightnessConfiguration var1);

    public abstract float getBrightness(float var1, String var2, int var3);

    public float getBrightness(float lux) {
        return this.getBrightness(lux, null, -1);
    }

    public abstract float getAutoBrightnessAdjustment();

    public abstract boolean setAutoBrightnessAdjustment(float var1);

    public abstract float convertToNits(int var1);

    public abstract void addUserDataPoint(float var1, float var2);

    public abstract void clearUserDataPoints();

    public abstract boolean hasUserDataPoints();

    public abstract boolean isDefaultConfig();

    public abstract BrightnessConfiguration getDefaultConfig();

    public abstract void dump(PrintWriter var1);

    protected float normalizeAbsoluteBrightness(int brightness) {
        brightness = MathUtils.constrain(brightness, 0, 255);
        return (float)brightness / 255.0f;
    }

    private Pair<float[], float[]> insertControlPoint(float[] luxLevels, float[] brightnessLevels, float lux, float brightness) {
        float[] newBrightnessLevels;
        float[] newLuxLevels;
        int idx = this.findInsertionPoint(luxLevels, lux);
        if (idx == luxLevels.length) {
            newLuxLevels = Arrays.copyOf(luxLevels, luxLevels.length + 1);
            newBrightnessLevels = Arrays.copyOf(brightnessLevels, brightnessLevels.length + 1);
            newLuxLevels[idx] = lux;
            newBrightnessLevels[idx] = brightness;
        } else if (luxLevels[idx] == lux) {
            newLuxLevels = Arrays.copyOf(luxLevels, luxLevels.length);
            newBrightnessLevels = Arrays.copyOf(brightnessLevels, brightnessLevels.length);
            newBrightnessLevels[idx] = brightness;
        } else {
            newLuxLevels = Arrays.copyOf(luxLevels, luxLevels.length + 1);
            System.arraycopy(newLuxLevels, idx, newLuxLevels, idx + 1, luxLevels.length - idx);
            newLuxLevels[idx] = lux;
            newBrightnessLevels = Arrays.copyOf(brightnessLevels, brightnessLevels.length + 1);
            System.arraycopy(newBrightnessLevels, idx, newBrightnessLevels, idx + 1, brightnessLevels.length - idx);
            newBrightnessLevels[idx] = brightness;
        }
        this.smoothCurve(newLuxLevels, newBrightnessLevels, idx);
        return Pair.create(newLuxLevels, newBrightnessLevels);
    }

    private int findInsertionPoint(float[] arr, float val) {
        for (int i = 0; i < arr.length; ++i) {
            if (!(val <= arr[i])) continue;
            return i;
        }
        return arr.length;
    }

    private void smoothCurve(float[] lux, float[] brightness, int idx) {
        float minBrightness;
        float currLux;
        float maxBrightness;
        float currBrightness;
        float newBrightness;
        int i;
        if (this.mLoggingEnabled) {
            PLOG.logCurve("unsmoothed curve", lux, brightness);
        }
        float prevLux = lux[idx];
        float prevBrightness = brightness[idx];
        for (i = idx + 1; i < lux.length && (newBrightness = MathUtils.constrain(currBrightness = brightness[i], prevBrightness, maxBrightness = prevBrightness * this.permissibleRatio(currLux = lux[i], prevLux))) != currBrightness; ++i) {
            prevLux = currLux;
            prevBrightness = newBrightness;
            brightness[i] = newBrightness;
        }
        prevLux = lux[idx];
        prevBrightness = brightness[idx];
        for (i = idx - 1; i >= 0 && (newBrightness = MathUtils.constrain(currBrightness = brightness[i], minBrightness = prevBrightness * this.permissibleRatio(currLux = lux[i], prevLux), prevBrightness)) != currBrightness; --i) {
            prevLux = currLux;
            prevBrightness = newBrightness;
            brightness[i] = newBrightness;
        }
        if (this.mLoggingEnabled) {
            PLOG.logCurve("smoothed curve", lux, brightness);
        }
    }

    private float permissibleRatio(float currLux, float prevLux) {
        return MathUtils.exp(1.0f * (MathUtils.log(currLux + 0.25f) - MathUtils.log(prevLux + 0.25f)));
    }

    protected float inferAutoBrightnessAdjustment(float maxGamma, float desiredBrightness, float currentBrightness) {
        float adjustment = 0.0f;
        float gamma = Float.NaN;
        if (currentBrightness <= 0.1f || currentBrightness >= 0.9f) {
            adjustment = desiredBrightness - currentBrightness;
        } else if (desiredBrightness == 0.0f) {
            adjustment = -1.0f;
        } else if (desiredBrightness == 1.0f) {
            adjustment = 1.0f;
        } else {
            gamma = MathUtils.log(desiredBrightness) / MathUtils.log(currentBrightness);
            adjustment = -MathUtils.log(gamma) / MathUtils.log(maxGamma);
        }
        adjustment = MathUtils.constrain(adjustment, -1.0f, 1.0f);
        if (this.mLoggingEnabled) {
            Slog.d(TAG, "inferAutoBrightnessAdjustment: " + maxGamma + "^" + -adjustment + "=" + MathUtils.pow(maxGamma, -adjustment) + " == " + gamma);
            Slog.d(TAG, "inferAutoBrightnessAdjustment: " + currentBrightness + "^" + gamma + "=" + MathUtils.pow(currentBrightness, gamma) + " == " + desiredBrightness);
        }
        return adjustment;
    }

    protected Pair<float[], float[]> getAdjustedCurve(float[] lux, float[] brightness, float userLux, float userBrightness, float adjustment, float maxGamma) {
        float[] newLux = lux;
        float[] newBrightness = Arrays.copyOf(brightness, brightness.length);
        if (this.mLoggingEnabled) {
            PLOG.logCurve("unadjusted curve", newLux, newBrightness);
        }
        adjustment = MathUtils.constrain(adjustment, -1.0f, 1.0f);
        float gamma = MathUtils.pow(maxGamma, -adjustment);
        if (this.mLoggingEnabled) {
            Slog.d(TAG, "getAdjustedCurve: " + maxGamma + "^" + -adjustment + "=" + MathUtils.pow(maxGamma, -adjustment) + " == " + gamma);
        }
        if (gamma != 1.0f) {
            for (int i = 0; i < newBrightness.length; ++i) {
                newBrightness[i] = MathUtils.pow(newBrightness[i], gamma);
            }
        }
        if (this.mLoggingEnabled) {
            PLOG.logCurve("gamma adjusted curve", newLux, newBrightness);
        }
        if (userLux != -1.0f) {
            Pair<float[], float[]> curve = this.insertControlPoint(newLux, newBrightness, userLux, userBrightness);
            newLux = (float[])curve.first;
            newBrightness = (float[])curve.second;
            if (this.mLoggingEnabled) {
                PLOG.logCurve("gamma and user adjusted curve", newLux, newBrightness);
                curve = this.insertControlPoint(lux, brightness, userLux, userBrightness);
                PLOG.logCurve("user adjusted curve", (float[])curve.first, (float[])curve.second);
            }
        }
        return Pair.create(newLux, newBrightness);
    }

    @VisibleForTesting
    static class PhysicalMappingStrategy
    extends BrightnessMappingStrategy {
        private BrightnessConfiguration mConfig;
        private Spline mBrightnessSpline;
        private final Spline mNitsToBacklightSpline;
        private final BrightnessConfiguration mDefaultConfig;
        private Spline mBacklightToNitsSpline;
        private float mMaxGamma;
        private float mAutoBrightnessAdjustment;
        private float mUserLux;
        private float mUserBrightness;

        public PhysicalMappingStrategy(BrightnessConfiguration config, float[] nits, int[] backlight, float maxGamma) {
            Preconditions.checkArgument(nits.length != 0 && backlight.length != 0, "Nits and backlight arrays must not be empty!");
            Preconditions.checkArgument(nits.length == backlight.length, "Nits and backlight arrays must be the same length!");
            Preconditions.checkNotNull(config);
            Preconditions.checkArrayElementsInRange(nits, 0.0f, Float.MAX_VALUE, "nits");
            Preconditions.checkArrayElementsInRange(backlight, 0, 255, "backlight");
            this.mMaxGamma = maxGamma;
            this.mAutoBrightnessAdjustment = 0.0f;
            this.mUserLux = -1.0f;
            this.mUserBrightness = -1.0f;
            int N = nits.length;
            float[] normalizedBacklight = new float[N];
            for (int i = 0; i < N; ++i) {
                normalizedBacklight[i] = this.normalizeAbsoluteBrightness(backlight[i]);
            }
            this.mNitsToBacklightSpline = Spline.createSpline(nits, normalizedBacklight);
            this.mBacklightToNitsSpline = Spline.createSpline(normalizedBacklight, nits);
            this.mDefaultConfig = config;
            if (this.mLoggingEnabled) {
                PLOG.start("physical mapping strategy");
            }
            this.mConfig = config;
            this.computeSpline();
        }

        @Override
        public boolean setBrightnessConfiguration(BrightnessConfiguration config) {
            if (config == null) {
                config = this.mDefaultConfig;
            }
            if (config.equals(this.mConfig)) {
                return false;
            }
            if (this.mLoggingEnabled) {
                PLOG.start("brightness configuration");
            }
            this.mConfig = config;
            this.computeSpline();
            return true;
        }

        @Override
        public float getBrightness(float lux, String packageName, int category) {
            float nits = this.mBrightnessSpline.interpolate(lux);
            float backlight = this.mNitsToBacklightSpline.interpolate(nits);
            if (this.mUserLux == -1.0f) {
                backlight = this.correctBrightness(backlight, packageName, category);
            } else if (this.mLoggingEnabled) {
                Slog.d(BrightnessMappingStrategy.TAG, "user point set, correction not applied");
            }
            return backlight;
        }

        @Override
        public float getAutoBrightnessAdjustment() {
            return this.mAutoBrightnessAdjustment;
        }

        @Override
        public boolean setAutoBrightnessAdjustment(float adjustment) {
            if ((adjustment = MathUtils.constrain(adjustment, -1.0f, 1.0f)) == this.mAutoBrightnessAdjustment) {
                return false;
            }
            if (this.mLoggingEnabled) {
                Slog.d(BrightnessMappingStrategy.TAG, "setAutoBrightnessAdjustment: " + this.mAutoBrightnessAdjustment + " => " + adjustment);
                PLOG.start("auto-brightness adjustment");
            }
            this.mAutoBrightnessAdjustment = adjustment;
            this.computeSpline();
            return true;
        }

        @Override
        public float convertToNits(int backlight) {
            return this.mBacklightToNitsSpline.interpolate(this.normalizeAbsoluteBrightness(backlight));
        }

        @Override
        public void addUserDataPoint(float lux, float brightness) {
            float unadjustedBrightness = this.getUnadjustedBrightness(lux);
            if (this.mLoggingEnabled) {
                Slog.d(BrightnessMappingStrategy.TAG, "addUserDataPoint: (" + lux + "," + brightness + ")");
                PLOG.start("add user data point").logPoint("user data point", lux, brightness).logPoint("current brightness", lux, unadjustedBrightness);
            }
            float adjustment = this.inferAutoBrightnessAdjustment(this.mMaxGamma, brightness, unadjustedBrightness);
            if (this.mLoggingEnabled) {
                Slog.d(BrightnessMappingStrategy.TAG, "addUserDataPoint: " + this.mAutoBrightnessAdjustment + " => " + adjustment);
            }
            this.mAutoBrightnessAdjustment = adjustment;
            this.mUserLux = lux;
            this.mUserBrightness = brightness;
            this.computeSpline();
        }

        @Override
        public void clearUserDataPoints() {
            if (this.mUserLux != -1.0f) {
                if (this.mLoggingEnabled) {
                    Slog.d(BrightnessMappingStrategy.TAG, "clearUserDataPoints: " + this.mAutoBrightnessAdjustment + " => 0");
                    PLOG.start("clear user data points").logPoint("user data point", this.mUserLux, this.mUserBrightness);
                }
                this.mAutoBrightnessAdjustment = 0.0f;
                this.mUserLux = -1.0f;
                this.mUserBrightness = -1.0f;
                this.computeSpline();
            }
        }

        @Override
        public boolean hasUserDataPoints() {
            return this.mUserLux != -1.0f;
        }

        @Override
        public boolean isDefaultConfig() {
            return this.mDefaultConfig.equals(this.mConfig);
        }

        @Override
        public BrightnessConfiguration getDefaultConfig() {
            return this.mDefaultConfig;
        }

        @Override
        public void dump(PrintWriter pw) {
            pw.println("PhysicalMappingStrategy");
            pw.println("  mConfig=" + this.mConfig);
            pw.println("  mBrightnessSpline=" + this.mBrightnessSpline);
            pw.println("  mNitsToBacklightSpline=" + this.mNitsToBacklightSpline);
            pw.println("  mMaxGamma=" + this.mMaxGamma);
            pw.println("  mAutoBrightnessAdjustment=" + this.mAutoBrightnessAdjustment);
            pw.println("  mUserLux=" + this.mUserLux);
            pw.println("  mUserBrightness=" + this.mUserBrightness);
            pw.println("  mDefaultConfig=" + this.mDefaultConfig);
        }

        private void computeSpline() {
            Pair<float[], float[]> defaultCurve = this.mConfig.getCurve();
            float[] defaultLux = (float[])defaultCurve.first;
            float[] defaultNits = (float[])defaultCurve.second;
            float[] defaultBacklight = new float[defaultNits.length];
            for (int i = 0; i < defaultBacklight.length; ++i) {
                defaultBacklight[i] = this.mNitsToBacklightSpline.interpolate(defaultNits[i]);
            }
            Pair<float[], float[]> curve = this.getAdjustedCurve(defaultLux, defaultBacklight, this.mUserLux, this.mUserBrightness, this.mAutoBrightnessAdjustment, this.mMaxGamma);
            float[] lux = (float[])curve.first;
            float[] backlight = (float[])curve.second;
            float[] nits = new float[backlight.length];
            for (int i = 0; i < nits.length; ++i) {
                nits[i] = this.mBacklightToNitsSpline.interpolate(backlight[i]);
            }
            this.mBrightnessSpline = Spline.createSpline(lux, nits);
        }

        private float getUnadjustedBrightness(float lux) {
            Pair<float[], float[]> curve = this.mConfig.getCurve();
            Spline spline = Spline.createSpline((float[])curve.first, (float[])curve.second);
            return this.mNitsToBacklightSpline.interpolate(spline.interpolate(lux));
        }

        private float correctBrightness(float brightness, String packageName, int category) {
            BrightnessCorrection correction;
            if (packageName != null && (correction = this.mConfig.getCorrectionByPackageName(packageName)) != null) {
                return correction.apply(brightness);
            }
            if (category != -1 && (correction = this.mConfig.getCorrectionByCategory(category)) != null) {
                return correction.apply(brightness);
            }
            return brightness;
        }
    }

    private static class SimpleMappingStrategy
    extends BrightnessMappingStrategy {
        private final float[] mLux;
        private final float[] mBrightness;
        private Spline mSpline;
        private float mMaxGamma;
        private float mAutoBrightnessAdjustment;
        private float mUserLux;
        private float mUserBrightness;

        public SimpleMappingStrategy(float[] lux, int[] brightness, float maxGamma) {
            Preconditions.checkArgument(lux.length != 0 && brightness.length != 0, "Lux and brightness arrays must not be empty!");
            Preconditions.checkArgument(lux.length == brightness.length, "Lux and brightness arrays must be the same length!");
            Preconditions.checkArrayElementsInRange(lux, 0.0f, Float.MAX_VALUE, "lux");
            Preconditions.checkArrayElementsInRange(brightness, 0, Integer.MAX_VALUE, "brightness");
            int N = brightness.length;
            this.mLux = new float[N];
            this.mBrightness = new float[N];
            for (int i = 0; i < N; ++i) {
                this.mLux[i] = lux[i];
                this.mBrightness[i] = this.normalizeAbsoluteBrightness(brightness[i]);
            }
            this.mMaxGamma = maxGamma;
            this.mAutoBrightnessAdjustment = 0.0f;
            this.mUserLux = -1.0f;
            this.mUserBrightness = -1.0f;
            if (this.mLoggingEnabled) {
                PLOG.start("simple mapping strategy");
            }
            this.computeSpline();
        }

        @Override
        public boolean setBrightnessConfiguration(BrightnessConfiguration config) {
            return false;
        }

        @Override
        public float getBrightness(float lux, String packageName, int category) {
            return this.mSpline.interpolate(lux);
        }

        @Override
        public float getAutoBrightnessAdjustment() {
            return this.mAutoBrightnessAdjustment;
        }

        @Override
        public boolean setAutoBrightnessAdjustment(float adjustment) {
            if ((adjustment = MathUtils.constrain(adjustment, -1.0f, 1.0f)) == this.mAutoBrightnessAdjustment) {
                return false;
            }
            if (this.mLoggingEnabled) {
                Slog.d(BrightnessMappingStrategy.TAG, "setAutoBrightnessAdjustment: " + this.mAutoBrightnessAdjustment + " => " + adjustment);
                PLOG.start("auto-brightness adjustment");
            }
            this.mAutoBrightnessAdjustment = adjustment;
            this.computeSpline();
            return true;
        }

        @Override
        public float convertToNits(int backlight) {
            return -1.0f;
        }

        @Override
        public void addUserDataPoint(float lux, float brightness) {
            float unadjustedBrightness = this.getUnadjustedBrightness(lux);
            if (this.mLoggingEnabled) {
                Slog.d(BrightnessMappingStrategy.TAG, "addUserDataPoint: (" + lux + "," + brightness + ")");
                PLOG.start("add user data point").logPoint("user data point", lux, brightness).logPoint("current brightness", lux, unadjustedBrightness);
            }
            float adjustment = this.inferAutoBrightnessAdjustment(this.mMaxGamma, brightness, unadjustedBrightness);
            if (this.mLoggingEnabled) {
                Slog.d(BrightnessMappingStrategy.TAG, "addUserDataPoint: " + this.mAutoBrightnessAdjustment + " => " + adjustment);
            }
            this.mAutoBrightnessAdjustment = adjustment;
            this.mUserLux = lux;
            this.mUserBrightness = brightness;
            this.computeSpline();
        }

        @Override
        public void clearUserDataPoints() {
            if (this.mUserLux != -1.0f) {
                if (this.mLoggingEnabled) {
                    Slog.d(BrightnessMappingStrategy.TAG, "clearUserDataPoints: " + this.mAutoBrightnessAdjustment + " => 0");
                    PLOG.start("clear user data points").logPoint("user data point", this.mUserLux, this.mUserBrightness);
                }
                this.mAutoBrightnessAdjustment = 0.0f;
                this.mUserLux = -1.0f;
                this.mUserBrightness = -1.0f;
                this.computeSpline();
            }
        }

        @Override
        public boolean hasUserDataPoints() {
            return this.mUserLux != -1.0f;
        }

        @Override
        public boolean isDefaultConfig() {
            return true;
        }

        @Override
        public BrightnessConfiguration getDefaultConfig() {
            return null;
        }

        @Override
        public void dump(PrintWriter pw) {
            pw.println("SimpleMappingStrategy");
            pw.println("  mSpline=" + this.mSpline);
            pw.println("  mMaxGamma=" + this.mMaxGamma);
            pw.println("  mAutoBrightnessAdjustment=" + this.mAutoBrightnessAdjustment);
            pw.println("  mUserLux=" + this.mUserLux);
            pw.println("  mUserBrightness=" + this.mUserBrightness);
        }

        private void computeSpline() {
            Pair<float[], float[]> curve = this.getAdjustedCurve(this.mLux, this.mBrightness, this.mUserLux, this.mUserBrightness, this.mAutoBrightnessAdjustment, this.mMaxGamma);
            this.mSpline = Spline.createSpline((float[])curve.first, (float[])curve.second);
        }

        private float getUnadjustedBrightness(float lux) {
            Spline spline = Spline.createSpline(this.mLux, this.mBrightness);
            return spline.interpolate(lux);
        }
    }
}

