/*
 * Decompiled with CFR 0.152.
 */
package com.android.server.hdmi;

import android.hardware.hdmi.HdmiDeviceInfo;
import android.util.Slog;
import com.android.internal.util.Preconditions;
import com.android.server.hdmi.HdmiCecFeatureAction;
import com.android.server.hdmi.HdmiCecLocalDevice;
import com.android.server.hdmi.HdmiCecMessage;
import com.android.server.hdmi.HdmiCecMessageBuilder;
import com.android.server.hdmi.HdmiControlService;
import com.android.server.hdmi.HdmiUtils;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.List;

final class DeviceDiscoveryAction
extends HdmiCecFeatureAction {
    private static final String TAG = "DeviceDiscoveryAction";
    private static final int STATE_WAITING_FOR_DEVICE_POLLING = 1;
    private static final int STATE_WAITING_FOR_PHYSICAL_ADDRESS = 2;
    private static final int STATE_WAITING_FOR_OSD_NAME = 3;
    private static final int STATE_WAITING_FOR_VENDOR_ID = 4;
    private static final int STATE_WAITING_FOR_DEVICES = 5;
    private static final int STATE_WAITING_FOR_POWER = 6;
    private final ArrayList<DeviceInfo> mDevices = new ArrayList();
    private final DeviceDiscoveryCallback mCallback;
    private int mProcessedDeviceCount = 0;
    private int mTimeoutRetry = 0;
    private boolean mIsTvDevice;
    private final int mDelayPeriod;

    DeviceDiscoveryAction(HdmiCecLocalDevice source, DeviceDiscoveryCallback callback, int delay) {
        super(source);
        this.mIsTvDevice = this.localDevice().mService.isTvDevice();
        this.mCallback = Preconditions.checkNotNull(callback);
        this.mDelayPeriod = delay;
    }

    DeviceDiscoveryAction(HdmiCecLocalDevice source, DeviceDiscoveryCallback callback) {
        this(source, callback, 0);
    }

    @Override
    boolean start() {
        this.mDevices.clear();
        this.mState = 1;
        this.pollDevices(new HdmiControlService.DevicePollingCallback(){

            @Override
            public void onPollingFinished(List<Integer> ackedAddress) {
                if (ackedAddress.isEmpty()) {
                    Slog.v(DeviceDiscoveryAction.TAG, "No device is detected.");
                    DeviceDiscoveryAction.this.wrapUpAndFinish();
                    return;
                }
                Slog.v(DeviceDiscoveryAction.TAG, "Device detected: " + ackedAddress);
                DeviceDiscoveryAction.this.allocateDevices(ackedAddress);
                if (DeviceDiscoveryAction.this.mDelayPeriod > 0) {
                    DeviceDiscoveryAction.this.startToDelayAction();
                } else {
                    DeviceDiscoveryAction.this.startPhysicalAddressStage();
                }
            }
        }, 131073, 1);
        return true;
    }

    private void allocateDevices(List<Integer> addresses) {
        for (Integer i : addresses) {
            DeviceInfo info = new DeviceInfo(i);
            this.mDevices.add(info);
        }
    }

    private void startToDelayAction() {
        Slog.v(TAG, "Waiting for connected devices to be ready");
        this.mState = 5;
        this.checkAndProceedStage();
    }

    private void startPhysicalAddressStage() {
        Slog.v(TAG, "Start [Physical Address Stage]:" + this.mDevices.size());
        this.mProcessedDeviceCount = 0;
        this.mState = 2;
        this.checkAndProceedStage();
    }

    private boolean verifyValidLogicalAddress(int address) {
        return address >= 0 && address < 15;
    }

    private void queryPhysicalAddress(int address) {
        if (!this.verifyValidLogicalAddress(address)) {
            this.checkAndProceedStage();
            return;
        }
        this.mActionTimer.clearTimerMessage();
        if (this.mayProcessMessageIfCached(address, 132)) {
            return;
        }
        this.sendCommand(HdmiCecMessageBuilder.buildGivePhysicalAddress(this.getSourceAddress(), address));
        this.addTimer(this.mState, 2000);
    }

    private void delayActionWithTimePeriod(int timeDelay) {
        this.mActionTimer.clearTimerMessage();
        this.addTimer(this.mState, timeDelay);
    }

    private void startOsdNameStage() {
        Slog.v(TAG, "Start [Osd Name Stage]:" + this.mDevices.size());
        this.mProcessedDeviceCount = 0;
        this.mState = 3;
        this.checkAndProceedStage();
    }

    private void queryOsdName(int address) {
        if (!this.verifyValidLogicalAddress(address)) {
            this.checkAndProceedStage();
            return;
        }
        this.mActionTimer.clearTimerMessage();
        if (this.mayProcessMessageIfCached(address, 71)) {
            return;
        }
        this.sendCommand(HdmiCecMessageBuilder.buildGiveOsdNameCommand(this.getSourceAddress(), address));
        this.addTimer(this.mState, 2000);
    }

    private void startVendorIdStage() {
        Slog.v(TAG, "Start [Vendor Id Stage]:" + this.mDevices.size());
        this.mProcessedDeviceCount = 0;
        this.mState = 4;
        this.checkAndProceedStage();
    }

    private void queryVendorId(int address) {
        if (!this.verifyValidLogicalAddress(address)) {
            this.checkAndProceedStage();
            return;
        }
        this.mActionTimer.clearTimerMessage();
        if (this.mayProcessMessageIfCached(address, 135)) {
            return;
        }
        this.sendCommand(HdmiCecMessageBuilder.buildGiveDeviceVendorIdCommand(this.getSourceAddress(), address));
        this.addTimer(this.mState, 2000);
    }

    private void startPowerStatusStage() {
        Slog.v(TAG, "Start [Power Status Stage]:" + this.mDevices.size());
        this.mProcessedDeviceCount = 0;
        this.mState = 6;
        this.checkAndProceedStage();
    }

    private void queryPowerStatus(int address) {
        if (!this.verifyValidLogicalAddress(address)) {
            this.checkAndProceedStage();
            return;
        }
        this.mActionTimer.clearTimerMessage();
        if (this.mayProcessMessageIfCached(address, 144)) {
            return;
        }
        this.sendCommand(HdmiCecMessageBuilder.buildGiveDevicePowerStatus(this.getSourceAddress(), address));
        this.addTimer(this.mState, 2000);
    }

    private boolean mayProcessMessageIfCached(int address, int opcode) {
        HdmiCecMessage message = this.getCecMessageCache().getMessage(address, opcode);
        if (message != null) {
            this.processCommand(message);
            return true;
        }
        return false;
    }

    @Override
    boolean processCommand(HdmiCecMessage cmd) {
        switch (this.mState) {
            case 2: {
                if (cmd.getOpcode() == 132) {
                    this.handleReportPhysicalAddress(cmd);
                    return true;
                }
                return false;
            }
            case 3: {
                if (cmd.getOpcode() == 71) {
                    this.handleSetOsdName(cmd);
                    return true;
                }
                if (cmd.getOpcode() == 0 && (cmd.getParams()[0] & 0xFF) == 70) {
                    this.handleSetOsdName(cmd);
                    return true;
                }
                return false;
            }
            case 4: {
                if (cmd.getOpcode() == 135) {
                    this.handleVendorId(cmd);
                    return true;
                }
                if (cmd.getOpcode() == 0 && (cmd.getParams()[0] & 0xFF) == 140) {
                    this.handleVendorId(cmd);
                    return true;
                }
                return false;
            }
            case 6: {
                if (cmd.getOpcode() == 144) {
                    this.handleReportPowerStatus(cmd);
                    return true;
                }
                if (cmd.getOpcode() == 0 && (cmd.getParams()[0] & 0xFF) == 144) {
                    this.handleReportPowerStatus(cmd);
                    return true;
                }
                return false;
            }
        }
        return false;
    }

    private void handleReportPhysicalAddress(HdmiCecMessage cmd) {
        Preconditions.checkState(this.mProcessedDeviceCount < this.mDevices.size());
        DeviceInfo current = this.mDevices.get(this.mProcessedDeviceCount);
        if (current.mLogicalAddress != cmd.getSource()) {
            Slog.w(TAG, "Unmatched address[expected:" + current.mLogicalAddress + ", actual:" + cmd.getSource());
            return;
        }
        byte[] params = cmd.getParams();
        current.mPhysicalAddress = HdmiUtils.twoBytesToInt(params);
        current.mPortId = this.getPortId(current.mPhysicalAddress);
        current.mDeviceType = params[2] & 0xFF;
        current.mDisplayName = HdmiUtils.getDefaultDeviceName(current.mDeviceType);
        if (this.mIsTvDevice) {
            this.tv().updateCecSwitchInfo(current.mLogicalAddress, current.mDeviceType, current.mPhysicalAddress);
        }
        this.increaseProcessedDeviceCount();
        this.checkAndProceedStage();
    }

    private int getPortId(int physicalAddress) {
        return this.mIsTvDevice ? this.tv().getPortId(physicalAddress) : this.source().getPortId(physicalAddress);
    }

    private void handleSetOsdName(HdmiCecMessage cmd) {
        Preconditions.checkState(this.mProcessedDeviceCount < this.mDevices.size());
        DeviceInfo current = this.mDevices.get(this.mProcessedDeviceCount);
        if (current.mLogicalAddress != cmd.getSource()) {
            Slog.w(TAG, "Unmatched address[expected:" + current.mLogicalAddress + ", actual:" + cmd.getSource());
            return;
        }
        String displayName = null;
        try {
            displayName = cmd.getOpcode() == 0 ? HdmiUtils.getDefaultDeviceName(current.mLogicalAddress) : new String(cmd.getParams(), "US-ASCII");
        }
        catch (UnsupportedEncodingException e) {
            Slog.w(TAG, "Failed to decode display name: " + cmd.toString());
            displayName = HdmiUtils.getDefaultDeviceName(current.mLogicalAddress);
        }
        current.mDisplayName = displayName;
        this.increaseProcessedDeviceCount();
        this.checkAndProceedStage();
    }

    private void handleVendorId(HdmiCecMessage cmd) {
        Preconditions.checkState(this.mProcessedDeviceCount < this.mDevices.size());
        DeviceInfo current = this.mDevices.get(this.mProcessedDeviceCount);
        if (current.mLogicalAddress != cmd.getSource()) {
            Slog.w(TAG, "Unmatched address[expected:" + current.mLogicalAddress + ", actual:" + cmd.getSource());
            return;
        }
        if (cmd.getOpcode() != 0) {
            byte[] params = cmd.getParams();
            int vendorId = HdmiUtils.threeBytesToInt(params);
            current.mVendorId = vendorId;
        }
        this.increaseProcessedDeviceCount();
        this.checkAndProceedStage();
    }

    private void handleReportPowerStatus(HdmiCecMessage cmd) {
        Preconditions.checkState(this.mProcessedDeviceCount < this.mDevices.size());
        DeviceInfo current = this.mDevices.get(this.mProcessedDeviceCount);
        if (current.mLogicalAddress != cmd.getSource()) {
            Slog.w(TAG, "Unmatched address[expected:" + current.mLogicalAddress + ", actual:" + cmd.getSource());
            return;
        }
        if (cmd.getOpcode() != 0) {
            byte[] params = cmd.getParams();
            int powerStatus = params[0] & 0xFF;
            current.mPowerStatus = powerStatus;
        }
        this.increaseProcessedDeviceCount();
        this.checkAndProceedStage();
    }

    private void increaseProcessedDeviceCount() {
        ++this.mProcessedDeviceCount;
        this.mTimeoutRetry = 0;
    }

    private void removeDevice(int index) {
        this.mDevices.remove(index);
    }

    private void wrapUpAndFinish() {
        Slog.v(TAG, "---------Wrap up Device Discovery:[" + this.mDevices.size() + "]---------");
        ArrayList<HdmiDeviceInfo> result = new ArrayList<HdmiDeviceInfo>();
        for (DeviceInfo info : this.mDevices) {
            HdmiDeviceInfo cecDeviceInfo = info.toHdmiDeviceInfo();
            Slog.v(TAG, " DeviceInfo: " + cecDeviceInfo);
            result.add(cecDeviceInfo);
        }
        Slog.v(TAG, "--------------------------------------------");
        this.mCallback.onDeviceDiscoveryDone(result);
        this.finish();
        if (this.mIsTvDevice) {
            this.tv().processAllDelayedMessages();
        }
    }

    private void checkAndProceedStage() {
        if (this.mDevices.isEmpty()) {
            this.wrapUpAndFinish();
            return;
        }
        if (this.mProcessedDeviceCount == this.mDevices.size()) {
            this.mProcessedDeviceCount = 0;
            switch (this.mState) {
                case 2: {
                    this.startOsdNameStage();
                    return;
                }
                case 3: {
                    this.startVendorIdStage();
                    return;
                }
                case 4: {
                    this.startPowerStatusStage();
                    return;
                }
                case 6: {
                    this.wrapUpAndFinish();
                    return;
                }
            }
            return;
        }
        this.sendQueryCommand();
    }

    private void sendQueryCommand() {
        int address = this.mDevices.get(this.mProcessedDeviceCount).mLogicalAddress;
        switch (this.mState) {
            case 5: {
                this.delayActionWithTimePeriod(this.mDelayPeriod);
                return;
            }
            case 2: {
                this.queryPhysicalAddress(address);
                return;
            }
            case 3: {
                this.queryOsdName(address);
                return;
            }
            case 4: {
                this.queryVendorId(address);
                return;
            }
            case 6: {
                this.queryPowerStatus(address);
                return;
            }
        }
    }

    @Override
    void handleTimerEvent(int state) {
        if (this.mState == 0 || this.mState != state) {
            return;
        }
        if (this.mState == 5) {
            this.startPhysicalAddressStage();
            return;
        }
        if (++this.mTimeoutRetry < 5) {
            this.sendQueryCommand();
            return;
        }
        this.mTimeoutRetry = 0;
        Slog.v(TAG, "Timeout[State=" + this.mState + ", Processed=" + this.mProcessedDeviceCount);
        if (this.mState != 6 && this.mState != 3) {
            this.removeDevice(this.mProcessedDeviceCount);
        } else {
            this.increaseProcessedDeviceCount();
        }
        this.checkAndProceedStage();
    }

    private static final class DeviceInfo {
        private final int mLogicalAddress;
        private int mPhysicalAddress = 65535;
        private int mPortId = -1;
        private int mVendorId = 0xFFFFFF;
        private int mPowerStatus = -1;
        private String mDisplayName = "";
        private int mDeviceType = -1;

        private DeviceInfo(int logicalAddress) {
            this.mLogicalAddress = logicalAddress;
        }

        private HdmiDeviceInfo toHdmiDeviceInfo() {
            return new HdmiDeviceInfo(this.mLogicalAddress, this.mPhysicalAddress, this.mPortId, this.mDeviceType, this.mVendorId, this.mDisplayName, this.mPowerStatus);
        }
    }

    static interface DeviceDiscoveryCallback {
        public void onDeviceDiscoveryDone(List<HdmiDeviceInfo> var1);
    }
}

