package org.specs2
package specification

import execute._
import main._
import scalaz.Scalaz._
import collection.Seqx._
import control.Functions._
import control.ImplicitParameters

/**
 * This trait provides function to create specification Fragments:
 *  - a SpecStart fragment with title
 *  - a Text fragment with text
 *  - an Example fragment with a body
 *  - a group of fragments (when including another specification for example)
 * 
 */
trait FragmentsBuilder extends ExamplesFactory with ImplicitParameters with FormattingTags { outer =>

  /**
   * Methods for chaining fragments
   */

  /** @return a Fragments object from a single Fragment */
  implicit def fragments(f: =>Fragment): Fragments = Fragments.createList(f)
  implicit def fragmentFragments(f: =>Fragment): FragmentsFragment = new FragmentsFragment(fragments(f))
  implicit def fragmentsFragments(fs: =>Fragments): FragmentsFragment = new FragmentsFragment(fs)
  /** reverse conversion from a Fragment containing a Fragments object to the Fragments object*/
  implicit def fragmentsFragmentToFragments(fs: FragmentsFragment): Fragments = fs.fragments

  /**
   * Methods for creating fragments
   */
  /** @return a Fragments object from a string */
  implicit def textStart(s: String): Fragments = Fragments.createList(Text(s))
  
  /** @return create a Text Fragment from a string and allow it to be chained to other fragments */
  implicit def textFragment(s: String): FragmentsFragment = textStart(s)
  /**
   * This method allows to add a title to the Specification. It can be used as an operation on a String:
   * `"spec title".title`
   *
   * @return a Fragments object from a string, with the given title 
   */
  implicit def title(s: String): SpecTitle = new SpecTitle(s)
  class SpecTitle(name: String) {
    def title = Fragments(SpecName(name))
  }
  /**
   * Example creation
   * @return an Example description from a string, to create a full Example once the body is defined
   */
  implicit def forExample(s: String): ExampleDesc = new ExampleDesc(s)

  /** transient class to hold an example description before creating a full Example */
  class ExampleDesc(s: String) {
    /** @return an Example, using anything that can be translated to a Result, e.g. a Boolean */
    def ![T : AsResult](t: =>T): Example = exampleFactory.newExample(s, t)
    /** @return an Example, using the example description */
    def ![T : AsResult](f: String => T): Example = exampleFactory.newExample(s, f(s))
  }

  /**
   * Arguments creation
   * @return a Fragments object which can be chained with other fragments
   */
  implicit def argumentsFragment(a: Arguments): FragmentsFragment = new Fragments().overrideArgs(a)

  /**
   * Links to other specifications creation
   * 
   * @see org.specs2.UserGuide
   */
  implicit def stringToHtmlLinkFragments(s: String): HtmlLinkFragments = new HtmlLinkFragments(HtmlLink(SpecName(""), s, "", "", ""))
  class HtmlLinkFragments(link: HtmlLink) {
    def ~(s: SpecificationStructure) = outer.link(HtmlLink(name(s), link.beforeText, title(s)), s)
    def ~(n: String, s: SpecificationStructure) = outer.link(HtmlLink(name(s), link.beforeText, n), s)
    def ~(n: String, s: SpecificationStructure, a: String) = outer.link(HtmlLink(name(s), link.beforeText, n, a), s)
    def ~(n: String, s: SpecificationStructure, m: String, a: String) = outer.link(HtmlLink(name(s), link.beforeText, n, m, a), s)

    def ~/(s: SpecificationStructure) = outer.see(HtmlLink(name(s), link.beforeText, title(s)), s)
    def ~/(n: String, s: SpecificationStructure) = outer.see(HtmlLink(name(s), link.beforeText, n), s)
    def ~/(n: String, s: SpecificationStructure, a: String) = outer.see(HtmlLink(name(s), link.beforeText, n, a), s)
    def ~/(n: String, s: SpecificationStructure, m: String, a: String) = outer.see(HtmlLink(name(s), link.beforeText, n, m, a), s)

    private def name(s: SpecificationStructure) = s.content.specName
    private def title(s: SpecificationStructure) = name(s).title
  }

  implicit def stringToHtmlLinkFragments2(s: String): HtmlLinkFragments2 = new HtmlLinkFragments2(HtmlLink(SpecName(""), s, "", "", ""))
  class HtmlLinkFragments2(link: HtmlLink) {
    def ~(s: SpecificationStructure, n: String) = outer.link(HtmlLink(name(s), link.beforeText, title(s), n), s)
    def ~(s: SpecificationStructure, n: String, a: String) = outer.link(HtmlLink(name(s), link.beforeText, title(s), n, a), s)
    
    def ~/(s: SpecificationStructure, n: String) = outer.see(HtmlLink(name(s), link.beforeText, title(s), n), s)
    def ~/(s: SpecificationStructure, n: String, a: String) = outer.see(HtmlLink(name(s), link.beforeText, title(s), n, a), s)

    private def name(s: SpecificationStructure) = s.content.specName
    private def title(s: SpecificationStructure) = name(s).title
  }

  /** this implicit allows to call some functions directly on a specification, like 'hide' */
  implicit def specificationStructureToFragments(s: SpecificationStructure): AsFragments = AsFragments(s.content)
  case class AsFragments(fs: Fragments) {
    def hide = fs.hide
    def overrideArgs(args: Arguments) = fs.overrideArgs(args)
  }

  /** create a link directly on a specification */
  def link(s: SpecificationStructure): Fragments                              = link(s.content)
  def link(s: SpecificationStructure, ss: SpecificationStructure*): Fragments = link(s +: ss)
  def link(ss: Seq[SpecificationStructure], dummy: Int = 0): Fragments        = link(ss.map(_.content))
  def link(fs: Fragments): Fragments                                          = link(HtmlLink(fs), fs)
  def link(fs: Fragments, fss: Fragments*): Fragments                         = link(fs +: fss)
  def link(fss: Seq[Fragments]): Fragments                                    = fss.map(link).sumr
  /** create a link directly on a specification, with a given link */
  def link(htmlLink: HtmlLink, s: SpecificationStructure): Fragments          = link(htmlLink, s.content)
  def link(htmlLink: HtmlLink, f: Fragments): Fragments                       = f.linkIs(htmlLink)

  /** create a html link without including the other specification fragments */
  def see(s: SpecificationStructure, ss: SpecificationStructure*): Fragments       = see(s.emptyContent, ss.map(_.emptyContent):_*)
  def see(ss: Seq[SpecificationStructure])(implicit p1: ImplicitParam1): Fragments = see(ss.map(_.emptyContent))
  def see(s: SpecificationStructure): Fragments                                    = see(s.emptyContent)
  def see(fs: Fragments): Fragments                                                = see(HtmlLink(fs), fs)
  def see(fs: Fragments, fss: Fragments*): Fragments                               = see(fs +: fss)
  def see(fss: Seq[Fragments]): Fragments                                          = fss.map(see).sumr
  /** create a see-only link directly on a specification, with a given link */
  def see(htmlLink: HtmlLink, s: SpecificationStructure): Fragments                = see(htmlLink, s.emptyContent)
  def see(htmlLink: HtmlLink, fs: Fragments): Fragments                            = fs.seeIs(htmlLink)

  /** create markdown links from string + spec identification */
  implicit def specIdentificationMarkdownLink(s: String): SpecIdentificationMarkdownLink = new SpecIdentificationMarkdownLink(s)
  case class SpecIdentificationMarkdownLink(s: String) {
    def markdownLink(specIdentification: SpecIdentification) = specIdentification.markdownLink(s)
  }
}

object FragmentsBuilder extends FragmentsBuilder

/**
 * This trait can be used to deactivate implicits for building fragments
 */
trait NoFragmentsBuilder extends FragmentsBuilder with NoToHtmlLinkFragments {
  override def textStart(s: String): Fragments = super.textStart(s)
  override def textFragment(s: String): FragmentsFragment = super.textFragment(s)
  override def title(s: String): SpecTitle = super.title(s)
  override def specificationStructureToFragments(s: SpecificationStructure): AsFragments = super.specificationStructureToFragments(s)
}

/**
 * This trait can be used to deactivate implicits for ~ and ~/ operators on strings
 */
trait NoToHtmlLinkFragments extends FragmentsBuilder {
  override def stringToHtmlLinkFragments(s: String): HtmlLinkFragments = super.stringToHtmlLinkFragments(s)
  override def stringToHtmlLinkFragments2(s: String): HtmlLinkFragments2 = super.stringToHtmlLinkFragments2(s)
}


import scalaz._
/**
 * Implementation of the Show trait to display Fragments
 */
private[specs2]
trait FragmentsShow {
  implicit object showFragments extends Show[Fragment] {
    override def shows(f: Fragment) = f.toString
  }
}

private[specs2]
object FragmentsShow extends FragmentsShow
