
package org.talend.uniserv;

import uniserv.cliserv.*;

import java.io.*;
import java.net.*;

public class AutoApi {
	private boolean myLoggingIsEnabled = false;
	private boolean myIKJLoggingIsEnabled = false;
	private String myLanguage = "D";
	private GenClient myClient = null;

	public AutoApi() {
	}

	public void enableLogging() {
		myLoggingIsEnabled = true;
	}

	public void enableIKJLogging() {
		myIKJLoggingIsEnabled = true;
	}

	public void setLanguage(String language) {
		myLanguage = language;
	}

	private void abortOnError(Uniresult result) throws IOException,
			GatewayInternalException {
		if (result.getRetType() != GenClient.UNI_OK) {
			StringBuffer msg = new StringBuffer();
			Uniresult re = myClient.getErrorMsg(result.getRetType(), result
					.getRetInfo(), myLanguage, msg);
			if (re.getRetType() != GenClient.UNI_OK) {
				throw new IOException("abort (no message found)");
			}
			System.out.println(msg.toString());
			throw new IOException("abort");
		}
	}

	@SuppressWarnings("deprecation")
	public void openConnection(String autoApiHost, int masterPort,
			String mandant, String user, String password) throws Exception {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.openConnection: create client");
		myClient = new GenClient(myIKJLoggingIsEnabled);
		if (myLoggingIsEnabled)
			System.out
					.println("AutoApi.openConnection: connect to master at port "
							+ masterPort);
		Uniresult result = myClient.open(autoApiHost, masterPort, "dqsmaster");
		abortOnError(result);
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.openConnection: query server port");
		myClient.setArg("in_mandant", mandant);
		result = myClient.execRequest("query_port");
		abortOnError(result);
		StringBuffer port = new StringBuffer();
		result = myClient.getArg("out_port", port);
		abortOnError(result);
		int serverPort = Integer.parseInt(port.toString());
		if (myLoggingIsEnabled)
			System.out
					.println("AutoApi.openConnection: disconnect from master");
		if (myLoggingIsEnabled)
			System.out
					.println("AutoApi.openConnection: connect to server at port "
							+ serverPort);
		result = myClient.open(autoApiHost, serverPort, "dqsserver");
		abortOnError(result);
		if (myLoggingIsEnabled)
			System.out
					.println("AutoApi.openConnection: connected successfully");
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.authenticate");
		myClient.setArg("par_username", user);
		myClient.setArg("par_passwd", password);
	}

	public void closeConnection() throws Exception {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.closeConnection");
		myClient.close();
	}

	public void lockJob(String jobDirectory, String job) throws IOException,
			GatewayInternalException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.lockJob: " + jobDirectory + "/" + job);
		myClient.setArg("in_path", jobDirectory);
		myClient.setArg("in_name", job);
		Uniresult result = myClient.execRequest("lock_job");
		abortOnError(result);
	}

	public void unlockJob(String jobDirectory, String job) throws IOException,
			GatewayInternalException {
		if (myLoggingIsEnabled)
			System.out
					.println("AutoApi.unlockJob: " + jobDirectory + "/" + job);
		myClient.setArg("in_path", jobDirectory);
		myClient.setArg("in_name", job);
		Uniresult result = myClient.execRequest("unlock_job");
		abortOnError(result);
	}

	public void validateJob(String jobDirectory, String job)
			throws IOException, NoSuchArgException, GatewayInternalException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.validateJob: " + jobDirectory + "/"
					+ job);
		myClient.setArg("in_path", jobDirectory);
		myClient.setArg("in_name", job);
		Uniresult result = myClient.execRequest("job_validate");
		abortOnError(result);
		StringBuffer isValid = new StringBuffer();
		result = myClient.getArg("out_job_valid", isValid);
		abortOnError(result);
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.validateJob: is valid: "
					+ isValid.toString());
	}

	public void startJob(String jobDirectory, String job, String line,
			String step, String substep, String jobVariables)
			throws IOException, GatewayInternalException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.startJob: " + jobDirectory + "/" + job
					+ "(" + line + "," + step + "," + substep + ")");
		myClient.setArg("in_path", jobDirectory);
		myClient.setArg("in_name", job);
		myClient.setArg("in_line", line);
		myClient.setArg("in_step", step);
		myClient.setArg("in_substep", substep);
		myClient.setArg("in_variables", jobVariables);
		Uniresult result = myClient.execRequest("job_start");
		abortOnError(result);
	}

	public void stopJob(String jobDirectory, String job, String line,
			String step, String substep) throws IOException,
			GatewayInternalException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.stopJob: " + jobDirectory + "/" + job
					+ "(" + line + "," + step + "," + substep + ")");
		myClient.setArg("in_path", jobDirectory);
		myClient.setArg("in_name", job);
		myClient.setArg("in_line", line);
		myClient.setArg("in_step", step);
		myClient.setArg("in_substep", substep);
		Uniresult result = myClient.execRequest("job_stop");
		abortOnError(result);
	}

	public void resetJob(String jobDirectory, String job, String line,
			String step, String substep) throws IOException,
			GatewayInternalException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.resetJob: " + jobDirectory + "/" + job
					+ "(" + line + "," + step + "," + substep + ")");
		myClient.setArg("in_path", jobDirectory);
		myClient.setArg("in_name", job);
		myClient.setArg("in_line", line);
		myClient.setArg("in_step", step);
		myClient.setArg("in_substep", substep);
		Uniresult result = myClient.execRequest("job_reset");
		abortOnError(result);
	}

	public void deleteJobFiles(String jobDirectory, String job)
			throws IOException, GatewayInternalException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.deleteJobFiles: " + jobDirectory + "/"
					+ job);
		myClient.setArg("in_path", jobDirectory);
		myClient.setArg("in_name", job);
		Uniresult result = myClient.execRequest("delete_job_files");
		abortOnError(result);
	}

	public void waitForJobCompletion(String jobDirectory, String job)
			throws IOException, NoSuchArgException, GatewayInternalException,
			InterruptedException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.waitForJobCompletion:" + jobDirectory
					+ "/" + job);
		int state = 0;
		int version = 0;
		do {
			myClient.setArg("in_path", jobDirectory);
			myClient.setArg("in_name", job);
			myClient.setArg("in_version", "" + version);
			Uniresult result = myClient.execRequest("get_jobelement_state");
			abortOnError(result);
			StringBuffer newVersionAsString = new StringBuffer();
			result = myClient.getArg("out_version", newVersionAsString);
			abortOnError(result);
			int newVersion = Integer.parseInt(newVersionAsString.toString());
			if (newVersion > version) {
				StringBuffer stateAsString = new StringBuffer();
				result = myClient.getArg("out_state", stateAsString);
				state = Integer.parseInt(stateAsString.toString());
				version = newVersion;
			}
			if (state == 1) {
				System.out.println("still waiting for start");
				java.lang.Thread.sleep(5000); /* check anew after five seconds */
			} else if (state == 2) {
				System.out.println("still running");
				java.lang.Thread.sleep(5000); /* check anew after five seconds */
			} else if (state == 6) {
				System.out.println("still resetting");
				java.lang.Thread.sleep(5000); /* check anew after five seconds */
			}
		} while (state == 1 || state == 2 || state == 6);
		System.out.print("terminated with state ");
		if (state == 0) {
			System.out.println("idle");
		} else if (state == 4) {
			System.out.println("success");
		} else {
			System.out.println("error: " + state);
		}
	}

	public void createEmptyJob(String jobDirectory, String job)
			throws IOException, GatewayInternalException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.createEmptyJob: " + jobDirectory + "/"
					+ job);
		myClient.setArg("in_path", jobDirectory);
		myClient.setArg("in_name", job);
		Uniresult result = myClient.execRequest("create_job");
		abortOnError(result);
	}

	public void deleteJob(String jobDirectory, String job) throws IOException,
			GatewayInternalException {
		if (myLoggingIsEnabled)
			System.out
					.println("AutoApi.deleteJob: " + jobDirectory + "/" + job);
		myClient.setArg("in_path", jobDirectory);
		myClient.setArg("in_name", job);
		Uniresult result = myClient.execRequest("delete_job");
		abortOnError(result);
	}

	public void moveJob(String oldJobDirectory, String job,
			String newJobDirectory) throws IOException,
			GatewayInternalException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.moveJob: " + oldJobDirectory + "/"
					+ job + " to " + newJobDirectory + "/" + job);
		myClient.setArg("in_old_path", oldJobDirectory);
		myClient.setArg("in_name", job);
		myClient.setArg("in_new_path", newJobDirectory);
		Uniresult result = myClient.execRequest("move_job");
		abortOnError(result);
	}

	public void renameJob(String jobDirectory, String oldJob, String newJob)
			throws IOException, GatewayInternalException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.renameJob: " + jobDirectory + "/"
					+ oldJob + " to " + jobDirectory + "/" + newJob);
		myClient.setArg("in_path", jobDirectory);
		myClient.setArg("in_old_name", oldJob);
		myClient.setArg("in_new_name", newJob);
		Uniresult result = myClient.execRequest("rename_job");
		abortOnError(result);
	}

	public void copyJob(String sourceJobDirectory, String job,
			String targetJobDirectory) throws IOException,
			GatewayInternalException, NoSuchArgException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.copyJob: " + sourceJobDirectory + "/"
					+ job + " to " + targetJobDirectory + "/");
		myClient.setArg("in_old_path", sourceJobDirectory);
		myClient.setArg("in_name", job);
		myClient.setArg("in_new_path", targetJobDirectory);
		Uniresult result = myClient.execRequest("copy_job");
		abortOnError(result);
		StringBuffer newName = new StringBuffer();
		result = myClient.getArg("out_new_name", newName);
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.copyJob: " + sourceJobDirectory + "/"
					+ job + " to " + targetJobDirectory + "/"
					+ newName.toString());
		abortOnError(result);
	}

	public void createJobDirectory(String jobDirectory, String jobSubDirectory)
			throws IOException, GatewayInternalException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.createJobDirectory: " + jobDirectory
					+ "/" + jobSubDirectory);
		myClient.setArg("in_path", jobDirectory);
		myClient.setArg("in_name", jobSubDirectory);
		Uniresult result = myClient.execRequest("create_directory");
		abortOnError(result);
	}

	public void deleteJobDirectory(String jobDirectory, String jobSubDirectory)
			throws IOException, GatewayInternalException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.deleteJobDirectory: " + jobDirectory
					+ "/" + jobSubDirectory);
		myClient.setArg("in_path", jobDirectory);
		myClient.setArg("in_name", jobSubDirectory);
		Uniresult result = myClient.execRequest("delete_directory");
		abortOnError(result);
	}

	public void moveJobDirectory(String oldJobDirectory,
			String jobSubDirectory, String newJobDirectory) throws IOException,
			GatewayInternalException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.moveJobDirectory: " + oldJobDirectory
					+ "/" + jobSubDirectory + " to " + newJobDirectory + "/"
					+ jobSubDirectory);
		myClient.setArg("in_old_path", oldJobDirectory);
		myClient.setArg("in_name", jobSubDirectory);
		myClient.setArg("in_new_path", newJobDirectory);
		Uniresult result = myClient.execRequest("move_directory");
		abortOnError(result);
	}

	public void renameJobDirectory(String jobDirectory,
			String oldJobSubDirectory, String newJobSubDirectory)
			throws IOException, GatewayInternalException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.renameJobDirectory: " + jobDirectory
					+ "/" + oldJobSubDirectory + " to " + jobDirectory + "/"
					+ newJobSubDirectory);
		myClient.setArg("in_path", jobDirectory);
		myClient.setArg("in_old_name", oldJobSubDirectory);
		myClient.setArg("in_new_name", newJobSubDirectory);
		Uniresult result = myClient.execRequest("rename_directory");
		abortOnError(result);
	}

	public void copyFile(String oldJobPath, String oldSharePath,
			String newJobPath, String newSharePath, String oldFilename,
			String newFilename) throws IOException, GatewayInternalException {
		myClient.setArg("in_old_job_path", oldJobPath);
		myClient.setArg("in_old_frg_path", oldSharePath);
		myClient.setArg("in_new_job_path", newJobPath);
		myClient.setArg("in_new_frg_path", newSharePath);
		myClient.setArg("in_old_filename", oldFilename);
		myClient.setArg("in_new_filename", newFilename);
		Uniresult result = myClient.execRequest("copy_file");
		abortOnError(result);
	}

	public void deleteFile(String jobPath, String sharePath, String filename)
			throws IOException, GatewayInternalException {
		myClient.setArg("in_job_path", jobPath);
		myClient.setArg("in_frg_path", sharePath);
		myClient.setArg("in_filename", filename);
		Uniresult result = myClient.execRequest("delete_file");
		abortOnError(result);
	}

	public void moveFile(String oldJobPath, String oldSharePath,
			String newJobPath, String newSharePath, String filename)
			throws IOException, GatewayInternalException {
		myClient.setArg("in_old_job_path", oldJobPath);
		myClient.setArg("in_old_frg_path", oldSharePath);
		myClient.setArg("in_new_job_path", newJobPath);
		myClient.setArg("in_new_frg_path", newSharePath);
		myClient.setArg("in_filename", filename);
		Uniresult result = myClient.execRequest("move_file");
		abortOnError(result);
	}

	public void renameFile(String jobPath, String sharePath,
			String oldFilename, String newFilename) throws IOException,
			GatewayInternalException {
		myClient.setArg("in_job_path", jobPath);
		myClient.setArg("in_frg_path", sharePath);
		myClient.setArg("in_old_filename", oldFilename);
		myClient.setArg("in_new_filename", newFilename);
		Uniresult result = myClient.execRequest("rename_file");
		abortOnError(result);
	}

	public void createServerDirectory(String share, String path, String name)
			throws IOException, GatewayInternalException {
		myClient.setArg("in_frg", share);
		myClient.setArg("in_path", path);
		myClient.setArg("in_name", name);
		Uniresult result = myClient.execRequest("create_srv_dir");
		abortOnError(result);
	}

	public void deleteServerDirectory(String share, String path, String name)
			throws IOException, GatewayInternalException {
		myClient.setArg("in_frg", share);
		myClient.setArg("in_path", path);
		myClient.setArg("in_name", name);
		Uniresult result = myClient.execRequest("delete_srv_dir");
		abortOnError(result);
	}

	public void moveServerDirectory(String oldShare, String oldPath,
			String newShare, String newPath, String name) throws IOException,
			GatewayInternalException {
		myClient.setArg("in_old_frg", oldShare);
		myClient.setArg("in_old_path", oldPath);
		myClient.setArg("in_new_frg", newShare);
		myClient.setArg("in_new_path", newPath);
		myClient.setArg("in_name", name);
		Uniresult result = myClient.execRequest("move_srv_dir");
		abortOnError(result);
	}

	public void renameServerDirectory(String share, String path,
			String oldName, String newName) throws IOException,
			GatewayInternalException {
		myClient.setArg("in_frg", share);
		myClient.setArg("in_path", path);
		myClient.setArg("in_old_name", oldName);
		myClient.setArg("in_new_name", newName);
		Uniresult result = myClient.execRequest("rename_srv_dir");
		abortOnError(result);
	}

	private static String sixBitsIntToBase64CharTable = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";
	@SuppressWarnings("unused")
	private static int[] fromBase64CharToSixBitsIntTable = { 64, 64, 64, 64,
			64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
			64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
			64, 64, 64, 64, 64, 62, 64, 64, 64, 63, 52, 53, 54, 55, 56, 57, 58,
			59, 60, 61, 64, 64, 64, 64, 64, 64, 64, 0, 1, 2, 3, 4, 5, 6, 7, 8,
			9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
			64, 64, 64, 64, 64, 64, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36,
			37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 64, 64,
			64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
			64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
			64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
			64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
			64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
			64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
			64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
			64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64 };
	private int length = 0;
	private boolean eof = false;

	private void uploadDataBlock(String dataBlockToUpload) throws IOException,
			GatewayInternalException {
		String eofAsString;
		if (eof) {
			eofAsString = "1";
		} else {
			eofAsString = "0";
		}
		myClient.setArg("in_length", Integer.toString(length));
		myClient.setArg("in_data", dataBlockToUpload);
		myClient.setArg("in_eof", eofAsString);
		Uniresult result = myClient.execRequest("put_any_file_data");
		abortOnError(result);
	}

	public void encodeAndUploadFileContent(String fileToUpload,
			int blockSizeToUse) throws IOException, GatewayInternalException {
		FileInputStream fileToUploadStream = new FileInputStream(fileToUpload);
		byte[] buffer = new byte[3 * 1024]; // multiple of three
		length = 0;
		eof = false;
		StringBuffer encodedBlock = new StringBuffer();
		for (int len; (len = fileToUploadStream.read(buffer)) != -1;) {
			for (int i = 0; i < len; i += 3) {
				if (length + 4 > blockSizeToUse) {
					uploadDataBlock(encodedBlock.toString());
					encodedBlock.setLength(0);
					length = 0;
				}
				// add one or two zero fill bytes at the end
				if (i + 1 == len) {
					// add two zero fill bytes
					buffer[len] = 0x0;
					buffer[len + 1] = 0x0;
				} else if (i + 2 == len) {
					// add one zero fill byte
					buffer[len] = 0x0;
				}
				// encode three bytes (i, i+1, i+2) into four bytes (Base64)
				encodedBlock.append(sixBitsIntToBase64CharTable
						.charAt(((buffer[i] >> 2) & 0x3f)));
				encodedBlock.append(sixBitsIntToBase64CharTable
						.charAt(((buffer[i] & 0x03) << 4)
								| ((buffer[i + 1] & 0xf0) >> 4)));
				// number of fill bytes corresponds to number of '=' at the end
				if (i + 1 == len) {
					encodedBlock.append("=");
				} else {
					encodedBlock.append(sixBitsIntToBase64CharTable
							.charAt(((buffer[i + 1] & 0x0f) << 2)
									| ((buffer[i + 2] & 0xc0) >> 6)));
				}
				if (i + 2 == len || i + 1 == len) {
					encodedBlock.append("=");
				} else {
					encodedBlock.append(sixBitsIntToBase64CharTable
							.charAt((buffer[i + 2] & 0x3f)));
				}
				length += 4;
			}
		}
		eof = true;
		uploadDataBlock(encodedBlock.toString());
		fileToUploadStream.close();
	}

	private String downloadDataBlock() throws IOException,
			GatewayInternalException, NoSuchArgException {
		Uniresult result = myClient.execRequest("get_any_file_data");
		abortOnError(result);
		StringBuffer lengthAsString = new StringBuffer();
		result = myClient.getArg("out_length", lengthAsString);
		abortOnError(result);
		length = Integer.parseInt(lengthAsString.toString());
		StringBuffer dataBlockAsString = new StringBuffer();
		result = myClient.getArg("out_data", dataBlockAsString);
		abortOnError(result);
		StringBuffer eofAsString = new StringBuffer();
		result = myClient.getArg("out_eof", eofAsString);
		abortOnError(result);
		if (eofAsString.toString().equals("1")) {
			eof = true;
		} else {
			eof = false;
		}
		return dataBlockAsString.toString();
	}

	public void downloadFileContent(String fileToDownload, int blockSizeToUse)
			throws IOException, GatewayInternalException, NoSuchArgException {
		FileWriter fileToDownloadWriter = new FileWriter(fileToDownload);
		length = 0;
		eof = false;
		do {
			String dataBlock = downloadDataBlock();
			if (length > 0) {
				fileToDownloadWriter.write(dataBlock);
			}
		} while (eof == false);
		fileToDownloadWriter.close();
	}

	public void downloadAndDecodeFileContent(String fileToDownload,
			int blockSizeToUse) throws IOException, GatewayInternalException,
			NoSuchArgException {
		FileOutputStream fileToDownloadStream = new FileOutputStream(
				fileToDownload);
		byte[] buffer = new byte[3 * 1024]; // multiple of three
		int lengthToWrite = 0;
		eof = false;
		do {
			String encodedDataBlock = downloadDataBlock();
			for (int i = 0; i < encodedDataBlock.length(); i += 4) {
				if (length + 3 > 3 * 1024) {
					fileToDownloadStream.write(buffer, 0, lengthToWrite);
					lengthToWrite = 0;
				}
				int firstSixBits = sixBitsIntToBase64CharTable
						.indexOf(encodedDataBlock.charAt(i));
				int secondSixBits = sixBitsIntToBase64CharTable
						.indexOf(encodedDataBlock.charAt(i + 1));
				buffer[lengthToWrite++] = (byte) ((firstSixBits << 2) | (secondSixBits >> 4));
				if (encodedDataBlock.charAt(i + 2) == '=') {
					break;
				}
				int thirdSixBits = sixBitsIntToBase64CharTable
						.indexOf(encodedDataBlock.charAt(i + 2));
				buffer[lengthToWrite++] = (byte) ((secondSixBits << 4) | (thirdSixBits >> 2));
				if (encodedDataBlock.charAt(i + 3) == '=') {
					break;
				}
				int forthSixBits = sixBitsIntToBase64CharTable
						.indexOf(encodedDataBlock.charAt(i + 3));
				buffer[lengthToWrite++] = (byte) ((thirdSixBits << 6) | (forthSixBits));
			}
		} while (eof == false);
		if (lengthToWrite > 0) {
			fileToDownloadStream.write(buffer, 0, lengthToWrite);
		}
		fileToDownloadStream.close();
	}

	public void putInputFile(String jobDirectory, String job, String line,
			String step, String substep, String fileToUpload)
			throws IOException, GatewayInternalException, NoSuchArgException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.putInputFile: " + jobDirectory + "/"
					+ job + "(" + line + "," + step + "," + substep + "), "
					+ fileToUpload);
		myClient.setArg("in_pos", "input");
		myClient.setArg("in_path", jobDirectory);
		myClient.setArg("in_name", job);
		myClient.setArg("in_line", line);
		myClient.setArg("in_step", step);
		myClient.setArg("in_substep", substep);
		myClient.setArg("in_encoding", "base64");
		Uniresult result = myClient.execRequest("put_any_file_open");
		abortOnError(result);
		StringBuffer blockSizeToUseAsString = new StringBuffer();
		result = myClient.getArg("out_block_size_to_use",
				blockSizeToUseAsString);
		abortOnError(result);
		int blockSizeToUse = Integer
				.parseInt(blockSizeToUseAsString.toString());
		encodeAndUploadFileContent(fileToUpload, blockSizeToUse);
	}

	public void getInputFile(String jobDirectory, String job, String line,
			String step, String substep, String fileToDownload)
			throws IOException, GatewayInternalException, NoSuchArgException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.getInputFile: " + jobDirectory + "/"
					+ job + "(" + line + "," + step + "," + substep + "), "
					+ fileToDownload);
		myClient.setArg("in_pos", "input");
		myClient.setArg("in_path", jobDirectory);
		myClient.setArg("in_name", job);
		myClient.setArg("in_line", line);
		myClient.setArg("in_step", step);
		myClient.setArg("in_substep", substep);
		myClient.setArg("in_encoding", "base64");
		Uniresult result = myClient.execRequest("get_any_file_open");
		abortOnError(result);
		StringBuffer blockSizeToUseAsString = new StringBuffer();
		result = myClient.getArg("out_block_size_to_use",
				blockSizeToUseAsString);
		abortOnError(result);
		int blockSizeToUse = Integer
				.parseInt(blockSizeToUseAsString.toString());
		downloadAndDecodeFileContent(fileToDownload, blockSizeToUse);
	}

	public void getOutputFile(String jobDirectory, String job, String line,
			String step, String substep, String fileToDownload)
			throws IOException, GatewayInternalException, NoSuchArgException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.getOutputFile: " + jobDirectory + "/"
					+ job + "(" + line + "," + step + "," + substep + "), "
					+ fileToDownload);
		myClient.setArg("in_pos", "output");
		myClient.setArg("in_path", jobDirectory);
		myClient.setArg("in_name", job);
		myClient.setArg("in_line", line);
		myClient.setArg("in_step", step);
		myClient.setArg("in_substep", substep);
		myClient.setArg("in_encoding", "base64");
		Uniresult result = myClient.execRequest("get_any_file_open");
		abortOnError(result);
		StringBuffer blockSizeToUseAsString = new StringBuffer();
		result = myClient.getArg("out_block_size_to_use",
				blockSizeToUseAsString);
		abortOnError(result);
		int blockSizeToUse = Integer
				.parseInt(blockSizeToUseAsString.toString());
		downloadAndDecodeFileContent(fileToDownload, blockSizeToUse);
	}

	public void putJobDefinitionFile(String jobDirectory, String job,
			String fileToUpload) throws IOException, GatewayInternalException,
			NoSuchArgException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.putJobDefinitionFile: " + jobDirectory
					+ "/" + job + ", " + fileToUpload);
		myClient.setArg("in_pos", "jobdef");
		myClient.setArg("in_path", jobDirectory);
		myClient.setArg("in_name", job);
		myClient.setArg("in_encoding", "base64");
		Uniresult result = myClient.execRequest("put_any_file_open");
		abortOnError(result);
		StringBuffer blockSizeToUseAsString = new StringBuffer();
		result = myClient.getArg("out_block_size_to_use",
				blockSizeToUseAsString);
		abortOnError(result);
		int blockSizeToUse = Integer
				.parseInt(blockSizeToUseAsString.toString());
		encodeAndUploadFileContent(fileToUpload, blockSizeToUse);
	}

	public void getJobDefinitionFile(String jobDirectory, String job,
			String fileToDownload) throws Exception{
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.getJobDefinitionFile: " + jobDirectory
					+ "/" + job + ", " + fileToDownload);
		myClient.setArg("in_pos", "jobdef");
		myClient.setArg("in_path", jobDirectory);
		myClient.setArg("in_name", job);
		myClient.setArg("in_encoding", "base64");
		Uniresult result = myClient.execRequest("get_any_file_open");
		abortOnError(result);
		StringBuffer blockSizeToUseAsString = new StringBuffer();
		result = myClient.getArg("out_block_size_to_use",
				blockSizeToUseAsString);
		abortOnError(result);
		int blockSizeToUse = Integer
				.parseInt(blockSizeToUseAsString.toString());
		downloadAndDecodeFileContent(fileToDownload, blockSizeToUse);
	}

	public void getValidationResultFile(String jobDirectory, String job,
			String fileToDownload) throws IOException,
			GatewayInternalException, NoSuchArgException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.getValidationResultFile: "
					+ jobDirectory + "/" + job + ", " + fileToDownload);
		myClient.setArg("in_pos", "val");
		myClient.setArg("in_path", jobDirectory);
		myClient.setArg("in_name", job);
		myClient.setArg("in_encoding", "base64");
		Uniresult result = myClient.execRequest("get_any_file_open");
		abortOnError(result);
		StringBuffer blockSizeToUseAsString = new StringBuffer();
		result = myClient.getArg("out_block_size_to_use",
				blockSizeToUseAsString);
		abortOnError(result);
		int blockSizeToUse = Integer
				.parseInt(blockSizeToUseAsString.toString());
		downloadAndDecodeFileContent(fileToDownload, blockSizeToUse);
	}

	public void putImportFile(String jobDirectory, String job,
			String fileToUpload) throws IOException, GatewayInternalException,
			NoSuchArgException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.putImportFile: " + jobDirectory + "/"
					+ job + ", " + fileToUpload);
		myClient.setArg("in_pos", "impex");
		myClient.setArg("in_path", jobDirectory);
		myClient.setArg("in_name", job);
		myClient.setArg("in_encoding", "base64");
		Uniresult result = myClient.execRequest("put_any_file_open");
		abortOnError(result);
		StringBuffer blockSizeToUseAsString = new StringBuffer();
		result = myClient.getArg("out_block_size_to_use",
				blockSizeToUseAsString);
		abortOnError(result);
		int blockSizeToUse = Integer
				.parseInt(blockSizeToUseAsString.toString());
		encodeAndUploadFileContent(fileToUpload, blockSizeToUse);
	}

	public void getExportFile(String jobDirectory, String job,
			String fileToDownload) throws IOException,
			GatewayInternalException, NoSuchArgException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.getExportFile: " + jobDirectory + "/"
					+ job + ", " + fileToDownload);
		myClient.setArg("in_pos", "impex");
		myClient.setArg("in_path", jobDirectory);
		myClient.setArg("in_name", job);
		myClient.setArg("in_encoding", "base64");
		Uniresult result = myClient.execRequest("get_any_file_open");
		abortOnError(result);
		StringBuffer blockSizeToUseAsString = new StringBuffer();
		result = myClient.getArg("out_block_size_to_use",
				blockSizeToUseAsString);
		abortOnError(result);
		int blockSizeToUse = Integer
				.parseInt(blockSizeToUseAsString.toString());
		downloadAndDecodeFileContent(fileToDownload, blockSizeToUse);
	}

	public void putFileToShare(String share, String pathAndFileName,
			String fileToUpload) throws IOException, GatewayInternalException,
			NoSuchArgException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.putFileToShare: " + share + "/"
					+ pathAndFileName + ", " + fileToUpload);
		myClient.setArg("in_pos", "dir");
		myClient.setArg("in_frg", share);
		myClient.setArg("in_filename", pathAndFileName);
		myClient.setArg("in_encoding", "base64");
		Uniresult result = myClient.execRequest("put_any_file_open");
		abortOnError(result);
		StringBuffer blockSizeToUseAsString = new StringBuffer();
		result = myClient.getArg("out_block_size_to_use",
				blockSizeToUseAsString);
		abortOnError(result);
		int blockSizeToUse = Integer
				.parseInt(blockSizeToUseAsString.toString());
		encodeAndUploadFileContent(fileToUpload, blockSizeToUse);
	}

	public void getFileFromShare(String share, String pathAndFileName,
			String fileToDownload) throws IOException,
			GatewayInternalException, NoSuchArgException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.getFileFromShare: " + share + "/"
					+ pathAndFileName + ", " + fileToDownload);
		myClient.setArg("in_pos", "dir");
		myClient.setArg("in_frg", share);
		myClient.setArg("in_filename", pathAndFileName);
		myClient.setArg("in_encoding", "base64");
		Uniresult result = myClient.execRequest("get_any_file_open");
		abortOnError(result);
		StringBuffer blockSizeToUseAsString = new StringBuffer();
		result = myClient.getArg("out_block_size_to_use",
				blockSizeToUseAsString);
		abortOnError(result);
		int blockSizeToUse = Integer
				.parseInt(blockSizeToUseAsString.toString());
		downloadAndDecodeFileContent(fileToDownload, blockSizeToUse);
	}

	public void putFileToPool(String pathAndFileName, String fileToUpload)
			throws IOException, GatewayInternalException, NoSuchArgException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.putFileToPool: " + pathAndFileName
					+ ", " + fileToUpload);
		myClient.setArg("in_pos", "dir");
		myClient.setArg("in_frg", "Pool");
		myClient.setArg("in_filename", pathAndFileName);
		myClient.setArg("in_encoding", "base64");
		Uniresult result = myClient.execRequest("put_any_file_open");
		abortOnError(result);
		StringBuffer blockSizeToUseAsString = new StringBuffer();
		result = myClient.getArg("out_block_size_to_use",
				blockSizeToUseAsString);
		abortOnError(result);
		int blockSizeToUse = Integer
				.parseInt(blockSizeToUseAsString.toString());
		encodeAndUploadFileContent(fileToUpload, blockSizeToUse);
	}

	public void getFileFromPool(String pathAndFileName, String fileToDownload)
			throws IOException, GatewayInternalException, NoSuchArgException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.getFileFromPool: " + pathAndFileName
					+ ", " + fileToDownload);
		myClient.setArg("in_pos", "dir");
		myClient.setArg("in_frg", "Pool");
		myClient.setArg("in_filename", pathAndFileName);
		myClient.setArg("in_encoding", "base64");
		Uniresult result = myClient.execRequest("get_any_file_open");
		abortOnError(result);
		StringBuffer blockSizeToUseAsString = new StringBuffer();
		result = myClient.getArg("out_block_size_to_use",
				blockSizeToUseAsString);
		abortOnError(result);
		int blockSizeToUse = Integer
				.parseInt(blockSizeToUseAsString.toString());
		downloadAndDecodeFileContent(fileToDownload, blockSizeToUse);
	}

	public void putFileToTrans(String jobDirectory, String job,
			String pathAndFileName, String fileToUpload) throws IOException,
			GatewayInternalException, NoSuchArgException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.putFileToTrans: " + jobDirectory + "/"
					+ job + "/" + pathAndFileName + ", " + fileToUpload);
		myClient.setArg("in_pos", "trans");
		myClient.setArg("in_path", jobDirectory);
		myClient.setArg("in_name", job);
		myClient.setArg("in_filename", pathAndFileName);
		myClient.setArg("in_encoding", "base64");
		Uniresult result = myClient.execRequest("put_any_file_open");
		abortOnError(result);
		StringBuffer blockSizeToUseAsString = new StringBuffer();
		result = myClient.getArg("out_block_size_to_use",
				blockSizeToUseAsString);
		abortOnError(result);
		int blockSizeToUse = Integer
				.parseInt(blockSizeToUseAsString.toString());
		encodeAndUploadFileContent(fileToUpload, blockSizeToUse);
	}

	public void getFileFromTrans(String jobDirectory, String job,
			String pathAndFileName, String fileToDownload) throws IOException,
			GatewayInternalException, NoSuchArgException {
		if (myLoggingIsEnabled)
			System.out
					.println("AutoApi.getFileFromTrans: " + jobDirectory + "/"
							+ job + "/" + pathAndFileName + ", "
							+ fileToDownload);
		myClient.setArg("in_pos", "trans");
		myClient.setArg("in_path", jobDirectory);
		myClient.setArg("in_name", job);
		myClient.setArg("in_filename", pathAndFileName);
		myClient.setArg("in_encoding", "base64");
		Uniresult result = myClient.execRequest("get_any_file_open");
		abortOnError(result);
		StringBuffer blockSizeToUseAsString = new StringBuffer();
		result = myClient.getArg("out_block_size_to_use",
				blockSizeToUseAsString);
		abortOnError(result);
		int blockSizeToUse = Integer
				.parseInt(blockSizeToUseAsString.toString());
		downloadAndDecodeFileContent(fileToDownload, blockSizeToUse);
	}

	public void getStepFileInUtf8(String jobDirectory, String job, String line,
			String step, String substep, String pathAndFileName,
			String fileToDownload) throws IOException,
			GatewayInternalException, NoSuchArgException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.getStepFileInUtf8: " + jobDirectory
					+ "/" + job + "(" + line + "," + step + "," + substep
					+ ")/" + pathAndFileName + ", " + fileToDownload);
		myClient.setArg("in_pos", "list");
		myClient.setArg("in_path", jobDirectory);
		myClient.setArg("in_name", job);
		myClient.setArg("in_line", line);
		myClient.setArg("in_step", step);
		myClient.setArg("in_substep", substep);
		myClient.setArg("in_filename", pathAndFileName);
		myClient.setArg("in_encoding", "base64");
		Uniresult result = myClient.execRequest("get_any_file_open");
		abortOnError(result);
		StringBuffer blockSizeToUseAsString = new StringBuffer();
		result = myClient.getArg("out_block_size_to_use",
				blockSizeToUseAsString);
		abortOnError(result);
		int blockSizeToUse = Integer
				.parseInt(blockSizeToUseAsString.toString());
		downloadAndDecodeFileContent(fileToDownload, blockSizeToUse);
	}

	public void getStepFileInCodepage(String jobDirectory, String job,
			String line, String step, String substep, String pathAndFileName,
			String codepage, String fileToDownload) throws IOException,
			GatewayInternalException, NoSuchArgException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.getStepFileInCodepage: " + jobDirectory
					+ "/" + job + "(" + line + "," + step + "," + substep
					+ ")/" + pathAndFileName + ",codepage=" + codepage + ", "
					+ fileToDownload);
		myClient.setArg("in_pos", "list");
		myClient.setArg("in_path", jobDirectory);
		myClient.setArg("in_name", job);
		myClient.setArg("in_line", line);
		myClient.setArg("in_step", step);
		myClient.setArg("in_substep", substep);
		myClient.setArg("in_filename", pathAndFileName);
		myClient.setArg("in_result_codepage", codepage);
		myClient.setArg("in_encoding", "base64");
		Uniresult result = myClient.execRequest("get_any_file_open");
		abortOnError(result);
		StringBuffer blockSizeToUseAsString = new StringBuffer();
		result = myClient.getArg("out_block_size_to_use",
				blockSizeToUseAsString);
		abortOnError(result);
		int blockSizeToUse = Integer
				.parseInt(blockSizeToUseAsString.toString());
		downloadAndDecodeFileContent(fileToDownload, blockSizeToUse);
	}

	public void exportJob(String jobDirectory, String job, String useInput,
			String useOutput, String useDynamic, String useTable,
			String useBinary) throws IOException, GatewayInternalException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.exportJob: " + jobDirectory + "/" + job
					+ "(input:" + useInput + ",output:" + useOutput
					+ ",dynamic:" + useDynamic + ",table:" + useTable
					+ ",binary:" + useBinary + ")");
		myClient.setArg("in_path", jobDirectory);
		myClient.setArg("in_name", job);
		myClient.setArg("in_use_input", useInput);
		myClient.setArg("in_use_output", useOutput);
		myClient.setArg("in_use_dynamic", useDynamic);
		myClient.setArg("in_use_table", useTable);
		myClient.setArg("in_use_binary", useBinary);
		Uniresult result = myClient.execRequest("export_job");
		abortOnError(result);
	}

	public void importJob(String jobDirectory, String job, String useInput,
			String useOutput, String useDynamic, String useTable,
			String useBinary) throws IOException, GatewayInternalException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.importJob: " + jobDirectory + "/" + job
					+ "(input:" + useInput + ",output:" + useOutput
					+ ",dynamic:" + useDynamic + ",table:" + useTable
					+ ",binary:" + useBinary + ")");
		myClient.setArg("in_path", jobDirectory);
		myClient.setArg("in_name", job);
		myClient.setArg("in_use_input", useInput);
		myClient.setArg("in_use_output", useOutput);
		myClient.setArg("in_use_dynamic", useDynamic);
		myClient.setArg("in_use_table", useTable);
		myClient.setArg("in_use_binary", useBinary);
		Uniresult result = myClient.execRequest("import_job");
		abortOnError(result);
	}

	public void waitForImExportCompletion(String jobDirectory, String job)
			throws IOException, GatewayInternalException, NoSuchArgException,
			InterruptedException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.waitForImExportCompletion: "
					+ jobDirectory + "/" + job);
		StringBuffer state;
		do {
			myClient.setArg("in_path", jobDirectory);
			myClient.setArg("in_name", job);
			Uniresult result = myClient.execRequest("get_imexport_progress");
			state = new StringBuffer();
			result = myClient.getArg("out_status", state);
			abortOnError(result);
			StringBuffer totalBytes = new StringBuffer();
			result = myClient.getArg("out_total_bytes", totalBytes);
			abortOnError(result);
			StringBuffer progressInBytes = new StringBuffer();
			result = myClient.getArg("out_progress", progressInBytes);
			abortOnError(result);
			StringBuffer estimatedRemainingTimeInSeconds = new StringBuffer();
			result = myClient.getArg("out_time_remain",
					estimatedRemainingTimeInSeconds);
			abortOnError(result);
			if (myLoggingIsEnabled)
				System.out.println("AutoApi.waitForImExportCompletion: state="
						+ state.toString() + ", " + progressInBytes + "/"
						+ totalBytes + " bytes, still remaining "
						+ estimatedRemainingTimeInSeconds + "s");
			if (state.toString().equals("1")) {
				if (myLoggingIsEnabled)
					System.out.println("still running");
				java.lang.Thread.sleep(5000); /* check anew after five seconds */
			}
		} while (state.toString().equals("1"));
		if (state.toString().equals("0")) {
			if (myLoggingIsEnabled)
				System.out.println("not running");
		} else if (state.toString().equals("2")) {
			if (myLoggingIsEnabled)
				System.out.println("terminated with success");
		} else if (state.toString().equals("3")) {
			if (myLoggingIsEnabled)
				System.out.println("aborted by user");
		} else if (state.toString().equals("4")) {
			if (myLoggingIsEnabled)
				System.out.println("aborted with error");
		} else {
			if (myLoggingIsEnabled)
				System.out.println("invalid state " + state.toString());
		}
	}

	public void cancelImExport(String jobDirectory, String job)
			throws IOException, GatewayInternalException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.cancelImExport: " + jobDirectory + "/"
					+ job);
		myClient.setArg("in_path", jobDirectory);
		myClient.setArg("in_name", job);
		Uniresult result = myClient.execRequest("cancel_imexport");
		abortOnError(result);
	}

	public void cleanupImExport(String jobDirectory, String job)
			throws IOException, GatewayInternalException {
		if (myLoggingIsEnabled)
			System.out.println("AutoApi.cleanupImExport: " + jobDirectory + "/"
					+ job);
		myClient.setArg("in_path", jobDirectory);
		myClient.setArg("in_name", job);
		Uniresult result = myClient.execRequest("cleanup_imexport");
		abortOnError(result);
	}

	public static void main(String[] args) throws Exception {
boolean connected = false;
String host = null;
int port = 0;
String mandant = null;
String user = null;
String password = null;
if (args.length == 0) {
System.out.println("logging:");
System.out.println("-log #enable logging");
System.out.println("-IKJlog #enable interface kit logging");
System.out.println("language for error messages:");
System.out.println("-language <language> #de, en, fr");
System.out.println("connection control:");
System.out.println("-host <host>");
System.out.println("-port <port>");
System.out.println("-mandant <mandant>");
System.out.println("-user <user>");
System.out.println("-password <password>");
System.out.println("-open");
System.out.println("-close");
System.out.println("job control:");
System.out.println("-lock_job <jobdirectory> <jobname>");
System.out.println("-unlock_job <jobdirectory> <jobname>");
System.out.println("-validate_job <jobdirectory> <jobname>");
System.out.println("-start_job <jobdirectory> <jobname>");
System.out.println("-start_line <jobdirectory> <jobname> <line>");
System.out
.println("-start_step <jobdirectory> <jobname> <line> <step>");
System.out
.println("-start_substep <jobdirectory> <jobname> <line> <step> <substep>");
System.out.println("-stop_job <jobdirectory> <jobname>");
System.out.println("-stop_line <jobdirectory> <jobname> <line>");
System.out
.println("-stop_step <jobdirectory> <jobname> <line> <step>");
System.out
.println("-stop_substep <jobdirectory> <jobname> <line> <step> <substep>");
System.out.println("-reset_job <jobdirectory> <jobname>");
System.out.println("-reset_line <jobdirectory> <jobname> <line>");
System.out
.println("-reset_step <jobdirectory> <jobname> <line> <step>");
System.out
.println("-reset_substep <jobdirectory> <jobname> <line> <step> <substep>");
System.out.println("-delete_job_files <jobdirectory> <jobname>");
System.out
.println("-wait_for_job_completion <jobdirectory> <jobname>");
System.out.println("job tree manipulation:");
System.out.println("-create_empty_job <jobdirectory> <jobname>");
System.out.println("-delete_job <jobdirectory> <jobname>");
System.out
.println("-move_job <oldjobdirectory> <jobname> <newjobdirectory>");
System.out
.println("-rename_job <jobdirectory> <oldjobname> <newjobname>");
System.out
.println("-copy_job <oldjobdirectory> <jobname> <newjobdirectory>");
System.out
.println("-create_job_directory <jobdirectory> <jobsubdirectory>");
System.out
.println("-delete_job_directory <jobdirectory> <jobsubdirectory>");
System.out
.println("-move_job_directory <oldjobdirectory> <jobsubdirectory> <newjobdirectory>");
System.out
.println("-rename_job_directory <jobdirectory> <oldjobsubdirectory> <newjobsubdirectory>");
System.out.println("file tree manipulation:");
System.out.println("-copy_file <oldjobpath> <oldsharepath> <newjobpath> <newsharepath><oldfilename> <newfilename>");
System.out.println("-move_file <oldjobpath> <oldsharepath> <newjobpath> <newsharepath><filename>");
System.out
.println("-rename_file <jobpath> <sharepath> <oldfilename> <newfilename>");
System.out.println("-create_srv_dir <jobpath> <sharepath> <name>");
System.out.println("-delete_srv_dir <jobpath> <sharepath> <name>");
System.out.println("-move_srv_dir <oldjobpath> <oldsharepath> <newjobpath> <newsharepath><name>");
System.out.println("-rename_srv_dir <jobpath> <sharepath> <olddname> <newname>");
System.out.println("file transfer:");
System.out.println("-put_input_file <jobpath> <jobname> <line> <step> <substep> <localfilename>");
System.out.println("-get_input_file <jobpath> <jobname> <line> <step> <substep> <localfilename>");
System.out.println("-get_output_file <jobpath> <jobname> <line> <step> <substep> <localfilename>");
System.out.println("-put_job_definition_file <jobpath> <jobname> <localfilename>");
System.out.println("-get_job_definition_file <jobpath> <jobname> <localfilename>");
System.out.println("-get_validation_result_file <jobpath> <jobname> <localfilename>");
System.out.println("-put_import_file <jobpath> <jobname> <localfilename>");
System.out
.println("-get_export_file <jobpath> <jobname> <localfilename>");
System.out
.println("-put_file_to_share <share> <pathandfilename> <localfilename>");
System.out
.println("-get_file_from_share <share> <pathandfilename> <localfilename>");
System.out
.println("-put_file_to_pool <pathandfilename> <localfilename>");
System.out
.println("-get_file_from_pool <pathandfilename> <localfilename>");
System.out
.println("-put_file_to_trans <jobpath> <jobname> <pathandfilename> <localfilename>");
System.out
.println("-get_file_from_trans <jobpath> <jobname> <pathandfilename> <localfilename>");
System.out.println("-get_step_file_in_utf8 <jobpath> <jobname> <line> <step> <substep><pathandfilename> <localfilename>");
System.out
.println("-get_step_file_in_codepage <jobpath> <jobname> <line> <step> <substep><pathandfilename> <codepage> <localfilename>");
System.out.println("job import/export:");
System.out
.println("-export_job <jobdirectory> <jobname> <input:1/0> <output:1/0> <dynamic:1/0> <table:1/0> <binary:1/0>");
System.out
.println("-import_job <jobdirectory> <jobname> <input:1/0> <output:1/0> <dynamic:1/0> <table:1/0> <binary:1/0>");
System.out
.println("-wait_for_imexport_completion <jobdirectory> <jobname>");
System.out.println("-cancel_imexport <jobdirectory> <jobname>");
System.out.println("-cleanup_imexport <jobdirectory> <jobname>");
System.out.println("-uc1 # simple job control");
System.out.println("-uc2 # job control with file transfer");
System.out.println("-uc3 # job modification");
System.out.println("-uc4 # export and import of jobs");
}
AutoApi autoApiClient = new AutoApi();
try {
for (int i = 0; i < args.length; i++) {
if (args[i].equals("-log")) {
autoApiClient.enableLogging();
} else if (args[i].equals("-IKJlog")) {
autoApiClient.enableIKJLogging();
} else if (args[i].equals("-language") && i + 1 < args.length) {
autoApiClient.setLanguage(args[++i]);
} else if (args[i].equals("-host")) {
host = args[++i];} else if (args[i].equals("-port") && i + 1 < args.length) {
port = Integer.parseInt(args[++i]);
} else if (args[i].equals("-mandant") && i + 1 < args.length) {
mandant = args[++i];
} else if (args[i].equals("-user") && i + 1 < args.length) {
user = args[++i];
} else if (args[i].equals("-password") && i + 1 < args.length) {
password = args[++i];
} else if (args[i].equals("-open")) {
if (!connected) {
autoApiClient.openConnection(host, port, mandant, user,
password);
connected = true;
}
} else if (args[i].equals("-close")) {
if (connected) {
autoApiClient.closeConnection();
connected = false;
}
} else if (args[i].equals("-lock_job") && i + 2 < args.length) {
autoApiClient.lockJob(args[++i], args[++i]);
} else if (args[i].equals("-unlock_job") && i + 2 < args.length) {
autoApiClient.unlockJob(args[++i], args[++i]);
} else if (args[i].equals("-validate_job") && i + 2 < args.length) {
autoApiClient.validateJob(args[++i], args[++i]);
} else if (args[i].equals("-start_job") && i + 2 < args.length) {
autoApiClient.startJob(args[++i], args[++i], "", "", "", "");
} else if (args[i].equals("-start_line") && i + 3 < args.length) {
autoApiClient.startJob(args[++i], args[++i], args[++i], "", "",
"");
} else if (args[i].equals("-start_step") && i + 4 < args.length) {
autoApiClient.startJob(args[++i], args[++i], args[++i],
args[++i], "", "");
} else if (args[i].equals("-start_substep") && i + 5 < args.length) {
autoApiClient.startJob(args[++i], args[++i], args[++i],
args[++i], args[++i], "");
} else if (args[i].equals("-stop_job") && i + 2 < args.length) {
autoApiClient.stopJob(args[++i], args[++i], "", "", "");
} else if (args[i].equals("-stop_line") && i + 3 < args.length) {
autoApiClient.stopJob(args[++i], args[++i], args[++i], "", "");
} else if (args[i].equals("-stop_step") && i + 4 < args.length) {
autoApiClient.stopJob(args[++i], args[++i], args[++i],
args[++i], "");
} else if (args[i].equals("-stop_substep") && i + 5 < args.length) {
autoApiClient.stopJob(args[++i], args[++i], args[++i],
args[++i], args[++i]);
} else if (args[i].equals("-reset_job") && i + 2 < args.length) {
autoApiClient.resetJob(args[++i], args[++i], "", "", "");
} else if (args[i].equals("-reset_line") && i + 3 < args.length) {
autoApiClient.resetJob(args[++i], args[++i], args[++i], "", "");
} else if (args[i].equals("-reset_step") && i + 4 < args.length) {
autoApiClient.resetJob(args[++i], args[++i], args[++i],
args[++i], "");
} else if (args[i].equals("-reset_substep") && i + 5 < args.length) {
autoApiClient.resetJob(args[++i], args[++i], args[++i],
args[++i], args[++i]);
} else if (args[i].equals("-delete_job_files")
&& i + 2 < args.length) {
autoApiClient.deleteJobFiles(args[++i], args[++i]);
} else if (args[i].equals("-wait_for_job_completion")
&& i + 2 < args.length) {
autoApiClient.waitForJobCompletion(args[++i], args[++i]);
} else if (args[i].equals("-create_empty_job")
&& i + 2 < args.length) {
autoApiClient.createEmptyJob(args[++i], args[++i]);} else if (args[i].equals("-delete_job") && i + 2 < args.length) {
autoApiClient.deleteJob(args[++i], args[++i]);
} else if (args[i].equals("-move_job") && i + 3 < args.length) {
autoApiClient.moveJob(args[++i], args[++i], args[++i]);
} else if (args[i].equals("-rename_job") && i + 3 < args.length) {
autoApiClient.renameJob(args[++i], args[++i], args[++i]);
} else if (args[i].equals("-copy_job") && i + 3 < args.length) {
autoApiClient.copyJob(args[++i], args[++i], args[++i]);
} else if (args[i].equals("-create_job_directory")
&& i + 2 < args.length) {
autoApiClient.createJobDirectory(args[++i], args[++i]);
} else if (args[i].equals("-delete_job_directory")
&& i + 2 < args.length) {
autoApiClient.deleteJobDirectory(args[++i], args[++i]);
} else if (args[i].equals("-move_job_directory")
&& i + 3 < args.length) {
autoApiClient.moveJobDirectory(args[++i], args[++i], args[++i]);
} else if (args[i].equals("-rename_job_directory")
&& i + 3 < args.length) {
autoApiClient
.renameJobDirectory(args[++i], args[++i], args[++i]);
} else if (args[i].equals("-copy_file") && i + 6 < args.length) {
autoApiClient.copyFile(args[++i], args[++i], args[++i],
args[++i], args[++i], args[++i]);
} else if (args[i].equals("-move_file") && i + 5 < args.length) {
autoApiClient.moveFile(args[++i], args[++i], args[++i],
args[++i], args[++i]);
} else if (args[i].equals("-rename_file") && i + 4 < args.length) {
autoApiClient.renameFile(args[++i], args[++i], args[++i],
args[++i]);
} else if (args[i].equals("-create_srv_dir") && i + 3 < args.length) {
autoApiClient.createServerDirectory(args[++i], args[++i],
args[++i]);
} else if (args[i].equals("-delete_srv_dir") && i + 3 < args.length) {
autoApiClient.deleteServerDirectory(args[++i], args[++i],
args[++i]);
} else if (args[i].equals("-move_srv_dir") && i + 5 < args.length) {
autoApiClient.moveServerDirectory(args[++i], args[++i],
args[++i], args[++i], args[++i]);
} else if (args[i].equals("-rename_srv_dir") && i + 4 < args.length) {
autoApiClient.renameServerDirectory(args[++i], args[++i],
args[++i], args[++i]);
}
else if (args[i].equals("-put_input_file") && i + 6 < args.length) {
autoApiClient.putInputFile(args[++i], args[++i], args[++i],
args[++i], args[++i], args[++i]);
} else if (args[i].equals("-get_input_file") && i + 6 < args.length) {
autoApiClient.getInputFile(args[++i], args[++i], args[++i],
args[++i], args[++i], args[++i]);
} else if (args[i].equals("-get_output_file")
&& i + 6 < args.length) {
autoApiClient.getOutputFile(args[++i], args[++i], args[++i],
args[++i], args[++i], args[++i]);
} else if (args[i].equals("-put_job_definition_file")
&& i + 3 < args.length) {
autoApiClient.putJobDefinitionFile(args[++i], args[++i],
args[++i]);
} else if (args[i].equals("-get_job_definition_file")
&& i + 3 < args.length) {
autoApiClient.getJobDefinitionFile(args[++i], args[++i],
args[++i]);
} else if (args[i].equals("-get_validation_result_file")
&& i + 3 < args.length) {
autoApiClient.getValidationResultFile(args[++i], args[++i],args[++i]);
} else if (args[i].equals("-put_import_file")
&& i + 3 < args.length) {
autoApiClient.putImportFile(args[++i], args[++i], args[++i]);
} else if (args[i].equals("-get_export_file")
&& i + 3 < args.length) {
autoApiClient.getExportFile(args[++i], args[++i], args[++i]);
} else if (args[i].equals("-put_file_to_share")
&& i + 3 < args.length) {
autoApiClient.putFileToShare(args[++i], args[++i], args[++i]);
} else if (args[i].equals("-get_file_from_share")
&& i + 3 < args.length) {
autoApiClient.getFileFromShare(args[++i], args[++i], args[++i]);
} else if (args[i].equals("-put_file_to_pool")
&& i + 2 < args.length) {
autoApiClient.putFileToPool(args[++i], args[++i]);
} else if (args[i].equals("-get_file_from_pool")
&& i + 2 < args.length) {
autoApiClient.getFileFromPool(args[++i], args[++i]);
} else if (args[i].equals("-put_file_to_trans")
&& i + 4 < args.length) {
autoApiClient.putFileToTrans(args[++i], args[++i], args[++i],
args[++i]);
} else if (args[i].equals("-get_file_from_trans")
&& i + 4 < args.length) {
autoApiClient.getFileFromTrans(args[++i], args[++i], args[++i],
args[++i]);
} else if (args[i].equals("-get_step_file_in_utf8")
&& i + 7 < args.length) {
autoApiClient.getStepFileInUtf8(args[++i], args[++i], args[++i],
args[++i], args[++i], args[++i], args[++i]);
} else if (args[i].equals("-get_step_file_in_codepage")
&& i + 8 < args.length) {
autoApiClient.getStepFileInCodepage(args[++i], args[++i],
args[++i], args[++i], args[++i], args[++i], args[++i],
args[++i]);
}
else if (args[i].equals("-export_job") && i + 7 < args.length) {
autoApiClient.exportJob(args[++i], args[++i], args[++i],
args[++i], args[++i], args[++i], args[++i]);
} else if (args[i].equals("-import_job") && i + 7 < args.length) {
autoApiClient.importJob(args[++i], args[++i], args[++i],
args[++i], args[++i], args[++i], args[++i]);
} else if (args[i].equals("-wait_for_imexport_completion")
&& i + 2 < args.length) {
autoApiClient.waitForImExportCompletion(args[++i], args[++i]);
} else if (args[i].equals("-cancel_imexport")
&& i + 2 < args.length) {
autoApiClient.cancelImExport(args[++i], args[++i]);
} else if (args[i].equals("-cleanup_imexport")
&& i + 2 < args.length) {
autoApiClient.cleanupImExport(args[++i], args[++i]);
}
else if (args[i].equals("-uc1")) {
autoApiClient.resetJob("examples", "JobVariable", "", "", "");
autoApiClient.waitForJobCompletion("examples", "JobVariable");
autoApiClient
.startJob("examples", "JobVariable", "", "", "", "");
autoApiClient.waitForJobCompletion("examples", "JobVariable");
} else if (args[i].equals("-uc2")) {
autoApiClient.resetJob("examples", "JobVariable", "", "", "");
autoApiClient.waitForJobCompletion("examples", "JobVariable");
autoApiClient.putFileToPool("examples/inputNew.txt",
"inputNew.txt");
autoApiClient.startJob("examples", "JobVariable", "", "", "",
"InputFileName\u0009inputNew.txt");
autoApiClient.waitForJobCompletion("examples", "JobVariable");
autoApiClient.getStepFileInCodepage("examples", "JobVariable",
"Line1", "Validation1", "", "message.log", "ISO-8859-1",
"postalValidation.log");
autoApiClient.getStepFileInCodepage("examples", "JobVariable",
"Line1", "MailBatch1", "", "message.log", "ISO-8859-1",
"mailBatch.log");
autoApiClient.getStepFileInCodepage("examples", "JobVariable",
"Line1", "MailBatch1", "", "list1001.lst", "ISO-8859-1",
"mailBatchDuplicates.lst");
autoApiClient.getOutputFile("examples", "JobVariable", "Line1",
"Output1", "Target1", "output.txt");
} else if (args[i].equals("-uc3")) {
autoApiClient.createJobDirectory("", "AutoApi_UC3");
autoApiClient.copyJob("examples", "JobVariable", "AutoApi_UC3");
autoApiClient.getJobDefinitionFile("AutoApi_UC3", "JobVariable",
"jobDefinitionJobVariables.xml");
autoApiClient.putJobDefinitionFile("AutoApi_UC3", "JobVariable",
"jobDefinitionJobVariables.xml");
autoApiClient.validateJob("AutoApi_UC3", "JobVariable");
autoApiClient.startJob("AutoApi_UC3", "JobVariable", "", "", "",
"");
autoApiClient.waitForJobCompletion("AutoApi_UC3", "JobVariable");
} else if (args[i].equals("-uc4")) {
autoApiClient.exportJob("examples", "JobVariable", "1", "1",
"1", "1", "1");
autoApiClient.waitForImExportCompletion("examples",
"JobVariable");
autoApiClient.getExportFile("examples", "JobVariable",
"exportFileJobVariable.dqa");
autoApiClient.cleanupImExport("examples", "JobVariable");
autoApiClient.createJobDirectory("", "AutoApi_UC4");
autoApiClient.createEmptyJob("AutoApi_UC4", "JobVariable");
autoApiClient.putImportFile("AutoApi_UC4", "JobVariable",
"exportFileJobVariable.dqa");
autoApiClient.importJob("AutoApi_UC4", "JobVariable", "1", "1",
"1", "1", "1");
autoApiClient.waitForImExportCompletion("AutoApi_UC4",
"JobVariable");
autoApiClient.cleanupImExport("AutoApi_UC4", "JobVariable");
} else {
System.out.println("ERROR: " + args[i] + " not supported");
}
}
} catch (Exception e) {
e.printStackTrace();
} finally {
if (connected) {
autoApiClient.closeConnection();
}
}
}
}