/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.accessanalyzer.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A proposed access control list grant configuration for an Amazon S3 bucket. For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#setting-acls">How to Specify an ACL</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3BucketAclGrantConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<S3BucketAclGrantConfiguration.Builder, S3BucketAclGrantConfiguration> {
    private static final SdkField<AclGrantee> GRANTEE_FIELD = SdkField.<AclGrantee> builder(MarshallingType.SDK_POJO)
            .memberName("grantee").getter(getter(S3BucketAclGrantConfiguration::grantee)).setter(setter(Builder::grantee))
            .constructor(AclGrantee::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("grantee").build()).build();

    private static final SdkField<String> PERMISSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("permission").getter(getter(S3BucketAclGrantConfiguration::permissionAsString))
            .setter(setter(Builder::permission))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("permission").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(GRANTEE_FIELD,
            PERMISSION_FIELD));

    private static final long serialVersionUID = 1L;

    private final AclGrantee grantee;

    private final String permission;

    private S3BucketAclGrantConfiguration(BuilderImpl builder) {
        this.grantee = builder.grantee;
        this.permission = builder.permission;
    }

    /**
     * <p>
     * The grantee to whom you’re assigning access rights.
     * </p>
     * 
     * @return The grantee to whom you’re assigning access rights.
     */
    public final AclGrantee grantee() {
        return grantee;
    }

    /**
     * <p>
     * The permissions being granted.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #permission} will
     * return {@link AclPermission#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #permissionAsString}.
     * </p>
     * 
     * @return The permissions being granted.
     * @see AclPermission
     */
    public final AclPermission permission() {
        return AclPermission.fromValue(permission);
    }

    /**
     * <p>
     * The permissions being granted.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #permission} will
     * return {@link AclPermission#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #permissionAsString}.
     * </p>
     * 
     * @return The permissions being granted.
     * @see AclPermission
     */
    public final String permissionAsString() {
        return permission;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(grantee());
        hashCode = 31 * hashCode + Objects.hashCode(permissionAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3BucketAclGrantConfiguration)) {
            return false;
        }
        S3BucketAclGrantConfiguration other = (S3BucketAclGrantConfiguration) obj;
        return Objects.equals(grantee(), other.grantee()) && Objects.equals(permissionAsString(), other.permissionAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("S3BucketAclGrantConfiguration").add("Grantee", grantee())
                .add("Permission", permissionAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "grantee":
            return Optional.ofNullable(clazz.cast(grantee()));
        case "permission":
            return Optional.ofNullable(clazz.cast(permissionAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<S3BucketAclGrantConfiguration, T> g) {
        return obj -> g.apply((S3BucketAclGrantConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3BucketAclGrantConfiguration> {
        /**
         * <p>
         * The grantee to whom you’re assigning access rights.
         * </p>
         * 
         * @param grantee
         *        The grantee to whom you’re assigning access rights.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder grantee(AclGrantee grantee);

        /**
         * <p>
         * The grantee to whom you’re assigning access rights.
         * </p>
         * This is a convenience that creates an instance of the {@link AclGrantee.Builder} avoiding the need to create
         * one manually via {@link AclGrantee#builder()}.
         *
         * When the {@link Consumer} completes, {@link AclGrantee.Builder#build()} is called immediately and its result
         * is passed to {@link #grantee(AclGrantee)}.
         * 
         * @param grantee
         *        a consumer that will call methods on {@link AclGrantee.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #grantee(AclGrantee)
         */
        default Builder grantee(Consumer<AclGrantee.Builder> grantee) {
            return grantee(AclGrantee.builder().applyMutation(grantee).build());
        }

        /**
         * <p>
         * The permissions being granted.
         * </p>
         * 
         * @param permission
         *        The permissions being granted.
         * @see AclPermission
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AclPermission
         */
        Builder permission(String permission);

        /**
         * <p>
         * The permissions being granted.
         * </p>
         * 
         * @param permission
         *        The permissions being granted.
         * @see AclPermission
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AclPermission
         */
        Builder permission(AclPermission permission);
    }

    static final class BuilderImpl implements Builder {
        private AclGrantee grantee;

        private String permission;

        private BuilderImpl() {
        }

        private BuilderImpl(S3BucketAclGrantConfiguration model) {
            grantee(model.grantee);
            permission(model.permission);
        }

        public final AclGrantee.Builder getGrantee() {
            return grantee != null ? grantee.toBuilder() : null;
        }

        public final void setGrantee(AclGrantee.BuilderImpl grantee) {
            this.grantee = grantee != null ? grantee.build() : null;
        }

        @Override
        @Transient
        public final Builder grantee(AclGrantee grantee) {
            this.grantee = grantee;
            return this;
        }

        public final String getPermission() {
            return permission;
        }

        public final void setPermission(String permission) {
            this.permission = permission;
        }

        @Override
        @Transient
        public final Builder permission(String permission) {
            this.permission = permission;
            return this;
        }

        @Override
        @Transient
        public final Builder permission(AclPermission permission) {
            this.permission(permission == null ? null : permission.toString());
            return this;
        }

        @Override
        public S3BucketAclGrantConfiguration build() {
            return new S3BucketAclGrantConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
