/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backupgateway;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.backupgateway.model.AccessDeniedException;
import software.amazon.awssdk.services.backupgateway.model.AssociateGatewayToServerRequest;
import software.amazon.awssdk.services.backupgateway.model.AssociateGatewayToServerResponse;
import software.amazon.awssdk.services.backupgateway.model.BackupGatewayException;
import software.amazon.awssdk.services.backupgateway.model.BackupGatewayRequest;
import software.amazon.awssdk.services.backupgateway.model.ConflictException;
import software.amazon.awssdk.services.backupgateway.model.CreateGatewayRequest;
import software.amazon.awssdk.services.backupgateway.model.CreateGatewayResponse;
import software.amazon.awssdk.services.backupgateway.model.DeleteGatewayRequest;
import software.amazon.awssdk.services.backupgateway.model.DeleteGatewayResponse;
import software.amazon.awssdk.services.backupgateway.model.DeleteHypervisorRequest;
import software.amazon.awssdk.services.backupgateway.model.DeleteHypervisorResponse;
import software.amazon.awssdk.services.backupgateway.model.DisassociateGatewayFromServerRequest;
import software.amazon.awssdk.services.backupgateway.model.DisassociateGatewayFromServerResponse;
import software.amazon.awssdk.services.backupgateway.model.ImportHypervisorConfigurationRequest;
import software.amazon.awssdk.services.backupgateway.model.ImportHypervisorConfigurationResponse;
import software.amazon.awssdk.services.backupgateway.model.InternalServerException;
import software.amazon.awssdk.services.backupgateway.model.ListGatewaysRequest;
import software.amazon.awssdk.services.backupgateway.model.ListGatewaysResponse;
import software.amazon.awssdk.services.backupgateway.model.ListHypervisorsRequest;
import software.amazon.awssdk.services.backupgateway.model.ListHypervisorsResponse;
import software.amazon.awssdk.services.backupgateway.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.backupgateway.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.backupgateway.model.ListVirtualMachinesRequest;
import software.amazon.awssdk.services.backupgateway.model.ListVirtualMachinesResponse;
import software.amazon.awssdk.services.backupgateway.model.PutMaintenanceStartTimeRequest;
import software.amazon.awssdk.services.backupgateway.model.PutMaintenanceStartTimeResponse;
import software.amazon.awssdk.services.backupgateway.model.ResourceNotFoundException;
import software.amazon.awssdk.services.backupgateway.model.TagResourceRequest;
import software.amazon.awssdk.services.backupgateway.model.TagResourceResponse;
import software.amazon.awssdk.services.backupgateway.model.TestHypervisorConfigurationRequest;
import software.amazon.awssdk.services.backupgateway.model.TestHypervisorConfigurationResponse;
import software.amazon.awssdk.services.backupgateway.model.UntagResourceRequest;
import software.amazon.awssdk.services.backupgateway.model.UntagResourceResponse;
import software.amazon.awssdk.services.backupgateway.model.UpdateGatewayInformationRequest;
import software.amazon.awssdk.services.backupgateway.model.UpdateGatewayInformationResponse;
import software.amazon.awssdk.services.backupgateway.model.UpdateHypervisorRequest;
import software.amazon.awssdk.services.backupgateway.model.UpdateHypervisorResponse;
import software.amazon.awssdk.services.backupgateway.model.ValidationException;
import software.amazon.awssdk.services.backupgateway.paginators.ListGatewaysIterable;
import software.amazon.awssdk.services.backupgateway.paginators.ListHypervisorsIterable;
import software.amazon.awssdk.services.backupgateway.paginators.ListVirtualMachinesIterable;
import software.amazon.awssdk.services.backupgateway.transform.AssociateGatewayToServerRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.CreateGatewayRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.DeleteGatewayRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.DeleteHypervisorRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.DisassociateGatewayFromServerRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.ImportHypervisorConfigurationRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.ListGatewaysRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.ListHypervisorsRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.ListVirtualMachinesRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.PutMaintenanceStartTimeRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.TestHypervisorConfigurationRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.UpdateGatewayInformationRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.UpdateHypervisorRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link BackupGatewayClient}.
 *
 * @see BackupGatewayClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultBackupGatewayClient implements BackupGatewayClient {
    private static final Logger log = Logger.loggerFor(DefaultBackupGatewayClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultBackupGatewayClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Associates a backup gateway with your server. After you complete the association process, you can back up and
     * restore your VMs through the gateway.
     * </p>
     *
     * @param associateGatewayToServerRequest
     * @return Result of the AssociateGatewayToServer operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws ConflictException
     *         The operation cannot proceed because it is not supported.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.AssociateGatewayToServer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/AssociateGatewayToServer"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateGatewayToServerResponse associateGatewayToServer(
            AssociateGatewayToServerRequest associateGatewayToServerRequest) throws ValidationException, ConflictException,
            InternalServerException, AwsServiceException, SdkClientException, BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateGatewayToServerResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateGatewayToServerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateGatewayToServerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateGatewayToServer");

            return clientHandler
                    .execute(new ClientExecutionParams<AssociateGatewayToServerRequest, AssociateGatewayToServerResponse>()
                            .withOperationName("AssociateGatewayToServer").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(associateGatewayToServerRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AssociateGatewayToServerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a backup gateway. After you create a gateway, you can associate it with a server using the
     * <code>AssociateGatewayToServer</code> operation.
     * </p>
     *
     * @param createGatewayRequest
     * @return Result of the CreateGateway operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.CreateGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/CreateGateway" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateGatewayResponse createGateway(CreateGatewayRequest createGatewayRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateGatewayResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateGatewayResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGateway");

            return clientHandler.execute(new ClientExecutionParams<CreateGatewayRequest, CreateGatewayResponse>()
                    .withOperationName("CreateGateway").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createGatewayRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateGatewayRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a backup gateway.
     * </p>
     *
     * @param deleteGatewayRequest
     * @return Result of the DeleteGateway operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action wasn't found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.DeleteGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/DeleteGateway" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteGatewayResponse deleteGateway(DeleteGatewayRequest deleteGatewayRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException, BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteGatewayResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteGatewayResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteGateway");

            return clientHandler.execute(new ClientExecutionParams<DeleteGatewayRequest, DeleteGatewayResponse>()
                    .withOperationName("DeleteGateway").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteGatewayRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteGatewayRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a hypervisor.
     * </p>
     *
     * @param deleteHypervisorRequest
     * @return Result of the DeleteHypervisor operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws AccessDeniedException
     *         The operation cannot proceed because you have insufficient permissions.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action wasn't found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.DeleteHypervisor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/DeleteHypervisor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteHypervisorResponse deleteHypervisor(DeleteHypervisorRequest deleteHypervisorRequest) throws ValidationException,
            InternalServerException, AccessDeniedException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteHypervisorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteHypervisorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteHypervisorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteHypervisor");

            return clientHandler.execute(new ClientExecutionParams<DeleteHypervisorRequest, DeleteHypervisorResponse>()
                    .withOperationName("DeleteHypervisor").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteHypervisorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteHypervisorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disassociates a backup gateway from the specified server. After the disassociation process finishes, the gateway
     * can no longer access the virtual machines on the server.
     * </p>
     *
     * @param disassociateGatewayFromServerRequest
     * @return Result of the DisassociateGatewayFromServer operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws ConflictException
     *         The operation cannot proceed because it is not supported.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action wasn't found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.DisassociateGatewayFromServer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/DisassociateGatewayFromServer"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateGatewayFromServerResponse disassociateGatewayFromServer(
            DisassociateGatewayFromServerRequest disassociateGatewayFromServerRequest) throws ValidationException,
            ConflictException, InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateGatewayFromServerResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateGatewayFromServerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateGatewayFromServerRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateGatewayFromServer");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateGatewayFromServerRequest, DisassociateGatewayFromServerResponse>()
                            .withOperationName("DisassociateGatewayFromServer").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(disassociateGatewayFromServerRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateGatewayFromServerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Connect to a hypervisor by importing its configuration.
     * </p>
     *
     * @param importHypervisorConfigurationRequest
     * @return Result of the ImportHypervisorConfiguration operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws AccessDeniedException
     *         The operation cannot proceed because you have insufficient permissions.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.ImportHypervisorConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/ImportHypervisorConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ImportHypervisorConfigurationResponse importHypervisorConfiguration(
            ImportHypervisorConfigurationRequest importHypervisorConfigurationRequest) throws ValidationException,
            InternalServerException, AccessDeniedException, AwsServiceException, SdkClientException, BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ImportHypervisorConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ImportHypervisorConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                importHypervisorConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ImportHypervisorConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<ImportHypervisorConfigurationRequest, ImportHypervisorConfigurationResponse>()
                            .withOperationName("ImportHypervisorConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(importHypervisorConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ImportHypervisorConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists backup gateways owned by an Amazon Web Services account in an Amazon Web Services Region. The returned list
     * is ordered by gateway Amazon Resource Name (ARN).
     * </p>
     *
     * @param listGatewaysRequest
     * @return Result of the ListGateways operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.ListGateways
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/ListGateways" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListGatewaysResponse listGateways(ListGatewaysRequest listGatewaysRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListGatewaysResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListGatewaysResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGatewaysRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGateways");

            return clientHandler.execute(new ClientExecutionParams<ListGatewaysRequest, ListGatewaysResponse>()
                    .withOperationName("ListGateways").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listGatewaysRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListGatewaysRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists backup gateways owned by an Amazon Web Services account in an Amazon Web Services Region. The returned list
     * is ordered by gateway Amazon Resource Name (ARN).
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listGateways(software.amazon.awssdk.services.backupgateway.model.ListGatewaysRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupgateway.paginators.ListGatewaysIterable responses = client.listGatewaysPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.backupgateway.paginators.ListGatewaysIterable responses = client
     *             .listGatewaysPaginator(request);
     *     for (software.amazon.awssdk.services.backupgateway.model.ListGatewaysResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupgateway.paginators.ListGatewaysIterable responses = client.listGatewaysPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listGateways(software.amazon.awssdk.services.backupgateway.model.ListGatewaysRequest)} operation.</b>
     * </p>
     *
     * @param listGatewaysRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.ListGateways
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/ListGateways" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListGatewaysIterable listGatewaysPaginator(ListGatewaysRequest listGatewaysRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, BackupGatewayException {
        return new ListGatewaysIterable(this, applyPaginatorUserAgent(listGatewaysRequest));
    }

    /**
     * <p>
     * Lists your hypervisors.
     * </p>
     *
     * @param listHypervisorsRequest
     * @return Result of the ListHypervisors operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.ListHypervisors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/ListHypervisors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListHypervisorsResponse listHypervisors(ListHypervisorsRequest listHypervisorsRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListHypervisorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListHypervisorsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listHypervisorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListHypervisors");

            return clientHandler.execute(new ClientExecutionParams<ListHypervisorsRequest, ListHypervisorsResponse>()
                    .withOperationName("ListHypervisors").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listHypervisorsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListHypervisorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists your hypervisors.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listHypervisors(software.amazon.awssdk.services.backupgateway.model.ListHypervisorsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupgateway.paginators.ListHypervisorsIterable responses = client.listHypervisorsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.backupgateway.paginators.ListHypervisorsIterable responses = client
     *             .listHypervisorsPaginator(request);
     *     for (software.amazon.awssdk.services.backupgateway.model.ListHypervisorsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupgateway.paginators.ListHypervisorsIterable responses = client.listHypervisorsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listHypervisors(software.amazon.awssdk.services.backupgateway.model.ListHypervisorsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listHypervisorsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.ListHypervisors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/ListHypervisors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListHypervisorsIterable listHypervisorsPaginator(ListHypervisorsRequest listHypervisorsRequest)
            throws ValidationException, InternalServerException, AwsServiceException, SdkClientException, BackupGatewayException {
        return new ListHypervisorsIterable(this, applyPaginatorUserAgent(listHypervisorsRequest));
    }

    /**
     * <p>
     * Lists the tags applied to the resource identified by its Amazon Resource Name (ARN).
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action wasn't found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists your virtual machines.
     * </p>
     *
     * @param listVirtualMachinesRequest
     * @return Result of the ListVirtualMachines operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.ListVirtualMachines
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/ListVirtualMachines"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListVirtualMachinesResponse listVirtualMachines(ListVirtualMachinesRequest listVirtualMachinesRequest)
            throws ValidationException, InternalServerException, AwsServiceException, SdkClientException, BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListVirtualMachinesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListVirtualMachinesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVirtualMachinesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVirtualMachines");

            return clientHandler.execute(new ClientExecutionParams<ListVirtualMachinesRequest, ListVirtualMachinesResponse>()
                    .withOperationName("ListVirtualMachines").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listVirtualMachinesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListVirtualMachinesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists your virtual machines.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listVirtualMachines(software.amazon.awssdk.services.backupgateway.model.ListVirtualMachinesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupgateway.paginators.ListVirtualMachinesIterable responses = client.listVirtualMachinesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.backupgateway.paginators.ListVirtualMachinesIterable responses = client
     *             .listVirtualMachinesPaginator(request);
     *     for (software.amazon.awssdk.services.backupgateway.model.ListVirtualMachinesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupgateway.paginators.ListVirtualMachinesIterable responses = client.listVirtualMachinesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listVirtualMachines(software.amazon.awssdk.services.backupgateway.model.ListVirtualMachinesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listVirtualMachinesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.ListVirtualMachines
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/ListVirtualMachines"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListVirtualMachinesIterable listVirtualMachinesPaginator(ListVirtualMachinesRequest listVirtualMachinesRequest)
            throws ValidationException, InternalServerException, AwsServiceException, SdkClientException, BackupGatewayException {
        return new ListVirtualMachinesIterable(this, applyPaginatorUserAgent(listVirtualMachinesRequest));
    }

    /**
     * <p>
     * Set the maintenance start time for a gateway.
     * </p>
     *
     * @param putMaintenanceStartTimeRequest
     * @return Result of the PutMaintenanceStartTime operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws ConflictException
     *         The operation cannot proceed because it is not supported.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action wasn't found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.PutMaintenanceStartTime
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/PutMaintenanceStartTime"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutMaintenanceStartTimeResponse putMaintenanceStartTime(PutMaintenanceStartTimeRequest putMaintenanceStartTimeRequest)
            throws ValidationException, ConflictException, InternalServerException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutMaintenanceStartTimeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutMaintenanceStartTimeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putMaintenanceStartTimeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutMaintenanceStartTime");

            return clientHandler
                    .execute(new ClientExecutionParams<PutMaintenanceStartTimeRequest, PutMaintenanceStartTimeResponse>()
                            .withOperationName("PutMaintenanceStartTime").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(putMaintenanceStartTimeRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutMaintenanceStartTimeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Tag the resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action wasn't found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException, BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Tests your hypervisor configuration to validate that backup gateway can connect with the hypervisor and its
     * resources.
     * </p>
     *
     * @param testHypervisorConfigurationRequest
     * @return Result of the TestHypervisorConfiguration operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws ConflictException
     *         The operation cannot proceed because it is not supported.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action wasn't found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.TestHypervisorConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/TestHypervisorConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public TestHypervisorConfigurationResponse testHypervisorConfiguration(
            TestHypervisorConfigurationRequest testHypervisorConfigurationRequest) throws ValidationException, ConflictException,
            InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException, BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TestHypervisorConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, TestHypervisorConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, testHypervisorConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TestHypervisorConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<TestHypervisorConfigurationRequest, TestHypervisorConfigurationResponse>()
                            .withOperationName("TestHypervisorConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(testHypervisorConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new TestHypervisorConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from the resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action wasn't found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException, BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a gateway's name. Specify which gateway to update using the Amazon Resource Name (ARN) of the gateway in
     * your request.
     * </p>
     *
     * @param updateGatewayInformationRequest
     * @return Result of the UpdateGatewayInformation operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws ConflictException
     *         The operation cannot proceed because it is not supported.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action wasn't found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.UpdateGatewayInformation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/UpdateGatewayInformation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateGatewayInformationResponse updateGatewayInformation(
            UpdateGatewayInformationRequest updateGatewayInformationRequest) throws ValidationException, ConflictException,
            InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException, BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateGatewayInformationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateGatewayInformationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGatewayInformationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGatewayInformation");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateGatewayInformationRequest, UpdateGatewayInformationResponse>()
                            .withOperationName("UpdateGatewayInformation").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateGatewayInformationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateGatewayInformationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a hypervisor metadata, including its host, username, and password. Specify which hypervisor to update
     * using the Amazon Resource Name (ARN) of the hypervisor in your request.
     * </p>
     *
     * @param updateHypervisorRequest
     * @return Result of the UpdateHypervisor operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws AccessDeniedException
     *         The operation cannot proceed because you have insufficient permissions.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action wasn't found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.UpdateHypervisor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/UpdateHypervisor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateHypervisorResponse updateHypervisor(UpdateHypervisorRequest updateHypervisorRequest) throws ValidationException,
            InternalServerException, AccessDeniedException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateHypervisorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateHypervisorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateHypervisorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateHypervisor");

            return clientHandler.execute(new ClientExecutionParams<UpdateHypervisorRequest, UpdateHypervisorResponse>()
                    .withOperationName("UpdateHypervisor").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateHypervisorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateHypervisorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(BackupGatewayException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BackupGatewayRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
