/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrock.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Details about topics for the guardrail to identify and deny.
 * </p>
 * <p>
 * This data type is used in the following API operations:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a href=
 * "https://docs.aws.amazon.com/bedrock/latest/APIReference/API_CreateGuardrail.html#API_CreateGuardrail_RequestSyntax"
 * >CreateGuardrail request body</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href=
 * "https://docs.aws.amazon.com/bedrock/latest/APIReference/API_UpdateGuardrail.html#API_UpdateGuardrail_RequestSyntax"
 * >UpdateGuardrail request body</a>
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GuardrailTopicConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<GuardrailTopicConfig.Builder, GuardrailTopicConfig> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(GuardrailTopicConfig::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> DEFINITION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("definition").getter(getter(GuardrailTopicConfig::definition)).setter(setter(Builder::definition))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("definition").build()).build();

    private static final SdkField<List<String>> EXAMPLES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("examples")
            .getter(getter(GuardrailTopicConfig::examples))
            .setter(setter(Builder::examples))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("examples").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(GuardrailTopicConfig::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, DEFINITION_FIELD,
            EXAMPLES_FIELD, TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String definition;

    private final List<String> examples;

    private final String type;

    private GuardrailTopicConfig(BuilderImpl builder) {
        this.name = builder.name;
        this.definition = builder.definition;
        this.examples = builder.examples;
        this.type = builder.type;
    }

    /**
     * <p>
     * The name of the topic to deny.
     * </p>
     * 
     * @return The name of the topic to deny.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * A definition of the topic to deny.
     * </p>
     * 
     * @return A definition of the topic to deny.
     */
    public final String definition() {
        return definition;
    }

    /**
     * For responses, this returns true if the service returned a value for the Examples property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasExamples() {
        return examples != null && !(examples instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of prompts, each of which is an example of a prompt that can be categorized as belonging to the topic.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExamples} method.
     * </p>
     * 
     * @return A list of prompts, each of which is an example of a prompt that can be categorized as belonging to the
     *         topic.
     */
    public final List<String> examples() {
        return examples;
    }

    /**
     * <p>
     * Specifies to deny the topic.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link GuardrailTopicType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Specifies to deny the topic.
     * @see GuardrailTopicType
     */
    public final GuardrailTopicType type() {
        return GuardrailTopicType.fromValue(type);
    }

    /**
     * <p>
     * Specifies to deny the topic.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link GuardrailTopicType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Specifies to deny the topic.
     * @see GuardrailTopicType
     */
    public final String typeAsString() {
        return type;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(definition());
        hashCode = 31 * hashCode + Objects.hashCode(hasExamples() ? examples() : null);
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GuardrailTopicConfig)) {
            return false;
        }
        GuardrailTopicConfig other = (GuardrailTopicConfig) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(definition(), other.definition())
                && hasExamples() == other.hasExamples() && Objects.equals(examples(), other.examples())
                && Objects.equals(typeAsString(), other.typeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GuardrailTopicConfig").add("Name", name() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Definition", definition() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Examples", examples() == null ? null : "*** Sensitive Data Redacted ***").add("Type", typeAsString())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "definition":
            return Optional.ofNullable(clazz.cast(definition()));
        case "examples":
            return Optional.ofNullable(clazz.cast(examples()));
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GuardrailTopicConfig, T> g) {
        return obj -> g.apply((GuardrailTopicConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GuardrailTopicConfig> {
        /**
         * <p>
         * The name of the topic to deny.
         * </p>
         * 
         * @param name
         *        The name of the topic to deny.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A definition of the topic to deny.
         * </p>
         * 
         * @param definition
         *        A definition of the topic to deny.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder definition(String definition);

        /**
         * <p>
         * A list of prompts, each of which is an example of a prompt that can be categorized as belonging to the topic.
         * </p>
         * 
         * @param examples
         *        A list of prompts, each of which is an example of a prompt that can be categorized as belonging to the
         *        topic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder examples(Collection<String> examples);

        /**
         * <p>
         * A list of prompts, each of which is an example of a prompt that can be categorized as belonging to the topic.
         * </p>
         * 
         * @param examples
         *        A list of prompts, each of which is an example of a prompt that can be categorized as belonging to the
         *        topic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder examples(String... examples);

        /**
         * <p>
         * Specifies to deny the topic.
         * </p>
         * 
         * @param type
         *        Specifies to deny the topic.
         * @see GuardrailTopicType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GuardrailTopicType
         */
        Builder type(String type);

        /**
         * <p>
         * Specifies to deny the topic.
         * </p>
         * 
         * @param type
         *        Specifies to deny the topic.
         * @see GuardrailTopicType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GuardrailTopicType
         */
        Builder type(GuardrailTopicType type);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String definition;

        private List<String> examples = DefaultSdkAutoConstructList.getInstance();

        private String type;

        private BuilderImpl() {
        }

        private BuilderImpl(GuardrailTopicConfig model) {
            name(model.name);
            definition(model.definition);
            examples(model.examples);
            type(model.type);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDefinition() {
            return definition;
        }

        public final void setDefinition(String definition) {
            this.definition = definition;
        }

        @Override
        public final Builder definition(String definition) {
            this.definition = definition;
            return this;
        }

        public final Collection<String> getExamples() {
            if (examples instanceof SdkAutoConstructList) {
                return null;
            }
            return examples;
        }

        public final void setExamples(Collection<String> examples) {
            this.examples = GuardrailTopicExamplesCopier.copy(examples);
        }

        @Override
        public final Builder examples(Collection<String> examples) {
            this.examples = GuardrailTopicExamplesCopier.copy(examples);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder examples(String... examples) {
            examples(Arrays.asList(examples));
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(GuardrailTopicType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        @Override
        public GuardrailTopicConfig build() {
            return new GuardrailTopicConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
