/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateStackInstancesRequest extends CloudFormationRequest implements
        ToCopyableBuilder<CreateStackInstancesRequest.Builder, CreateStackInstancesRequest> {
    private static final SdkField<String> STACK_SET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StackSetName").getter(getter(CreateStackInstancesRequest::stackSetName))
            .setter(setter(Builder::stackSetName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackSetName").build()).build();

    private static final SdkField<List<String>> ACCOUNTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Accounts")
            .getter(getter(CreateStackInstancesRequest::accounts))
            .setter(setter(Builder::accounts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Accounts").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<DeploymentTargets> DEPLOYMENT_TARGETS_FIELD = SdkField
            .<DeploymentTargets> builder(MarshallingType.SDK_POJO).memberName("DeploymentTargets")
            .getter(getter(CreateStackInstancesRequest::deploymentTargets)).setter(setter(Builder::deploymentTargets))
            .constructor(DeploymentTargets::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeploymentTargets").build()).build();

    private static final SdkField<List<String>> REGIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Regions")
            .getter(getter(CreateStackInstancesRequest::regions))
            .setter(setter(Builder::regions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Regions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Parameter>> PARAMETER_OVERRIDES_FIELD = SdkField
            .<List<Parameter>> builder(MarshallingType.LIST)
            .memberName("ParameterOverrides")
            .getter(getter(CreateStackInstancesRequest::parameterOverrides))
            .setter(setter(Builder::parameterOverrides))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ParameterOverrides").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Parameter> builder(MarshallingType.SDK_POJO)
                                            .constructor(Parameter::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<StackSetOperationPreferences> OPERATION_PREFERENCES_FIELD = SdkField
            .<StackSetOperationPreferences> builder(MarshallingType.SDK_POJO).memberName("OperationPreferences")
            .getter(getter(CreateStackInstancesRequest::operationPreferences)).setter(setter(Builder::operationPreferences))
            .constructor(StackSetOperationPreferences::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperationPreferences").build())
            .build();

    private static final SdkField<String> OPERATION_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("OperationId")
            .getter(getter(CreateStackInstancesRequest::operationId))
            .setter(setter(Builder::operationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperationId").build(),
                    DefaultValueTrait.idempotencyToken()).build();

    private static final SdkField<String> CALL_AS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("CallAs")
            .getter(getter(CreateStackInstancesRequest::callAsAsString)).setter(setter(Builder::callAs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CallAs").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STACK_SET_NAME_FIELD,
            ACCOUNTS_FIELD, DEPLOYMENT_TARGETS_FIELD, REGIONS_FIELD, PARAMETER_OVERRIDES_FIELD, OPERATION_PREFERENCES_FIELD,
            OPERATION_ID_FIELD, CALL_AS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String stackSetName;

    private final List<String> accounts;

    private final DeploymentTargets deploymentTargets;

    private final List<String> regions;

    private final List<Parameter> parameterOverrides;

    private final StackSetOperationPreferences operationPreferences;

    private final String operationId;

    private final String callAs;

    private CreateStackInstancesRequest(BuilderImpl builder) {
        super(builder);
        this.stackSetName = builder.stackSetName;
        this.accounts = builder.accounts;
        this.deploymentTargets = builder.deploymentTargets;
        this.regions = builder.regions;
        this.parameterOverrides = builder.parameterOverrides;
        this.operationPreferences = builder.operationPreferences;
        this.operationId = builder.operationId;
        this.callAs = builder.callAs;
    }

    /**
     * <p>
     * The name or unique ID of the StackSet that you want to create stack instances from.
     * </p>
     * 
     * @return The name or unique ID of the StackSet that you want to create stack instances from.
     */
    public final String stackSetName() {
        return stackSetName;
    }

    /**
     * For responses, this returns true if the service returned a value for the Accounts property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAccounts() {
        return accounts != null && !(accounts instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * [Self-managed permissions] The account IDs of one or more Amazon Web Services accounts that you want to create
     * stack instances in the specified Region(s) for.
     * </p>
     * <p>
     * You can specify <code>Accounts</code> or <code>DeploymentTargets</code>, but not both.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAccounts} method.
     * </p>
     * 
     * @return [Self-managed permissions] The account IDs of one or more Amazon Web Services accounts that you want to
     *         create stack instances in the specified Region(s) for.</p>
     *         <p>
     *         You can specify <code>Accounts</code> or <code>DeploymentTargets</code>, but not both.
     */
    public final List<String> accounts() {
        return accounts;
    }

    /**
     * <p>
     * [Service-managed permissions] The Organizations accounts in which to create stack instances in the specified
     * Amazon Web Services Regions.
     * </p>
     * <p>
     * You can specify <code>Accounts</code> or <code>DeploymentTargets</code>, but not both.
     * </p>
     * 
     * @return [Service-managed permissions] The Organizations accounts in which to create stack instances in the
     *         specified Amazon Web Services Regions.</p>
     *         <p>
     *         You can specify <code>Accounts</code> or <code>DeploymentTargets</code>, but not both.
     */
    public final DeploymentTargets deploymentTargets() {
        return deploymentTargets;
    }

    /**
     * For responses, this returns true if the service returned a value for the Regions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasRegions() {
        return regions != null && !(regions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The names of one or more Amazon Web Services Regions where you want to create stack instances using the specified
     * Amazon Web Services accounts.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRegions} method.
     * </p>
     * 
     * @return The names of one or more Amazon Web Services Regions where you want to create stack instances using the
     *         specified Amazon Web Services accounts.
     */
    public final List<String> regions() {
        return regions;
    }

    /**
     * For responses, this returns true if the service returned a value for the ParameterOverrides property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasParameterOverrides() {
        return parameterOverrides != null && !(parameterOverrides instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of StackSet parameters whose values you want to override in the selected stack instances.
     * </p>
     * <p>
     * Any overridden parameter values will be applied to all stack instances in the specified accounts and Amazon Web
     * Services Regions. When specifying parameters and their values, be aware of how CloudFormation sets parameter
     * values during stack instance operations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * To override the current value for a parameter, include the parameter and specify its value.
     * </p>
     * </li>
     * <li>
     * <p>
     * To leave an overridden parameter set to its present value, include the parameter and specify
     * <code>UsePreviousValue</code> as <code>true</code>. (You can't specify both a value and set
     * <code>UsePreviousValue</code> to <code>true</code>.)
     * </p>
     * </li>
     * <li>
     * <p>
     * To set an overridden parameter back to the value specified in the StackSet, specify a parameter list but don't
     * include the parameter in the list.
     * </p>
     * </li>
     * <li>
     * <p>
     * To leave all parameters set to their present values, don't specify this property at all.
     * </p>
     * </li>
     * </ul>
     * <p>
     * During StackSet updates, any parameter values overridden for a stack instance aren't updated, but retain their
     * overridden value.
     * </p>
     * <p>
     * You can only override the parameter <i>values</i> that are specified in the StackSet; to add or delete a
     * parameter itself, use <a
     * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UpdateStackSet.html"
     * >UpdateStackSet</a> to update the StackSet template.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasParameterOverrides} method.
     * </p>
     * 
     * @return A list of StackSet parameters whose values you want to override in the selected stack instances.</p>
     *         <p>
     *         Any overridden parameter values will be applied to all stack instances in the specified accounts and
     *         Amazon Web Services Regions. When specifying parameters and their values, be aware of how CloudFormation
     *         sets parameter values during stack instance operations:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         To override the current value for a parameter, include the parameter and specify its value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To leave an overridden parameter set to its present value, include the parameter and specify
     *         <code>UsePreviousValue</code> as <code>true</code>. (You can't specify both a value and set
     *         <code>UsePreviousValue</code> to <code>true</code>.)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To set an overridden parameter back to the value specified in the StackSet, specify a parameter list but
     *         don't include the parameter in the list.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To leave all parameters set to their present values, don't specify this property at all.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         During StackSet updates, any parameter values overridden for a stack instance aren't updated, but retain
     *         their overridden value.
     *         </p>
     *         <p>
     *         You can only override the parameter <i>values</i> that are specified in the StackSet; to add or delete a
     *         parameter itself, use <a
     *         href="https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UpdateStackSet.html"
     *         >UpdateStackSet</a> to update the StackSet template.
     */
    public final List<Parameter> parameterOverrides() {
        return parameterOverrides;
    }

    /**
     * <p>
     * Preferences for how CloudFormation performs this StackSet operation.
     * </p>
     * 
     * @return Preferences for how CloudFormation performs this StackSet operation.
     */
    public final StackSetOperationPreferences operationPreferences() {
        return operationPreferences;
    }

    /**
     * <p>
     * The unique identifier for this StackSet operation.
     * </p>
     * <p>
     * The operation ID also functions as an idempotency token, to ensure that CloudFormation performs the StackSet
     * operation only once, even if you retry the request multiple times. You might retry StackSet operation requests to
     * ensure that CloudFormation successfully received them.
     * </p>
     * <p>
     * If you don't specify an operation ID, the SDK generates one automatically.
     * </p>
     * <p>
     * Repeating this StackSet operation with a new operation ID retries all stack instances whose status is
     * <code>OUTDATED</code>.
     * </p>
     * 
     * @return The unique identifier for this StackSet operation.</p>
     *         <p>
     *         The operation ID also functions as an idempotency token, to ensure that CloudFormation performs the
     *         StackSet operation only once, even if you retry the request multiple times. You might retry StackSet
     *         operation requests to ensure that CloudFormation successfully received them.
     *         </p>
     *         <p>
     *         If you don't specify an operation ID, the SDK generates one automatically.
     *         </p>
     *         <p>
     *         Repeating this StackSet operation with a new operation ID retries all stack instances whose status is
     *         <code>OUTDATED</code>.
     */
    public final String operationId() {
        return operationId;
    }

    /**
     * <p>
     * [Service-managed permissions] Specifies whether you are acting as an account administrator in the organization's
     * management account or as a delegated administrator in a member account.
     * </p>
     * <p>
     * By default, <code>SELF</code> is specified. Use <code>SELF</code> for StackSets with self-managed permissions.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you are signed in to the management account, specify <code>SELF</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you are signed in to a delegated administrator account, specify <code>DELEGATED_ADMIN</code>.
     * </p>
     * <p>
     * Your Amazon Web Services account must be registered as a delegated administrator in the management account. For
     * more information, see <a
     * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-orgs-delegated-admin.html"
     * >Register a delegated administrator</a> in the <i>CloudFormation User Guide</i>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #callAs} will
     * return {@link CallAs#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #callAsAsString}.
     * </p>
     * 
     * @return [Service-managed permissions] Specifies whether you are acting as an account administrator in the
     *         organization's management account or as a delegated administrator in a member account.</p>
     *         <p>
     *         By default, <code>SELF</code> is specified. Use <code>SELF</code> for StackSets with self-managed
     *         permissions.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If you are signed in to the management account, specify <code>SELF</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you are signed in to a delegated administrator account, specify <code>DELEGATED_ADMIN</code>.
     *         </p>
     *         <p>
     *         Your Amazon Web Services account must be registered as a delegated administrator in the management
     *         account. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-orgs-delegated-admin.html"
     *         >Register a delegated administrator</a> in the <i>CloudFormation User Guide</i>.
     *         </p>
     *         </li>
     * @see CallAs
     */
    public final CallAs callAs() {
        return CallAs.fromValue(callAs);
    }

    /**
     * <p>
     * [Service-managed permissions] Specifies whether you are acting as an account administrator in the organization's
     * management account or as a delegated administrator in a member account.
     * </p>
     * <p>
     * By default, <code>SELF</code> is specified. Use <code>SELF</code> for StackSets with self-managed permissions.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you are signed in to the management account, specify <code>SELF</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you are signed in to a delegated administrator account, specify <code>DELEGATED_ADMIN</code>.
     * </p>
     * <p>
     * Your Amazon Web Services account must be registered as a delegated administrator in the management account. For
     * more information, see <a
     * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-orgs-delegated-admin.html"
     * >Register a delegated administrator</a> in the <i>CloudFormation User Guide</i>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #callAs} will
     * return {@link CallAs#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #callAsAsString}.
     * </p>
     * 
     * @return [Service-managed permissions] Specifies whether you are acting as an account administrator in the
     *         organization's management account or as a delegated administrator in a member account.</p>
     *         <p>
     *         By default, <code>SELF</code> is specified. Use <code>SELF</code> for StackSets with self-managed
     *         permissions.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If you are signed in to the management account, specify <code>SELF</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you are signed in to a delegated administrator account, specify <code>DELEGATED_ADMIN</code>.
     *         </p>
     *         <p>
     *         Your Amazon Web Services account must be registered as a delegated administrator in the management
     *         account. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-orgs-delegated-admin.html"
     *         >Register a delegated administrator</a> in the <i>CloudFormation User Guide</i>.
     *         </p>
     *         </li>
     * @see CallAs
     */
    public final String callAsAsString() {
        return callAs;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(stackSetName());
        hashCode = 31 * hashCode + Objects.hashCode(hasAccounts() ? accounts() : null);
        hashCode = 31 * hashCode + Objects.hashCode(deploymentTargets());
        hashCode = 31 * hashCode + Objects.hashCode(hasRegions() ? regions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasParameterOverrides() ? parameterOverrides() : null);
        hashCode = 31 * hashCode + Objects.hashCode(operationPreferences());
        hashCode = 31 * hashCode + Objects.hashCode(operationId());
        hashCode = 31 * hashCode + Objects.hashCode(callAsAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateStackInstancesRequest)) {
            return false;
        }
        CreateStackInstancesRequest other = (CreateStackInstancesRequest) obj;
        return Objects.equals(stackSetName(), other.stackSetName()) && hasAccounts() == other.hasAccounts()
                && Objects.equals(accounts(), other.accounts()) && Objects.equals(deploymentTargets(), other.deploymentTargets())
                && hasRegions() == other.hasRegions() && Objects.equals(regions(), other.regions())
                && hasParameterOverrides() == other.hasParameterOverrides()
                && Objects.equals(parameterOverrides(), other.parameterOverrides())
                && Objects.equals(operationPreferences(), other.operationPreferences())
                && Objects.equals(operationId(), other.operationId()) && Objects.equals(callAsAsString(), other.callAsAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateStackInstancesRequest").add("StackSetName", stackSetName())
                .add("Accounts", hasAccounts() ? accounts() : null).add("DeploymentTargets", deploymentTargets())
                .add("Regions", hasRegions() ? regions() : null)
                .add("ParameterOverrides", hasParameterOverrides() ? parameterOverrides() : null)
                .add("OperationPreferences", operationPreferences()).add("OperationId", operationId())
                .add("CallAs", callAsAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StackSetName":
            return Optional.ofNullable(clazz.cast(stackSetName()));
        case "Accounts":
            return Optional.ofNullable(clazz.cast(accounts()));
        case "DeploymentTargets":
            return Optional.ofNullable(clazz.cast(deploymentTargets()));
        case "Regions":
            return Optional.ofNullable(clazz.cast(regions()));
        case "ParameterOverrides":
            return Optional.ofNullable(clazz.cast(parameterOverrides()));
        case "OperationPreferences":
            return Optional.ofNullable(clazz.cast(operationPreferences()));
        case "OperationId":
            return Optional.ofNullable(clazz.cast(operationId()));
        case "CallAs":
            return Optional.ofNullable(clazz.cast(callAsAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("StackSetName", STACK_SET_NAME_FIELD);
        map.put("Accounts", ACCOUNTS_FIELD);
        map.put("DeploymentTargets", DEPLOYMENT_TARGETS_FIELD);
        map.put("Regions", REGIONS_FIELD);
        map.put("ParameterOverrides", PARAMETER_OVERRIDES_FIELD);
        map.put("OperationPreferences", OPERATION_PREFERENCES_FIELD);
        map.put("OperationId", OPERATION_ID_FIELD);
        map.put("CallAs", CALL_AS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CreateStackInstancesRequest, T> g) {
        return obj -> g.apply((CreateStackInstancesRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends CloudFormationRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, CreateStackInstancesRequest> {
        /**
         * <p>
         * The name or unique ID of the StackSet that you want to create stack instances from.
         * </p>
         * 
         * @param stackSetName
         *        The name or unique ID of the StackSet that you want to create stack instances from.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackSetName(String stackSetName);

        /**
         * <p>
         * [Self-managed permissions] The account IDs of one or more Amazon Web Services accounts that you want to
         * create stack instances in the specified Region(s) for.
         * </p>
         * <p>
         * You can specify <code>Accounts</code> or <code>DeploymentTargets</code>, but not both.
         * </p>
         * 
         * @param accounts
         *        [Self-managed permissions] The account IDs of one or more Amazon Web Services accounts that you want
         *        to create stack instances in the specified Region(s) for.</p>
         *        <p>
         *        You can specify <code>Accounts</code> or <code>DeploymentTargets</code>, but not both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accounts(Collection<String> accounts);

        /**
         * <p>
         * [Self-managed permissions] The account IDs of one or more Amazon Web Services accounts that you want to
         * create stack instances in the specified Region(s) for.
         * </p>
         * <p>
         * You can specify <code>Accounts</code> or <code>DeploymentTargets</code>, but not both.
         * </p>
         * 
         * @param accounts
         *        [Self-managed permissions] The account IDs of one or more Amazon Web Services accounts that you want
         *        to create stack instances in the specified Region(s) for.</p>
         *        <p>
         *        You can specify <code>Accounts</code> or <code>DeploymentTargets</code>, but not both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accounts(String... accounts);

        /**
         * <p>
         * [Service-managed permissions] The Organizations accounts in which to create stack instances in the specified
         * Amazon Web Services Regions.
         * </p>
         * <p>
         * You can specify <code>Accounts</code> or <code>DeploymentTargets</code>, but not both.
         * </p>
         * 
         * @param deploymentTargets
         *        [Service-managed permissions] The Organizations accounts in which to create stack instances in the
         *        specified Amazon Web Services Regions.</p>
         *        <p>
         *        You can specify <code>Accounts</code> or <code>DeploymentTargets</code>, but not both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deploymentTargets(DeploymentTargets deploymentTargets);

        /**
         * <p>
         * [Service-managed permissions] The Organizations accounts in which to create stack instances in the specified
         * Amazon Web Services Regions.
         * </p>
         * <p>
         * You can specify <code>Accounts</code> or <code>DeploymentTargets</code>, but not both.
         * </p>
         * This is a convenience method that creates an instance of the {@link DeploymentTargets.Builder} avoiding the
         * need to create one manually via {@link DeploymentTargets#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DeploymentTargets.Builder#build()} is called immediately and its
         * result is passed to {@link #deploymentTargets(DeploymentTargets)}.
         * 
         * @param deploymentTargets
         *        a consumer that will call methods on {@link DeploymentTargets.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #deploymentTargets(DeploymentTargets)
         */
        default Builder deploymentTargets(Consumer<DeploymentTargets.Builder> deploymentTargets) {
            return deploymentTargets(DeploymentTargets.builder().applyMutation(deploymentTargets).build());
        }

        /**
         * <p>
         * The names of one or more Amazon Web Services Regions where you want to create stack instances using the
         * specified Amazon Web Services accounts.
         * </p>
         * 
         * @param regions
         *        The names of one or more Amazon Web Services Regions where you want to create stack instances using
         *        the specified Amazon Web Services accounts.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder regions(Collection<String> regions);

        /**
         * <p>
         * The names of one or more Amazon Web Services Regions where you want to create stack instances using the
         * specified Amazon Web Services accounts.
         * </p>
         * 
         * @param regions
         *        The names of one or more Amazon Web Services Regions where you want to create stack instances using
         *        the specified Amazon Web Services accounts.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder regions(String... regions);

        /**
         * <p>
         * A list of StackSet parameters whose values you want to override in the selected stack instances.
         * </p>
         * <p>
         * Any overridden parameter values will be applied to all stack instances in the specified accounts and Amazon
         * Web Services Regions. When specifying parameters and their values, be aware of how CloudFormation sets
         * parameter values during stack instance operations:
         * </p>
         * <ul>
         * <li>
         * <p>
         * To override the current value for a parameter, include the parameter and specify its value.
         * </p>
         * </li>
         * <li>
         * <p>
         * To leave an overridden parameter set to its present value, include the parameter and specify
         * <code>UsePreviousValue</code> as <code>true</code>. (You can't specify both a value and set
         * <code>UsePreviousValue</code> to <code>true</code>.)
         * </p>
         * </li>
         * <li>
         * <p>
         * To set an overridden parameter back to the value specified in the StackSet, specify a parameter list but
         * don't include the parameter in the list.
         * </p>
         * </li>
         * <li>
         * <p>
         * To leave all parameters set to their present values, don't specify this property at all.
         * </p>
         * </li>
         * </ul>
         * <p>
         * During StackSet updates, any parameter values overridden for a stack instance aren't updated, but retain
         * their overridden value.
         * </p>
         * <p>
         * You can only override the parameter <i>values</i> that are specified in the StackSet; to add or delete a
         * parameter itself, use <a
         * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UpdateStackSet.html"
         * >UpdateStackSet</a> to update the StackSet template.
         * </p>
         * 
         * @param parameterOverrides
         *        A list of StackSet parameters whose values you want to override in the selected stack instances.</p>
         *        <p>
         *        Any overridden parameter values will be applied to all stack instances in the specified accounts and
         *        Amazon Web Services Regions. When specifying parameters and their values, be aware of how
         *        CloudFormation sets parameter values during stack instance operations:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        To override the current value for a parameter, include the parameter and specify its value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To leave an overridden parameter set to its present value, include the parameter and specify
         *        <code>UsePreviousValue</code> as <code>true</code>. (You can't specify both a value and set
         *        <code>UsePreviousValue</code> to <code>true</code>.)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To set an overridden parameter back to the value specified in the StackSet, specify a parameter list
         *        but don't include the parameter in the list.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To leave all parameters set to their present values, don't specify this property at all.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        During StackSet updates, any parameter values overridden for a stack instance aren't updated, but
         *        retain their overridden value.
         *        </p>
         *        <p>
         *        You can only override the parameter <i>values</i> that are specified in the StackSet; to add or delete
         *        a parameter itself, use <a
         *        href="https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UpdateStackSet.html"
         *        >UpdateStackSet</a> to update the StackSet template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameterOverrides(Collection<Parameter> parameterOverrides);

        /**
         * <p>
         * A list of StackSet parameters whose values you want to override in the selected stack instances.
         * </p>
         * <p>
         * Any overridden parameter values will be applied to all stack instances in the specified accounts and Amazon
         * Web Services Regions. When specifying parameters and their values, be aware of how CloudFormation sets
         * parameter values during stack instance operations:
         * </p>
         * <ul>
         * <li>
         * <p>
         * To override the current value for a parameter, include the parameter and specify its value.
         * </p>
         * </li>
         * <li>
         * <p>
         * To leave an overridden parameter set to its present value, include the parameter and specify
         * <code>UsePreviousValue</code> as <code>true</code>. (You can't specify both a value and set
         * <code>UsePreviousValue</code> to <code>true</code>.)
         * </p>
         * </li>
         * <li>
         * <p>
         * To set an overridden parameter back to the value specified in the StackSet, specify a parameter list but
         * don't include the parameter in the list.
         * </p>
         * </li>
         * <li>
         * <p>
         * To leave all parameters set to their present values, don't specify this property at all.
         * </p>
         * </li>
         * </ul>
         * <p>
         * During StackSet updates, any parameter values overridden for a stack instance aren't updated, but retain
         * their overridden value.
         * </p>
         * <p>
         * You can only override the parameter <i>values</i> that are specified in the StackSet; to add or delete a
         * parameter itself, use <a
         * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UpdateStackSet.html"
         * >UpdateStackSet</a> to update the StackSet template.
         * </p>
         * 
         * @param parameterOverrides
         *        A list of StackSet parameters whose values you want to override in the selected stack instances.</p>
         *        <p>
         *        Any overridden parameter values will be applied to all stack instances in the specified accounts and
         *        Amazon Web Services Regions. When specifying parameters and their values, be aware of how
         *        CloudFormation sets parameter values during stack instance operations:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        To override the current value for a parameter, include the parameter and specify its value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To leave an overridden parameter set to its present value, include the parameter and specify
         *        <code>UsePreviousValue</code> as <code>true</code>. (You can't specify both a value and set
         *        <code>UsePreviousValue</code> to <code>true</code>.)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To set an overridden parameter back to the value specified in the StackSet, specify a parameter list
         *        but don't include the parameter in the list.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To leave all parameters set to their present values, don't specify this property at all.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        During StackSet updates, any parameter values overridden for a stack instance aren't updated, but
         *        retain their overridden value.
         *        </p>
         *        <p>
         *        You can only override the parameter <i>values</i> that are specified in the StackSet; to add or delete
         *        a parameter itself, use <a
         *        href="https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UpdateStackSet.html"
         *        >UpdateStackSet</a> to update the StackSet template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameterOverrides(Parameter... parameterOverrides);

        /**
         * <p>
         * A list of StackSet parameters whose values you want to override in the selected stack instances.
         * </p>
         * <p>
         * Any overridden parameter values will be applied to all stack instances in the specified accounts and Amazon
         * Web Services Regions. When specifying parameters and their values, be aware of how CloudFormation sets
         * parameter values during stack instance operations:
         * </p>
         * <ul>
         * <li>
         * <p>
         * To override the current value for a parameter, include the parameter and specify its value.
         * </p>
         * </li>
         * <li>
         * <p>
         * To leave an overridden parameter set to its present value, include the parameter and specify
         * <code>UsePreviousValue</code> as <code>true</code>. (You can't specify both a value and set
         * <code>UsePreviousValue</code> to <code>true</code>.)
         * </p>
         * </li>
         * <li>
         * <p>
         * To set an overridden parameter back to the value specified in the StackSet, specify a parameter list but
         * don't include the parameter in the list.
         * </p>
         * </li>
         * <li>
         * <p>
         * To leave all parameters set to their present values, don't specify this property at all.
         * </p>
         * </li>
         * </ul>
         * <p>
         * During StackSet updates, any parameter values overridden for a stack instance aren't updated, but retain
         * their overridden value.
         * </p>
         * <p>
         * You can only override the parameter <i>values</i> that are specified in the StackSet; to add or delete a
         * parameter itself, use <a
         * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UpdateStackSet.html"
         * >UpdateStackSet</a> to update the StackSet template.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.cloudformation.model.Parameter.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.cloudformation.model.Parameter#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.cloudformation.model.Parameter.Builder#build()} is called immediately
         * and its result is passed to {@link #parameterOverrides(List<Parameter>)}.
         * 
         * @param parameterOverrides
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.cloudformation.model.Parameter.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parameterOverrides(java.util.Collection<Parameter>)
         */
        Builder parameterOverrides(Consumer<Parameter.Builder>... parameterOverrides);

        /**
         * <p>
         * Preferences for how CloudFormation performs this StackSet operation.
         * </p>
         * 
         * @param operationPreferences
         *        Preferences for how CloudFormation performs this StackSet operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationPreferences(StackSetOperationPreferences operationPreferences);

        /**
         * <p>
         * Preferences for how CloudFormation performs this StackSet operation.
         * </p>
         * This is a convenience method that creates an instance of the {@link StackSetOperationPreferences.Builder}
         * avoiding the need to create one manually via {@link StackSetOperationPreferences#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link StackSetOperationPreferences.Builder#build()} is called
         * immediately and its result is passed to {@link #operationPreferences(StackSetOperationPreferences)}.
         * 
         * @param operationPreferences
         *        a consumer that will call methods on {@link StackSetOperationPreferences.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #operationPreferences(StackSetOperationPreferences)
         */
        default Builder operationPreferences(Consumer<StackSetOperationPreferences.Builder> operationPreferences) {
            return operationPreferences(StackSetOperationPreferences.builder().applyMutation(operationPreferences).build());
        }

        /**
         * <p>
         * The unique identifier for this StackSet operation.
         * </p>
         * <p>
         * The operation ID also functions as an idempotency token, to ensure that CloudFormation performs the StackSet
         * operation only once, even if you retry the request multiple times. You might retry StackSet operation
         * requests to ensure that CloudFormation successfully received them.
         * </p>
         * <p>
         * If you don't specify an operation ID, the SDK generates one automatically.
         * </p>
         * <p>
         * Repeating this StackSet operation with a new operation ID retries all stack instances whose status is
         * <code>OUTDATED</code>.
         * </p>
         * 
         * @param operationId
         *        The unique identifier for this StackSet operation.</p>
         *        <p>
         *        The operation ID also functions as an idempotency token, to ensure that CloudFormation performs the
         *        StackSet operation only once, even if you retry the request multiple times. You might retry StackSet
         *        operation requests to ensure that CloudFormation successfully received them.
         *        </p>
         *        <p>
         *        If you don't specify an operation ID, the SDK generates one automatically.
         *        </p>
         *        <p>
         *        Repeating this StackSet operation with a new operation ID retries all stack instances whose status is
         *        <code>OUTDATED</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationId(String operationId);

        /**
         * <p>
         * [Service-managed permissions] Specifies whether you are acting as an account administrator in the
         * organization's management account or as a delegated administrator in a member account.
         * </p>
         * <p>
         * By default, <code>SELF</code> is specified. Use <code>SELF</code> for StackSets with self-managed
         * permissions.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If you are signed in to the management account, specify <code>SELF</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you are signed in to a delegated administrator account, specify <code>DELEGATED_ADMIN</code>.
         * </p>
         * <p>
         * Your Amazon Web Services account must be registered as a delegated administrator in the management account.
         * For more information, see <a
         * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-orgs-delegated-admin.html"
         * >Register a delegated administrator</a> in the <i>CloudFormation User Guide</i>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param callAs
         *        [Service-managed permissions] Specifies whether you are acting as an account administrator in the
         *        organization's management account or as a delegated administrator in a member account.</p>
         *        <p>
         *        By default, <code>SELF</code> is specified. Use <code>SELF</code> for StackSets with self-managed
         *        permissions.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If you are signed in to the management account, specify <code>SELF</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you are signed in to a delegated administrator account, specify <code>DELEGATED_ADMIN</code>.
         *        </p>
         *        <p>
         *        Your Amazon Web Services account must be registered as a delegated administrator in the management
         *        account. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-orgs-delegated-admin.html"
         *        >Register a delegated administrator</a> in the <i>CloudFormation User Guide</i>.
         *        </p>
         *        </li>
         * @see CallAs
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CallAs
         */
        Builder callAs(String callAs);

        /**
         * <p>
         * [Service-managed permissions] Specifies whether you are acting as an account administrator in the
         * organization's management account or as a delegated administrator in a member account.
         * </p>
         * <p>
         * By default, <code>SELF</code> is specified. Use <code>SELF</code> for StackSets with self-managed
         * permissions.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If you are signed in to the management account, specify <code>SELF</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you are signed in to a delegated administrator account, specify <code>DELEGATED_ADMIN</code>.
         * </p>
         * <p>
         * Your Amazon Web Services account must be registered as a delegated administrator in the management account.
         * For more information, see <a
         * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-orgs-delegated-admin.html"
         * >Register a delegated administrator</a> in the <i>CloudFormation User Guide</i>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param callAs
         *        [Service-managed permissions] Specifies whether you are acting as an account administrator in the
         *        organization's management account or as a delegated administrator in a member account.</p>
         *        <p>
         *        By default, <code>SELF</code> is specified. Use <code>SELF</code> for StackSets with self-managed
         *        permissions.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If you are signed in to the management account, specify <code>SELF</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you are signed in to a delegated administrator account, specify <code>DELEGATED_ADMIN</code>.
         *        </p>
         *        <p>
         *        Your Amazon Web Services account must be registered as a delegated administrator in the management
         *        account. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-orgs-delegated-admin.html"
         *        >Register a delegated administrator</a> in the <i>CloudFormation User Guide</i>.
         *        </p>
         *        </li>
         * @see CallAs
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CallAs
         */
        Builder callAs(CallAs callAs);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CloudFormationRequest.BuilderImpl implements Builder {
        private String stackSetName;

        private List<String> accounts = DefaultSdkAutoConstructList.getInstance();

        private DeploymentTargets deploymentTargets;

        private List<String> regions = DefaultSdkAutoConstructList.getInstance();

        private List<Parameter> parameterOverrides = DefaultSdkAutoConstructList.getInstance();

        private StackSetOperationPreferences operationPreferences;

        private String operationId;

        private String callAs;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateStackInstancesRequest model) {
            super(model);
            stackSetName(model.stackSetName);
            accounts(model.accounts);
            deploymentTargets(model.deploymentTargets);
            regions(model.regions);
            parameterOverrides(model.parameterOverrides);
            operationPreferences(model.operationPreferences);
            operationId(model.operationId);
            callAs(model.callAs);
        }

        public final String getStackSetName() {
            return stackSetName;
        }

        public final void setStackSetName(String stackSetName) {
            this.stackSetName = stackSetName;
        }

        @Override
        public final Builder stackSetName(String stackSetName) {
            this.stackSetName = stackSetName;
            return this;
        }

        public final Collection<String> getAccounts() {
            if (accounts instanceof SdkAutoConstructList) {
                return null;
            }
            return accounts;
        }

        public final void setAccounts(Collection<String> accounts) {
            this.accounts = AccountListCopier.copy(accounts);
        }

        @Override
        public final Builder accounts(Collection<String> accounts) {
            this.accounts = AccountListCopier.copy(accounts);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder accounts(String... accounts) {
            accounts(Arrays.asList(accounts));
            return this;
        }

        public final DeploymentTargets.Builder getDeploymentTargets() {
            return deploymentTargets != null ? deploymentTargets.toBuilder() : null;
        }

        public final void setDeploymentTargets(DeploymentTargets.BuilderImpl deploymentTargets) {
            this.deploymentTargets = deploymentTargets != null ? deploymentTargets.build() : null;
        }

        @Override
        public final Builder deploymentTargets(DeploymentTargets deploymentTargets) {
            this.deploymentTargets = deploymentTargets;
            return this;
        }

        public final Collection<String> getRegions() {
            if (regions instanceof SdkAutoConstructList) {
                return null;
            }
            return regions;
        }

        public final void setRegions(Collection<String> regions) {
            this.regions = RegionListCopier.copy(regions);
        }

        @Override
        public final Builder regions(Collection<String> regions) {
            this.regions = RegionListCopier.copy(regions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder regions(String... regions) {
            regions(Arrays.asList(regions));
            return this;
        }

        public final List<Parameter.Builder> getParameterOverrides() {
            List<Parameter.Builder> result = ParametersCopier.copyToBuilder(this.parameterOverrides);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setParameterOverrides(Collection<Parameter.BuilderImpl> parameterOverrides) {
            this.parameterOverrides = ParametersCopier.copyFromBuilder(parameterOverrides);
        }

        @Override
        public final Builder parameterOverrides(Collection<Parameter> parameterOverrides) {
            this.parameterOverrides = ParametersCopier.copy(parameterOverrides);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameterOverrides(Parameter... parameterOverrides) {
            parameterOverrides(Arrays.asList(parameterOverrides));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameterOverrides(Consumer<Parameter.Builder>... parameterOverrides) {
            parameterOverrides(Stream.of(parameterOverrides).map(c -> Parameter.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final StackSetOperationPreferences.Builder getOperationPreferences() {
            return operationPreferences != null ? operationPreferences.toBuilder() : null;
        }

        public final void setOperationPreferences(StackSetOperationPreferences.BuilderImpl operationPreferences) {
            this.operationPreferences = operationPreferences != null ? operationPreferences.build() : null;
        }

        @Override
        public final Builder operationPreferences(StackSetOperationPreferences operationPreferences) {
            this.operationPreferences = operationPreferences;
            return this;
        }

        public final String getOperationId() {
            return operationId;
        }

        public final void setOperationId(String operationId) {
            this.operationId = operationId;
        }

        @Override
        public final Builder operationId(String operationId) {
            this.operationId = operationId;
            return this;
        }

        public final String getCallAs() {
            return callAs;
        }

        public final void setCallAs(String callAs) {
            this.callAs = callAs;
        }

        @Override
        public final Builder callAs(String callAs) {
            this.callAs = callAs;
            return this;
        }

        @Override
        public final Builder callAs(CallAs callAs) {
            this.callAs(callAs == null ? null : callAs.toString());
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateStackInstancesRequest build() {
            return new CreateStackInstancesRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
