/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A public key that you can use with <a
 * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html">signed URLs and signed
 * cookies</a>, or with <a
 * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/field-level-encryption.html">field-level
 * encryption</a>.
 * </p>
 * <p>
 * CloudFront supports signed URLs and signed cookies with RSA 2048 or ECDSA 256 key signatures. Field-level encryption
 * is only compatible with RSA 2048 key signatures.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PublicKey implements SdkPojo, Serializable, ToCopyableBuilder<PublicKey.Builder, PublicKey> {
    private static final SdkField<String> ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Id")
            .getter(getter(PublicKey::id))
            .setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").unmarshallLocationName("Id")
                    .build()).build();

    private static final SdkField<Instant> CREATED_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedTime")
            .getter(getter(PublicKey::createdTime))
            .setter(setter(Builder::createdTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedTime")
                    .unmarshallLocationName("CreatedTime").build()).build();

    private static final SdkField<PublicKeyConfig> PUBLIC_KEY_CONFIG_FIELD = SdkField
            .<PublicKeyConfig> builder(MarshallingType.SDK_POJO)
            .memberName("PublicKeyConfig")
            .getter(getter(PublicKey::publicKeyConfig))
            .setter(setter(Builder::publicKeyConfig))
            .constructor(PublicKeyConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PublicKeyConfig")
                    .unmarshallLocationName("PublicKeyConfig").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, CREATED_TIME_FIELD,
            PUBLIC_KEY_CONFIG_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String id;

    private final Instant createdTime;

    private final PublicKeyConfig publicKeyConfig;

    private PublicKey(BuilderImpl builder) {
        this.id = builder.id;
        this.createdTime = builder.createdTime;
        this.publicKeyConfig = builder.publicKeyConfig;
    }

    /**
     * <p>
     * The identifier of the public key.
     * </p>
     * 
     * @return The identifier of the public key.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The date and time when the public key was uploaded.
     * </p>
     * 
     * @return The date and time when the public key was uploaded.
     */
    public final Instant createdTime() {
        return createdTime;
    }

    /**
     * <p>
     * Configuration information about a public key that you can use with <a
     * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html">signed URLs and
     * signed cookies</a>, or with <a
     * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/field-level-encryption.html">field-level
     * encryption</a>.
     * </p>
     * 
     * @return Configuration information about a public key that you can use with <a
     *         href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html">signed URLs
     *         and signed cookies</a>, or with <a
     *         href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/field-level-encryption.html"
     *         >field-level encryption</a>.
     */
    public final PublicKeyConfig publicKeyConfig() {
        return publicKeyConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(createdTime());
        hashCode = 31 * hashCode + Objects.hashCode(publicKeyConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PublicKey)) {
            return false;
        }
        PublicKey other = (PublicKey) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(createdTime(), other.createdTime())
                && Objects.equals(publicKeyConfig(), other.publicKeyConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PublicKey").add("Id", id()).add("CreatedTime", createdTime())
                .add("PublicKeyConfig", publicKeyConfig()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "CreatedTime":
            return Optional.ofNullable(clazz.cast(createdTime()));
        case "PublicKeyConfig":
            return Optional.ofNullable(clazz.cast(publicKeyConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Id", ID_FIELD);
        map.put("CreatedTime", CREATED_TIME_FIELD);
        map.put("PublicKeyConfig", PUBLIC_KEY_CONFIG_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<PublicKey, T> g) {
        return obj -> g.apply((PublicKey) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PublicKey> {
        /**
         * <p>
         * The identifier of the public key.
         * </p>
         * 
         * @param id
         *        The identifier of the public key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The date and time when the public key was uploaded.
         * </p>
         * 
         * @param createdTime
         *        The date and time when the public key was uploaded.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdTime(Instant createdTime);

        /**
         * <p>
         * Configuration information about a public key that you can use with <a
         * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html">signed URLs and
         * signed cookies</a>, or with <a
         * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/field-level-encryption.html"
         * >field-level encryption</a>.
         * </p>
         * 
         * @param publicKeyConfig
         *        Configuration information about a public key that you can use with <a
         *        href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html">signed
         *        URLs and signed cookies</a>, or with <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/field-level-encryption.html"
         *        >field-level encryption</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publicKeyConfig(PublicKeyConfig publicKeyConfig);

        /**
         * <p>
         * Configuration information about a public key that you can use with <a
         * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html">signed URLs and
         * signed cookies</a>, or with <a
         * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/field-level-encryption.html"
         * >field-level encryption</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link PublicKeyConfig.Builder} avoiding the
         * need to create one manually via {@link PublicKeyConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PublicKeyConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #publicKeyConfig(PublicKeyConfig)}.
         * 
         * @param publicKeyConfig
         *        a consumer that will call methods on {@link PublicKeyConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #publicKeyConfig(PublicKeyConfig)
         */
        default Builder publicKeyConfig(Consumer<PublicKeyConfig.Builder> publicKeyConfig) {
            return publicKeyConfig(PublicKeyConfig.builder().applyMutation(publicKeyConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private Instant createdTime;

        private PublicKeyConfig publicKeyConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(PublicKey model) {
            id(model.id);
            createdTime(model.createdTime);
            publicKeyConfig(model.publicKeyConfig);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final Instant getCreatedTime() {
            return createdTime;
        }

        public final void setCreatedTime(Instant createdTime) {
            this.createdTime = createdTime;
        }

        @Override
        public final Builder createdTime(Instant createdTime) {
            this.createdTime = createdTime;
            return this;
        }

        public final PublicKeyConfig.Builder getPublicKeyConfig() {
            return publicKeyConfig != null ? publicKeyConfig.toBuilder() : null;
        }

        public final void setPublicKeyConfig(PublicKeyConfig.BuilderImpl publicKeyConfig) {
            this.publicKeyConfig = publicKeyConfig != null ? publicKeyConfig.build() : null;
        }

        @Override
        public final Builder publicKeyConfig(PublicKeyConfig publicKeyConfig) {
            this.publicKeyConfig = publicKeyConfig;
            return this;
        }

        @Override
        public PublicKey build() {
            return new PublicKey(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
