/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.crt;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.regex.Pattern;
import software.amazon.awssdk.crt.CrtPlatform;
import software.amazon.awssdk.crt.CrtRuntimeException;
import software.amazon.awssdk.crt.Log;

public final class CRT {
    private static final String CRT_ARCH_OVERRIDE_SYSTEM_PROPERTY = "aws.crt.arch";
    private static final String CRT_ARCH_OVERRIDE_ENVIRONMENT_VARIABLE = "AWS_CRT_ARCH";
    private static final String CRT_LIB_NAME = "aws-crt-jni";
    public static final int AWS_CRT_SUCCESS = 0;
    private static final CrtPlatform s_platform = CRT.findPlatformImpl();
    private static final String NON_LINUX_RUNTIME_TAG = "cruntime";
    private static final String MUSL_RUNTIME_TAG = "musl";
    private static final String GLIBC_RUNTIME_TAG = "glibc";

    private static String normalize(String value) {
        if (value == null) {
            return "";
        }
        return value.toLowerCase(Locale.US).replaceAll("[^a-z0-9]+", "");
    }

    public static String getOSIdentifier() throws UnknownPlatformException {
        CrtPlatform platform = CRT.getPlatformImpl();
        String name = CRT.normalize(platform != null ? platform.getOSIdentifier() : System.getProperty("os.name"));
        if (name.contains("windows")) {
            return "windows";
        }
        if (name.contains("linux")) {
            return "linux";
        }
        if (name.contains("freebsd")) {
            return "freebsd";
        }
        if (name.contains("macosx")) {
            return "osx";
        }
        if (name.contains("sun os") || name.contains("sunos") || name.contains("solaris")) {
            return "solaris";
        }
        if (name.contains("android")) {
            return "android";
        }
        throw new UnknownPlatformException("AWS CRT: OS not supported: " + name);
    }

    public static String getArchIdentifier() throws UnknownPlatformException {
        String systemPropertyOverride = System.getProperty(CRT_ARCH_OVERRIDE_SYSTEM_PROPERTY);
        if (systemPropertyOverride != null && systemPropertyOverride.length() > 0) {
            return systemPropertyOverride;
        }
        String environmentOverride = System.getProperty(CRT_ARCH_OVERRIDE_ENVIRONMENT_VARIABLE);
        if (environmentOverride != null && environmentOverride.length() > 0) {
            return environmentOverride;
        }
        CrtPlatform platform = CRT.getPlatformImpl();
        String arch = CRT.normalize(platform != null ? platform.getArchIdentifier() : System.getProperty("os.arch"));
        if (arch.matches("^(x8664|amd64|ia32e|em64t|x64|x86_64)$")) {
            return "x86_64";
        }
        if (arch.matches("^(x8632|x86|i[3-6]86|ia32|x32)$")) {
            return "x86_32";
        }
        if (arch.startsWith("armeabi")) {
            if (arch.contains("v7")) {
                return "armv7";
            }
            return "armv6";
        }
        if (arch.startsWith("arm64") || arch.startsWith("aarch64")) {
            return "armv8";
        }
        if (arch.equals("armv7l")) {
            return "armv7";
        }
        if (arch.startsWith("arm")) {
            return "armv6";
        }
        throw new UnknownPlatformException("AWS CRT: architecture not supported: " + arch);
    }

    public static String getCRuntime(String osIdentifier) {
        File javaExecutable;
        if (!osIdentifier.equals("linux")) {
            return NON_LINUX_RUNTIME_TAG;
        }
        String systemPropertyOverride = System.getProperty("aws.crt.libc");
        if (systemPropertyOverride != null && !(systemPropertyOverride = systemPropertyOverride.toLowerCase().trim()).isEmpty()) {
            return systemPropertyOverride;
        }
        Pattern muslWord = Pattern.compile("\\bmusl\\b", 2);
        Pattern libcWord = Pattern.compile("\\blibc\\b", 2);
        String javaHome = System.getProperty("java.home");
        if (javaHome != null && (javaExecutable = new File(new File(javaHome, "bin"), "java")).exists()) {
            try {
                String[] lddJavaCmd = new String[]{"ldd", javaExecutable.toString()};
                List<String> lddJavaOutput = CRT.runProcess(lddJavaCmd);
                for (String line : lddJavaOutput) {
                    if (!libcWord.matcher(line).find()) continue;
                    if (muslWord.matcher(line).find()) {
                        return MUSL_RUNTIME_TAG;
                    }
                    return GLIBC_RUNTIME_TAG;
                }
            }
            catch (IOException lddJavaCmd) {
                // empty catch block
            }
        }
        try {
            String[] lddVersionCmd = new String[]{"ldd", "--version"};
            List<String> lddVersionOutput = CRT.runProcess(lddVersionCmd);
            for (String line : lddVersionOutput) {
                if (!muslWord.matcher(line).find()) continue;
                return MUSL_RUNTIME_TAG;
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
        return GLIBC_RUNTIME_TAG;
    }

    private static List<String> runProcess(String[] cmdArray) throws IOException {
        String line;
        Process proc = new ProcessBuilder(cmdArray).redirectErrorStream(true).start();
        BufferedReader outputReader = new BufferedReader(new InputStreamReader(proc.getInputStream()));
        ArrayList<String> output = new ArrayList<String>();
        while ((line = outputReader.readLine()) != null) {
            output.add(line);
        }
        return output;
    }

    private static void extractAndLoadLibrary(String path) {
        try {
            File tmpdirFile;
            try {
                tmpdirFile = new File(path).getAbsoluteFile();
                String tmpdirPath = tmpdirFile.getAbsolutePath();
                if (tmpdirFile.exists()) {
                    if (!tmpdirFile.isDirectory()) {
                        throw new IOException("not a directory: " + tmpdirPath);
                    }
                } else {
                    tmpdirFile.mkdirs();
                }
                if (!tmpdirFile.canRead() || !tmpdirFile.canWrite()) {
                    throw new IOException("access denied: " + tmpdirPath);
                }
            }
            catch (Exception ex) {
                String msg = "Invalid directory: " + path;
                throw new IOException(msg, ex);
            }
            String libraryName = System.mapLibraryName(CRT_LIB_NAME);
            String tempSharedLibPrefix = "AWSCRT_";
            File tempSharedLib = File.createTempFile(tempSharedLibPrefix, libraryName, tmpdirFile);
            if (!tempSharedLib.setExecutable(true, true)) {
                throw new CrtRuntimeException("Unable to make shared library executable by owner only");
            }
            if (!tempSharedLib.setWritable(true, true)) {
                throw new CrtRuntimeException("Unable to make shared library writeable by owner only");
            }
            if (!tempSharedLib.setReadable(true, true)) {
                throw new CrtRuntimeException("Unable to make shared library readable by owner only");
            }
            tempSharedLib.deleteOnExit();
            String os = CRT.getOSIdentifier();
            if (os.equals("windows")) {
                CRT.tryDeleteOldLibrariesFromTempDir(tmpdirFile, tempSharedLibPrefix, libraryName);
            }
            String libResourcePath = "/" + os + "/" + CRT.getArchIdentifier() + "/" + CRT.getCRuntime(os) + "/" + libraryName;
            try (InputStream in = CRT.class.getResourceAsStream(libResourcePath);){
                if (in == null) {
                    throw new IOException("Unable to open library in jar for AWS CRT: " + libResourcePath);
                }
                try (FileOutputStream out = new FileOutputStream(tempSharedLib);){
                    int read;
                    byte[] bytes = new byte[1024];
                    while ((read = in.read(bytes)) != -1) {
                        out.write(bytes, 0, read);
                    }
                }
            }
            if (!tempSharedLib.setWritable(false)) {
                throw new CrtRuntimeException("Unable to make shared library read-only");
            }
            System.load(tempSharedLib.getAbsolutePath());
        }
        catch (CrtRuntimeException crtex) {
            System.err.println("Unable to initialize AWS CRT: " + crtex);
            crtex.printStackTrace();
            throw crtex;
        }
        catch (UnknownPlatformException upe) {
            System.err.println("Unable to determine platform for AWS CRT: " + upe);
            upe.printStackTrace();
            CrtRuntimeException rex = new CrtRuntimeException("Unable to determine platform for AWS CRT");
            rex.initCause(upe);
            throw rex;
        }
        catch (Exception ex) {
            System.err.println("Unable to unpack AWS CRT lib: " + ex);
            ex.printStackTrace();
            CrtRuntimeException rex = new CrtRuntimeException("Unable to unpack AWS CRT library");
            rex.initCause(ex);
            throw rex;
        }
    }

    private static void loadLibraryFromJar() {
        LinkedList<String> pathsToTry = new LinkedList<String>();
        pathsToTry.add(System.getProperty("java.io.tmpdir"));
        String overrideLibDir = System.getProperty("aws.crt.lib.dir");
        if (overrideLibDir != null) {
            pathsToTry.add(0, overrideLibDir);
        }
        LinkedList<CrtRuntimeException> exceptions = new LinkedList<CrtRuntimeException>();
        for (String path : pathsToTry) {
            try {
                CRT.extractAndLoadLibrary(path);
                return;
            }
            catch (CrtRuntimeException ex) {
                exceptions.add(ex);
            }
        }
        StringBuilder failureMessage = new StringBuilder();
        exceptions.stream().map(Throwable::toString).forEach(failureMessage::append);
        throw new CrtRuntimeException(failureMessage.toString());
    }

    private static void tryDeleteOldLibrariesFromTempDir(File tmpDir, final String libNamePrefix, final String libNameSuffix) {
        try {
            File[] oldLibs = tmpDir.listFiles(new FilenameFilter(){

                @Override
                public boolean accept(File dir, String name) {
                    return name.startsWith(libNamePrefix) && name.endsWith(libNameSuffix);
                }
            });
            long aFewMomentsAgo = System.currentTimeMillis() - 10000L;
            for (File oldLib : oldLibs) {
                try {
                    if (oldLib.lastModified() >= aFewMomentsAgo) continue;
                    oldLib.delete();
                }
                catch (Exception exception) {
                    // empty catch block
                }
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    private static CrtPlatform findPlatformImpl() {
        String[] platforms;
        ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
        for (String platformImpl : platforms = new String[]{String.format("software.amazon.awssdk.crt.test.%s.CrtPlatformImpl", CRT.getOSIdentifier()), "software.amazon.awssdk.crt.test.android.CrtPlatformImpl", String.format("software.amazon.awssdk.crt.%s.CrtPlatformImpl", CRT.getOSIdentifier())}) {
            try {
                Class<?> platformClass = classLoader.loadClass(platformImpl);
                CrtPlatform instance = (CrtPlatform)platformClass.getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
                return instance;
            }
            catch (ClassNotFoundException platformClass) {
            }
            catch (IllegalAccessException | InstantiationException | NoSuchMethodException | InvocationTargetException ex) {
                throw new CrtRuntimeException(ex.toString());
            }
        }
        return null;
    }

    public static CrtPlatform getPlatformImpl() {
        return s_platform;
    }

    private static void jvmInit() {
        CrtPlatform platform = CRT.getPlatformImpl();
        if (platform != null) {
            platform.jvmInit();
        }
    }

    private static native void awsCrtInit(int var0, boolean var1, boolean var2) throws CrtRuntimeException;

    public static native int awsLastError();

    public static native String awsErrorString(int var0);

    public static native String awsErrorName(int var0);

    public static long nativeMemory() {
        return CRT.awsNativeMemory();
    }

    public static native void dumpNativeMemory();

    private static native long awsNativeMemory();

    static void testJniException(boolean throwException) {
        if (throwException) {
            throw new RuntimeException("Testing");
        }
    }

    public static void checkJniExceptionContract(boolean clearException) {
        CRT.nativeCheckJniExceptionContract(clearException);
    }

    private static native void nativeCheckJniExceptionContract(boolean var0);

    private static native void onJvmShutdown();

    static {
        CRT.jvmInit();
        try {
            System.loadLibrary(CRT_LIB_NAME);
        }
        catch (UnsatisfiedLinkError e) {
            CRT.loadLibraryFromJar();
        }
        int memoryTracingLevel = 0;
        try {
            memoryTracingLevel = Integer.parseInt(System.getProperty("aws.crt.memory.tracing"));
        }
        catch (Exception exception) {
            // empty catch block
        }
        boolean debugWait = System.getProperty("aws.crt.debugwait") != null;
        boolean strictShutdown = System.getProperty("aws.crt.strictshutdown") != null;
        CRT.awsCrtInit(memoryTracingLevel, debugWait, strictShutdown);
        Runtime.getRuntime().addShutdownHook(new Thread(){

            @Override
            public void run() {
                CRT.onJvmShutdown();
            }
        });
        try {
            Log.initLoggingFromSystemProperties();
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    public static class UnknownPlatformException
    extends RuntimeException {
        UnknownPlatformException(String message) {
            super(message);
        }
    }
}

