/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.databrew.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents one run of a DataBrew job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class JobRun implements SdkPojo, Serializable, ToCopyableBuilder<JobRun.Builder, JobRun> {
    private static final SdkField<Integer> ATTEMPT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Attempt").getter(getter(JobRun::attempt)).setter(setter(Builder::attempt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Attempt").build()).build();

    private static final SdkField<Instant> COMPLETED_ON_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CompletedOn").getter(getter(JobRun::completedOn)).setter(setter(Builder::completedOn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompletedOn").build()).build();

    private static final SdkField<String> DATASET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DatasetName").getter(getter(JobRun::datasetName)).setter(setter(Builder::datasetName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DatasetName").build()).build();

    private static final SdkField<String> ERROR_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ErrorMessage").getter(getter(JobRun::errorMessage)).setter(setter(Builder::errorMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ErrorMessage").build()).build();

    private static final SdkField<Integer> EXECUTION_TIME_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ExecutionTime").getter(getter(JobRun::executionTime)).setter(setter(Builder::executionTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutionTime").build()).build();

    private static final SdkField<String> JOB_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("JobName").getter(getter(JobRun::jobName)).setter(setter(Builder::jobName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JobName").build()).build();

    private static final SdkField<String> RUN_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("RunId")
            .getter(getter(JobRun::runId)).setter(setter(Builder::runId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RunId").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(JobRun::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<String> LOG_SUBSCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LogSubscription").getter(getter(JobRun::logSubscriptionAsString))
            .setter(setter(Builder::logSubscription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogSubscription").build()).build();

    private static final SdkField<String> LOG_GROUP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LogGroupName").getter(getter(JobRun::logGroupName)).setter(setter(Builder::logGroupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogGroupName").build()).build();

    private static final SdkField<List<Output>> OUTPUTS_FIELD = SdkField
            .<List<Output>> builder(MarshallingType.LIST)
            .memberName("Outputs")
            .getter(getter(JobRun::outputs))
            .setter(setter(Builder::outputs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Outputs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Output> builder(MarshallingType.SDK_POJO)
                                            .constructor(Output::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<DataCatalogOutput>> DATA_CATALOG_OUTPUTS_FIELD = SdkField
            .<List<DataCatalogOutput>> builder(MarshallingType.LIST)
            .memberName("DataCatalogOutputs")
            .getter(getter(JobRun::dataCatalogOutputs))
            .setter(setter(Builder::dataCatalogOutputs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataCatalogOutputs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DataCatalogOutput> builder(MarshallingType.SDK_POJO)
                                            .constructor(DataCatalogOutput::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<DatabaseOutput>> DATABASE_OUTPUTS_FIELD = SdkField
            .<List<DatabaseOutput>> builder(MarshallingType.LIST)
            .memberName("DatabaseOutputs")
            .getter(getter(JobRun::databaseOutputs))
            .setter(setter(Builder::databaseOutputs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DatabaseOutputs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DatabaseOutput> builder(MarshallingType.SDK_POJO)
                                            .constructor(DatabaseOutput::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<RecipeReference> RECIPE_REFERENCE_FIELD = SdkField
            .<RecipeReference> builder(MarshallingType.SDK_POJO).memberName("RecipeReference")
            .getter(getter(JobRun::recipeReference)).setter(setter(Builder::recipeReference))
            .constructor(RecipeReference::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecipeReference").build()).build();

    private static final SdkField<String> STARTED_BY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StartedBy").getter(getter(JobRun::startedBy)).setter(setter(Builder::startedBy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartedBy").build()).build();

    private static final SdkField<Instant> STARTED_ON_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartedOn").getter(getter(JobRun::startedOn)).setter(setter(Builder::startedOn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartedOn").build()).build();

    private static final SdkField<JobSample> JOB_SAMPLE_FIELD = SdkField.<JobSample> builder(MarshallingType.SDK_POJO)
            .memberName("JobSample").getter(getter(JobRun::jobSample)).setter(setter(Builder::jobSample))
            .constructor(JobSample::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JobSample").build()).build();

    private static final SdkField<List<ValidationConfiguration>> VALIDATION_CONFIGURATIONS_FIELD = SdkField
            .<List<ValidationConfiguration>> builder(MarshallingType.LIST)
            .memberName("ValidationConfigurations")
            .getter(getter(JobRun::validationConfigurations))
            .setter(setter(Builder::validationConfigurations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ValidationConfigurations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ValidationConfiguration> builder(MarshallingType.SDK_POJO)
                                            .constructor(ValidationConfiguration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ATTEMPT_FIELD,
            COMPLETED_ON_FIELD, DATASET_NAME_FIELD, ERROR_MESSAGE_FIELD, EXECUTION_TIME_FIELD, JOB_NAME_FIELD, RUN_ID_FIELD,
            STATE_FIELD, LOG_SUBSCRIPTION_FIELD, LOG_GROUP_NAME_FIELD, OUTPUTS_FIELD, DATA_CATALOG_OUTPUTS_FIELD,
            DATABASE_OUTPUTS_FIELD, RECIPE_REFERENCE_FIELD, STARTED_BY_FIELD, STARTED_ON_FIELD, JOB_SAMPLE_FIELD,
            VALIDATION_CONFIGURATIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer attempt;

    private final Instant completedOn;

    private final String datasetName;

    private final String errorMessage;

    private final Integer executionTime;

    private final String jobName;

    private final String runId;

    private final String state;

    private final String logSubscription;

    private final String logGroupName;

    private final List<Output> outputs;

    private final List<DataCatalogOutput> dataCatalogOutputs;

    private final List<DatabaseOutput> databaseOutputs;

    private final RecipeReference recipeReference;

    private final String startedBy;

    private final Instant startedOn;

    private final JobSample jobSample;

    private final List<ValidationConfiguration> validationConfigurations;

    private JobRun(BuilderImpl builder) {
        this.attempt = builder.attempt;
        this.completedOn = builder.completedOn;
        this.datasetName = builder.datasetName;
        this.errorMessage = builder.errorMessage;
        this.executionTime = builder.executionTime;
        this.jobName = builder.jobName;
        this.runId = builder.runId;
        this.state = builder.state;
        this.logSubscription = builder.logSubscription;
        this.logGroupName = builder.logGroupName;
        this.outputs = builder.outputs;
        this.dataCatalogOutputs = builder.dataCatalogOutputs;
        this.databaseOutputs = builder.databaseOutputs;
        this.recipeReference = builder.recipeReference;
        this.startedBy = builder.startedBy;
        this.startedOn = builder.startedOn;
        this.jobSample = builder.jobSample;
        this.validationConfigurations = builder.validationConfigurations;
    }

    /**
     * <p>
     * The number of times that DataBrew has attempted to run the job.
     * </p>
     * 
     * @return The number of times that DataBrew has attempted to run the job.
     */
    public final Integer attempt() {
        return attempt;
    }

    /**
     * <p>
     * The date and time when the job completed processing.
     * </p>
     * 
     * @return The date and time when the job completed processing.
     */
    public final Instant completedOn() {
        return completedOn;
    }

    /**
     * <p>
     * The name of the dataset for the job to process.
     * </p>
     * 
     * @return The name of the dataset for the job to process.
     */
    public final String datasetName() {
        return datasetName;
    }

    /**
     * <p>
     * A message indicating an error (if any) that was encountered when the job ran.
     * </p>
     * 
     * @return A message indicating an error (if any) that was encountered when the job ran.
     */
    public final String errorMessage() {
        return errorMessage;
    }

    /**
     * <p>
     * The amount of time, in seconds, during which a job run consumed resources.
     * </p>
     * 
     * @return The amount of time, in seconds, during which a job run consumed resources.
     */
    public final Integer executionTime() {
        return executionTime;
    }

    /**
     * <p>
     * The name of the job being processed during this run.
     * </p>
     * 
     * @return The name of the job being processed during this run.
     */
    public final String jobName() {
        return jobName;
    }

    /**
     * <p>
     * The unique identifier of the job run.
     * </p>
     * 
     * @return The unique identifier of the job run.
     */
    public final String runId() {
        return runId;
    }

    /**
     * <p>
     * The current state of the job run entity itself.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link JobRunState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the job run entity itself.
     * @see JobRunState
     */
    public final JobRunState state() {
        return JobRunState.fromValue(state);
    }

    /**
     * <p>
     * The current state of the job run entity itself.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link JobRunState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the job run entity itself.
     * @see JobRunState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The current status of Amazon CloudWatch logging for the job run.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #logSubscription}
     * will return {@link LogSubscription#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #logSubscriptionAsString}.
     * </p>
     * 
     * @return The current status of Amazon CloudWatch logging for the job run.
     * @see LogSubscription
     */
    public final LogSubscription logSubscription() {
        return LogSubscription.fromValue(logSubscription);
    }

    /**
     * <p>
     * The current status of Amazon CloudWatch logging for the job run.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #logSubscription}
     * will return {@link LogSubscription#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #logSubscriptionAsString}.
     * </p>
     * 
     * @return The current status of Amazon CloudWatch logging for the job run.
     * @see LogSubscription
     */
    public final String logSubscriptionAsString() {
        return logSubscription;
    }

    /**
     * <p>
     * The name of an Amazon CloudWatch log group, where the job writes diagnostic messages when it runs.
     * </p>
     * 
     * @return The name of an Amazon CloudWatch log group, where the job writes diagnostic messages when it runs.
     */
    public final String logGroupName() {
        return logGroupName;
    }

    /**
     * For responses, this returns true if the service returned a value for the Outputs property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasOutputs() {
        return outputs != null && !(outputs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One or more output artifacts from a job run.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOutputs} method.
     * </p>
     * 
     * @return One or more output artifacts from a job run.
     */
    public final List<Output> outputs() {
        return outputs;
    }

    /**
     * For responses, this returns true if the service returned a value for the DataCatalogOutputs property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDataCatalogOutputs() {
        return dataCatalogOutputs != null && !(dataCatalogOutputs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One or more artifacts that represent the Glue Data Catalog output from running the job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDataCatalogOutputs} method.
     * </p>
     * 
     * @return One or more artifacts that represent the Glue Data Catalog output from running the job.
     */
    public final List<DataCatalogOutput> dataCatalogOutputs() {
        return dataCatalogOutputs;
    }

    /**
     * For responses, this returns true if the service returned a value for the DatabaseOutputs property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDatabaseOutputs() {
        return databaseOutputs != null && !(databaseOutputs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Represents a list of JDBC database output objects which defines the output destination for a DataBrew recipe job
     * to write into.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDatabaseOutputs} method.
     * </p>
     * 
     * @return Represents a list of JDBC database output objects which defines the output destination for a DataBrew
     *         recipe job to write into.
     */
    public final List<DatabaseOutput> databaseOutputs() {
        return databaseOutputs;
    }

    /**
     * <p>
     * The set of steps processed by the job.
     * </p>
     * 
     * @return The set of steps processed by the job.
     */
    public final RecipeReference recipeReference() {
        return recipeReference;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the user who initiated the job run.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the user who initiated the job run.
     */
    public final String startedBy() {
        return startedBy;
    }

    /**
     * <p>
     * The date and time when the job run began.
     * </p>
     * 
     * @return The date and time when the job run began.
     */
    public final Instant startedOn() {
        return startedOn;
    }

    /**
     * <p>
     * A sample configuration for profile jobs only, which determines the number of rows on which the profile job is
     * run. If a <code>JobSample</code> value isn't provided, the default is used. The default value is CUSTOM_ROWS for
     * the mode parameter and 20,000 for the size parameter.
     * </p>
     * 
     * @return A sample configuration for profile jobs only, which determines the number of rows on which the profile
     *         job is run. If a <code>JobSample</code> value isn't provided, the default is used. The default value is
     *         CUSTOM_ROWS for the mode parameter and 20,000 for the size parameter.
     */
    public final JobSample jobSample() {
        return jobSample;
    }

    /**
     * For responses, this returns true if the service returned a value for the ValidationConfigurations property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasValidationConfigurations() {
        return validationConfigurations != null && !(validationConfigurations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List of validation configurations that are applied to the profile job run.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasValidationConfigurations} method.
     * </p>
     * 
     * @return List of validation configurations that are applied to the profile job run.
     */
    public final List<ValidationConfiguration> validationConfigurations() {
        return validationConfigurations;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(attempt());
        hashCode = 31 * hashCode + Objects.hashCode(completedOn());
        hashCode = 31 * hashCode + Objects.hashCode(datasetName());
        hashCode = 31 * hashCode + Objects.hashCode(errorMessage());
        hashCode = 31 * hashCode + Objects.hashCode(executionTime());
        hashCode = 31 * hashCode + Objects.hashCode(jobName());
        hashCode = 31 * hashCode + Objects.hashCode(runId());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(logSubscriptionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(logGroupName());
        hashCode = 31 * hashCode + Objects.hashCode(hasOutputs() ? outputs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasDataCatalogOutputs() ? dataCatalogOutputs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasDatabaseOutputs() ? databaseOutputs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(recipeReference());
        hashCode = 31 * hashCode + Objects.hashCode(startedBy());
        hashCode = 31 * hashCode + Objects.hashCode(startedOn());
        hashCode = 31 * hashCode + Objects.hashCode(jobSample());
        hashCode = 31 * hashCode + Objects.hashCode(hasValidationConfigurations() ? validationConfigurations() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof JobRun)) {
            return false;
        }
        JobRun other = (JobRun) obj;
        return Objects.equals(attempt(), other.attempt()) && Objects.equals(completedOn(), other.completedOn())
                && Objects.equals(datasetName(), other.datasetName()) && Objects.equals(errorMessage(), other.errorMessage())
                && Objects.equals(executionTime(), other.executionTime()) && Objects.equals(jobName(), other.jobName())
                && Objects.equals(runId(), other.runId()) && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(logSubscriptionAsString(), other.logSubscriptionAsString())
                && Objects.equals(logGroupName(), other.logGroupName()) && hasOutputs() == other.hasOutputs()
                && Objects.equals(outputs(), other.outputs()) && hasDataCatalogOutputs() == other.hasDataCatalogOutputs()
                && Objects.equals(dataCatalogOutputs(), other.dataCatalogOutputs())
                && hasDatabaseOutputs() == other.hasDatabaseOutputs()
                && Objects.equals(databaseOutputs(), other.databaseOutputs())
                && Objects.equals(recipeReference(), other.recipeReference()) && Objects.equals(startedBy(), other.startedBy())
                && Objects.equals(startedOn(), other.startedOn()) && Objects.equals(jobSample(), other.jobSample())
                && hasValidationConfigurations() == other.hasValidationConfigurations()
                && Objects.equals(validationConfigurations(), other.validationConfigurations());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("JobRun").add("Attempt", attempt()).add("CompletedOn", completedOn())
                .add("DatasetName", datasetName()).add("ErrorMessage", errorMessage()).add("ExecutionTime", executionTime())
                .add("JobName", jobName()).add("RunId", runId()).add("State", stateAsString())
                .add("LogSubscription", logSubscriptionAsString()).add("LogGroupName", logGroupName())
                .add("Outputs", hasOutputs() ? outputs() : null)
                .add("DataCatalogOutputs", hasDataCatalogOutputs() ? dataCatalogOutputs() : null)
                .add("DatabaseOutputs", hasDatabaseOutputs() ? databaseOutputs() : null)
                .add("RecipeReference", recipeReference()).add("StartedBy", startedBy()).add("StartedOn", startedOn())
                .add("JobSample", jobSample())
                .add("ValidationConfigurations", hasValidationConfigurations() ? validationConfigurations() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Attempt":
            return Optional.ofNullable(clazz.cast(attempt()));
        case "CompletedOn":
            return Optional.ofNullable(clazz.cast(completedOn()));
        case "DatasetName":
            return Optional.ofNullable(clazz.cast(datasetName()));
        case "ErrorMessage":
            return Optional.ofNullable(clazz.cast(errorMessage()));
        case "ExecutionTime":
            return Optional.ofNullable(clazz.cast(executionTime()));
        case "JobName":
            return Optional.ofNullable(clazz.cast(jobName()));
        case "RunId":
            return Optional.ofNullable(clazz.cast(runId()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "LogSubscription":
            return Optional.ofNullable(clazz.cast(logSubscriptionAsString()));
        case "LogGroupName":
            return Optional.ofNullable(clazz.cast(logGroupName()));
        case "Outputs":
            return Optional.ofNullable(clazz.cast(outputs()));
        case "DataCatalogOutputs":
            return Optional.ofNullable(clazz.cast(dataCatalogOutputs()));
        case "DatabaseOutputs":
            return Optional.ofNullable(clazz.cast(databaseOutputs()));
        case "RecipeReference":
            return Optional.ofNullable(clazz.cast(recipeReference()));
        case "StartedBy":
            return Optional.ofNullable(clazz.cast(startedBy()));
        case "StartedOn":
            return Optional.ofNullable(clazz.cast(startedOn()));
        case "JobSample":
            return Optional.ofNullable(clazz.cast(jobSample()));
        case "ValidationConfigurations":
            return Optional.ofNullable(clazz.cast(validationConfigurations()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<JobRun, T> g) {
        return obj -> g.apply((JobRun) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, JobRun> {
        /**
         * <p>
         * The number of times that DataBrew has attempted to run the job.
         * </p>
         * 
         * @param attempt
         *        The number of times that DataBrew has attempted to run the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attempt(Integer attempt);

        /**
         * <p>
         * The date and time when the job completed processing.
         * </p>
         * 
         * @param completedOn
         *        The date and time when the job completed processing.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completedOn(Instant completedOn);

        /**
         * <p>
         * The name of the dataset for the job to process.
         * </p>
         * 
         * @param datasetName
         *        The name of the dataset for the job to process.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder datasetName(String datasetName);

        /**
         * <p>
         * A message indicating an error (if any) that was encountered when the job ran.
         * </p>
         * 
         * @param errorMessage
         *        A message indicating an error (if any) that was encountered when the job ran.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorMessage(String errorMessage);

        /**
         * <p>
         * The amount of time, in seconds, during which a job run consumed resources.
         * </p>
         * 
         * @param executionTime
         *        The amount of time, in seconds, during which a job run consumed resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionTime(Integer executionTime);

        /**
         * <p>
         * The name of the job being processed during this run.
         * </p>
         * 
         * @param jobName
         *        The name of the job being processed during this run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobName(String jobName);

        /**
         * <p>
         * The unique identifier of the job run.
         * </p>
         * 
         * @param runId
         *        The unique identifier of the job run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder runId(String runId);

        /**
         * <p>
         * The current state of the job run entity itself.
         * </p>
         * 
         * @param state
         *        The current state of the job run entity itself.
         * @see JobRunState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobRunState
         */
        Builder state(String state);

        /**
         * <p>
         * The current state of the job run entity itself.
         * </p>
         * 
         * @param state
         *        The current state of the job run entity itself.
         * @see JobRunState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobRunState
         */
        Builder state(JobRunState state);

        /**
         * <p>
         * The current status of Amazon CloudWatch logging for the job run.
         * </p>
         * 
         * @param logSubscription
         *        The current status of Amazon CloudWatch logging for the job run.
         * @see LogSubscription
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogSubscription
         */
        Builder logSubscription(String logSubscription);

        /**
         * <p>
         * The current status of Amazon CloudWatch logging for the job run.
         * </p>
         * 
         * @param logSubscription
         *        The current status of Amazon CloudWatch logging for the job run.
         * @see LogSubscription
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogSubscription
         */
        Builder logSubscription(LogSubscription logSubscription);

        /**
         * <p>
         * The name of an Amazon CloudWatch log group, where the job writes diagnostic messages when it runs.
         * </p>
         * 
         * @param logGroupName
         *        The name of an Amazon CloudWatch log group, where the job writes diagnostic messages when it runs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logGroupName(String logGroupName);

        /**
         * <p>
         * One or more output artifacts from a job run.
         * </p>
         * 
         * @param outputs
         *        One or more output artifacts from a job run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputs(Collection<Output> outputs);

        /**
         * <p>
         * One or more output artifacts from a job run.
         * </p>
         * 
         * @param outputs
         *        One or more output artifacts from a job run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputs(Output... outputs);

        /**
         * <p>
         * One or more output artifacts from a job run.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Output>.Builder} avoiding the need to
         * create one manually via {@link List<Output>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Output>.Builder#build()} is called immediately and its
         * result is passed to {@link #outputs(List<Output>)}.
         * 
         * @param outputs
         *        a consumer that will call methods on {@link List<Output>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #outputs(List<Output>)
         */
        Builder outputs(Consumer<Output.Builder>... outputs);

        /**
         * <p>
         * One or more artifacts that represent the Glue Data Catalog output from running the job.
         * </p>
         * 
         * @param dataCatalogOutputs
         *        One or more artifacts that represent the Glue Data Catalog output from running the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataCatalogOutputs(Collection<DataCatalogOutput> dataCatalogOutputs);

        /**
         * <p>
         * One or more artifacts that represent the Glue Data Catalog output from running the job.
         * </p>
         * 
         * @param dataCatalogOutputs
         *        One or more artifacts that represent the Glue Data Catalog output from running the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataCatalogOutputs(DataCatalogOutput... dataCatalogOutputs);

        /**
         * <p>
         * One or more artifacts that represent the Glue Data Catalog output from running the job.
         * </p>
         * This is a convenience that creates an instance of the {@link List<DataCatalogOutput>.Builder} avoiding the
         * need to create one manually via {@link List<DataCatalogOutput>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<DataCatalogOutput>.Builder#build()} is called immediately
         * and its result is passed to {@link #dataCatalogOutputs(List<DataCatalogOutput>)}.
         * 
         * @param dataCatalogOutputs
         *        a consumer that will call methods on {@link List<DataCatalogOutput>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dataCatalogOutputs(List<DataCatalogOutput>)
         */
        Builder dataCatalogOutputs(Consumer<DataCatalogOutput.Builder>... dataCatalogOutputs);

        /**
         * <p>
         * Represents a list of JDBC database output objects which defines the output destination for a DataBrew recipe
         * job to write into.
         * </p>
         * 
         * @param databaseOutputs
         *        Represents a list of JDBC database output objects which defines the output destination for a DataBrew
         *        recipe job to write into.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databaseOutputs(Collection<DatabaseOutput> databaseOutputs);

        /**
         * <p>
         * Represents a list of JDBC database output objects which defines the output destination for a DataBrew recipe
         * job to write into.
         * </p>
         * 
         * @param databaseOutputs
         *        Represents a list of JDBC database output objects which defines the output destination for a DataBrew
         *        recipe job to write into.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databaseOutputs(DatabaseOutput... databaseOutputs);

        /**
         * <p>
         * Represents a list of JDBC database output objects which defines the output destination for a DataBrew recipe
         * job to write into.
         * </p>
         * This is a convenience that creates an instance of the {@link List<DatabaseOutput>.Builder} avoiding the need
         * to create one manually via {@link List<DatabaseOutput>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<DatabaseOutput>.Builder#build()} is called immediately and
         * its result is passed to {@link #databaseOutputs(List<DatabaseOutput>)}.
         * 
         * @param databaseOutputs
         *        a consumer that will call methods on {@link List<DatabaseOutput>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #databaseOutputs(List<DatabaseOutput>)
         */
        Builder databaseOutputs(Consumer<DatabaseOutput.Builder>... databaseOutputs);

        /**
         * <p>
         * The set of steps processed by the job.
         * </p>
         * 
         * @param recipeReference
         *        The set of steps processed by the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recipeReference(RecipeReference recipeReference);

        /**
         * <p>
         * The set of steps processed by the job.
         * </p>
         * This is a convenience that creates an instance of the {@link RecipeReference.Builder} avoiding the need to
         * create one manually via {@link RecipeReference#builder()}.
         *
         * When the {@link Consumer} completes, {@link RecipeReference.Builder#build()} is called immediately and its
         * result is passed to {@link #recipeReference(RecipeReference)}.
         * 
         * @param recipeReference
         *        a consumer that will call methods on {@link RecipeReference.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #recipeReference(RecipeReference)
         */
        default Builder recipeReference(Consumer<RecipeReference.Builder> recipeReference) {
            return recipeReference(RecipeReference.builder().applyMutation(recipeReference).build());
        }

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the user who initiated the job run.
         * </p>
         * 
         * @param startedBy
         *        The Amazon Resource Name (ARN) of the user who initiated the job run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startedBy(String startedBy);

        /**
         * <p>
         * The date and time when the job run began.
         * </p>
         * 
         * @param startedOn
         *        The date and time when the job run began.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startedOn(Instant startedOn);

        /**
         * <p>
         * A sample configuration for profile jobs only, which determines the number of rows on which the profile job is
         * run. If a <code>JobSample</code> value isn't provided, the default is used. The default value is CUSTOM_ROWS
         * for the mode parameter and 20,000 for the size parameter.
         * </p>
         * 
         * @param jobSample
         *        A sample configuration for profile jobs only, which determines the number of rows on which the profile
         *        job is run. If a <code>JobSample</code> value isn't provided, the default is used. The default value
         *        is CUSTOM_ROWS for the mode parameter and 20,000 for the size parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobSample(JobSample jobSample);

        /**
         * <p>
         * A sample configuration for profile jobs only, which determines the number of rows on which the profile job is
         * run. If a <code>JobSample</code> value isn't provided, the default is used. The default value is CUSTOM_ROWS
         * for the mode parameter and 20,000 for the size parameter.
         * </p>
         * This is a convenience that creates an instance of the {@link JobSample.Builder} avoiding the need to create
         * one manually via {@link JobSample#builder()}.
         *
         * When the {@link Consumer} completes, {@link JobSample.Builder#build()} is called immediately and its result
         * is passed to {@link #jobSample(JobSample)}.
         * 
         * @param jobSample
         *        a consumer that will call methods on {@link JobSample.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #jobSample(JobSample)
         */
        default Builder jobSample(Consumer<JobSample.Builder> jobSample) {
            return jobSample(JobSample.builder().applyMutation(jobSample).build());
        }

        /**
         * <p>
         * List of validation configurations that are applied to the profile job run.
         * </p>
         * 
         * @param validationConfigurations
         *        List of validation configurations that are applied to the profile job run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validationConfigurations(Collection<ValidationConfiguration> validationConfigurations);

        /**
         * <p>
         * List of validation configurations that are applied to the profile job run.
         * </p>
         * 
         * @param validationConfigurations
         *        List of validation configurations that are applied to the profile job run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validationConfigurations(ValidationConfiguration... validationConfigurations);

        /**
         * <p>
         * List of validation configurations that are applied to the profile job run.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ValidationConfiguration>.Builder} avoiding
         * the need to create one manually via {@link List<ValidationConfiguration>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ValidationConfiguration>.Builder#build()} is called
         * immediately and its result is passed to {@link #validationConfigurations(List<ValidationConfiguration>)}.
         * 
         * @param validationConfigurations
         *        a consumer that will call methods on {@link List<ValidationConfiguration>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #validationConfigurations(List<ValidationConfiguration>)
         */
        Builder validationConfigurations(Consumer<ValidationConfiguration.Builder>... validationConfigurations);
    }

    static final class BuilderImpl implements Builder {
        private Integer attempt;

        private Instant completedOn;

        private String datasetName;

        private String errorMessage;

        private Integer executionTime;

        private String jobName;

        private String runId;

        private String state;

        private String logSubscription;

        private String logGroupName;

        private List<Output> outputs = DefaultSdkAutoConstructList.getInstance();

        private List<DataCatalogOutput> dataCatalogOutputs = DefaultSdkAutoConstructList.getInstance();

        private List<DatabaseOutput> databaseOutputs = DefaultSdkAutoConstructList.getInstance();

        private RecipeReference recipeReference;

        private String startedBy;

        private Instant startedOn;

        private JobSample jobSample;

        private List<ValidationConfiguration> validationConfigurations = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(JobRun model) {
            attempt(model.attempt);
            completedOn(model.completedOn);
            datasetName(model.datasetName);
            errorMessage(model.errorMessage);
            executionTime(model.executionTime);
            jobName(model.jobName);
            runId(model.runId);
            state(model.state);
            logSubscription(model.logSubscription);
            logGroupName(model.logGroupName);
            outputs(model.outputs);
            dataCatalogOutputs(model.dataCatalogOutputs);
            databaseOutputs(model.databaseOutputs);
            recipeReference(model.recipeReference);
            startedBy(model.startedBy);
            startedOn(model.startedOn);
            jobSample(model.jobSample);
            validationConfigurations(model.validationConfigurations);
        }

        public final Integer getAttempt() {
            return attempt;
        }

        public final void setAttempt(Integer attempt) {
            this.attempt = attempt;
        }

        @Override
        @Transient
        public final Builder attempt(Integer attempt) {
            this.attempt = attempt;
            return this;
        }

        public final Instant getCompletedOn() {
            return completedOn;
        }

        public final void setCompletedOn(Instant completedOn) {
            this.completedOn = completedOn;
        }

        @Override
        @Transient
        public final Builder completedOn(Instant completedOn) {
            this.completedOn = completedOn;
            return this;
        }

        public final String getDatasetName() {
            return datasetName;
        }

        public final void setDatasetName(String datasetName) {
            this.datasetName = datasetName;
        }

        @Override
        @Transient
        public final Builder datasetName(String datasetName) {
            this.datasetName = datasetName;
            return this;
        }

        public final String getErrorMessage() {
            return errorMessage;
        }

        public final void setErrorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
        }

        @Override
        @Transient
        public final Builder errorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
            return this;
        }

        public final Integer getExecutionTime() {
            return executionTime;
        }

        public final void setExecutionTime(Integer executionTime) {
            this.executionTime = executionTime;
        }

        @Override
        @Transient
        public final Builder executionTime(Integer executionTime) {
            this.executionTime = executionTime;
            return this;
        }

        public final String getJobName() {
            return jobName;
        }

        public final void setJobName(String jobName) {
            this.jobName = jobName;
        }

        @Override
        @Transient
        public final Builder jobName(String jobName) {
            this.jobName = jobName;
            return this;
        }

        public final String getRunId() {
            return runId;
        }

        public final void setRunId(String runId) {
            this.runId = runId;
        }

        @Override
        @Transient
        public final Builder runId(String runId) {
            this.runId = runId;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        @Transient
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        @Transient
        public final Builder state(JobRunState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getLogSubscription() {
            return logSubscription;
        }

        public final void setLogSubscription(String logSubscription) {
            this.logSubscription = logSubscription;
        }

        @Override
        @Transient
        public final Builder logSubscription(String logSubscription) {
            this.logSubscription = logSubscription;
            return this;
        }

        @Override
        @Transient
        public final Builder logSubscription(LogSubscription logSubscription) {
            this.logSubscription(logSubscription == null ? null : logSubscription.toString());
            return this;
        }

        public final String getLogGroupName() {
            return logGroupName;
        }

        public final void setLogGroupName(String logGroupName) {
            this.logGroupName = logGroupName;
        }

        @Override
        @Transient
        public final Builder logGroupName(String logGroupName) {
            this.logGroupName = logGroupName;
            return this;
        }

        public final List<Output.Builder> getOutputs() {
            List<Output.Builder> result = OutputListCopier.copyToBuilder(this.outputs);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setOutputs(Collection<Output.BuilderImpl> outputs) {
            this.outputs = OutputListCopier.copyFromBuilder(outputs);
        }

        @Override
        @Transient
        public final Builder outputs(Collection<Output> outputs) {
            this.outputs = OutputListCopier.copy(outputs);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder outputs(Output... outputs) {
            outputs(Arrays.asList(outputs));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder outputs(Consumer<Output.Builder>... outputs) {
            outputs(Stream.of(outputs).map(c -> Output.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<DataCatalogOutput.Builder> getDataCatalogOutputs() {
            List<DataCatalogOutput.Builder> result = DataCatalogOutputListCopier.copyToBuilder(this.dataCatalogOutputs);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDataCatalogOutputs(Collection<DataCatalogOutput.BuilderImpl> dataCatalogOutputs) {
            this.dataCatalogOutputs = DataCatalogOutputListCopier.copyFromBuilder(dataCatalogOutputs);
        }

        @Override
        @Transient
        public final Builder dataCatalogOutputs(Collection<DataCatalogOutput> dataCatalogOutputs) {
            this.dataCatalogOutputs = DataCatalogOutputListCopier.copy(dataCatalogOutputs);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder dataCatalogOutputs(DataCatalogOutput... dataCatalogOutputs) {
            dataCatalogOutputs(Arrays.asList(dataCatalogOutputs));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder dataCatalogOutputs(Consumer<DataCatalogOutput.Builder>... dataCatalogOutputs) {
            dataCatalogOutputs(Stream.of(dataCatalogOutputs).map(c -> DataCatalogOutput.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<DatabaseOutput.Builder> getDatabaseOutputs() {
            List<DatabaseOutput.Builder> result = DatabaseOutputListCopier.copyToBuilder(this.databaseOutputs);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDatabaseOutputs(Collection<DatabaseOutput.BuilderImpl> databaseOutputs) {
            this.databaseOutputs = DatabaseOutputListCopier.copyFromBuilder(databaseOutputs);
        }

        @Override
        @Transient
        public final Builder databaseOutputs(Collection<DatabaseOutput> databaseOutputs) {
            this.databaseOutputs = DatabaseOutputListCopier.copy(databaseOutputs);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder databaseOutputs(DatabaseOutput... databaseOutputs) {
            databaseOutputs(Arrays.asList(databaseOutputs));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder databaseOutputs(Consumer<DatabaseOutput.Builder>... databaseOutputs) {
            databaseOutputs(Stream.of(databaseOutputs).map(c -> DatabaseOutput.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final RecipeReference.Builder getRecipeReference() {
            return recipeReference != null ? recipeReference.toBuilder() : null;
        }

        public final void setRecipeReference(RecipeReference.BuilderImpl recipeReference) {
            this.recipeReference = recipeReference != null ? recipeReference.build() : null;
        }

        @Override
        @Transient
        public final Builder recipeReference(RecipeReference recipeReference) {
            this.recipeReference = recipeReference;
            return this;
        }

        public final String getStartedBy() {
            return startedBy;
        }

        public final void setStartedBy(String startedBy) {
            this.startedBy = startedBy;
        }

        @Override
        @Transient
        public final Builder startedBy(String startedBy) {
            this.startedBy = startedBy;
            return this;
        }

        public final Instant getStartedOn() {
            return startedOn;
        }

        public final void setStartedOn(Instant startedOn) {
            this.startedOn = startedOn;
        }

        @Override
        @Transient
        public final Builder startedOn(Instant startedOn) {
            this.startedOn = startedOn;
            return this;
        }

        public final JobSample.Builder getJobSample() {
            return jobSample != null ? jobSample.toBuilder() : null;
        }

        public final void setJobSample(JobSample.BuilderImpl jobSample) {
            this.jobSample = jobSample != null ? jobSample.build() : null;
        }

        @Override
        @Transient
        public final Builder jobSample(JobSample jobSample) {
            this.jobSample = jobSample;
            return this;
        }

        public final List<ValidationConfiguration.Builder> getValidationConfigurations() {
            List<ValidationConfiguration.Builder> result = ValidationConfigurationListCopier
                    .copyToBuilder(this.validationConfigurations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setValidationConfigurations(Collection<ValidationConfiguration.BuilderImpl> validationConfigurations) {
            this.validationConfigurations = ValidationConfigurationListCopier.copyFromBuilder(validationConfigurations);
        }

        @Override
        @Transient
        public final Builder validationConfigurations(Collection<ValidationConfiguration> validationConfigurations) {
            this.validationConfigurations = ValidationConfigurationListCopier.copy(validationConfigurations);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder validationConfigurations(ValidationConfiguration... validationConfigurations) {
            validationConfigurations(Arrays.asList(validationConfigurations));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder validationConfigurations(Consumer<ValidationConfiguration.Builder>... validationConfigurations) {
            validationConfigurations(Stream.of(validationConfigurations)
                    .map(c -> ValidationConfiguration.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public JobRun build() {
            return new JobRun(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
