/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.databrew.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a single data quality requirement that should be validated in the scope of this dataset.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Rule implements SdkPojo, Serializable, ToCopyableBuilder<Rule.Builder, Rule> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Rule::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<Boolean> DISABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Disabled").getter(getter(Rule::disabled)).setter(setter(Builder::disabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Disabled").build()).build();

    private static final SdkField<String> CHECK_EXPRESSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CheckExpression").getter(getter(Rule::checkExpression)).setter(setter(Builder::checkExpression))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CheckExpression").build()).build();

    private static final SdkField<Map<String, String>> SUBSTITUTION_MAP_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("SubstitutionMap")
            .getter(getter(Rule::substitutionMap))
            .setter(setter(Builder::substitutionMap))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubstitutionMap").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Threshold> THRESHOLD_FIELD = SdkField.<Threshold> builder(MarshallingType.SDK_POJO)
            .memberName("Threshold").getter(getter(Rule::threshold)).setter(setter(Builder::threshold))
            .constructor(Threshold::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Threshold").build()).build();

    private static final SdkField<List<ColumnSelector>> COLUMN_SELECTORS_FIELD = SdkField
            .<List<ColumnSelector>> builder(MarshallingType.LIST)
            .memberName("ColumnSelectors")
            .getter(getter(Rule::columnSelectors))
            .setter(setter(Builder::columnSelectors))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ColumnSelectors").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ColumnSelector> builder(MarshallingType.SDK_POJO)
                                            .constructor(ColumnSelector::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, DISABLED_FIELD,
            CHECK_EXPRESSION_FIELD, SUBSTITUTION_MAP_FIELD, THRESHOLD_FIELD, COLUMN_SELECTORS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final Boolean disabled;

    private final String checkExpression;

    private final Map<String, String> substitutionMap;

    private final Threshold threshold;

    private final List<ColumnSelector> columnSelectors;

    private Rule(BuilderImpl builder) {
        this.name = builder.name;
        this.disabled = builder.disabled;
        this.checkExpression = builder.checkExpression;
        this.substitutionMap = builder.substitutionMap;
        this.threshold = builder.threshold;
        this.columnSelectors = builder.columnSelectors;
    }

    /**
     * <p>
     * The name of the rule.
     * </p>
     * 
     * @return The name of the rule.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * A value that specifies whether the rule is disabled. Once a rule is disabled, a profile job will not validate it
     * during a job run. Default value is false.
     * </p>
     * 
     * @return A value that specifies whether the rule is disabled. Once a rule is disabled, a profile job will not
     *         validate it during a job run. Default value is false.
     */
    public final Boolean disabled() {
        return disabled;
    }

    /**
     * <p>
     * The expression which includes column references, condition names followed by variable references, possibly
     * grouped and combined with other conditions. For example,
     * <code>(:col1 starts_with :prefix1 or :col1 starts_with :prefix2) and (:col1 ends_with :suffix1 or :col1 ends_with :suffix2)</code>
     * . Column and value references are substitution variables that should start with the ':' symbol. Depending on the
     * context, substitution variables' values can be either an actual value or a column name. These values are defined
     * in the SubstitutionMap. If a CheckExpression starts with a column reference, then ColumnSelectors in the rule
     * should be null. If ColumnSelectors has been defined, then there should be no columnn reference in the left side
     * of a condition, for example, <code>is_between :val1 and :val2</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/databrew/latest/dg/profile.data-quality-available-checks.html">Available
     * checks</a>
     * </p>
     * 
     * @return The expression which includes column references, condition names followed by variable references,
     *         possibly grouped and combined with other conditions. For example,
     *         <code>(:col1 starts_with :prefix1 or :col1 starts_with :prefix2) and (:col1 ends_with :suffix1 or :col1 ends_with :suffix2)</code>
     *         . Column and value references are substitution variables that should start with the ':' symbol. Depending
     *         on the context, substitution variables' values can be either an actual value or a column name. These
     *         values are defined in the SubstitutionMap. If a CheckExpression starts with a column reference, then
     *         ColumnSelectors in the rule should be null. If ColumnSelectors has been defined, then there should be no
     *         columnn reference in the left side of a condition, for example, <code>is_between :val1 and :val2</code>
     *         .</p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/databrew/latest/dg/profile.data-quality-available-checks.html"
     *         >Available checks</a>
     */
    public final String checkExpression() {
        return checkExpression;
    }

    /**
     * For responses, this returns true if the service returned a value for the SubstitutionMap property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSubstitutionMap() {
        return substitutionMap != null && !(substitutionMap instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The map of substitution variable names to their values used in a check expression. Variable names should start
     * with a ':' (colon). Variable values can either be actual values or column names. To differentiate between the
     * two, column names should be enclosed in backticks, for example, <code>":col1": "`Column A`".</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSubstitutionMap} method.
     * </p>
     * 
     * @return The map of substitution variable names to their values used in a check expression. Variable names should
     *         start with a ':' (colon). Variable values can either be actual values or column names. To differentiate
     *         between the two, column names should be enclosed in backticks, for example,
     *         <code>":col1": "`Column A`".</code>
     */
    public final Map<String, String> substitutionMap() {
        return substitutionMap;
    }

    /**
     * <p>
     * The threshold used with a non-aggregate check expression. Non-aggregate check expressions will be applied to each
     * row in a specific column, and the threshold will be used to determine whether the validation succeeds.
     * </p>
     * 
     * @return The threshold used with a non-aggregate check expression. Non-aggregate check expressions will be applied
     *         to each row in a specific column, and the threshold will be used to determine whether the validation
     *         succeeds.
     */
    public final Threshold threshold() {
        return threshold;
    }

    /**
     * For responses, this returns true if the service returned a value for the ColumnSelectors property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasColumnSelectors() {
        return columnSelectors != null && !(columnSelectors instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List of column selectors. Selectors can be used to select columns using a name or regular expression from the
     * dataset. Rule will be applied to selected columns.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasColumnSelectors} method.
     * </p>
     * 
     * @return List of column selectors. Selectors can be used to select columns using a name or regular expression from
     *         the dataset. Rule will be applied to selected columns.
     */
    public final List<ColumnSelector> columnSelectors() {
        return columnSelectors;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(disabled());
        hashCode = 31 * hashCode + Objects.hashCode(checkExpression());
        hashCode = 31 * hashCode + Objects.hashCode(hasSubstitutionMap() ? substitutionMap() : null);
        hashCode = 31 * hashCode + Objects.hashCode(threshold());
        hashCode = 31 * hashCode + Objects.hashCode(hasColumnSelectors() ? columnSelectors() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Rule)) {
            return false;
        }
        Rule other = (Rule) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(disabled(), other.disabled())
                && Objects.equals(checkExpression(), other.checkExpression())
                && hasSubstitutionMap() == other.hasSubstitutionMap()
                && Objects.equals(substitutionMap(), other.substitutionMap()) && Objects.equals(threshold(), other.threshold())
                && hasColumnSelectors() == other.hasColumnSelectors()
                && Objects.equals(columnSelectors(), other.columnSelectors());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Rule").add("Name", name()).add("Disabled", disabled()).add("CheckExpression", checkExpression())
                .add("SubstitutionMap", hasSubstitutionMap() ? substitutionMap() : null).add("Threshold", threshold())
                .add("ColumnSelectors", hasColumnSelectors() ? columnSelectors() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Disabled":
            return Optional.ofNullable(clazz.cast(disabled()));
        case "CheckExpression":
            return Optional.ofNullable(clazz.cast(checkExpression()));
        case "SubstitutionMap":
            return Optional.ofNullable(clazz.cast(substitutionMap()));
        case "Threshold":
            return Optional.ofNullable(clazz.cast(threshold()));
        case "ColumnSelectors":
            return Optional.ofNullable(clazz.cast(columnSelectors()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Rule, T> g) {
        return obj -> g.apply((Rule) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Rule> {
        /**
         * <p>
         * The name of the rule.
         * </p>
         * 
         * @param name
         *        The name of the rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A value that specifies whether the rule is disabled. Once a rule is disabled, a profile job will not validate
         * it during a job run. Default value is false.
         * </p>
         * 
         * @param disabled
         *        A value that specifies whether the rule is disabled. Once a rule is disabled, a profile job will not
         *        validate it during a job run. Default value is false.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder disabled(Boolean disabled);

        /**
         * <p>
         * The expression which includes column references, condition names followed by variable references, possibly
         * grouped and combined with other conditions. For example,
         * <code>(:col1 starts_with :prefix1 or :col1 starts_with :prefix2) and (:col1 ends_with :suffix1 or :col1 ends_with :suffix2)</code>
         * . Column and value references are substitution variables that should start with the ':' symbol. Depending on
         * the context, substitution variables' values can be either an actual value or a column name. These values are
         * defined in the SubstitutionMap. If a CheckExpression starts with a column reference, then ColumnSelectors in
         * the rule should be null. If ColumnSelectors has been defined, then there should be no columnn reference in
         * the left side of a condition, for example, <code>is_between :val1 and :val2</code>.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/databrew/latest/dg/profile.data-quality-available-checks.html">Available
         * checks</a>
         * </p>
         * 
         * @param checkExpression
         *        The expression which includes column references, condition names followed by variable references,
         *        possibly grouped and combined with other conditions. For example,
         *        <code>(:col1 starts_with :prefix1 or :col1 starts_with :prefix2) and (:col1 ends_with :suffix1 or :col1 ends_with :suffix2)</code>
         *        . Column and value references are substitution variables that should start with the ':' symbol.
         *        Depending on the context, substitution variables' values can be either an actual value or a column
         *        name. These values are defined in the SubstitutionMap. If a CheckExpression starts with a column
         *        reference, then ColumnSelectors in the rule should be null. If ColumnSelectors has been defined, then
         *        there should be no columnn reference in the left side of a condition, for example,
         *        <code>is_between :val1 and :val2</code>.</p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/databrew/latest/dg/profile.data-quality-available-checks.html"
         *        >Available checks</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder checkExpression(String checkExpression);

        /**
         * <p>
         * The map of substitution variable names to their values used in a check expression. Variable names should
         * start with a ':' (colon). Variable values can either be actual values or column names. To differentiate
         * between the two, column names should be enclosed in backticks, for example,
         * <code>":col1": "`Column A`".</code>
         * </p>
         * 
         * @param substitutionMap
         *        The map of substitution variable names to their values used in a check expression. Variable names
         *        should start with a ':' (colon). Variable values can either be actual values or column names. To
         *        differentiate between the two, column names should be enclosed in backticks, for example,
         *        <code>":col1": "`Column A`".</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder substitutionMap(Map<String, String> substitutionMap);

        /**
         * <p>
         * The threshold used with a non-aggregate check expression. Non-aggregate check expressions will be applied to
         * each row in a specific column, and the threshold will be used to determine whether the validation succeeds.
         * </p>
         * 
         * @param threshold
         *        The threshold used with a non-aggregate check expression. Non-aggregate check expressions will be
         *        applied to each row in a specific column, and the threshold will be used to determine whether the
         *        validation succeeds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder threshold(Threshold threshold);

        /**
         * <p>
         * The threshold used with a non-aggregate check expression. Non-aggregate check expressions will be applied to
         * each row in a specific column, and the threshold will be used to determine whether the validation succeeds.
         * </p>
         * This is a convenience that creates an instance of the {@link Threshold.Builder} avoiding the need to create
         * one manually via {@link Threshold#builder()}.
         *
         * When the {@link Consumer} completes, {@link Threshold.Builder#build()} is called immediately and its result
         * is passed to {@link #threshold(Threshold)}.
         * 
         * @param threshold
         *        a consumer that will call methods on {@link Threshold.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #threshold(Threshold)
         */
        default Builder threshold(Consumer<Threshold.Builder> threshold) {
            return threshold(Threshold.builder().applyMutation(threshold).build());
        }

        /**
         * <p>
         * List of column selectors. Selectors can be used to select columns using a name or regular expression from the
         * dataset. Rule will be applied to selected columns.
         * </p>
         * 
         * @param columnSelectors
         *        List of column selectors. Selectors can be used to select columns using a name or regular expression
         *        from the dataset. Rule will be applied to selected columns.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder columnSelectors(Collection<ColumnSelector> columnSelectors);

        /**
         * <p>
         * List of column selectors. Selectors can be used to select columns using a name or regular expression from the
         * dataset. Rule will be applied to selected columns.
         * </p>
         * 
         * @param columnSelectors
         *        List of column selectors. Selectors can be used to select columns using a name or regular expression
         *        from the dataset. Rule will be applied to selected columns.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder columnSelectors(ColumnSelector... columnSelectors);

        /**
         * <p>
         * List of column selectors. Selectors can be used to select columns using a name or regular expression from the
         * dataset. Rule will be applied to selected columns.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ColumnSelector>.Builder} avoiding the need
         * to create one manually via {@link List<ColumnSelector>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ColumnSelector>.Builder#build()} is called immediately and
         * its result is passed to {@link #columnSelectors(List<ColumnSelector>)}.
         * 
         * @param columnSelectors
         *        a consumer that will call methods on {@link List<ColumnSelector>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #columnSelectors(List<ColumnSelector>)
         */
        Builder columnSelectors(Consumer<ColumnSelector.Builder>... columnSelectors);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private Boolean disabled;

        private String checkExpression;

        private Map<String, String> substitutionMap = DefaultSdkAutoConstructMap.getInstance();

        private Threshold threshold;

        private List<ColumnSelector> columnSelectors = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Rule model) {
            name(model.name);
            disabled(model.disabled);
            checkExpression(model.checkExpression);
            substitutionMap(model.substitutionMap);
            threshold(model.threshold);
            columnSelectors(model.columnSelectors);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Boolean getDisabled() {
            return disabled;
        }

        public final void setDisabled(Boolean disabled) {
            this.disabled = disabled;
        }

        @Override
        @Transient
        public final Builder disabled(Boolean disabled) {
            this.disabled = disabled;
            return this;
        }

        public final String getCheckExpression() {
            return checkExpression;
        }

        public final void setCheckExpression(String checkExpression) {
            this.checkExpression = checkExpression;
        }

        @Override
        @Transient
        public final Builder checkExpression(String checkExpression) {
            this.checkExpression = checkExpression;
            return this;
        }

        public final Map<String, String> getSubstitutionMap() {
            if (substitutionMap instanceof SdkAutoConstructMap) {
                return null;
            }
            return substitutionMap;
        }

        public final void setSubstitutionMap(Map<String, String> substitutionMap) {
            this.substitutionMap = ValuesMapCopier.copy(substitutionMap);
        }

        @Override
        @Transient
        public final Builder substitutionMap(Map<String, String> substitutionMap) {
            this.substitutionMap = ValuesMapCopier.copy(substitutionMap);
            return this;
        }

        public final Threshold.Builder getThreshold() {
            return threshold != null ? threshold.toBuilder() : null;
        }

        public final void setThreshold(Threshold.BuilderImpl threshold) {
            this.threshold = threshold != null ? threshold.build() : null;
        }

        @Override
        @Transient
        public final Builder threshold(Threshold threshold) {
            this.threshold = threshold;
            return this;
        }

        public final List<ColumnSelector.Builder> getColumnSelectors() {
            List<ColumnSelector.Builder> result = ColumnSelectorListCopier.copyToBuilder(this.columnSelectors);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setColumnSelectors(Collection<ColumnSelector.BuilderImpl> columnSelectors) {
            this.columnSelectors = ColumnSelectorListCopier.copyFromBuilder(columnSelectors);
        }

        @Override
        @Transient
        public final Builder columnSelectors(Collection<ColumnSelector> columnSelectors) {
            this.columnSelectors = ColumnSelectorListCopier.copy(columnSelectors);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder columnSelectors(ColumnSelector... columnSelectors) {
            columnSelectors(Arrays.asList(columnSelectors));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder columnSelectors(Consumer<ColumnSelector.Builder>... columnSelectors) {
            columnSelectors(Stream.of(columnSelectors).map(c -> ColumnSelector.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public Rule build() {
            return new Rule(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
